#GPL
#GPL  libwhisker copyright 2000,2001,2002 by rfp.labs
#GPL
#GPL  This program is free software; you can redistribute it and/or
#GPL  modify it under the terms of the GNU General Public License
#GPL  as published by the Free Software Foundation; either version 2
#GPL  of the License, or (at your option) any later version.
#GPL
#GPL  This program is distributed in the hope that it will be useful,
#GPL  but WITHOUT ANY WARRANTY; without even the implied warranty of
#GPL  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GPL  GNU General Public License for more details.
#GPL

=pod

=head1 ++ Sub package: utils

The utils subpackage contains various utility functions which serve
different purposes.

=cut

########################################################################

=pod

=head1 - Function: LW::utils_recperm
  
Params: $uri, $depth, \@dir_parts, \@valid, \&func, \%track, \%arrays, \&cfunc
Return: nothing

This is a special function which is used to recursively-permutate through
a given directory listing.  This is really only used by whisker, in order
to traverse down directories, testing them as it goes.  See whisker 2.0 for
exact usage examples.

=cut

# '/', 0, \@dir.split, \@valid, \&func, \%track, \%arrays, \&cfunc
sub utils_recperm {
 my ($p, $pp, $pn, $r, $fr, $dr, $ar, $cr)=(shift,shift,@_);
 $p=~s#/+#/#g; if($pp >= @$pn) { push @$r, $p if &$cr($$dr{$p});
 } else { my $c=$$pn[$pp];
  if($c!~/^\@/){ utils_recperm($p.$c.'/',$pp+1,@_) if(&$fr($p.$c.'/'));
  } else {	$c=~tr/\@//d; if(defined $$ar{$c}){
		foreach $d (@{$$ar{$c}}){
			if(&$fr($p.$d.'/')){
                  utils_recperm($p.$d.'/',$pp+1,@_);}}}}}}


#################################################################

=pod

=head1 - Function: LW::utils_array_shuffle
  
Params: @array
Return: nothing

This function will randomize the order of the elements in the given array.

=cut

sub utils_array_shuffle { # fisher yates shuffle....w00p!
        my $array=shift; my $i;
        for ($i = @$array; --$i;){
                my $j = int rand ($i+1);
                next if $i==$j;
                @$array[$i,$j]=@$array[$j,$i];
}} # end array_shuffle, from Perl Cookbook (rock!)


#################################################################

=pod

=head1 - Function: LW::utils_randstr
  
Params: [ $size, $chars ]
Return: $random_string

This function generates a random string between 10 and 20 characters
long, or of $size if specified.  If $chars is specified, then the
random function picks characters from the supplied string.  For example,
to have a random string of 10 characters, composed of only the characters
'abcdef', then you would run:

LW::utils_randstr(10,'abcdef');

The default character string is alphanumeric.

=cut

sub utils_randstr {
        my $str;
        my $drift=shift||((rand() * 10) % 10)+10; 

	# 'a'..'z' doesn't seem to work on string assignment :(
	my $CHARS = shift || 'abcdefghijklmnopqrstuvwxyz' .
			'ABCDEFGHIJKLMNOPQRSTUVWXYZ' .
			'0123456789';

	my $L = length($CHARS);
        for(1..$drift){
	        $str .= substr($CHARS,((rand() * $L) % $L),1);
	}
        return $str;}

#################################################################

=pod

=head1 - Function: LW::utils_get_dir
  
Params: $uri
Return: $uri_directory

Will take a URI and return the directory base of it, i.e. /rfp/page.php 
will return /rfp/.

=cut

sub utils_get_dir {
        my ($w,$URL)=(0,shift);

	return undef if(!defined $URL);

	$URL=substr($URL,0,$w) if( ($w=index($URL,'#')) >= 0);
	$URL=substr($URL,0,$w) if( ($w=index($URL,'?')) >= 0);

	if( ($w=rindex($URL,'/')) >= 0){
		$URL = substr($URL,0,$w+1);
	}
        return $URL; 
}


#################################################################

=pod

=head1 - Function: LW::utils_port_open
  
Params: $host, $port
Return: $result

Quick function to attempt to make a connection to the given host and
port.  If a connection was successfully made, function will return true
(1).  Otherwise it returns false (0).

Note: this uses standard TCP connections, thus is not recommended for use
in port-scanning type applications.  Extremely slow.

=cut

sub utils_port_open {  # this should be platform-safe
        my ($target,$port)=@_;

	return 0 if(!defined $target || !defined $port);

        if(!(socket(S,PF_INET,SOCK_STREAM,0))){ return 0;}
        if(connect(S,sockaddr_in($port,inet_aton($target)))){
                close(S); return 1;
        } else { return 0;}}


#################################################################

=pod

=head1 - Function: LW::utils_split_uri
  
Params: $uri_string [, \%hin_request]
Return: @uri_parts

Return an array of the following values, in order:  uri, protocol, host,
port, params, frag, user, password.  Values not defined are given an undef
value.  If a %hin_request hash is passed in, then utils_split_uri() will
also set the appropriate values in the hash.  While it attempts to do
RFC-compliant URI parsing, it still caters to HTTP[S] only.

Note:  utils_split_uri() will only set the %hin_request if the protocol
is HTTP or HTTPS!

=cut

sub utils_split_uri {
	my ($uri,$w)=(shift,'',0);
	my ($hr)=shift;
	my @res=(undef,'http',undef,0,undef,undef,undef,undef);

	return undef if(!defined $uri);

	# remove fragments
	($uri,$res[5])=split('#',$uri,2) if(index($uri,'#',0) >=0);

	# get scheme and net_loc
	my $net_loc = undef;
	if($uri=~s/^([-+.a-z0-9A-Z]+)://){
		$res[1]=lc($1);
		if(substr($uri,0,2) eq '//'){
			$w=index($uri,'/',2);
			if($w >= 0){
				$net_loc=substr($uri,2,$w-2);
				$uri=substr($uri,$w,length($uri)-$w);
			} else {
				($net_loc=$uri)=~tr#/##d;
				$uri='/';
			}
		}
	}


	# parse net_loc info
	if(defined $net_loc){
		if(index($net_loc,'@',0) >=0){
			($res[6],$net_loc)=split('@',$net_loc,2);
			if(index($res[6],':',0) >=0){
				($res[6],$res[7])=split(':',$res[6],2);
			}
		}
		$res[3]=$1 if($net_loc=~s/:([0-9]+)$//);
		$res[2]=$net_loc;
	}

	# remove query info
	($uri,$res[4])=split('\?',$uri,2) if(index($uri,'?',0) >=0);

	# whatever is left over is the uri
	$res[0]=$uri;

	if($res[3]==0 && defined $res[1]){
		$res[3]=80 if($res[1] eq 'http');
		$res[3]=443 if($res[1] eq 'https');
	}

	return @res if($res[3]==0);

	# setup whisker hash
	if(defined $hr && ref($hr)){
		# these must always exist
		$$hr{whisker}->{uri}=$res[0] 		if(defined $res[0]);
		$$hr{whisker}->{ssl}=1			if($res[1] eq 'https');
		$$hr{whisker}->{host}=$res[2]		if(defined $res[2]);
		$$hr{whisker}->{port}=$res[3]		;

		# set/delete parameter attributes
		if(defined $res[4]){
			$$hr{whisker}->{uri_param}=$res[4];
		} else { delete $$hr{whisker}->{uri_param}; }
		if(defined $res[6]){
			$$hr{whisker}->{uri_user}=$res[6];
		} else { delete $$hr{whisker}->{uri_user}; }
		if(defined $res[7]){
			$$hr{whisker}->{uri_password}=$res[7];
		} else { delete $$hr{whisker}->{uri_password}; }
	}
		
	return @res;
}

#################################################################
=pod

=head1 - Function: LW::utils_lowercase_headers
  
Params: \%hash
Return: nothing

Will lowercase all the header names (but not values) of the given hash.

=cut

sub utils_lowercase_headers {
	goto &utils_lowercase_hashkeys;
}

#################################################################
=pod

=head1 - Function: LW::utils_lowercase_hashkeys
  
Params: \%hash
Return: nothing

Will lowercase all the header names (but not values) of the given hash.

=cut

sub utils_lowercase_hashkeys {
	my $href=shift;

	return if(!(defined $href && ref($href)));

	while( my ($key,$val)=each %$href ){
		delete $$href{$key};
		$$href{lc($key)}=$val;
	}
}

#################################################################
=pod

=head1 - Function: LW::utils_find_lowercase_key
  
Params: \%hash, $key
Return: $value, undef on error or not exist

Searches the given hash for the $key (regardless of case), and
returns the value.

=cut

sub utils_find_lowercase_key {
	my ($href,$key)=(shift,lc(shift));

	return undef if(!(defined $href && ref($href)));
	return undef if(!defined $key);	

	while( my ($k,$v)=each %$href ){
		return $v if(lc($k) eq $key);
	}
	return undef;
}

#################################################################

=pod

=head1 - Function: LW::utils_join_uri
  
Params: @vals
Return: $url

Takes the @vals array output from utils_split_uri, and returns a single 
scalar/string with them joined again, in the form of:
protocol://user:password@host:port/uri?params#frag

=cut

sub utils_join_uri {
	my @V=@_;
	my $URL;

	$URL.=$V[1].':' if defined $V[1];
	if(defined $V[2]){
		$URL.='//';
		if(defined $V[6]){
			$URL.=$V[6];
			$URL.=':'.$V[7] if defined $V[7];
			$URL.='@';
		}
		$URL.=$V[2];
	}
	if($V[3]>0){
		my $no = 0;
		$no++ if($V[3]==80 && defined $V[1] && $V[1] eq 'http');
		$no++ if($V[3]==443 && defined $V[1] && $V[1] eq 'https');
		$URL .= ':'.$V[3] if(!$no);
	}
	$URL.=$V[0];
	$URL .= '?'.$V[4] if defined $V[4];
	$URL .= '#'.$V[5] if defined $V[5];
	return $URL;
}

#################################################################

=pod

=head1 - Function: LW::utils_getline
  
Params: \$data [, $resetpos ]
Return: $line (undef if no more data)

Fetches the next \n terminated line from the given data.  Use
the optional $resetpos to reset the internal position pointer.
Does *NOT* return trialing \n.

=cut

{ $POS=0;
sub utils_getline {
	my ($dr, $rp)=@_;

	return undef if(!(defined $dr && ref($dr)));
	$POS=$rp if(defined $rp);

	my $where=index($$dr,"\n",$POS);
	return undef if($where==-1);

	my $str=substr($$dr,$POS,$where-$POS);
	$POS=$where+1;

	return $str;
}}

#################################################################

=pod

=head1 - Function: LW::utils_getline_crlf
  
Params: \$data [, $resetpos ]
Return: $line (undef if no more data)

Fetches the next \r\n terminated line from the given data.  Use
the optional $resetpos to reset the internal position pointer.
Does *NOT* return trialing \r\n.

=cut

{ $POS=0;
sub utils_getline_crlf {
	my ($dr, $rp)=@_;

	return undef if(!(defined $dr && ref($dr)));
	$POS=$rp if(defined $rp);

	my $tpos=$POS;
	while(1){
		my $where=index($$dr,"\n",$tpos);
		return undef if($where==-1);

		if(substr($$dr,$where-1,1) eq "\r"){
			my $str=substr($$dr,$POS,$where-$POS-1);
			$POS=$where+1;
			return $str;
		} else {
			$tpos=$where+1;
		}
	}
}}

#################################################################

=pod

=head1 - Function: LW::utils_absolute_uri
  
Params: $uri, $base_uri [, $normalize_flag ]
Return: $absolute_$url

Double checks that the given $uri is in absolute form (that is,
"http://host/file"), and if not (it's in the form "/file"), then
it will append the given $base_uri to make it absolute.  This
provides a compatibility similar to that found in the URI
subpackage.

If $normalize_flag is set to 1, then the output will be passed
through utils_normalize_uri before being returned.

=cut

sub utils_absolute_uri {
        my ($uri, $buri, $norm)=@_;
        return undef if(!defined $uri || !defined $buri);
	return $uri if($uri=~m#^[a-zA-Z]+://#);

	if(substr($uri,0,1) eq '/'){
		if($buri=~m#^[a-zA-Z]+://#){
			my @p=utils_split_uri($buri);
			$buri="$p[1]://$p[2]";
			$buri.=":$p[3]" if($p[3]!=80);
			$buri.='/';
		} else { # ah suck, base URI isn't absolute...
			return $uri;
		}
	} else {
		$buri=~s/[?#].*$//; # remove params and frags
		$buri.='/' if($buri=~m#^[a-z]+://[^/]+$#i);
		$buri=~s#/[^/]*$#/#;
	}
	return utils_normalize_uri("$buri$uri") 
		if(defined $norm && $norm > 0);
        return $buri.$uri;
}

#################################################################

=pod

=head1 - Function: LW::utils_normalize_uri
  
Params: $uri [, $fix_windows_slashes ]
Return: $normalized_uri

Takes the given $uri and does any /./ and /../ dereferencing in
order to come up with the correct absolute URL.  If the $fix_
windows_slashes parameter is set to 1, all \ (back slashes) will
be converted to / (forward slashes).

=cut

sub utils_normalize_uri {
	my ($host,$uri, $win)=('',@_);

	$uri=~tr#\\#/# if(defined $win && $win>0);

	if($uri=~s#^([-+.a-z0-9A-Z]+:)##){
		return undef if($1 ne 'http:' && $1 ne 'https:');
		$host=$1;
		return undef unless($uri=~s#^(//[^/]+)##);
		$host.=$1; }
	return "$host/" if($uri eq '' || $uri eq '/');

	# fast path check
	return "$host$uri" if(index($uri,'/.')<0);

	# parse order/steps as defined in RFC 1808
	1 while($uri=~s#/\./#/# || $uri=~s#//#/#);
	$uri=~s#/\.$#/#;
	1 while($uri=~s#[^/]+/\.\./##);
	1 while($uri=~s#^/\.\./#/#);
	$uri=~s#[^/]*/\.\.$##;
	$uri||='/';
	return $host.$uri;
}

#################################################################

=pod

=head1 - Function: LW::utils_save_page
  
Params: $file, \%response
Return: 0 on success, 1 on error

Saves the data portion of the given whisker %response hash to the
indicated file.  Can technically save the data portion of a
%request hash too.  A file is not written if there is no data.

Note: LW does not do any special file checking; files are opened
in overwrite mode.

=cut

sub utils_save_page {
	my ($file, $hr)=@_;
	return 1 if(!ref($hr) || ref($file));
	return 0 if(!defined $$hr{'whisker'} || 
		!defined $$hr{'whisker'}->{'data'});
	open(OUT,">$file") || return 1;
	print OUT $$hr{'whisker'}->{'data'};
	close(OUT);
	return 0;
}

#################################################################

=pod

=head1 - Function: LW::utils_getopts
  
Params: $opt_str, \%opt_results
Return: 0 on success, 1 on error

This function is a general implementation of GetOpts::Std.  It will
parse @ARGV, looking for the options specified in $opt_str, and will
put the results in %opt_results.  Behavior/parameter values are
similar to GetOpts::Std's getopts().

Note: this function does *not* support long options (--option),
option grouping (-opq), or options with immediate values (-ovalue).
If an option is indicated as having a value, it will take the next
argument regardless.

=cut

sub utils_getopts {
        my ($str,$ref)=@_;
        my (%O,$l);
        my @left;

        return 1 if($str=~tr/-:a-zA-Z0-9//c);

        while($str=~m/([a-z0-9]:{0,1})/ig){
                $l=$1;
                if($l=~tr/://d){        $O{$l}=1;
                } else {                $O{$l}=0; }
        }

        while($l=shift(@ARGV)){
                push(@left,$l)&&next if(substr($l,0,1) ne '-');
                push(@left,$l)&&next if($l eq '-');
                substr($l,0,1)='';
                if(length($l)!=1){
                        %$ref=();
                        return 1; }
                if($O{$l}==1){
                        my $x=shift(@ARGV);
                        $$ref{$l}=$x;
                } else { $$ref{$l}=1; }
        }

        @ARGV=@left;
        return 0;
}

#################################################################

=pod

=head1 - Function: LW::utils_unidecode_uri
  
Params: $unicode_string
Return: $decoded_string

This function attempts to decode a unicode (UTF-8) string by
converting it into a single-byte-character string.  Overlong 
characters are converted to their standard characters in place; 
non-overlong (aka multi-byte) characters are substituted with the 
0xff; invalid encoding characters are left as-is.

Note: this function is useful for dealing with the various unicode
exploits/vulnerabilities found in web servers; it is *not* good for
doing actual UTF-8 parsing, since characters over a single byte are
basically dropped/replaced with a placeholder.

=cut

sub utils_unidecode_uri {
        my $str = $_[0];
        return $str if($str!~tr/!-~//c); # fastpath
        my ($lead,$count,$idx);
        my $out='';
        my $len = length($str);
        my ($ptr,$no,$nu)=(0,0,0);

        while($ptr < $len){
                my $c=substr($str,$ptr,1);
                if( ord($c) >= 0xc0 && ord($c) <= 0xfd){
                        $count=0;
                        $c=ord($c)<<1;
                        while( ($c & 0x80) == 0x80){
                                $c<<=1;
                                last if($count++ ==4);
                        }
                        $c = ($c & 0xff);
                        for( $idx=1; $idx<$count; $idx++){
                                my $o=ord(substr($str,$ptr+$idx,1));
                                $no=1 if($o != 0x80);
                                $nu=1 if($o <0x80 || $o >0xbf);
                        }
                        my $o=ord(substr($str,$ptr+$idx,1));
                        $nu=1 if( $o < 0x80 || $o > 0xbf);
                        if($nu){
                                $out.=substr($str,$ptr++,1);
                        } else {
                                if($no){
                                        $out.="\xff"; # generic replacement char
                                } else {
                                        my $prior=ord(substr($str,$ptr+$count-1,1))<<6;
                                        $out.= pack("C", (ord(substr($str,$ptr+$count,1) )&0x7f)+$prior);
                                }
                                $ptr += $count+1;
                        }
                        $no=$nu=0;
                } else {
                        $out.=$c;
                        $ptr++;
                }
        }
        return $out;
}

#################################################################

=pod

=head1 - Function: LW::utils_text_wrapper
  
Params: $long_text_string [, $crlf, $width ]
Return: $formatted_test_string

This is a simple function used to format a long line of text for
display on a typical limited-character screen, such as a unix
shell console.

$crlf defaults to "\n", and $width defaults to 76.

=cut

sub utils_text_wrapper {
        my ($out,$w,$str,$crlf,$width)=('',0,@_);
	$crlf||="\n";	$width||=76;
        $str.=$crlf if($str!~/$crlf$/);
        return $str if(length($str)<=$width);
        while(length($str)>$width){
                my $w1=rindex($str,' ',$width);
                my $w2=rindex($str,"\t",$width);
                if($w1>$w2){ $w=$w1; } else { $w=$w2; }
                if($w==-1){	$w=$width;
	        } else {	substr($str,$w,1)=''; }
                $out.=substr($str,0,$w,'');
                $out.=$crlf;
        }
        return $out.$str;
}

#################################################################

