/*
   Copyright (C) 2006-2007 KovoKs <info@kovoks.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <sys/time.h>

#include <qpainter.h>
#include <qdragobject.h>

#include <klistview.h>
#include <kdebug.h>

#include "global.h"
#include "messagedata.h"
#include "headerlistview.h"

namespace Mailody {

HeaderListView::HeaderListView( QWidget* parent)
    : KListView( parent ),
    m_isolationActive(false)
{
    setShowSortIndicator(true);
    addColumn( i18n("Subject"), 150 );
    addColumn( i18n("Sender"), 150 );
    addColumn( i18n("Date"), 150 );
    addColumn( i18n("Size"), 150 );
    setColumnWidthMode(0, QListView::Manual);
    setColumnWidthMode(1, QListView::Manual);
    setColumnWidthMode(2, QListView::Manual);
    setColumnWidthMode(3, QListView::Manual);
    setSorting(2);
    setColumnAlignment(3, Qt::AlignRight);
    setAllColumnsShowFocus(true);
    setSelectionMode(QListView::Extended);
    setDragEnabled(true);
    setRootIsDecorated(true);
}

QDragObject *HeaderListView::dragObject()
{
    // kdDebug() << "DragObject called" << endl;

    QString items;
    QListViewItemIterator it( this );
    while ( it.current() )
    {
        if ( it.current()->isSelected() )
            dragInfo((HeaderListViewItem*)it.current(), items, true);
        ++it;
    }
    return new QTextDrag( items, viewport() );
}

void HeaderListView::dragInfo(HeaderListViewItem* i, QString& items,
                              bool visible)
{
    // the visible parameter is used to indicate if there is a need to check
    // if the item is visible (as in closed/open). We only care to select items
    // which are not visible at the top level. The rest should be selected
    // anyhow.

    items.append( QString::number(i->msg()->uid() )+'\n');
    items.append( i->msg()->mb()+'\n' );

    //if the item is closed and has childeren, move those too....
    if (!i->isOpen() || !visible)
    {
        QListViewItem * myChild = i->firstChild();
        while ( myChild )
        {
            dragInfo((HeaderListViewItem*)myChild, items, false);
            myChild = myChild->nextSibling();
        }
    }
}

void HeaderListView::keyPressEvent( QKeyEvent* e )
{
    if (e->key() == Qt::Key_Left)
    {
        e->accept();
        QListViewItem* lvi = currentItem()->itemAbove();

        //no item, return;
        if (!lvi)
            return;

        setSelected(currentItem(), false);

        setCurrentItem(lvi);
        setSelected(lvi, true);
        setSelectionAnchor(lvi);
        ensureItemVisible(lvi);

        // try to let other items be visible as well...
        center(0,lvi->itemPos());
    }
    else if (e->key() == Qt::Key_Right)
    {
        e->accept();
        QListViewItem* lvi = currentItem()->itemBelow();

        //no item, return;
        if (!lvi)
            return;

        setSelected(currentItem(), false);

        setCurrentItem(lvi);
        setSelected(lvi, true);
        setSelectionAnchor(lvi);
        ensureItemVisible(lvi);

        // try to let other items be visible as well...
        center(0,lvi->itemPos());
    }
    else if (e->key() == Qt::Key_Down)
    {
        e->accept();
        emit scrollDown();
    }
    else if (e->key() == Qt::Key_Up)
    {
        e->accept();
        emit scrollUp();
    }
    else if (e->key() == Qt::Key_PageDown)
    {
        e->accept();
        emit scrollPageDown();
    }
    else if (e->key() == Qt::Key_PageUp)
    {
        e->accept();
        emit scrollPageUp();
    }
    else
        KListView::keyPressEvent(e);
}

void HeaderListView::setHideDeleted(bool i)
{
    struct timeval tv1, tv2;
    gettimeofday(&tv1, 0);

    m_hideDeleted = i;

    // first we going to assemble a list of what should be visible or not
    QValueList<HeaderListViewItem*> visibleMap;

    QListViewItemIterator it( this );
    while ( it.current() )
    {
        HeaderListViewItem* item = (HeaderListViewItem*)it.current();
        if ( m_hideDeleted )
        {
            if (item->msg() &&
                item->msg()->isDeleted() && !item->msg()->isNew())
                item->setVisible(false);
            else
                if (!m_isolationActive)
                    visibleMap.append(item);
        }
        else
            if (!m_isolationActive)
                visibleMap.append(item);
        ++it;
    }

    gettimeofday(&tv2, 0);
    // Global::timing("Half way hiding", tv1, tv2);

    // now we know that we can set items visibility, but we need to adjust
    // the parents where a child needs to be visible.
    // kdDebug() << "Should be visibile: " << visibleMap.count() << endl;
    QValueList<HeaderListViewItem*>::ConstIterator ita;
    for (ita = visibleMap.begin(); ita != visibleMap.end(); ++ita)
    {
        // if we need to show this item, we better show all parents as well.
        HeaderListViewItem* item = *ita;
        item->setVisible(true);
        HeaderListViewItem* parent = (HeaderListViewItem*)item->parent();
        while (parent)
        {
            parent->setVisible(true);
            parent = (HeaderListViewItem*)parent->parent();
        }
    }

    gettimeofday(&tv2, 0);
    //Global::timing("Hiding deleted", tv1, tv2);
}

void HeaderListView::isolateItem( HeaderListViewItem* showItem )
{
    m_isolationActive = true;
    m_hiddenItems.clear();

    //Hide it all.
    QListViewItemIterator it( this, QListViewItemIterator::Visible );
    while ( it.current() )
    {
        HeaderListViewItem* item = (HeaderListViewItem*)it.current();
        if (item != showItem)
        {
            item->setVisible(false);
            m_hiddenItems.append(item);
        }
        ++it;
    }

    // Show the parents, its better and needed to show showItem if it is a child
    HeaderListViewItem* parent = (HeaderListViewItem*)showItem->parent();
    while (parent)
    {
        parent->setVisible(true);
        parent = (HeaderListViewItem*)parent->parent();
    }
}

void HeaderListView::clearIsolation()
{
    if (!m_isolationActive)
        return;

    // save this item to center it later.
    QListViewItem* lvi = currentItem();

    QValueList<HeaderListViewItem*>::Iterator it = m_hiddenItems.begin();
    while (it != m_hiddenItems.end())
    {
        (*it)->setVisible(true);
        ++it;
    }
    m_isolationActive = false;

    // center it
    if (lvi)
        ensureItemVisible(lvi);
}


void HeaderListView::allHeadersVisible()
{
    QListViewItemIterator it( this );
    while ( it.current() )
    {
        ((HeaderListViewItem*)it.current())->parseHeaders();
        ++it;
    }
}

HeaderListViewItem* HeaderListView::nextUnread()
{
    // iterate from the selected item if its there or from the started.
    QListViewItem* start = selectedItems( false ).first();

    if (!start)
    {
        QListViewItemIterator it( this );
        while ( it.current() )
        {
            HeaderListViewItem *item =
                    static_cast<HeaderListViewItem*>(it.current());
            if (item->msg()->isNew())
                return item;

            ++it;
        }
    }
    else
    {
        // start from the selected one.
        QListViewItemIterator it( start );
        while ( it.current() )
        {
            HeaderListViewItem *item =
                    static_cast<HeaderListViewItem*>(it.current());
            if (item->msg()->isNew())
                return item;

            ++it;
        }

        // not found. start at the beginning...
        it = this;
        while ( it.current() )
        {
            HeaderListViewItem *item =
                    static_cast<HeaderListViewItem*>(it.current());
            if (item == start)
                break;

            if (item->msg()->isNew())
                return item;

            ++it;
        }
    }
    return 0;
}

//-------------------------------------------------------------------

HeaderListViewItem::HeaderListViewItem( KListView* parent, MessageData* msg)
    : MailodyBaseListViewItem( parent ),
    m_dataNotSet(true)
{
    m_msg = msg;
    setDragEnabled(true);
}

HeaderListViewItem::HeaderListViewItem( HeaderListViewItem* parent,
                                        MessageData* msg)
    : MailodyBaseListViewItem( parent ),
    m_dataNotSet(true)
{
    m_msg = msg;
    setDragEnabled(true);
}

HeaderListViewItem::~HeaderListViewItem()
{
    if (m_msg)
        m_msg->done();
}

void HeaderListViewItem::parseHeaders()
{
    if (m_dataNotSet)
    {
        setText(0, m_msg->subject() );
        setText(1, m_msg->sender() );
        setText(2, m_msg->vDate() );
        setText(3, m_msg->vSize() );
        m_dataNotSet = false;
    }
}

void HeaderListViewItem::paintCell( QPainter * p, const QColorGroup & cg,
                            int column, int width, int align )
{
    if (m_dataNotSet && m_msg)
        parseHeaders();

    QColorGroup _cg( cg );
    QColor c = _cg.text();

    if (m_msg && m_msg->isDeleted())
        _cg.setColor( QColorGroup::Text, Qt::lightGray );

    if (m_msg && m_msg->hasUserTag(1))
    {
        _cg.setColor( QColorGroup::Text, Qt::red  );
        _cg.setColor( QColorGroup::Highlight, Qt::red );
        _cg.setColor( QColorGroup::HighlightedText, Qt::black );
    }
    else if (m_msg && m_msg->hasUserTag(2))
    {
        _cg.setColor( QColorGroup::Text, QColor(255,125,0));
        _cg.setColor( QColorGroup::Highlight, QColor(255,125,0) );
        _cg.setColor( QColorGroup::HighlightedText, Qt::white );
    }
    else if (m_msg && m_msg->hasUserTag(3))
    {
        _cg.setColor( QColorGroup::Text, Qt::darkGreen );
        _cg.setColor( QColorGroup::Highlight, Qt::darkGreen );
        _cg.setColor( QColorGroup::HighlightedText, Qt::white);
    }
    else if (m_msg && m_msg->hasUserTag(4))
    {
        _cg.setColor( QColorGroup::Text, Qt::darkBlue );
        _cg.setColor( QColorGroup::Highlight, Qt::darkBlue );
        _cg.setColor( QColorGroup::HighlightedText, Qt::white );
    }
    else if (m_msg && m_msg->hasUserTag(5))
    {
        _cg.setColor( QColorGroup::Text, Qt::darkMagenta );
        _cg.setColor( QColorGroup::Highlight, Qt::darkMagenta );
        _cg.setColor( QColorGroup::HighlightedText, Qt::white);
    }
    if (m_msg && m_msg->isNew())
    {
        _cg.setColor( QColorGroup::Text, Qt::red );
    }

    KListViewItem::paintCell( p, _cg, column, width, align );

    if (m_msg && m_msg->isDeleted())
        p->drawLine( 0, height()/2, width, height()/2);
}

int HeaderListViewItem::compare( QListViewItem *i, int col,
                              bool ascending ) const
{
    HeaderListViewItem* t = static_cast<HeaderListViewItem*>(i);

    if (level() > t->level())
        return 1;

    if (level() < t->level())
        return -1;

    if (m_msg && col == 1)
    {
        QString one = m_msg->sender();
        QString two = t->msg()->sender();
        if (one.startsWith("To: "))
            one = one.mid(4);
        if (two.startsWith("To: "))
            two = two.mid(4);
        return one.compare(two);
    }
    else if (m_msg && col == 2)
    {
        if (m_msg->sortDate() == t->msg()->sortDate())
        {
            if (m_msg->date() == t->msg()->date())
                return 0;
            else if (m_msg->date() > t->msg()->date())
                return 1;
            else
                return -1;
        }
        else if (m_msg->sortDate() > t->msg()->sortDate())
            return 1;
        else
            return -1;
    }
    else if (m_msg && col == 3)
    {
        int one = m_msg->size();
        int two = t->msg()->size();
        if (one == two)
            return 0;
        else if (one > two)
            return 1;
        else
            return -1;
    }
    else
        return key( col, ascending ).lower()
                .compare( i->key( col, ascending).lower() );

}

}
