""" The module importer used during the loading of plugin definitions. """


# Standard library imports.
from ihooks import ModuleImporter, VERBOSE
from imp import PY_SOURCE
import sys


class Importer(ModuleImporter):
    """ The module importer used during the loading of plugin definitions. """

    ###########################################################################
    # 'object' imterface.
    ###########################################################################

    def __init__(self, module_name_to_filename_map, verbose=VERBOSE):
        """ Creates a new importer. """

        # Inherited constructore.
        ModuleImporter.__init__(self, verbose=verbose)

        # Mapping from module names to filenames.
        self._module_name_to_filename_map = module_name_to_filename_map

        return
    
    ###########################################################################
    # 'ModuleImporter' imterface.
    ###########################################################################
    
    def import_module(self, name, globals=None, locals=None, fromlist=None):
        """ Imports a module. """

        # If this a plugin definition module then we suck it in without
        # importing anything else from the package that it is in.
        if name in self._module_name_to_filename_map:
            # If the module has already been imported then simply return it.
            if name in sys.modules:
                module = sys.modules[name]

            else:
                description = ('.py', 'r', PY_SOURCE)
                filename    = self._module_name_to_filename_map[name]
                info        = (file(filename), filename, description)

                module = self.loader.load_module(name, info)
                
        else:
            module = ModuleImporter.import_module(
                self, name, globals, locals, fromlist
            )

        return module

    def determine_parent(self, globals):
        """ Determine the parent of the import. """
        
        if not globals or not "__name__" in globals:
            return None

        # This is the name of the module that the import is being executed
        # in.
        pname = globals['__name__']

        # This is our change to the standard behavior. If the parent is a
        # plugin definition then there is no parent.
        if pname in self._module_name_to_filename_map:
            return None
        
        if "__path__" in globals:
            parent = self.modules[pname]
            assert globals is parent.__dict__
            return parent

        if '.' in pname:
            i = pname.rfind('.')
            pname = pname[:i]
            parent = self.modules[pname]
            assert parent.__name__ == pname
            return parent
        
        return None

#### EOF ######################################################################
