#-----------------------------------------------------------------------------
#
#  Copyright (c) 2005, Enthought, Inc.
#  All rights reserved.
#
#  Author: Scott Swarts <swarts@enthought.com>
#  Author: Dave Peterson <dpeterson@enthought.com>
#
#-----------------------------------------------------------------------------

""" Provides help for binding a trait to a preference. """


# Enthought library imports
from enthought.traits.api import Any, Bool, HasTraits, Instance, Str

# Local imports.
from preferences import Preferences, PreferenceChangedEvent


class PreferenceBinding(HasTraits):
    """ A helper object for managing the binding of a trait to a preference.
    """

    #### 'PreferenceBinding' interface ########################################

    # The object whose trait this object manages
    obj = Instance(HasTraits)

    # The name of the trait that is bound.
    trait_name = Str

    # The preferences object.
    preferences = Instance(Preferences)

    # The name of the preference.
    pref_name = Str

    # The default value of the preference.
    default_value = Any

    #### Private interface ####################################################

    # Guard for setting the trait from event.
    _handling_event = Bool

    ###########################################################################
    # 'object' interface
    ###########################################################################

    def __init__(self, **kw):
        """ Creates a new preference binding. 
        
        This method sets the value of `obj.trait_name` to the current 
        preference value. It sets up an event handler so that if the preference 
        value changes, the trait will be updated. Finally, it sets up a trait 
        change handler so that if the trait is otherwise set, the binding is
        removed.
        """

        super(PreferenceBinding, self).__init__( **kw )

        # Set the current trait value.
        setattr(
            self.obj,
            self.trait_name,
            self.preferences.get(self.pref_name, self.default_value)
        )

        # Set up a handler for when the preferences changes.
        self.preferences.on_trait_event(
            self._handle_preference_changed, 'preference_changed'
        )

        # Set up a handler for when the trait is set.
        self.obj.on_trait_change(self._handle_trait_changed, self.trait_name)

        return

    ###########################################################################
    # 'PreferenceBinding' interface.
    ###########################################################################

    #### Trait change handlers ################################################

    def _handle_preference_changed(self, event):
        """ Dynamic trait change hander. """

        if event.key == self.pref_name:
            # Set the trait with _handling_event set to True.
            try:
                self._handling_event = True
                setattr(self.obj, self.trait_name, event.new)
                
            finally:
                self._handling_event = False

        return

    def _handle_trait_changed(self):
        """ Dynamic trait change handler. """

        # If some other code is setting the trait value, then remove the
        # handlers.
        if not self._handling_event:
            self.preferences.on_trait_event(
                self._handle_preference_changed,
                'preference_changed',
                remove=True
            )

            self.obj.on_trait_change(
                self._handle_trait_changed, self.trait_name, remove=True
            )

        return 

##############################################################################
# Function 'bind_preference'
##############################################################################

def bind_preference(
    obj, trait_name, preferences, pref_name, deflt, save_binding=True
):
    """ Binds a trait to a preference value.

        Parameters
        ----------
        obj : instance
           The object whose trait is being bound.
        trait_name : string
           The name of the trait to bind.
        preferences: Preferences
            The set of preferences in which to look for the preference to
            bind to.
        pref_name : string
           The name of the preference to bind.
        deflt
           The default value
        save_binding : Boolean
           Whether to save the binding in a private trait on
           *obj*.  If this is False, then the caller must save the binding.

        Description
        -----------
        This method sets the value of `obj.trait_name` to the preference value
        and sets up a listener so that if the preference value is changed, the
        trait is updated. It also sets up a trait handler so that if the trait
        value is otherwise set, the preference listener is removed.

        This method returns a binding object. If *save_binding* is False, the
        binding object must be saved by the caller; if it is True, this method
        saves the binding object in a private trait on *obj* before returning.
    """
    # Do nothing is we have no preferences
    if preferences is None:
        return None

    # Set up a binding
    binding = PreferenceBinding(obj=obj, trait_name=trait_name,
        preferences=preferences, pref_name=pref_name, default_value=deflt)

    # Save the new binding on the object if we were told to do so.
    if save_binding:

        # Build the dictionary, if it is not there
        if not hasattr(obj, "_bind_preference_bindings") or \
           getattr(obj, "_bind_preference_bindings") is None:
            obj._bind_preference_bindings = {}

        # Save the binding
        obj._bind_preference_bindings[(trait_name, pref_name)] = binding

    # Return the binding
    return binding

#### EOF #####################################################################
