#-------------------------------------------------------------------------------
#
#  Define a view of all application wxPython windows.
#
#  Written by: David C. Morrill
#
#  Date: 10/18/2005
#
#  (c) Copyright 2005 by Enthought, Inc.
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

import wx

from enthought.traits.api \
    import HasPrivateTraits, Instance, Property, List, Code, Str, Button

from enthought.traits.ui.api \
    import View, Item, TreeEditor, TreeNodeObject, ObjectTreeNode, \
           TableEditor, VSplit

from enthought.traits.ui.table_column \
    import ObjectColumn

from enthought.envisage.workbench \
    import View as EnvisageView

#-------------------------------------------------------------------------------
#  Constants:
#-------------------------------------------------------------------------------

# Map from wxPython bit to corresponding name:
all_flags = [
    ( wx.EXPAND,                  'EXPAND' ),
    ( wx.ALL,                     'ALL' ),
    ( wx.TOP,                     'TOP' ),
    ( wx.BOTTOM,                  'BOTTOM' ),
    ( wx.LEFT,                    'LEFT' ),
    ( wx.RIGHT,                   'RIGHT' ),
    ( wx.FIXED_MINSIZE,           'FIXED_MINSIZE' ),
    ( wx.ALIGN_CENTER,            'ALIGN_CENTER' ),
    ( wx.ALIGN_RIGHT,             'ALIGN_RIGHT' ),
    ( wx.ALIGN_BOTTOM,            'ALIGN_BOTTOM' ),
    ( wx.ALIGN_CENTER_VERTICAL,   'ALIGN_CENTER_VERTICAL' ),
    ( wx.ALIGN_CENTER_HORIZONTAL, 'ALIGN_CENTER_HORIZONTAL' )
]

#-------------------------------------------------------------------------------
#  'WXView' class:
#-------------------------------------------------------------------------------

class WXView ( EnvisageView ):
    """ A view of all application wxPython windows.

        This view is intended for developers to help in debugging etc.  It is
        NOT intended to be seen by the end user!
    """

    #---------------------------------------------------------------------------
    #  'View' interface:
    #---------------------------------------------------------------------------

    def create_control ( self, parent ):
        """ Creates the toolkit-specific control that represents the view.

        'parent' is the toolkit-specific control that is the view's parent.
        """
        control = parent
        while True:
            root = control.GetParent()
            if root is None:
                break
            control = root
        self._ui = UIDebugger( root = WXWindow( window = control )
                       ).edit_traits( parent = parent )
        return self._ui.control

    #---------------------------------------------------------------------------
    #  Handles the view being closed:
    #---------------------------------------------------------------------------

    def _closing_changed ( self ):
        self._ui.dispose()

#-------------------------------------------------------------------------------
#  'SizerItem' class:
#-------------------------------------------------------------------------------

class WXSizerItem ( HasPrivateTraits ):

    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    item       = Instance( wx.SizerItem )
    kind       = Property
    calc_min   = Property
    proportion = Property
    flags      = Property
    border     = Property

    #---------------------------------------------------------------------------
    #  Property implementations:
    #---------------------------------------------------------------------------

    def _get_kind ( self ):
        obj = self.item.GetWindow()
        if obj is not None:
            return obj.__class__.__name__
        return self.item.GetSizer().__class__.__name__

    def _get_calc_min ( self ):
        dx, dy = self.item.CalcMin()
        return str( ( dx, dy ) )

    def _get_proportion ( self ):
        return str( self.item.GetProportion() )

    def _get_flags ( self ):
        flags = self.item.GetFlag()
        names = []
        for bit, name in all_flags:
            if (flags & bit) == bit:
                names.append( name )
                flags &= ~bit
        if flags != 0:
            names.append( '%8X' % flags )
        return ', '.join( names )

    def _get_border ( self ):
        return str( self.item.GetBorder() )

#-------------------------------------------------------------------------------
#  Tabele editor to use for displaying Sizer item information:
#-------------------------------------------------------------------------------

sizer_item_table_editor = TableEditor(
    columns = [ ObjectColumn( name = 'kind',       editable = False ),
                ObjectColumn( name = 'calc_min',   editable = False ),
                ObjectColumn( name = 'proportion', editable = False ),
                ObjectColumn( name = 'border',     editable = False ),
                ObjectColumn( name = 'flags',      editable = False ) ],
    editable     = False,
    configurable = False,
    sortable     = False )

#-------------------------------------------------------------------------------
#  'WXWindow' class:
#-------------------------------------------------------------------------------

class WXWindow ( TreeNodeObject ):

    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    window    = Instance( wx.Window )
    name      = Property
    position  = Property
    size      = Property
    sizer     = Property
    min_size  = Property
    best_size = Property
    items     = Property( List )
    result    = Property( Code )
    evaluate  = Str

    #---------------------------------------------------------------------------
    #  Traits view definitions:
    #---------------------------------------------------------------------------

    view = View( VSplit( [ 'position~', 'size~', 'sizer~', 'min_size~',
                           'best_size~' ],
                         [ Item( 'items', editor = sizer_item_table_editor ),
                           '|<>' ],
                         [ 'evaluate', 'result#~' ],
                         id = 'splitter' ),
                 kind = 'subpanel',
                 id   = 'window_info',
                 dock = 'horizontal' )

    #---------------------------------------------------------------------------
    #  Handles 'evaluate' being changed:
    #---------------------------------------------------------------------------

    def _evaluate_changed ( self ):
        self.trait_property_changed( 'result', None, None )

    #---------------------------------------------------------------------------
    #  Implementation of the various trait properties:
    #---------------------------------------------------------------------------

    def _get_name ( self ):
        w = self.window
        try:
            label = w.GetLabel()
        except:
            try:
                label = w.GetValue()
            except:
                try:
                    label = w.GetTitle()
                except:
                    label = ''
        if label != '':
            label = ' (%s)' % label
        return self.window.__class__.__name__ + label

    def _get_size ( self ):
        dx, dy = self.window.GetSizeTuple()
        return str( ( dx, dy ) )

    def _get_position ( self ):
        x, y = self.window.GetPositionTuple()
        return str( ( x, y ) )

    def _get_sizer ( self ):
        sizer = self.window.GetSizer()
        if sizer is None:
            return ''
        dx, dy = sizer.CalcMin()
        return '%s( %d, %d )' % ( sizer.__class__.__name__, dx, dy )

    def _get_min_size ( self ):
        dx, dy = self.window.GetMinSize()
        return str( ( dx, dy ) )

    def _get_best_size ( self ):
        dx, dy = self.window.GetBestFittingSize()
        return str( ( dx, dy ) )

    def _get_result ( self ):
        try:
            result = eval( self.evaluate, { '_':  self.window,
                                            '__': self.window.GetSizer() } )
            if isinstance( result, ( list, tuple ) ):
                return '\n'.join( [ '[%d]: %s' % ( i, str( x ) )
                                      for i, x in enumerate( result ) ] )
            return str( result )
        except:
            return '???'

    def _get_items ( self ):
        sizer = self.window.GetSizer()
        if sizer is None:
            return []
        items = []
        try:
            for i in range( 10000 ):
                items.append( WXSizerItem( item = sizer.GetItem( i ) ) )
        except:
            pass
        return items

    #---------------------------------------------------------------------------
    #  Returns whether chidren of this object are allowed or not:
    #---------------------------------------------------------------------------

    def tno_allows_children ( self, node ):
        """ Returns whether chidren of this object are allowed or not.
        """
        return True

    #---------------------------------------------------------------------------
    #  Returns whether or not the object has children:
    #---------------------------------------------------------------------------

    def tno_has_children ( self, node = None ):
        """ Returns whether or not the object has children.
        """
        return (len( self.window.GetChildren() ) > 0)

    #---------------------------------------------------------------------------
    #  Gets the object's children:
    #---------------------------------------------------------------------------

    def tno_get_children ( self, node ):
        """ Gets the object's children.
        """
        return [ WXWindow( window = window )
                 for window in self.window.GetChildren() ]

#-------------------------------------------------------------------------------
#  Defines the window browser tree editor:
#-------------------------------------------------------------------------------

window_tree_editor = TreeEditor(
    nodes = [
        ObjectTreeNode( node_for = [ WXWindow ],
                        children = 'children',
                        label    = 'name' ),
    ]
)

#-------------------------------------------------------------------------------
#  'UIDebugger' class:
#-------------------------------------------------------------------------------

class UIDebugger ( HasPrivateTraits ):

    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    # Root of a wxWindow window hierarchy:
    root = Instance( WXWindow )

    # Event to force a view refresh:
    refresh = Button( 'Refresh' )

    #---------------------------------------------------------------------------
    #  Traits view definitions:
    #---------------------------------------------------------------------------

    view = View( [ [ Item( 'root', editor = window_tree_editor,
                                   id     = 'window_tree' ), '|<>' ],
                   [ 'refresh', '-<>' ],
                   '|<>' ],
                 kind = 'subpanel',
                 dock = 'vertical' )

    #---------------------------------------------------------------------------
    #  Handles the 'Refresh' button being pressed:
    #---------------------------------------------------------------------------

    def _refresh_fired ( self ):
        """ Handles the 'Refresh' button being pressed.
        """
        temp, self.root = self.root, None
        self.root = temp

