""" The project plugin. """


# Standard library imports.
import inspect, logging, os
from os.path import join

# Enthought library imports.
from enthought.envisage import Plugin, application
from enthought.envisage.resource import ResourceManager, ResourceType
from enthought.io import File
from enthought.naming.api import Context, InitialContext
from enthought.pyface.api import ConfirmationDialog, YES, NO, CANCEL
from enthought.traits.api import Bool, Directory, Event, HasTraits, Instance, \
                             List, Str
from enthought.traits.ui.api import View, Group, Item

# Local imports.
from cookie_manager import CookieManager
from workspace import Workspace


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class WorkspaceDetails(HasTraits):
    """ Used to prompt the user for the workspace location etc. """

    # The directory that the workspace will be stored in.
    workspace_location = Directory

    # Should the application always use this location?
    use_as_default = Bool(True)

    view = View(
        Group(
            Group(
                Item(name='workspace_location', label='Workspace'),
            ),

            Group(
                Item(
                    name  = 'use_as_default',
                    label = 'Use this as the default and do not ask again',
                ),

                show_left = False,
            )
        )
    )


class ProjectPlugin(Plugin):
    """ The project plugin. """

    # The shared plugin instance.
    instance = None

    IWORKSPACE_SERVICE = 'enthought.envisage.project.IWorkspace'

    #### 'ProjectPlugin' interface ############################################

    # The names of all open projects.
    open_projects = List(Str)

    # The workspace.
    workspace = Instance(Workspace)

    # Root of the template namespace.
    templates = Instance(Context, ())

    # The workspace has been saved
    saved = Event

    ###########################################################################
    # 'object' interface.
    ###########################################################################

    def __init__(self, **kw):
        """ Creates a new plugin. """

        # Base class constructor.
        super(ProjectPlugin, self).__init__(**kw)

        # Set the shared instance.
        ProjectPlugin.instance = self

        return

    ###########################################################################
    # 'Plugin' interface.
    ###########################################################################

    def start(self, application):
        """ Starts the plugin.

        Can be called manually, but is usually called exactly once when the
        plugin is first required.

        """

        # Initialize the workspace.
        self.workspace = self._initialize_workspace()
        logger.info('workspace location %s' % self.workspace.path)

        # Register the workspace a service.
        self.register_service(
            'enthought.envisage.project.IWorkspace', self.workspace
        )

        # The names of the projects that were open last time.
        self.open_projects = self.preferences.get('open_projects')
        logger.info('open projects %s' % self.open_projects)

        here = os.path.dirname(inspect.getsourcefile(ProjectPlugin))

        # Lookup our default templates.
        templates_dir = File(os.path.join(here, 'data/templates'))
        for category_dir in templates_dir.children:
            if '.svn' in category_dir.path:
                continue

            category = Context()

            for template_dir in category_dir.children:
                if '.svn' in template_dir.path:
                    continue

                from template import FileCopyTemplate
                fct = FileCopyTemplate(
                    source = template_dir.path,
                    target = os.path.join(
                        self.workspace.path, template_dir.name
                    )
                )

                category.bind(template_dir.name, fct)

            self.templates.bind(category_dir.name, category)

        # fixme: Major hack to hook the closing process.
        from enthought.envisage.ui.ui_plugin import UIPlugin
        UIPlugin._confirm_exit = self._confirm_exit

        return

    def stop(self, application):
        """ Stops the plugin.

        Can be called manually, but is usually called exactly once when the
        application exits.

        """

        self.save_preferences()

        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _confirm_exit(self, window):
        """ Returns True iff it is OK to exit the application. """

        dialog = ConfirmationDialog(
            parent    = window.control,
            title     = 'Confirm Exit',
            message   = 'Do you want to save before exiting?',
            cancel    = True,
            yes_label = 'Save and Exit',
            no_label  = 'Exit'
        )
        result = dialog.open()
        if result == YES:
            self.save_workspace(window)
            can_exit = True

        elif result == NO:
            can_exit = True

        else:
            can_exit = False

        return can_exit

    def save_workspace(self, window):
        """ Saves the workspace. """

        workspace = self.workspace

        def save(context, progress, total=1000, value=0):
            from enthought.envisage.resource import ResourceContext
            for binding in context.list_bindings():
                value += 50
                progress.Update(value % total, 'Saving %s' % binding.name)
                if isinstance(binding.obj, ResourceContext):
                    value = save(binding.obj, progress, total, value)
                else:
                    context._rebind(binding.name, binding.obj)


            return value

        import wx
        progress = wx.ProgressDialog(
            'Saving the workspace',
            'Finding items to save...',
            1000,
            window.control,
            wx.PD_APP_MODAL
        )

        # notify the world that we're saving
        self.saved = self.workspace

        # Make up some number to indicate progress ;^)
        save(self.workspace, progress)
        progress.Destroy()

        return

    def _initialize_workspace(self):
        """ Initializes the workspace. """

        # Get the workspace location.
        workspace_location = self._get_workspace_location()

        # If no such directory exists then create one.
        if not os.path.isdir(workspace_location):
            os.mkdir(workspace_location)

        # fixme: We should use the JNDI initial context API here!
        klass_name = "enthought.envisage.project." \
                     "workspace_initial_context_factory." \
                     "WorkspaceInitialContextFactory"

        environment = {
            'application'                   : self.application,
            Context.INITIAL_CONTEXT_FACTORY : klass_name
        }
        environment['root'] = workspace_location

        # Create an initial context.
        return InitialContext(environment)

    def _get_workspace_location(self):
        """ Returns the location of the workspace. """

        workspace_location = self.preferences.get('workspace_location')
        use_as_default     = self.preferences.get('use_as_default')

        # The workspace location will be 'None' the first time the application
        # is run.
        if workspace_location is None or not use_as_default:
            if workspace_location is None:
                # By default we put the workspace in this plugin's state
                # location.
                workspace_location = join(self.state_location, 'Workspace')

            details = WorkspaceDetails(
                workspace_location = workspace_location,
                use_as_default     = use_as_default
            )

            from wizard.welcome_wizard import WelcomeWizard
            wizard = WelcomeWizard(details)
            wizard.open()

            # By default we put the workspace in this plugin's state location.
            workspace_location = details.workspace_location

            # We save the preferences straight away in case the application
            # does not close down in an orderly fashion.
            self.preferences.set('workspace_location', workspace_location)
            self.preferences.set('use_as_default', details.use_as_default)
            self.save_preferences()

        return workspace_location

    #### Trait event handlers #################################################

    def _current_project_changed(self):
        """ Called when the current project is changed. """

        self.preferences.set('current_project', self.current_project)

        self.save_preferences()

        return

#### EOF ######################################################################
