""" The workspace. """


# Standard library imports.
import os

# Enthought library imports.
from enthought.envisage.resource import ResourceContext
from enthought.traits.api import Str

# Local imports.
from folder import Folder
from project import Project


class Workspace(ResourceContext):
    """ The workspace.

    There is exactly one workspace per application instance.  It contains all
    projects.

    """

    # The workspace name.
    name = Str
    
    ###########################################################################
    # 'Workspace' interface.
    ###########################################################################

    def create_project(self, name):
        """ Creates a project with the specified name. """

        project = self.create_subcontext(name)

        return project

    def create_project_at(self, name, path):
        """ Creates a project with the specified name. """

        project_dir = os.path.join(path, name)
        if not os.path.exists(project_dir):
            os.mkdir(project_dir)
            
        project = self._context_factory(name, project_dir)
        self.bind(name, project)

        return project

    def has_project(self, name):
        """ Does a project exist with the specified name. """
        
        return self._is_bound(name)
    
    def delete_project(self, name):
        """ Deletes the project with the specified name. """

        self.destroy_subcontext(name)

        return

    ###########################################################################
    # Protected 'Context' interface.
    ###########################################################################

    def _context_factory(self, name, path=None):
        """ Create a sub-context. """

        if path is None:
            path = os.path.join(self.path, name)

        return Project(path=path, workspace=self, environment=self.environment)

    ###########################################################################
    # Private interface.
    ###########################################################################

    #### Trait event handlers ################################################

    def _path_changed(self):
        """ Called when the context's path has changed. """

        super(Workspace, self)._path_changed()

        # fixme: Make sure that the workspace exists.  Is this a good idea?
        if not os.path.isdir(self.path):
            os.mkdir(self.path)
            
        return
    
##### EOF #####################################################################
