#-----------------------------------------------------------------------------
#
# Copyright (c) 2006 by Enthought, Inc.
# All rights reserved.
#
# This software is provided without warranty under the terms of the BSD
# license included in enthought/LICENSE.txt and may be redistributed only
# under the conditions described in the aforementioned license.  The license
# is also available online at http://www.enthought.com/licenses/BSD.txt
# Thanks for using Enthought open source!
#
#-----------------------------------------------------------------------------

"""
A base class for resources that represent actions.

"""

# Standard library imports
import logging

# Enthought library imports
from enthought.envisage import get_application
from enthought.envisage.action.api import ActionFactory, ActionSetManager, \
    DefaultMenuBuilder
from enthought.envisage.action.action_plugin_definition import ActionSet
from enthought.pyface.api import ImageResource
from enthought.pyface.action.api import ActionItem, MenuManager
from enthought.traits.api import Class, HasPrivateTraits, Instance, Str


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class ActionResource(HasPrivateTraits):
    """
    A base class for resources that represent actions.

    These resources are typically used in controls, such as tree controls, in
    order to lead a user to the appropriate actions to take within that
    control.  They are meant to be added programmatically into the context
    displayed within the control when the action is possible to perform.

    These resources will be recognized by the
        enthought.envisage.resource.action_resource_type.ActionResourceType
    implementation which will defer all 'look and feel' behavior to this, and
    derived, implementations.  The end result is that implementers only need
    to derive from this class and insert an instance of that into their
    context.

    """


    ##########################################################################
    # Attributes
    ##########################################################################

    #### public 'ActionResource' interface ###################################

    # The class of ActionSets we will search to find actions that can be
    # applied to this resource.
    actionset_class = Class(ActionSet)

    # The root location, within the *actionset_class*, used to identify the
    # context menu actions, groups, and menus that apply to this action
    # resource.  If not set, then no context menu applies to this resource.
    context_location_root = Str

    # The id of an action, rooted within *context_location_root*, to be
    # performed as our default action.
    default_action = Str

    # The image used to represent this action.
    image = Instance(ImageResource)

    # The display name for this action
    name = Str


    ##########################################################################
    # 'ActionResource' interface.
    ##########################################################################

    #### public interface ####################################################

    def get_context_menu(self):
        """
        Return the context menu for this action.

        """

        menu = self._get_context_menu_template()
        self._initialize_context_menu(menu)

        return menu


    def get_default_action(self):
        """
        Return the default action for this action resource.

        """

        application = get_application()
        action_sets = application.load_extensions(self.actionset_class)
        manager = ActionSetManager(action_sets=action_sets)
        action = manager.get_action(self.default_action,
            self.context_location_root)
        if action is not None:
            factory = ActionFactory(application=application)
            result = factory.create_action(action)
        else:
            logger.warn('Could not find default action for ActionResource [%s]',
                self)
            result = None

        return result


    #### protected interface #################################################

    def _get_context_menu_template(self):
        """
        Return the context menu template for the specified node.

        """

        # If an action location root is set, then build the menu from all
        # actions, groups, and menus having that root location across the
        # known action sets.
        menu = None
        if self.context_location_root is not None and \
            len(self.context_location_root.strip()) > 0:

            menu = MenuManager()
            try:
                application = get_application()
                action_sets = application.load_extensions(self.actionset_class)
                manager = ActionSetManager(action_sets=action_sets)
                menu_builder = DefaultMenuBuilder(application=application)
                menu_builder.initialize_menu_manager(menu, manager,
                    self.context_location_root)
            except:
                logger.exception('Unable to build context menu')

        return menu


    def _image_default(self):
        """
        Return the default image for an action resource.

        """

        return ImageResource('action')


    def _initialize_action_item(self, node, item):
        """
        Initialize the specified action item before it is displayed in a
        context menu for the specified node.

        Implemented this way to allow implementors to easily override what
        gets set on action items during initialization.

        """

        # fixme: this is extra gorpy!
        if hasattr(item.action, 'refresh'):
            item.action.refresh()

        return


    def _initialize_context_menu(self, menu, node=None):
        """
        Set the enabled/disabled state of actions in a context menu.

        """

        for group in menu.groups:
            for item in group.items:
                if isinstance(item, ActionItem):
                    self._initialize_action_item(node, item)
                elif isinstance(item, MenuManager):
                    self._initialize_context_menu(item, node)

        return


#### EOF #####################################################################

