#-----------------------------------------------------------------------------
#
#  Copyright (c) 2005, Enthought, Inc.
#  All rights reserved.
#
#-----------------------------------------------------------------------------

""" A base class for editors that need standard decorations such as
    toolbars.
"""

# Major package imports
import logging
import wx

# Enthought library imports.
from enthought.traits.api import Instance
from enthought.pyface.action.api import ToolBarManager

# Local imports
from editor import Editor


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class DecoratedEditor(Editor):
    """ A base class for editors that need standard decorations such as
        toolbars.
    """

    ##########################################################################
    # Traits
    ##########################################################################

    #### public 'DecoratedEditor' interface ##################################

    # The (optional) tool bar for the window.
    tool_bar_manager = Instance(ToolBarManager)


    #### protected 'DecoratedEditor' interface ###############################

    # The toolkit object that represents this instance's toolbar
    _tool_bar = Instance(wx.ToolBar)

    # The toolkit object that represents this instance's content
    _content = Instance(wx.Window)


    ##########################################################################
    # 'Editor' interface.
    ##########################################################################

    #### public interface ####################################################

    def create_control(self, parent):
        """ Creates the toolkit-specific control that represents the editor.

            'parent' is the toolkit-specific control that is the editor's
            parent.

            DecoratedEditor implementors should not override this, they should
            override '_create_contents' instead!

        """
        logger.debug("Creating control for DecoratedEditor [%s]", self)

        # Create a panel that we can add additional controls to besides just
        # the editor content itself.
        import wx
        control = wx.Panel(parent, -1, style=wx.CLIP_CHILDREN)
        sizer = wx.BoxSizer(wx.VERTICAL)
        control.SetSizer(sizer)
        control.SetAutoLayout(True)

        # Add a toolbar to the panel if we have one.
        if self.tool_bar_manager is not None:
            logger.debug("Creating toolbar for DecoratedEditor [%s]", self)
            self._tool_bar = self.tool_bar_manager.create_tool_bar(control)
            sizer.Add(self._tool_bar, 0, wx.EXPAND)

        # Create the actual editor content
        self._content = self._create_contents(control)
        sizer.Add(self._content, 1, wx.EXPAND)

        # Ensure the child controls are laid-out.
        sizer.Layout()

        return control


    def destroy_control(self):
        """ Destroys the toolkit-specific control that represents the editor.

            Overridden to delete references to decorations.
        """
        if self._tool_bar is not None:
            logger.debug('Destorying toolbar for DecoratedEditor [%s]', self)
            self._tool_bar.Destroy()
            self._tool_bar = None

        if self._content is not None:
            logger.debug('Destorying content for DecoratedEditor [%s]', self)
            self._destroy_content()
            self._content = None

        super(DecoratedEditor, self).destroy_control()


    def set_focus(self):
        """ Sets the focus to the appropriate control in the editor.

            Overridden to give focus to the content of the editor.
        """
        if self._content is not None:
            self._content.SetFocus()

        return


    ##########################################################################
    # 'DecoratedEditor' interface.
    ##########################################################################

    #### protected interface #################################################

    def _create_contents(self, parent):
        """ Create the toolkit-specific control that represents the content
            of this editor.

            'parent' is the toolkit-specific control that is the content's
            parent.

            DecoratedEditor implementors should override this method to create
            their specific content.  They only need to return the created
            tool-kit control, the DecoratedEditor code will store it in the
            _content trait.
        """
        logger.warn(
            'DECORATEDEDITOR [%s] FAILED TO OVERRIDE "_create_contents"',
            self)

        # By default we create a yellow panel!
        panel = wx.Panel(parent, -1)
        panel.SetBackgroundColour("yellow")
        panel.SetSize((100, 200))

        return panel


    def _destroy_content(self):
        """ Destroys the toolkit-specific control that represents the content
            of this editor.

            DecoratedEditor implementors can override this if they wish to
            do additional control cleanup.  Overridden methods should still
            call this method.
        """
        if self._content is not None:
            logger.debug('Destroying content for DecoratedEditor [%s]', self)
            self._content.Destroy()
            self._content = None


#### EOF #####################################################################
