""" A view whose content is provided by a traits UI. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.traits.api import Any, Instance, Str
from enthought.traits.ui.api import UI

# Local imports.
from view import View


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class TraitsUIView(View):
    """ An view whose content is provided by a traits UI. """

    #### 'TraitsUIView' interface #############################################

    # The traits UI that represents the view.
    #
    # The framework sets this to the value returned by 'create_ui'.
    ui = Instance(UI)

    # The UOL of the object that provides the view (if this is not set then
    # the view itself is the object that we call 'edit_traits' on).
    uol = Str

    # The name of the traits UI view used to create the UI (if not specified,
    # the default traits UI view is used).
    view = Str

    ###########################################################################
    # 'View' interface.
    ###########################################################################

    def create_control(self, parent):
        """ Creates the toolkit-specific control that represents the view.

        'parent' is the toolkit-specific control that is the view's parent.

        Overridden to call 'create_ui' to get the traits UI.

        """

        self.ui = self.create_ui(parent)

        return self.ui.control

    def destroy_control(self):
        """ Destroys the toolkit-specific control that represents the view.

        Overridden to call 'dispose' on the traits UI.

        """

        # Give the traits UI a chance to clean itself up.
        if self.ui is not None:
            logger.debug('disposing traits UI for view [%s]', self)
            self.ui.dispose()

        return

    ###########################################################################
    # 'TraitsUIView' interface.
    ###########################################################################

    def create_ui(self, parent):
        """ Creates the traits UI that represents the view. """

        obj = self._get_target_object()

        return obj.edit_traits(parent=parent, view=self.view, kind='subpanel')

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _get_target_object(self):
        """ Returns the object that we create the traits UI view of. """

        if len(self.uol) > 0:
            obj = self.window.application.lookup_application_object(self.uol)

        else:
            obj = self

        return obj

#### EOF ######################################################################
