#-------------------------------------------------------------------------------
#
#  Manage user defined perspectives.
#
#  Written by: David C. Morrill
#
#  Date: 01/15/2007
#
#  (c) Copyright 2007 by Enthought, Inc.
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

import logging

from os \
    import remove

from os.path \
    import join

from enthought.traits.api \
    import HasPrivateTraits, Property, List, Dict, Str, Int

from perspective \
    import Perspective


# Setup a logger for this module.
logger=logging.getLogger(__name__)


#-------------------------------------------------------------------------------
#  'UserPerspective' class:
#-------------------------------------------------------------------------------

class UserPerspective ( HasPrivateTraits ):

#--  Trait Definitions ---------------------------------------------------------

    # A directory on the local file system that we can read and write to at
    # will. This is used to persist window layout information, etc.
    state_location = Str

    # Next available user perspective id:
    next_id = Property( Int )

    # Dictionary mapping perspective id to user defined perspective definition:
    id_to_perspective = Property( Dict )

    # The list of user defined perspective definitions:
    perspectives = Property( List )

    # The name of the user defined perspectives definition file:
    file_name = Property( Str )

#-- Property Implementations ---------------------------------------------------

    def _get_next_id ( self ):
        """ Returns the next available perspective id.
        """
        # Get all of the current perspective ids:
        ids = self.id_to_perspective.keys()

        # If there are none:
        if len( ids ) == 0:
            # Return the starting id:
            return 1

        # Else return the current highest id + 1 as the next id:
        ids.sort()
        return int( ids[-1][19:-2] ) + 1

    def _get_id_to_perspective ( self ):
        """ Returns the dictionary mapping a perspective id to a user defined
            perpsective definition.
        """
        if self._id_to_perspective is None:
            self._id_to_perspective = dic = {}
            try:
                fh = file( self.file_name, 'rb' )
                for line in fh:
                    data = line.split( ':', 1 )
                    if len( data ) == 2:
                        id, name = data[0].strip(), data[1].strip()
                        dic[ id ] = Perspective(
                            id               = id,
                            name             = name,
                            show_editor_area = False
                        )
                fh.close()
            except:
                pass

        return self._id_to_perspective

    def _get_perspectives ( self ):
        """ Returns the current list of user defined perspectives.
        """
        return self.id_to_perspective.values()

    def _get_file_name ( self ):
        """ Returns the name of the user defined perspective definition file.
        """
        return join( self.state_location, '__user_perspective__' )

#-- Public Methods -------------------------------------------------------------

    def add ( self, perspective, name = None ):
        """ Adds a specified perspective, with an optional name.
        """
        # Define the id for the new perspective:
        perspective.id = id = '__user_perspective_%09d__' % self.next_id

        # Save the new name (if specified):
        if name is not None:
            perspective.name = name

        # Create the perspective:
        self.id_to_perspective[ id ] = perspective

        # Update the persistent file information:
        self._update_persistent_data()

        # Return the new perspective created:
        return perspective

    def new_perspective ( self, name ):
        """ Returns a new user-defined perspective with a specified name.
        """
        return self.add( Perspective( show_editor_area = False ), name )

    def rename ( self, id, name ):
        """ Renames the user perspective with a specified id to the specified
            name.
        """
        self.id_to_perspective[ id ].name = name

        # Update the persistent file information:
        self._update_persistent_data()

    def remove ( self, id ):
        """ Removes the user perspective whose id is specified by id, and
            updates the persistent data.
        """
        if id in self.id_to_perspective:
            del self.id_to_perspective[ id ]

            # Update the persistent file information:
            self._update_persistent_data()

            # Try to delete the associated perspective layout file:
            try:
                remove( join( self.state_location, id ) )
            except:
                pass

#-- Private Methods ------------------------------------------------------------

    def _update_persistent_data ( self ):
        """ Updates the persistent file information.
        """
        try:
            fh = file( self.file_name, 'wb' )
            fh.write( '\n'.join( [ '%s: %s' % ( p.id, p.name )
                                   for p in self.perspectives ] ) )
            fh.close()
        except:
            logger.error( "Could not write the user defined perspective "
                          "definition file: " + self.file_name )


