""" The plugin that creates the Envisage Workbench UI. """


# Enthought library imports.
from enthought.envisage.api import ExtensionItem, ExtensionPoint
from enthought.envisage.api import PluginDefinition
from enthought.traits.api import Bool, Enum, Float, Instance, List, Str, Tuple

# Plugin definition imports.
from enthought.envisage.core.core_plugin_definition import \
     Preferences, Runnable


# The plugin's globally unique identifier (also used as the prefix for all
# identifiers defined in this module).
ID = "enthought.envisage.workbench"


##############################################################################
# Extension points.
##############################################################################

class Branding(ExtensionPoint):
    """ Branding information (the application icon, 'About' box image etc). """

    # Additional text to display in the about box
    about_additions = List(Str)

    # The about box image.
    about_image = Str

    # The application icon.
    application_icon = Str

    # The application name (used for window titles etc).
    application_name = Str


class Feature(ExtensionPoint):
    """ A Workbench DockWindow extension feature. """

    # The name of the class that implements the feature (must be a subclass
    # of enthought.pyface.dock.dock_window_feature.DockWindowFeature).
    class_name = Str


class Perspective(ExtensionItem):
    """ A perspective. """

    # The perspective's globally unique identifier.
    id = Str

    # The name of the class that implements the perspective. This is NOT
    # required iff you specify a list of contents.
    class_name = Str

    # The perspective name (this appears on the 'Perspective' menu etc).
    name = Str

    # The size of the editor area in this perspective. The default value of
    # (-1, -1) indicates that the workbench window should choose an appropriate
    # size based on the sizes of the views in the perspective.
    #
    # The tuple is of the form Tuple(Float width, Float height) where 'width'
    # and 'height' are expressed as fractions of the window width and height.
    #
    # e.g. (0.5, 0.5) == half the window width and height.
    #
    # Note that this is treated as a suggestion, and it may not be possible for
    # the workbench to allocate the space requested.
    editor_area_size = Tuple((-1, -1))

    # Should the editor are be shown in this perspective?
    show_editor_area = Bool(True)

    class Item(ExtensionItem):
        """ An item in the perspective contents list. """

        # The Id of the view to display in the perspective.
        id = Str

        # The position of the view relative to the item specified in the
        # 'relative_to' trait.
        #
        # 'top'    puts the view above the 'relative_to' item.
        # 'bottom' puts the view below the 'relative_to' item.
        # 'left'   puts the view to the left of  the 'relative_to' item.
        # 'right'  puts the view to the right of the 'relative_to' item.
        # 'with'   puts the view in the same region as the 'relative_to' item.
        #
        # If the position is specified as 'with' you must specify a
        # 'relative_to' item other than the editor area (i.e., you cannot
        # position a view 'with' the editor area).
        position = Enum('left', 'top', 'bottom', 'right', 'with')

        # The Id of the view to position relative to. If this is not specified
        # (or if no view exists with this Id) then the view will be placed
        # relative to the editor area.
        relative_to = Str

        # The width of the item (as a fraction of the window width).
        #
        # e.g. 0.5 == half the window width.
        #
        # Note that this is treated as a suggestion, and it may not be possible
        # for the workbench to allocate space requested.
        width = Float(-1)

        # The height of the item (as a fraction of the window height).
        #
        # e.g. 0.5 == half the window height.
        #
        # Note that this is treated as a suggestion, and it may not be possible
        # for the workbench to allocate space requested.
        height = Float(-1)

    # The contents of the perspective.
    contents = List(Item)

    # Is the perspective initally enabled?
    enabled = Bool(True)


class View(ExtensionItem):
    """ A view. """

    # The view's globally unique identifier.
    id = Str

    # There are currently 2 ways to specify the view.
    #
    # 1) Specify the name of the class that implements the view.
    # 2) Specify the UOL of an object and the name of a Traits UI view.
    #
    # The name of the class that implements the view.
    class_name = Str

    # The UOL of on object that provides the view.
    uol = Str

    # The name of the Traits UI view on the UOL object.
    traits_ui_view = Str
    
    # The view name (this appears on the View's tab etc.).
    name = Str

    # The image used to present the view visually.
    image = Str

    # Is the view visible by default?
    visible = Bool(True)

    # The following traits specify the *default* position of the view. This
    # information is only used if the view is added to a perspective that
    # it is not explicitly part of (i.e. it does not appear in the
    # perspective's 'contents'.
    #
    # This often happens because:-
    #
    # a) The application doesn't define any perspectives
    # b) The user can add/remove any view to/from any perspective
    #
    # fixme: These traits are idential to those in 'PerspectiveItem'. How can
    # we unify them?

    # The default position of the view relative to the item specified in the
    # 'relative_to' trait.
    #
    # 'top'    puts the view above the 'relative_to' item.
    # 'bottom' puts the view below the 'relative_to' item.
    # 'left'   puts the view to the left of  the 'relative_to' item.
    # 'right'  puts the view to the right of the 'relative_to' item.
    # 'with'   puts the view in the same region as the 'relative_to' item.
    #
    # If the position is specified as 'with' you must specify a 'relative_to'
    # item other than the editor area (i.e., you cannot position a view 'with'
    # the editor area).
    position = Enum('left', 'top', 'bottom', 'right', 'with')

    # The Id of the view to position this view relative to. If this is not
    # specified (or if no view exists with this Id) then the view will be
    # placed relative to the editor area.
    relative_to = Str

    # The default width of the view (as a fraction of the window width).
    #
    # e.g. 0.5 == half the window width.
    #
    # Note that this is treated as a suggestion, and it may not be possible for
    # the workbench to allocate space requested.
    width = Float(-1)

    # The default height of the view (as a fraction of the window height).
    #
    # e.g. 0.5 == half the window height.
    #
    # Note that this is treated as a suggestion, and it may not be possible for
    # the workbench to allocate space requested.
    height = Float(-1)


class Workbench(ExtensionPoint):
    """ Workbench contributions. """

    # The Id of the default perspective. By default the first contributed
    # perspective will be the default.
    default_perspective = Str

    # Perspectives.
    perspectives = List(Perspective)

    # Views.
    views = List(View)

##############################################################################
# Extensions.
##############################################################################

preferences = Preferences(
    defaults = {
        # The name of the perspective to show when the workbench starts up.
        'default_perspective' : '',

        # Should we prompt the user when they attempt to close the last open
        # window (and hence exit the application). This helps prevent the user
        # from exiting the application accidentally. However, we *never* prompt
        # if the user exits the workbench by choosing the 'Exit' action on the
        # 'File' menu.
        'prompt_on_exit' : True,

        # Should we display the name of each tool bar tool under its image?
        'show_tool_names'  : False,

        # The initial position of workbench windows (this is only used the
        # first time the application starts up. After that the workbench
        # remembers the position from the previous session).
        'window_position' : (300, 200),

        # The initial size of workbench windows (this is only used the first
        # time the application starts up. After that the workbench remembers
        # the size from the previous session).
        'window_size' : (1280, 1024)
    }
)

# A runnable that opens the first workbench window.
runnable = Runnable(
    class_name = ID + '.workbench_runnable.WorkbenchRunnable'
)

##############################################################################
# The plugin definition.
##############################################################################

class WorkbenchPluginDefinition(PluginDefinition):
    """ The workbench plugin definition. """

    # The plugin's globally unique identifier.
    id = ID

    # The name of the class that implements the plugin.
    class_name = ID + ".workbench_plugin.WorkbenchPlugin"

    # General information about the plugin.
    name          = "Workbench Plugin"
    version       = "1.0.0"
    provider_name = "Enthought Inc"
    provider_url  = "www.enthought.com"

    # The Id's of the plugins that this plugin requires.
    requires = [
        "enthought.envisage.core",
        "enthought.envisage.resource"
    ]

    # The extension points offered by this plugin.
    extension_points = [Branding, Feature, Workbench]

    # The contributions that this plugin makes to extension points offered by
    # either itself or other plugins.
    extensions = [preferences, runnable]

### EOF ######################################################################
