/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: console.c,v 1.11 2003/02/05 04:40:14 kos Exp $
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "llog.h"
#include "console.h"
#include "cli.h"
#include "medussa.h"
#include "common.h"

static int console_done = 0;

int
cmd_show(medussa_t *m, int argc, char **argv) {
  node_t *n;
  hash_t *hash;
  schedule_t *s;
  int i;
  char eta[HASH_LINELEN];
  
  fprintf(stdout, "hashpool: state %s, %d nodes, %d hashes, %d schedules\n",
       hashpool_state_ntoa(m->h->state),
       array_nelems(m->h->nodes),
       array_nelems(m->h->hashes),
       array_nelems(m->h->schedules));

  if(m->h->cur_schedule) {
    fprintf(stdout, "Current schedule %d: %s:",
	 m->h->cur_schedule->serial,
	 keyspace_ntoa(m->h->cur_schedule->start));
    fprintf(stdout, "%s:", keyspace_ntoa(m->h->cur_schedule->index));
    fprintf(stdout, "%s \n", keyspace_ntoa(m->h->cur_schedule->finish));
    fprintf(stdout, "  ETA %s, ", hashpool_stats(m->h, STATS_ETA, SCHED_CURRENT, eta, HASH_LINELEN));
    fprintf(stdout, "total cps %s, ", hashpool_stats(m->h, STATS_TOTALCPS, SCHED_CURRENT, eta, HASH_LINELEN));
    fprintf(stdout, "%s%% done\n", hashpool_stats(m->h, STATS_PERCENT, SCHED_CURRENT, eta, HASH_LINELEN));
  } else
    fprintf(stdout, "No current schedule.\n");
  
  for(i=0; (hash=array_get(m->h->hashes, i)); i++) {
    fprintf(stdout, "  hash %s: method %s, hash %s, key %s, status %s\n",
	 hash->name,
	 hash->method,
	 hash->hash.p,
	 hash->key.l?(char *)hash->key.p:"None",
	 hashpool_state_ntoa(hash->state));
  }

  for(i=0; (s=array_get(m->h->schedules, i)); i++) {
    fprintf(stdout, "  schedule %d: hash %s, generator %s(%s), method %s, status %s\n",
	 s->serial,
	 s->hash->name,
	 s->generator?s->generator:"",
	 s->generator?s->generator_params:"",
	 s->method,
	 hashpool_state_ntoa(s->state));
  }

  for(i=0; (n=array_get(m->h->nodes, i)); i++) {
    fprintf(stdout, "  node %s: %s:", 
	 n->name,
	 keyspace_ntoa(n->start));
    fprintf(stdout, "%s, slice %d, state %s, %d c/s\n",
	 keyspace_ntoa(n->finish),
	 n->slice,
	 hashpool_state_ntoa(n->state),
	 n->cracks_per_sec);
  }
  
  return 0;
}

int
cmd_loglevel(medussa_t *m, int argc, char **argv) {
  
  if(argc==2)
    llog_level(atoi(argv[1]));
  fprintf(stdout, "loglevel = %d\n", llog_getlevel());
  return 0;
}

int
cmd_suspend(medussa_t *m, int argc, char **argv) {
  msg *s;

  s = msg_new();
  if(hashpool_manage(m->h, "pool", "disable", s))
    fprintf(stdout, "hashpool_manage(pool, disable): %s\n", hashpool_geterr(m->h));
  msg_destroy(s);
  return 0;
}

int
cmd_resume(medussa_t *m, int argc, char **argv) {
  msg *s;

  s = msg_new();
  if(hashpool_manage(m->h, "pool", "enable", s))
    fprintf(stdout, "hashpool_manage(itself, enable): %s\n", hashpool_geterr(m->h));
  msg_destroy(s);
  return 0;
}

int
cmd_unset(medussa_t *m, int argc, char **argv) {
  config_unset(m->conf, argv[1]);
  return 0;
}

int
cmd_quit(medussa_t *m, int argc, char **argv) {
  console_done = 1;
  return 0;
}

int
cmd_set(medussa_t *m, int argc, char **argv) {
  int i;
  config_elem_t *el;
  
  if(argc==1) {
    for(i=0; i<config_nelems(m->conf); i++) {
      el = config_getbynum(m->conf, i);
      fprintf(stdout, "%-20s %s\n", el->lhs, el->rhs);
    }
  } else if(argc==2) {
    for(i=0; i<config_nelems(m->conf); i++) {
      el = config_getbynum(m->conf, i);
      if(!strcmp(argv[1], el->lhs))
	fprintf(stdout, "%-20s %s\n", el->lhs, el->rhs);
    }
  } else {
    config_set(m->conf, argv[1], argv[2]);
  }

  return 0;
}

console_t *
console_new(void *up) {
  console_t *c;

  c = xcalloc(1, sizeof(console_t));

  c->up = up;
  c->cli = cli_init(up);
  cli_set(c->cli, CLI_PROMPT, "medussa console> ");
  cli_set(c->cli, CLI_PARSESEP, " ");
  cli_register(c->cli, CLI_PARSE, (cmd_proc)cmd_show, "show", 0, 1, "Produces information about hashes, schedules and nodes");
  cli_register(c->cli, CLI_PARSE, (cmd_proc)cmd_set, "set", 1, 3, "Messes with the internal variable store");
  cli_register(c->cli, CLI_PARSE, (cmd_proc)cmd_unset, "unset", 2, 2, "UnMesses with the internal variable store");
  cli_register(c->cli, CLI_PARSE, (cmd_proc)cmd_suspend, "suspend", 0, 1, "Suspends the pool, ie all clients");
  cli_register(c->cli, CLI_PARSE, (cmd_proc)cmd_resume, "resume", 0, 1, "Resumes normal operation");
  cli_register(c->cli, CLI_PARSE, (cmd_proc)cmd_loglevel, "loglevel", 1, 2, "Gets/Sets the loglevel");
  cli_register(c->cli, CLI_PARSE, (cmd_proc)cmd_quit, "quit", 0, 0, "Quits");
  return c;
}

int
console_destroy(console_t *c) {
  cli_destroy(c->cli);
  xfree(c);
  return 0;
}

int
console_main(console_t *c) {
  medussa_t *m;
  
  m = (medussa_t *)c->up;
  while(!console_done && cli_main(c->cli)!=1)
    ;
  return 0;
}
