//
// =================================================================
//
//  MacroMolecular Data Base (MMDB)
//
//  File  sel_exmp1.cpp
//
//  Example:
//     Selection of atoms and looking for contacts between on of
//  them and all others.
//
//  15 May 2001
//
//  SGI make:  f77 -o sel_exmp1 sel_exmp1.cpp mmdb.a -lm -lC
//
// =================================================================
//

#ifndef  __STRING_H
#include <string.h>
#endif

#ifndef  __MMDB_Manager__
#include "mmdb_manager.h"
#endif



void  PrintInstructions()  {
  printf ( 
    "\n"
    "A MacroMolecular Data Base (MMDB) example:\n"
    "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "Selecting atoms and looking for contacts between one of them\n"
    "and all others.\n"
    "\n"
    "Command line syntax:\n"
    "~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "sel_exmp1 coor_file [sel_coor_file]\n"
    "\n"
    "where 'coor_file' is a coordinate file in PDB, mmCIF or\n"
    "BINary format, [] means an optional parameter.\n"
    "\n"
    "If 'sel_coor_file' is given, it will receive selected\n"
    "atoms in PDB format.\n"
    "\n"
         );
}


int main ( int argc, char ** argv, char ** env )  {
PCMMDBManager  MMDBManager1,MMDBManager2;
int            RC,lcount;
char           S[500];
int            selHnd,nSelAtoms,i;
PPCAtom        SelAtom;
PSContact      contact;
int            ncontacts;

  if (argc<2)  {
    PrintInstructions();
    return 1;
  }

  //  1.  Make routine initializations, which must always be done
  //      before working with MMDB
  InitMatType();

  MMDBManager1 = NULL;  // all pointers should be either NULL or
  MMDBManager2 = NULL;  // allocated

  //  2.  Create an instance of MMDB
  MMDBManager1 = new CMMDBManager();

  //  3.  Read coordinate file.
  //    3.1 Set all necessary read flags -- check with the top of
  //        file  mmdb_file.h  as needed
  MMDBManager1->SetFlag ( MMDBF_PrintCIFWarnings |
                          MMDBF_IgnoreDuplSeqNum );

  //    3.2 MMDB provides the following function for reading
  //        coordinate files with automatic format recognition
  //        for PDB, mmCIF and MMDB BINary files:
  RC = MMDBManager1->ReadCoorFile ( argv[1] );

  //    3.3 Check for possible errors:
  if (RC) {
    //  An error was encountered. MMDB provides an error messenger
    //  function for easy error message printing.
    printf ( " ***** ERROR #%i READ:\n\n %s\n\n",RC,GetErrorDescription(RC) );
    //  Location of the error may be identified as precise as line
    //  number and the line itself (PDB only. Errors in mmCIF are
    //  located by category/item name. Errors of reading BINary files
    //  are not locatable and the files are not editable). This
    //  information is now retrieved from MMDB input buffer:
    MMDBManager1->GetInputBuffer ( S,lcount );
    if (lcount>=0) 
      printf ( "       LINE #%i:\n%s\n\n",lcount,S );
    else if (lcount==-1)
      printf ( "       CIF ITEM: %s\n\n",S );
    //  dispose instance of MMDB and quit:
    delete MMDBManager1;
    return 2;
  } else  {
    //  MMDB allows to identify the type of file that has been just
    //  read:
    switch (MMDBManager1->GetFileType())  {
      case MMDB_FILE_PDB    :  printf ( " PDB"         );  break;
      case MMDB_FILE_CIF    :  printf ( " mmCIF"       );  break;
      case MMDB_FILE_Binary :  printf ( " MMDB binary" );  break;
      default : printf ( " Unknown (report as a bug!)" );
    }
    printf ( " file %s has been read in.\n",argv[1] );
  }


  //  4.  Select atoms.
  //    4.1 Each _new_ selection starts with creation of selection
  //        handle (a handle may be used in several selections)
  selHnd = MMDBManager1->NewSelection();

  //    4.2 Select atoms as needed. Check the selection function
  //        and its parameters in file  mmdb_selmngr.h
  MMDBManager1->SelectAtoms ( selHnd, 0,"A",ANY_RES,"*",ANY_RES,"*","*",
                              "CA","C","*" );

  //    4.3 Selected atoms may be accessed through the selection
  //        index. Selection index is merely a vector of pointers
  //        on the selected atoms. Check the function and its
  //        parameters in file  mmdb_selmngr.h
  MMDBManager1->GetSelIndex ( selHnd, SelAtom,nSelAtoms );
  printf ( "     %i atoms selected.\n",nSelAtoms );
  if (nSelAtoms>0)  {
    if (SelAtom[0]->isInSelection(selHnd))
         printf ( " Atom '%s' is in selection\n",SelAtom[0]->name );
    else printf ( " Atom->isInSelection() failed.\n" );
  }

  //    4.4 Find contacts between one of the selected atoms and all
  //        others. Here we use dynamical allocation of contacts' index.
  //        See details and the function description in file
  //        mmdb_selmngr.h
  contact = NULL;  // this is a _must_ for dynamically allocated index
  MMDBManager1->SeekContacts ( SelAtom,nSelAtoms,33,0.0,10.0,0,contact,
                               ncontacts );
  printf ( "  %i contacts found:\n",ncontacts );
  //  print contacts. Check with description of the structure SContact
  //  in file mmdb_selmngr.h .
  for (i=0;i<ncontacts;i++)
    printf ( " /%i/%s/%i(%s).%1s/%s[%s]:%1s <->"
             " /%i/%s/%i(%s).%1s/%s[%s]:%1s %10.4f A\n",
             SelAtom[contact[i].id1]->GetModelNum(),
             SelAtom[contact[i].id1]->GetChainID(),
             SelAtom[contact[i].id1]->GetSeqNum(),
             SelAtom[contact[i].id1]->GetResName(),
             SelAtom[contact[i].id1]->GetInsCode(),
             SelAtom[contact[i].id1]->name,
             SelAtom[contact[i].id1]->element,
             SelAtom[contact[i].id1]->altLoc,
             SelAtom[contact[i].id2]->GetModelNum(),
             SelAtom[contact[i].id2]->GetChainID(),
             SelAtom[contact[i].id2]->GetSeqNum(),
             SelAtom[contact[i].id2]->GetResName(),
             SelAtom[contact[i].id2]->GetInsCode(),
             SelAtom[contact[i].id2]->name,
             SelAtom[contact[i].id2]->element,
             SelAtom[contact[i].id2]->altLoc,
       contact[i].dist );

  //  since the contact index was dynamically allocated,
  //  the application must dispose it when it is no longer needed:
  if (contact)  delete contact;



  if (argc>2)  {

    //  5. An output file for selected atoms was specified.
    //     Collect selected atoms in a separate instance of MMDB
    //     and then purge them into the file.

    //    5.1  Create one more instance of MMDB
    MMDBManager2 = new CMMDBManager();

    //    5.2  It looks like a good idea to copy the header and
    //         crystallographic information into output file
    //         as well. However this is not needed by MMDB to
    //         function. Check the copy function and its parameters
    //         in file  mmdb_manager.h .
    MMDBManager2->Copy ( MMDBManager1,MMDBFCM_Title | MMDBFCM_Cryst );

    //    5.3  Stuffing an empty MMDB with atoms. Check the
    //         corresponding function and its parameters in file
    //         mmdb_file.h .
    for (i=0;i<nSelAtoms;i++)
      MMDBManager2->PutAtom ( i+1,SelAtom[i] );

    //    5.4  Purge the newly composed MMDB into the file
    printf ( " ...writing PDB file %s\n",argv[2] );
    RC = MMDBManager2->WritePDBASCII( argv[2] );

  }

  //  6. A formally correct style of programming requires you
  //     to explicitely release any allocated memory before
  //     your application quits.
  //     NOTE: we do not dispose the selection index SelAtom.
  //     It will be disposed by its manager automatically,
  //     therefore an attempt to dispose it from the application
  //     will cause a crash.
  if (MMDBManager1)  delete MMDBManager1;
  if (MMDBManager2)  delete MMDBManager2;

  return 0;

}



