################################################################
# gnome.tcl - gnome interface to mmucl
# 
# Copyright (C) 2003-2004 Mark Patton
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
###################################################################

package require Gnome
package require Gdk
package require GdkPixbuf
package require Vte

# FIXME: much of this should be refactored...
# FIXME: add can_beep, audio and visual
# FIXME: save term size on exit and restore

# Strangely enough vte seems to need the \r

proc mmucl::display {ses str} {
    variable Mmucl

    if {$ses != $Mmucl(cur)} {
	session_tab_set_text $ses $Mmucl(ses,$ses,host) blue3
    }

    set str [string map {\n \r\n} $str]
    vte::terminal feed $Mmucl(ses,$ses,term) $str [string length $str]
}

proc mmucl::event {ses event args} {
    variable Mmucl

    set host $Mmucl(ses,$ses,host)

    switch -exact $event {
	attempt {
	    session_tab_set_text $ses $host yellow2
	} connect {
	    session_tab_set_text $ses $host green4
	    session_status $ses "connected: $host"
	    set Mmucl(ses,$ses,connect_time) [clock seconds]
	    after 60000 mmucl::update_connect_time $ses
	} disconnect {
	    session_tab_set_text $ses $host red3
	    session_status $ses disconnected
	} timeout {
	    session_tab_set_text $ses $host red3
	    session_status $ses disconnected
	} ses_new {
	    session_tab_add
	} ses_switch {
	    set i [gtk::notebook page_num $Mmucl(notebook) \
		    $Mmucl(notebook,page,$ses)]
	    gtk::notebook set_current_page $Mmucl(notebook) $i
	    gtk::widget grab_focus $Mmucl(input)
	    
	    if {$host ne ""} {
		if {$Mmucl(ses,$ses,connect)} {
		    session_tab_set_text $ses $host green4
		} else {
		    session_tab_set_text $ses $host red3
		}
	    }
	    
	    session_status $ses $Mmucl(ses,$ses,status)
	} ses_close {
	    lassign {old_ses} $args
	    session_tab_remove $old_ses
	} default {
	    error "event $event not understood"
	}
    }
}

proc mmucl::MCclear {ses args} {
    variable Mmucl

    set syntax {}
    check clear $syntax $args

    vte::terminal reset $Mmucl(ses,$ses,term) true false
}

proc mmucl::MCbell {ses args} {
    set syntax {}
    check bell $syntax $args

    gdk::gdk beep
}

proc mmucl::MCstatus {ses args} {
    variable Mmucl

    set syntax {{+ text}}
    check status $syntax $args

    set Mmucl(ses,$ses,user_status) $arg(text)
    session_status $ses $Mmucl(ses,$ses,status)
}

proc mmucl::MChelp {ses args} {
    variable Mmucl

    set syntax {{? node ""}}
    check help $syntax $args

    if {![gnome::url show info:mmucl err]} {
	error "displaying help: [lindex $err 2]"
    }
}

proc mmucl::session_status {ses msg} {
    variable Mmucl

    set Mmucl(ses,$ses,status) $msg

    if {$ses == $Mmucl(cur)} {
	gtk::window set_title $Mmucl(app) "Mmucl - $msg"

	if {$Mmucl(ses,$ses,connect_duration) ne ""} {
	    append msg " \[" $Mmucl(ses,$ses,connect_duration) \]
	}

	if {$Mmucl(ses,$ses,user_status) ne ""} {
	    append msg " \{" $Mmucl(ses,$ses,user_status) \}
	}

	gnome::appbar set_status $Mmucl(statusbar) $msg
    }
}

proc mmucl::update_connect_time {ses} {
    variable Mmucl

    if {[info exists Mmucl(ses,$ses,connect)] && $Mmucl(ses,$ses,connect)} {
	set host $Mmucl(ses,$ses,host)
	set time [expr {[clock seconds] - $Mmucl(ses,$ses,connect_time)}]
	set Mmucl(ses,$ses,connect_duration) [duration $time]

	session_status $ses $Mmucl(ses,$ses,status)
	after 60000 mmucl::update_connect_time $ses
    }
}

proc mmucl::input_scroll_up {} {
    variable Mmucl

    if {$Mmucl(hist_loc)} {
	if {$Mmucl(hist_loc) == [llength $Mmucl(history)]} {
	    set Mmucl(hist_cur) [gtk::entry get_text $Mmucl(input)]
	}	
	gtk::entry set_text $Mmucl(input) \
		[lindex $Mmucl(history) [incr Mmucl(hist_loc) -1]]
    }

    return
}

proc mmucl::input_scroll_down {} {
    variable Mmucl
    
    if {[incr Mmucl(hist_loc)] > [llength $Mmucl(history)]} {
	incr Mmucl(hist_loc) -1
    } else {
	if {$Mmucl(hist_loc) == [llength $Mmucl(history)]} {
	    gtk::entry set_text $Mmucl(input) $Mmucl(hist_cur)
	} else {
	    gtk::entry set_text $Mmucl(input) \
		    [lindex $Mmucl(history) $Mmucl(hist_loc)]
	}
    }

    return
}

# Map from GdkModifierType to mmucl standard names
# Also map from some different keys from Gtk  reports to what Tk reports

set mmucl::modmap {
    control-mask  Control-
    mod1-mask     Alt-
    mod2-mask     Mod2-
    mod3-mask     Mod3-
    mod4-mask     Mod4-
    mod5-mask     Mod5-
    button1-mask  Button1-
    button2-mask  Button2-
    button3-mask  Button3-
    button3-mask  Button4-
    button3-mask  Button5-
    lock-mask     CapsLock-
    shift-mask    Shift-
    modifier-mask ""
    release-mask  ""
    Page_Up       Prior
    Page_Down     Next
}

proc mmucl::key_id {mod key} {
    variable modmap

    return [string map $modmap [append id [join $mod ""] $key]]
}

# FIXME: break and continue?

# handle a key press event
# FIXME: should query terminal rows for values on scrolling.
# FIXME per session?
proc mmucl::key_event {event} {
    variable Mmucl

    set id [key_id [gdk::event get_keymod $event] \
		[gdk::event get_keyval $event]]

    set ses $Mmucl(cur)
    set Mmucl(ses,$ses,current_key) $id

    key_eval Key

    if {[key_eval $id]} {
	return 1
    }

    switch -exact $id {
	Down {
	    input_scroll_down
	    return 1
	} Up {
	    input_scroll_up
	    return 1
	} Prior {
	    gtk::adjustment set_value $Mmucl(ses,$ses,termadj) \
             [expr {[gtk::adjustment get_value $Mmucl(ses,$ses,termadj)] - 25}]
	} Next {
	    gtk::adjustment set_value $Mmucl(ses,$ses,termadj) \
	     [expr {[gtk::adjustment get_value $Mmucl(ses,$ses,termadj)] + 25}]
	} Home {
	    gtk::adjustment set_value $Mmucl(ses,$ses,termadj) 0
	} End {
	    gtk::adjustment set_value $Mmucl(ses,$ses,termadj) \
		    $Mmucl(pref,scrollback)
	}
    }
	
    return 0
}

proc mmucl::key_eval {id} {
    variable Mmucl

    set ses $Mmucl(cur)
    set x ses,$ses,group

    if {[info exists Mmucl($x,forkey,key,$id)]} {
	foreach group $Mmucl($x,forkey,key,$id) {
	    set code [catch {$ses eval $Mmucl($x,val,$group,key,$id)} error]
    
	    if {$code == 1} {
		report $ses error "key $id: $error"
	    }
	}

	return 1
    }
    
    return 0
}

proc mmucl::MCcline {cur args} {
    variable Mmucl

    set w $Mmucl(input)

    set syntax {
	delete {{? first 0} {? last end}}
	get {{? first 0} {? last end}} 
	insert {{+ first} {+ str}}
	history {}
	hide {{? bool 1}}
    }

    set opt [check cline $syntax $args 1]

    foreach index {arg(first) arg(last)} {
	if {![info exists $index]} {
	    continue
	}
	set val [set $index]

	if {[string equal $val insert]} {
	    set $index [gtk::editable get_position $w]
	} elseif {[string equal $val end]} {
	    set $index -1
	} elseif {[string is int -strict $val]} {
	} else {
	    error "bad index: $val"
	}
    }

    switch -exact $opt {
        delete {
	    gtk::editable delete_text $w $arg(first) $arg(last)
	} get {
	    return [gtk::editable get_chars $w $arg(first) $arg(last)]
	} insert {
	    gtk::editable insert_text $w $arg(str) $arg(first)
	    gtk::editable set_position $w \
		    [expr {$arg(first) + [string length $arg(str)]}]
	} history {
	    return $Mmucl(history)
	} hide {
	    # FIXME: doesn't seem to work
	    gtk::entry set_visibility $w $arg(bool)
	}
    }

    return
}

proc mmucl::handle_input {w} {
    variable Mmucl

    meta_parse [gtk::entry get_text $w]

    if {$Mmucl(cfg,keep_line)} {
	gtk::editable select_region $w 0 -1
    } else {
	gtk::entry set_text $w ""
    }
}

proc mmucl::bind {w signal script} {
    g::signal connect $w $signal [namespace code $script]
}

# Create a submenu for a menuitem name.

proc submenu {name arglist accel_group} {
    set head [gtk::menu_item new_with_mnemonic $name]
    set menu [menu $arglist $accel_group]
    gtk::menu_item set_submenu $head $menu

    gtk::menu set_accel_group $menu $accel_group
    gtk::menu set_accel_path $menu <Mmucl>/$name

    return $head
}

proc submenu_dynamic {name script} {
    set head [gtk::menu_item new_with_mnemonic $name]

    _menu_item_dynamic_menu $head $script
    g::signal connect $head enter-notify-event \
	    [list _menu_item_dynamic_menu $head $script]

    return $head
}

proc menu {arglist accel_group} {
    set menu [gtk::menu new]

    foreach {type def} $arglist {
	gtk::menu_shell append $menu \
		[menu_item $type $def $accel_group]
    }

    return $menu
}

proc _menu_item_check_set_var {mi varname} {
    upvar 1 $varname val
    set val [g::object get $mi active]
}

proc _menu_item_dynamic_menu {mi script} {
    gtk::menu_item set_submenu $mi [eval $script]
    gtk::widget show_all $mi
    return 0
}

proc menu_item {type arglist accel_group} {
    switch -exact $type {
	button {
	    lassign {name script} $arglist

	    set mi [gtk::menu_item new_with_mnemonic $name]
	    g::signal connect $mi activate $script
	} stock {
	    lassign {name script} $arglist

	    set mi [gtk::image_menu_item new_from_stock $name $accel_group]
	    g::signal connect $mi activate $script
	} menu {
	    lassign {name script} $arglist
	    
	    set mi [gtk::menu_item new_with_mnemonic $name]
	    gtk::menu_item set_submenu $mi [eval $script]
	} menu_dynamic {
	    lassign {name script} $arglist

	    set mi [gtk::menu_item new_with_mnemonic $name]
	    _menu_item_dynamic_menu $mi $script
	    g::signal connect $mi enter-notify-event \
		    [list _menu_item_dynamic_menu $mi $script]
	} check {
	    lassign {name varname script} $arglist
	    upvar 1 $varname val
	    
	    set mi [gtk::check_menu_item new_with_mnemonic $name]
	    g::object set $mi active $val
	    
	    g::signal connect $mi activate \
		    [list _menu_item_check_set_var $mi $varname]
	    g::signal connect $mi activate $script
	} sep {
	    set mi [gtk::separator_menu_item new]
	} default {
	    error "unknown type: $type"
	}
    }

    return $mi
}


# layout widgets in a grid
# widgets are in {{row1} {row2} ...}

proc mmucl::layout_table {arglist widgets} {
    set cols [llength $widgets]
    set rows [llength [lindex $widgets 0]]

    lappend arglist n-rows [incr rows] n-columns [incr cols]
    set table [g::new GtkTable $arglist]

    set row 0
    foreach widget_row $widgets {
	set col 0
	foreach w $widget_row {
	    if {$w ne ""} {
		set al [gtk::alignment new 0.0 0.5 0.0 0.0]
		gtk::container add $al $w
		gtk::table attach $table $al \
			$col [expr {$col + 1}] $row [expr {$row + 1}] \
			{fill} {} 0 0
	    }
	    incr col
	}
	incr row
    }

    return $table
}

proc mmucl::vbox {arglist args} {
    set vbox [g::new GtkVBox $arglist]

    foreach w $args {
	gtk::box pack_start $vbox $w 0 0 0
    }

    return $vbox
}

proc mmucl::hbox {arglist args} {
    set hbox [g::new GtkHBox $arglist]

    foreach w $args {
	gtk::box pack_start $hbox $w 0 0 0
    }

    return $hbox
}

proc mmucl::error_dialog {parent msg} {
    set w [gtk::message_dialog new $parent modal error ok $msg]
    gtk::dialog run $w
    gtk::widget destroy $w
}

proc mmucl::handle_script {parent err_msg script {print_stack_trace 0}} {
    variable Mmucl

    if {[catch $script error]} {
	if {$print_stack_trace} {
	    report $Mmucl(cur) error "$error\nStack trace:\n$::errorInfo"
	}
	error_dialog $parent "$err_msg: $error"
	return 0
    }

    return 1
}

proc mmucl::gnome_quit {} {
    variable Mmucl

    foreach ses $Mmucl(sessions) {
	if {$Mmucl(ses,$ses,connect)} {
	    set w [gtk::message_dialog new $Mmucl(app) modal question \
		   ok-cancel "Session $ses is still connected.\n Really quit?"]
	    set response [gtk::dialog run $w]

	    switch -exact -- $response {
		ok {
		} cancel {
		    gtk::widget destroy $w
		    return
		} delete-event {
		    gtk::widget destroy $w
		    return
		} default {
		    error "response not understood: $response"
		}
	    }
	}
    }

    MCexit $Mmucl(cur)
}

proc mmucl::gnome_session_close {} {
    variable Mmucl

    set ses $Mmucl(cur)

    if {$Mmucl(ses,$ses,connect)} {
	set w [gtk::message_dialog new $Mmucl(app) modal question ok-cancel \
		"Session $ses is still connected.\nReally close session?"]
	set response [gtk::dialog run $w]

	switch -exact -- $response {
	    ok {
	    } cancel {
		gtk::widget destroy $w
		return
	    } delete-event {
		gtk::widget destroy $w
		return
	    } default {
		error "response not understood: $response"
	    }
	}
    }

    if {[llength $Mmucl(sessions)] == 1} {
	MCexit $ses
    } else {
	MCsession $ses close $ses
    }
}

proc mmucl::connect_dialog_attempt {ehost eport} {
    variable Mmucl

    if {[g::object get $Mmucl(dialog,connect,mchar) sensitive]} {
	set i [gtk::option_menu get_history $Mmucl(dialog,connect,mchar)]

	if {$i == -1} {
	    error_dialog $Mmucl(dialog,connect) "No characters defined."
	    return
	}

	set name [lindex [MCchar $Mmucl(cur) names] $i]

	set res [handle_script $Mmucl(dialog,connect) "loading $name" \
		[list MCchar $Mmucl(cur) load $name] 1]

    } else {
	set host [gtk::entry get_text $ehost]
	set port [gtk::entry get_text $eport]
	
	set res [handle_script $Mmucl(dialog,connect) "connecting to $host" \
		[list MCconnect $Mmucl(cur) $host $port]]
    }

    if {$res} {
	gtk::widget hide $Mmucl(dialog,connect)
    }
}

proc mmucl::connect_dialog_char_changed {ehost eport} {
    variable Mmucl

    if {![g::object get $Mmucl(dialog,connect,mchar) sensitive]} {
	return
    }

    set i [gtk::option_menu get_history $Mmucl(dialog,connect,mchar)]
	
    if {$i != -1} {
	set name [lindex [MCchar $Mmucl(cur) names] $i]

	lassign {host port login} [MCchar $Mmucl(cur) set $name]

	gtk::entry set_text $ehost $host
	gtk::entry set_text $eport $port
    }
}

proc mmucl::char_menu {} {
    variable Mmucl

    set menu [gtk::menu new]
    foreach name [MCchar $Mmucl(cur) names] {
	set mi [gtk::menu_item new_with_label $name]
	gtk::menu_shell append $menu $mi
    }

    return $menu
}

proc mmucl::snoop_menu_toggled {mi ses} {
    variable Mmucl

    MCsession $Mmucl(cur) snoop $ses [g::object get $mi active]
}

proc mmucl::snoop_menu {} {
    variable Mmucl

    set menu [gtk::menu new]
    foreach ses $Mmucl(sessions) {
	set mi [gtk::check_menu_item new_with_label \
		"$ses: $Mmucl(ses,$ses,host)"]
	g::object set $mi active [MCsession $Mmucl(cur) snoop $ses]
	bind $mi toggled [list snoop_menu_toggled $mi $ses]
	gtk::menu_shell append $menu $mi
    }

    return $menu
}

proc mmucl::connect_dialog_toggle {check_char ehost eport} {
    variable Mmucl

    set show_char [g::object get $check_char active]
    g::object set $Mmucl(dialog,connect,mchar) sensitive $show_char
    g::object set $ehost editable [expr {!$show_char}]
    g::object set $eport editable [expr {!$show_char}]

    connect_dialog_char_changed $ehost $eport
}

proc mmucl::connect_dialog {} {
    variable Mmucl
    
    if {[info exists Mmucl(dialog,connect)]} {
	gtk::option_menu set_menu $Mmucl(dialog,connect,mchar) [char_menu]
	gtk::widget show_all $Mmucl(dialog,connect)
	gtk::window present $Mmucl(dialog,connect)
	return
    }

    set Mmucl(dialog,connect) [g::new GtkWindow type toplevel \
	    title "Mmucl - Connect to host" "window-position" mouse]

    bind $Mmucl(dialog,connect) delete_event {
	gtk::widget hide $Mmucl(dialog,connect)
	return true
    }
    
    set vbox [g::new GtkVBox border-width 5]
    gtk::container add $Mmucl(dialog,connect) $vbox

    set frame [gtk::frame new ""]

    set lhost [gtk::label new Host:]
    set ehost [gtk::entry new]
    set lport [gtk::label new Port:]    
    set eport [gtk::entry new]
    set check_char [gtk::check_button new_with_label Char:]    
    set Mmucl(dialog,connect,mchar) [g::new GtkOptionMenu sensitive false \
	    menu [char_menu]]

    bind $check_char toggled \
	    [list connect_dialog_toggle $check_char $ehost $eport]
    bind $Mmucl(dialog,connect,mchar) changed \
	    [list connect_dialog_char_changed $ehost $eport]

    bind $ehost activate [list connect_dialog_attempt $ehost $eport]
    bind $eport activate [list connect_dialog_attempt $ehost $eport]

    set layout [list \
	    [list $check_char $Mmucl(dialog,connect,mchar)] \
	    [list $lhost $ehost] \
	    [list $lport $eport]]

    gtk::container add $frame [layout_table \
	    {homogeneous false border-width 10 row-spacing 5 \
	    column-spacing 10} $layout]

    gtk::box pack_start_defaults $vbox $frame
    
    set hbox [gtk::hbox new false 5]
    set b [gtk::button new_from_stock gtk-ok]
    bind $b clicked [list connect_dialog_attempt $ehost $eport]
    gtk::box pack_end $hbox $b 0 0 0

    set b [gtk::button new_from_stock gtk-cancel]
    bind $b clicked {
	gtk::widget hide $Mmucl(dialog,connect)
    }
    gtk::box pack_end $hbox $b 0 0 0
    gtk::box pack_start $vbox $hbox 0 0 5

    gtk::widget show_all $Mmucl(dialog,connect)
    gtk::widget grab_focus $ehost
}

proc mmucl::listview {colname} {
    set model [gtk::list_store new [g::type from_name gchararray]]

    set view [gtk::tree_view new_with_model $model]
    g::object unref $model

    set renderer [gtk::cell_renderer_text new]
    set col [g::new GtkTreeViewColumn title $colname sort-indicator true]
    gtk::tree_view_column pack_start $col $renderer false
    gtk::tree_view_column add_attribute $col $renderer text 0
    gtk::tree_view_column set_sort_column_id $col 0
    gtk::tree_view append_column $view $col

    return [list $model $view]
}

proc mmucl::user_dialog_refresh  {name} {
    variable Mmucl

    set model $Mmucl(dialog,$name,model)
    gtk::list_store clear $model

    set iter [gtk::tree_iter new]

    foreach name [MC$name $Mmucl(cur) names] {
	gtk::list_store append $model $iter
	gtk::list_store set $model $iter 0 $name
    }

    gtk::tree_iter free $iter
}

proc mmucl::user_dialog_add {name editables} {
    variable Mmucl

    foreach e $editables {
	lappend vals [gtk::entry get_text $e]
    }

    if {[llength $vals] == 2} {
	set value [lindex $vals 1]
    } else {
	set value [lrange $vals 1 end]
    }

    handle_script $Mmucl(dialog,$name) "setting $name" \
	    [list MC$name $Mmucl(cur) set [lindex $vals 0] $value]
    
    user_dialog_refresh $name
}

proc mmucl::user_dialog_remove {name ename} {
    variable Mmucl

    set key [gtk::entry get_text $ename]

    MC$name $Mmucl(cur) delete -exact $key
    user_dialog_refresh $name
}

proc mmucl::user_dialog_select {name sel view model editables} {
    variable Mmucl
   
    set iter [gtk::tree_iter new]

    if {[gtk::tree_selection get_selected $sel $iter]} {
	set key [gtk::tree_model get $model $iter 0]
	gtk::entry set_text [lindex $editables 0] $key

	set vals [MC$name $Mmucl(cur) set $key]

	if {[llength $editables] == 2} {
	    set vals [list $vals]
	}

	foreach e [lrange $editables 1 end] val $vals {
	    gtk::entry set_text $e $val
	}
    }

    gtk::tree_iter free $iter
}

proc mmucl::user_dialog {name title values} {
    variable Mmucl

    if {[info exists Mmucl(dialog,$name)]} {
	user_dialog_refresh $name
	gtk::window present $Mmucl(dialog,$name)
	return
    }

    set w [g::new GtkWindow default-width 500 default-height 300 \
	    type toplevel title "Mmucl - $title" window-position mouse]
    set Mmucl(dialog,$name) $w

    bind $w delete_event "
	gtk::widget hide \$Mmucl(dialog,$name)
	return true
    "

    set main [gtk::hbox new false 5]
    lassign {model view} [listview [lindex $values 0]]

    set Mmucl(dialog,$name,model) $model

    set sw [gtk::scrolled_window new null null]
    gtk::container add $sw $view
    gtk::box pack_start $main $sw 1 1 0

    set vbox [gtk::vbox new false 5]
    gtk::box pack_start $main $vbox 0 0 0

    foreach val $values {
	set label [gtk::label new $val:]    
	set edit [gtk::entry new]

	lappend editables $edit
	lappend layout [list $label $edit]
    }

    set badd [gtk::button new_from_stock gtk-add]
    bind $badd clicked [list user_dialog_add $name $editables]

    set brem [gtk::button new_from_stock gtk-remove]
    bind $brem clicked [list user_dialog_remove $name [lindex $editables 0]]

    lappend layout [list "" [hbox {border-width 5} $badd $brem]]

    set sel [gtk::tree_view get_selection $view]
    bind $sel changed \
	    [list user_dialog_select $name $sel $view $model $editables]

    foreach edit $editables {
	bind $edit activate [list user_dialog_add $name $editables]
    }

    gtk::box pack_start_defaults $vbox [layout_table \
	    {homogeneous false border-width 10 column-spacing 10} $layout]
    
    set hbox [gtk::hbox new false 5]
    set b [gtk::button new_from_stock gtk-close]
    bind $b clicked [list gtk::widget hide $Mmucl(dialog,$name)]
    
    gtk::box pack_end $hbox $b 0 0 0

    set b [gtk::button new_from_stock gtk-help]
    #bind $b clicked [list user_dialog_refresh $name]
    gtk::box pack_end $hbox $b 0 0 0

    gtk::box pack_start $vbox $hbox 0 0 0

    gtk::container add $Mmucl(dialog,$name) $main

    user_dialog_refresh $name
    gtk::widget show_all $Mmucl(dialog,$name)
    gtk::widget grab_focus [lindex [lindex $layout 0] 1]
}

proc mmucl::update_view {name} {
    variable Mmucl

    set w $Mmucl($name)
    
    if {$name eq "toolbar" || $name eq "menubar"} {
	set w [gtk::widget get_parent $w]
    }
    
    if {$Mmucl(pref,view,$name)} {
	gtk::widget show $w
    } else {
	gtk::widget hide $w
    }
}

proc mmucl::term_pref_update_color {arglist which} {
    variable Mmucl

    lassign {w r g b a} $arglist
    set color [list $r $g $b]

    if {$which == 16} {
	set Mmucl(pref,term,fg) $color
    } elseif {$which == 17} {
	set Mmucl(pref,term,bg) $color
    } else {
	lset Mmucl(pref,term,palette) $which $color
    }

    foreach ses $Mmucl(sessions) {
	vte::terminal set_colors  $Mmucl(ses,$ses,term) \
	    $Mmucl(pref,term,fg) $Mmucl(pref,term,bg) \
	    $Mmucl(pref,term,palette)
    }
}

proc mmucl::term_pref_update_font {gf check_bold} {
    variable Mmucl

    set Mmucl(pref,use_bold) [g::object get $check_bold active]
    set Mmucl(pref,font_name) [gnome::font_picker get_font_name $gf]

    foreach ses $Mmucl(sessions) {
	vte::terminal set_font_from_string $Mmucl(ses,$ses,term) \
	    $Mmucl(pref,font_name)
	vte::terminal set_allow_bold $Mmucl(ses,$ses,term) \
	    $Mmucl(pref,use_bold)
    }
}

proc mmucl::term_set_background {term imagefile transparent saturation} {
    if {$imagefile eq ""} {
	vte::terminal set_background_image $term null
    } else {
	vte::terminal set_background_image_file $term $imagefile
    }

    vte::terminal set_background_transparent $term $transparent
    vte::terminal set_background_saturation $term $saturation
}

# Called when a background value in the term pref dialog changes
proc mmucl::term_pref_update_bgtype {gpe r_normal r_image r_transparent sat} {
    variable Mmucl

    foreach v {normal image transparent} {
	if {[g::object get [set r_$v] active]} {
	    set Mmucl(pref,bg,type) $v
	}
    }

    gtk::widget set_sensitive $gpe [expr {$Mmucl(pref,bg,type) eq "image"}]
    gtk::widget set_sensitive $sat [expr {$Mmucl(pref,bg,type) ne "normal"}]

    set file ""
    set trans 0

    switch -exact $Mmucl(pref,bg,type) {
	normal {
	} image {
	    set file [gtk::entry get_text [gnome::file_entry gtk_entry $gpe]]
	} transparent {
	    set trans 1
	} default {
	    error "bad bg,type"
	}
    }

    set Mmucl(pref,bg,image) $file
    set Mmucl(pref,bg,transparent) $trans
    set Mmucl(pref,bg,saturation) [gtk::range get_value $sat]

    foreach ses $Mmucl(sessions) {
	mmucl::term_set_background $Mmucl(ses,$ses,term) \
	    $Mmucl(pref,bg,image) $Mmucl(pref,bg,transparent) \
	    $Mmucl(pref,bg,saturation)
    }
}

# gcs is palette followed by fg and bg  gnome color selector widgets

proc mmucl::term_pref_color_gcs {gcs} {
    variable Mmucl

    set colors $Mmucl(pref,term,palette)
    lappend colors $Mmucl(pref,term,fg) $Mmucl(pref,term,bg)

    foreach gc $gcs color $colors {
	lassign {r g b} $color
	gnome::color_picker set_i16 $gc $r $g $b 0
    }
}

proc mmucl::term_pref_reset_color {gcs} {
    variable Mmucl

    set Mmucl(pref,term,palette) $Mmucl(term,palette)
    set Mmucl(pref,term,fg) $Mmucl(term,fg)
    set Mmucl(pref,term,bg) $Mmucl(term,bg)

    term_pref_color_gcs $gcs

    foreach ses $Mmucl(sessions) {
	vte::terminal set_colors  $Mmucl(ses,$ses,term) \
	    $Mmucl(pref,term,fg) $Mmucl(pref,term,bg) \
	    $Mmucl(pref,term,palette)
    }
}

proc mmucl::term_pref_dialog {} {
    global term_color config
    variable Mmucl

    if {[info exists Mmucl(term_pref)]} {
	gtk::window present $Mmucl(term_pref)
	return
    }

    set Mmucl(term_pref) [g::new GtkWindow type toplevel \
	    title "Mmucl - Terminal preferences" "window-position" mouse]

    bind $Mmucl(term_pref) delete_event {
	gtk::widget hide $Mmucl(term_pref)
	return true
    }

    set vbox [g::new GtkVBox spacing 5 border-width 5]
    set frame [gtk::frame new Font]

    set gf [g::new GnomeFontPicker mode font-info show-size true \
	    use-font-in-label true font-name $Mmucl(pref,font_name)]
    set w [g::new GtkCheckButton label "Use bold" active $Mmucl(pref,use_bold)]

    bind $gf font-set [list term_pref_update_font $gf $w]
    bind $w toggled [list term_pref_update_font $gf $w]

    set layout [list \
	    [list $gf] \
	    [list $w] \
	    ]

    gtk::container add $frame [layout_table \
	    {homogeneous false border-width 10 column-spacing 10} $layout]

    gtk::box pack_start_defaults $vbox $frame

    set frame [gtk::frame new Color]

    set gc_fg [gnome::color_picker new]
    set gc_bg [gnome::color_picker new]

    bind $gc_fg color-set {term_pref_update_color $_ 16} 
    bind $gc_bg color-set {term_pref_update_color $_ 17}

    set hbox_fg [gtk::hbox new 0 0]
    set hbox_bg [gtk::hbox new 0 0]

    for {set i 0} {$i < 8} {incr i} {
	set gc [gnome::color_picker new]
	bind $gc color-set "term_pref_update_color \$_ $i" 
	lappend gcs $gc
	gtk::box pack_start_defaults $hbox_fg $gc
    }

    for {set i 8} {$i < 16} {incr i} {
	set gc [gnome::color_picker new]
	bind $gc color-set "term_pref_update_color \$_ $i" 
	gtk::box pack_start_defaults $hbox_bg $gc
	lappend gcs $gc
    }
    
    lappend gcs $gc_fg $gc_bg 

    term_pref_color_gcs $gcs

    set w [gtk::button new_with_label "Reset"]
    bind $w clicked [list term_pref_reset_color $gcs]
    
    set layout [list \
	    [list $w] \
	    [list [g::new GtkLabel label "Text:"] $gc_fg] \
	    [list [g::new GtkLabel label "Background:"] $gc_bg] \
	    [list [g::new GtkLabel label "Pallete:"] $hbox_fg] \
	    [list "" $hbox_bg] \
	    ]

    gtk::container add $frame [layout_table \
	    {border-width 10 column-spacing 10} $layout]
    gtk::box pack_start_defaults $vbox $frame


    set frame [gtk::frame new "Background type"]
    set r1 [g::new GtkRadioButton label Normal]
    set r2 [g::new GtkRadioButton label Image group $r1]
    set r3 [g::new GtkRadioButton label Transparent group $r1]
    set sep [g::new GtkHSeparator]

    g::object set $r1 active [expr {$Mmucl(pref,bg,type) eq "normal"}]
    g::object set $r2 active [expr {$Mmucl(pref,bg,type) eq "image"}]
    g::object set $r3 active [expr {$Mmucl(pref,bg,type) eq "transparent"}]

    set sat_label [gtk::label new "Shade transparent or image background"]
    set sat [gtk::hscale new_with_range 0.0 1.0 0.05]
    gtk::range set_value $sat $Mmucl(pref,bg,saturation)
    gtk::widget set_sensitive $sat [expr {$Mmucl(pref,bg,type) ne "normal"}]

    set w [gnome::pixmap_entry new id "Background image" false]
    gnome::file_entry set_filename $w $Mmucl(pref,bg,image)

    gtk::widget set_sensitive $w [expr {$Mmucl(pref,bg,type) eq "iamge"}]

    set script [list term_pref_update_bgtype $w $r1 $r2 $r3 $sat]
    
    bind $r1 toggled $script
    bind $r2 toggled $script
    bind $r3 toggled $script
    bind $sat value-changed $script
    bind $w activate $script

    gtk::container add $frame [vbox {border-width 10 spacing 2} \
				 $r1 $r2 $w $r3 $sep [hbox {} $sat_label] $sat]
    gtk::box pack_start_defaults $vbox $frame

    set hbox [gtk::hbox new 0 10]

    set w [gtk::button new_from_stock gtk-close]
    bind $w clicked [list gtk::widget hide $Mmucl(term_pref)]

    gtk::box pack_end $hbox $w 0 0 0
    gtk::box pack_start $vbox $hbox 0 0 0

    gtk::container add $Mmucl(term_pref) $vbox
    gtk::widget show_all $Mmucl(term_pref)
}

proc mmucl::file_dialog {name title} {
    variable Mmucl

    set w [g::new GtkFileSelection title "Mmucl - $title"]
    set response [gtk::dialog run $w]

    switch -exact -- $response {
	ok {
	    set file [gtk::file_selection get_filename $w]
	    
	    if {$name eq "load"} {
		set script [list $Mmucl(cur) invokehidden source $file]
		set err_msg "loading $file"
	    } elseif {$name eq "save"} {
		set script [list MCdump $Mmucl(cur) -all -- $file] 
		set err_msg "saving session state to $file"
	    } else {
		set script [list MCtextin $Mmucl(cur) $file] 
		set err_msg "sending $file to mud"
	    }
	    
	    handle_script $w $err_msg $script
	} cancel {
	} delete-event {
	} default {
	    error "response not understood: $response"
	}
    }

    gtk::widget destroy $w
}

proc mmucl::toolbar {arglist} {
    set toolbar [gtk::toolbar new]

    foreach {stockid tooltip verbose script} $arglist {
	set b [gtk::toolbar insert_stock $toolbar $stockid \
		$tooltip $verbose -1]
	bind $b clicked $script
    }

    return $toolbar
}

proc mmucl::session_tab_add {} {
    variable Mmucl

    set ses $Mmucl(cur)

    set Mmucl(ses,$ses,user_status) ""
    set Mmucl(ses,$ses,connect_duration) ""

    set term [set Mmucl(ses,$ses,term) [vte::terminal new]]

    vte::terminal set_font_from_string $term $Mmucl(pref,font_name)
    vte::terminal set_allow_bold $term $Mmucl(pref,use_bold)
    vte::terminal set_scrollback_lines $term $Mmucl(pref,scrollback)
    vte::terminal set_cursor_blinks $term false
    vte::terminal set_scroll_on_output $term false

    mmucl::term_set_background $Mmucl(ses,$ses,term) \
	$Mmucl(pref,bg,image) $Mmucl(pref,bg,transparent) \
	$Mmucl(pref,bg,saturation)

    set hbox [g::new GtkHBox]

    set Mmucl(ses,$ses,termadj) [vte::terminal get_adjustment $term]
    set sb [gtk::vscrollbar new $Mmucl(ses,$ses,termadj)]

    set l [g::new GtkLabel label "<b>$Mmucl(cur)</b>:" use-markup 1]
    gtk::box pack_start $hbox $term 1 1 0
    gtk::box pack_start $hbox $sb 0 0 0
    gtk::notebook append_page $Mmucl(notebook) $hbox $l

    set Mmucl(notebook,page,$Mmucl(cur)) $hbox
    set Mmucl(notebook,tab,$Mmucl(cur)) $l
    set Mmucl(notebook,ses,$hbox) $Mmucl(cur)

    gtk::widget show_all $Mmucl(notebook)

    gtk::notebook set_current_page $Mmucl(notebook) \
	    [gtk::notebook page_num $Mmucl(notebook) $hbox]

    # see coment for other set_show_tabs
    gtk::notebook set_show_tabs $Mmucl(notebook) 1

    gtk::widget realize $term

    vte::terminal set_colors $term \
	$Mmucl(pref,term,fg) $Mmucl(pref,term,bg) \
	$Mmucl(pref,term,palette)

    gtk::widget grab_focus $Mmucl(input)
    session_status $Mmucl(cur) disconnected

    bind $term button-press-event {
	return [popup_menu [lindex $_ 1]]
    }
}

proc mmucl::session_tab_set_text {ses msg {color ""}} {
    variable Mmucl

    append markup <b> $ses "</b>: "

    if {$color eq ""} {
	append markup $msg
    } else {
	append markup "<span color=\"" $color \"> $msg </span>
    }
    
    gtk::label set_markup $Mmucl(notebook,tab,$ses) $markup
    gtk::widget queue_draw $Mmucl(notebook) 
}

proc mmucl::session_tab_remove {ses} {
    variable Mmucl

    set page $Mmucl(notebook,page,$ses)
    set i [gtk::notebook page_num $Mmucl(notebook) $page]

    unset Mmucl(notebook,page,$ses) Mmucl(notebook,tab,$ses) \
	    Mmucl(notebook,ses,$page)

    gtk::notebook remove_page $Mmucl(notebook) $i

    if {[llength $Mmucl(sessions)] == 1} {
	gtk::notebook set_show_tabs $Mmucl(notebook) false
    }

    gtk::widget grab_focus $Mmucl(input)
}

proc mmucl::loadimage {file} {
    set pixbuf [gdk::pixbuf new_from_file $file err]
	
    if {[g::handle is_null $pixbuf]} {
    	error "loading $file: [lindex $err 2]"
    }

    return $pixbuf
}

proc mmucl::loadstock {dir arglist} {
    set fact [gtk::icon_factory new]
    gtk::icon_factory add_default $fact

    foreach {name id img mod key} $arglist {
	set pb [loadimage [file join $dir $img]]
	set set [gtk::icon_set new_from_pixbuf $pb]
	gtk::icon_factory add $fact $id $set
	gtk::stock add $id $name $mod $key
    }
}

proc mmucl::about_dialog {} {
    global config
    variable Mmucl

    if {[info exists Mmucl(dialog,about)]} {
	gtk::window present $Mmucl(dialog,about)
	return
    }

    set logo null    
    #set logo [loadimage [file join $config(lib_dir) images logo.png]]

    set cr "Copyright [format %c 0xA9] 1997-2001 Mark Patton"
    set com "http://mmucl.sourceforge.net"
    set auth  {"Mark Patton"}
    set vers $config(version)

    set Mmucl(dialog,about) \
	    [gnome::about new Mmucl $vers $cr $com $auth $auth "" $logo]
    bind $Mmucl(dialog,about) destroy {unset -nocomplain Mmucl(dialog,about)}
    gtk::widget show $Mmucl(dialog,about)
}

proc mmucl::popup_menu {ev} {
    variable Mmucl
   
    set b [gdk::event get_button $ev]

    if {$b != 3} {
	return false
    }

    set menu {
	check {Menubar mmucl::Mmucl(pref,view,menubar) \
		{mmucl::update_view menubar}}
	check {Toolbar mmucl::Mmucl(pref,view,toolbar) \
		{mmucl::update_view toolbar}}
	check {"Status bar" mmucl::Mmucl(pref,view,statusbar) \
		{mmucl::update_view statusbar}}
    }

    set main [gtk::menu new]
    gtk::menu_shell append $main [submenu View $menu null]
    gtk::widget show_all $main
    gtk::menu popup $main  null null 3 [gdk::event get_time $ev]

    return true
}

# called on Mmucl exit

proc mmucl::save_config {} {
    global config

    gtk::accel_map save [file join $config(rc_dir) .accelmap.gnome]
}

# FIXME: scrollback

proc mmucl::interface_init {} {
    global config
    variable Mmucl
    
    array set Mmucl {
	pref,use_bold       1
	pref,font_name      "courier medium 12"
	pref,bg,type         normal
	pref,bg,image       ""
	pref,bg,transparent false
	pref,bg,saturation  0.5
	pref,view,toolbar   1      
	pref,view,menubar   1      
	pref,view,statusbar 1
	pref,scrollback     500
	hist_cur        ""
	current_key        ""
	term,palette {
	    {0x0000 0x0000 0x0000} 
	    {0xaaaa 0x0000 0x0000}
	    {0x0000 0xaaaa 0x0000} 
	    {0xaaaa 0x5555 0x0000}
	    {0x0000 0x0000 0xaaaa}
	    {0xaaaa 0x0000 0xaaaa}
	    {0x0000 0xaaaa 0xaaaa} 
	    {0xaaaa 0xaaaa 0xaaaa}
	    {0x5555 0x5555 0x5555}
	    {0xffff 0x5555 0x5555}
	    {0x5555 0xffff 0x5555}
	    {0xffff 0xffff 0x5555}
	    {0x5555 0x5555 0xffff}
	    {0xffff 0x5555 0xffff}
	    {0x5555 0xffff 0xffff}
	    {0xffff 0xffff 0xffff}
	}
	term,fg {0xaaaa 0xaaaa 0xaaaa}
	term,bg {0x0000 0x0000 0x0000}
    }
    
    set Mmucl(pref,term,palette) $Mmucl(term,palette)
    set Mmucl(pref,term,fg) $Mmucl(term,fg)
    set Mmucl(pref,term,bg) $Mmucl(term,bg)

    set rc [file join $config(rc_dir) .pref.$config(interface)]
    if {[file exists $rc] && [catch {source $rc} err]} {
	puts stderr "BUG: loading $rc: $err"
    }

    gnome::program init mmucl $config(version) {}

    set Mmucl(app) [gnome::app new Mmucl Mmucl]
    bind $Mmucl(app) delete_event gnome_quit
    gtk::window set_default_size $Mmucl(app) 700 500

    set group [gtk::accel_group new]
    gtk::window add_accel_group $Mmucl(app) $group

    set stock {
	_Action mmucl-action action.gif   "" ""
	A_lias mmucl-alias alias.gif      "" ""
	C_har mmucl-char char.gif         "" ""
	"_Key"  mmucl-bind bind.gif       "" ""
	_Connect mmucl-connect connect.gif "" ""
	_Send mmucl-send send.gif "" ""
	S_ub mmucl-sub sub.gif "" ""
    }

    loadstock [file join $config(lib_dir) images] $stock
    set Mmucl(menubar) [g::new GtkMenuBar]

    set menu {
	stock  {mmucl-connect   mmucl::connect_dialog}
	sep {}
	stock  {gtk-open    {mmucl::file_dialog load "Load a script"}}
	stock  {mmucl-send  {mmucl::file_dialog send "Send a file to the mud"}}
	stock  {gtk-save-as {mmucl::file_dialog save "Save session state"}}
	sep {}
	stock  {gtk-quit    mmucl::gnome_quit}
    }

    gtk::menu_shell append $Mmucl(menubar) [submenu _Mmucl $menu $group]

    set menu {
	stock  {gtk-copy \
		{vte::terminal copy_clipboard \
		$mmucl::Mmucl(ses,$mmucl::Mmucl(cur),term)}}
	stock  {gtk-paste \
		{gtk::editable paste_clipboard $mmucl::Mmucl(input)}}
	sep {}
	stock  {mmucl-action \
		{mmucl::user_dialog action "Edit actions" {Pattern Script}}}
	stock  {mmucl-alias \
		{mmucl::user_dialog alias "Edit aliases" {Name Script}}}
	stock  {mmucl-char \
		{mmucl::user_dialog char "Edit chars" {Name Host Port Login}}}
	stock  {mmucl-bind \
		{mmucl::user_dialog key "Edit key bindings" {Key Script}}}
	stock  {mmucl-sub {mmucl::user_dialog sub "Edit subs" {Pattern Sub}}}
    }

    gtk::menu_shell append $Mmucl(menubar) [submenu _Edit $menu $group]

    set menu {
	check {Menubar mmucl::Mmucl(pref,view,menubar) \
		{mmucl::update_view menubar}}
	check {Toolbar mmucl::Mmucl(pref,view,toolbar) \
		{mmucl::update_view toolbar}}
	check {"Status bar" mmucl::Mmucl(pref,view,statusbar) \
		{mmucl::update_view statusbar}}
    }

    gtk::menu_shell append $Mmucl(menubar) \
	    [submenu_dynamic _View [list menu $menu $group]]

    set menu {
	stock  {gtk-new    {mmucl::MCsession $mmucl::Mmucl(cur) new}}
	menu_dynamic       {_Snoop    mmucl::snoop_menu}
	stock  {gtk-close  mmucl::gnome_session_close}
    }

    gtk::menu_shell append $Mmucl(menubar) [submenu _Session $menu $group]

    set menu {
	stock {gtk-clear {mmucl::MCclear $mmucl::Mmucl(cur)}}
	stock {gtk-preferences {mmucl::term_pref_dialog}}
    }

    gtk::menu_shell append $Mmucl(menubar) [submenu _Terminal $menu $group]

    set menu {
	check {_Reconnect mmucl::Mmucl(cfg,reconnect)}
	check {"_Keep line"  mmucl::Mmucl(cfg,keep_line)}
	check {_Actions mmucl::Mmucl(cfg,actions)}
	check {"Action by _line" mmucl::Mmucl(cfg,action_by_line)}
	check {S_ubs mmucl::Mmucl(cfg,subs)}
	check {"_Strip ANSI" mmucl::Mmucl(cfg,strip_ansi)}
	check {_Echo mmucl::Mmucl(cfg,echo)}
    }

    gtk::menu_shell append $Mmucl(menubar) \
	    [submenu_dynamic _Options [list menu $menu $group]]

    # FIXME: Doesn't look in right place for mmucl.info
    set menu {
	stock {gnome-stock-about mmucl::about_dialog}
	button  {_Contents {MChelp $mmucl::Mmucl(cur)}}
    }

    gtk::menu_shell append $Mmucl(menubar) [submenu _Help $menu $group]    

    set toolbar {
	mmucl-connect "Connect to a MUD" "" connect_dialog
	mmucl-action  "Edit Actions"  whee \
		{user_dialog action "Edit actions" {Pattern Script} }
	mmucl-alias   "Edit Aliases" whee \
		{user_dialog alias "Edit aliases" {Name Script}}
	mmucl-char    "Edit Characters" whee \
		{user_dialog char "Edit chars" {Name Host Port Login}}
	mmucl-bind    "Edit key bindings" whee \
		{user_dialog key "Edit key bindings" {Key Script}}
	mmucl-sub     "Edit Subs" whee \
		{user_dialog sub "Edit subs" {Pattern Sub}}
    }

    set Mmucl(toolbar) [toolbar $toolbar] 
    set Mmucl(notebook) [g::new GtkNotebook]
    set Mmucl(input) [gtk::entry new]

    set vbox [g::new GtkVBox]
    gtk::box pack_start $vbox $Mmucl(notebook) 1 1 0
    gtk::box pack_start $vbox $Mmucl(input) 0 0 0

    gnome::app set_toolbar $Mmucl(app) $Mmucl(toolbar)  
    gnome::app set_menus $Mmucl(app) $Mmucl(menubar)  
    gnome::app set_contents $Mmucl(app) $vbox

    set Mmucl(statusbar) [gnome::appbar new false true user]
    gnome::app set_statusbar $Mmucl(app) $Mmucl(statusbar)

    session_tab_add

    bind $Mmucl(notebook) switch-page {
	MCsession $Mmucl(cur) switch $Mmucl(notebook,ses,[gtk::notebook get_nth_page $Mmucl(notebook) [lindex $_ 2]])
    }

    set map [file join $config(rc_dir) .accelmap.gnome]
    if {[file exists $map]} {
	gtk::accel_map load $map
    }

    gtk::widget show_all $Mmucl(menubar)
    gtk::widget show_all $Mmucl(app)

    gtk::widget grab_focus $Mmucl(input)

    # FIXME: dunno why but can't switch the first time another tab is added
    # unless I do this
    gtk::notebook set_show_tabs $Mmucl(notebook) 0

    bind $Mmucl(input) activate [list handle_input $Mmucl(input)]
    bind $Mmucl(input) key-press-event {key_event [lindex $_ 1]}

    update_view menubar
    update_view toolbar
    update_view statusbar

    rename interface_init ""
}
