#include <cassert>
#include <list>
#include <map>
#include <sstream>

#include "Exception.h"

#include "AllObjects.h"
#include "ObjectVisitors.h"
#include "ObjectRepository.h"


//----------------------------------------------------------------------------
ObjectRepository ObjectRepository::sm_instance;


//----------------------------------------------------------------------------
class ObjectRepository::PImpl
{
    //------------------------------------------------------------------------
    /**
     * This visitor is used to insert an object into
     * m_objectList or m_particleList, dependend on its type.
     */
    class ShowObjectVisitor : public ObjectVisitor
    {
      public:
        //--------------------------------------------------------------------
        ShowObjectVisitor(ObjectRepository::PImpl *r)
                : m_objectRepository(r) {}
        ~ShowObjectVisitor() { m_objectRepository = NULL; }

      private:
        //--------------------------------------------------------------------
        void do_visit(Barrier *b) { do_do_visit(b); }
        void do_visit(BlackHole *b) { do_do_visit(b); }
        void do_visit(Crate *c) { do_do_visit(c); }
        void do_visit(Grenade *g) { do_do_visit(g); }
        void do_visit(Grinder *g) { do_do_visit(g); }
        void do_visit(MagnetBase *m) { do_do_visit(m); }
        void do_visit(Missile *m) { do_do_visit(m); }
        void do_visit(MortarBase *m) { do_do_visit(m); }
        void do_visit(ParticleBase *p) { m_objectRepository->m_particleList.push_back(p); }
        void do_visit(ParticleFountainBase *f) { do_do_visit(f); }
        void do_visit(Platform *p) { do_do_visit(p); }
        void do_visit(ProjectileBase *p) { do_do_visit(p); }
        void do_visit(SAMBatteryBase *s) { do_do_visit(s); }
        void do_visit(Ship *s) { do_do_visit(s); }
        void do_visit(SwitchBase *s) { do_do_visit(s); }
        void do_visit(Tank *t) { do_do_visit(t); }
        void do_visit(Thorn *t) { do_do_visit(t); }
        void do_visit(TurretBase *t) { do_do_visit(t); }

        //--------------------------------------------------------------------
        void do_do_visit(ObjectBase *o)
        {
            ObjectEntry e = { o, true };
            m_objectRepository->m_objectList.push_back(e);
        }

        //--------------------------------------------------------------------
        ObjectRepository::PImpl *m_objectRepository;
    };


    //------------------------------------------------------------------------
    typedef std::map<unsigned, ObjectBase*> Container;

    //------------------------------------------------------------------------
    struct ObjectEntry
    {
        /// The object itself.
        ObjectBase *object;

        /// Set to true, if the object shall be deleted when removing.
        bool autoDelete;
    };

    typedef std::list<ObjectEntry> ObjectList;

    typedef std::list<ParticleBase*> ParticleList;
    
public:
    //------------------------------------------------------------------------
    PImpl();
    ~PImpl();

    //------------------------------------------------------------------------
    inline size_t size() const 
    {
        return m_id2Object.size();
    }

    //------------------------------------------------------------------------
    void clear();

    //------------------------------------------------------------------------
    inline bool doesExist(const unsigned id) const
    {
        return m_id2Object.find(id) != m_id2Object.end();
    }

    //------------------------------------------------------------------------
    unsigned getDynamicObjectId() const;


    //------------------------------------------------------------------------
    void addObject(ObjectBase *o);

    //------------------------------------------------------------------------
    void showObject(ObjectBase *o);
    void showObjectId(const unsigned id);
    
    //------------------------------------------------------------------------
    void markObjectToRemove(ObjectBase *o, bool autoDelete);
    void removeObjectsToRemove();

    //------------------------------------------------------------------------
    inline ObjectBase *getObject(const unsigned id)
    {
        Container::iterator iter = m_id2Object.find(id);
        return iter != m_id2Object.end() ? iter->second : NULL;
    }

    inline const ObjectBase *getObject(const unsigned id) const
    {
        Container::const_iterator iter = m_id2Object.find(id);
        return iter != m_id2Object.end() ? iter->second : NULL;
    }

    //------------------------------------------------------------------------
    void acceptParticles(ObjectVisitor &v);
    void acceptParticles(ObjectConstVisitor &v) const;

    //------------------------------------------------------------------------
    void acceptObjects(ObjectVisitor &v);
    void acceptObjects(ObjectConstVisitor &v) const;

private:
    //------------------------------------------------------------------------
    void do_showObject(ObjectBase *o);

    //------------------------------------------------------------------------
    mutable unsigned m_dynamicObjectIdCounter;

    //------------------------------------------------------------------------
    /**
     * m_id2Object[id] is a pointer to the ObjectBase with the given id.
     */
    Container m_id2Object;

    /**
     * A list of all objects (except for particles) visible on the playground.
     */
    ObjectList m_objectList;

    /**
     * A list of all particles visible on the playground.
     * They are stored in an own list for a better performance
     * in the CollisionDetectionVisitor.
     */
    ParticleList m_particleList;
};


//----------------------------------------------------------------------------
ObjectRepository::PImpl::PImpl()
{
}

//----------------------------------------------------------------------------
ObjectRepository::PImpl::~PImpl()
{
    clear();
}

//----------------------------------------------------------------------------
void ObjectRepository::PImpl::clear()
{
    m_objectList.clear();
    m_particleList.clear();

    for (Container::iterator
             iter = m_id2Object.begin(),
             end = m_id2Object.end();
         iter != end; ++iter)
    {
        delete iter->second;
    }
    m_id2Object.clear();

    m_dynamicObjectIdCounter = 1024;
}

//----------------------------------------------------------------------------
unsigned ObjectRepository::PImpl::getDynamicObjectId() const
{
    for (;;)
    {
        if (++m_dynamicObjectIdCounter == 0x10000)
        {
            m_dynamicObjectIdCounter = 1024;
        }

        if (!doesExist(m_dynamicObjectIdCounter))
        {
            return m_dynamicObjectIdCounter;
        }
    }
}

//----------------------------------------------------------------------------
void ObjectRepository::PImpl::addObject(ObjectBase *o)
{
    assert(o != NULL);
    assert(o->getId() != 0);

    if (doesExist(o->getId()))
    {
        // Not an assert(), because a duplicate id might be given
        // via the XML level file.

        std::ostringstream s;
        s << "An object with the id '" << o->getId()
          << "' already exists in the object repository";
        throw Exception(s.str());
    }

    m_id2Object[o->getId()] = o;

    if (!o->isHidden())
    {
        do_showObject(o);
    }
}

//----------------------------------------------------------------------------
void ObjectRepository::PImpl::showObject(ObjectBase *o)
{
    assert(o != NULL);
    assert(doesExist(o->getId()));

    if (o->isHidden())
    {
        o->setHidden(false);
        do_showObject(o);
    }
}

//----------------------------------------------------------------------------
void ObjectRepository::PImpl::showObjectId(const unsigned id)
{
    Container::const_iterator iter = m_id2Object.find(id);
    assert(iter != m_id2Object.end());

    showObject(iter->second);
}

//----------------------------------------------------------------------------
void ObjectRepository::PImpl::do_showObject(ObjectBase *o)
{
    ShowObjectVisitor v(this);
    o->accept(v);
}

//----------------------------------------------------------------------------
void ObjectRepository::PImpl::markObjectToRemove(ObjectBase *o, bool autoDelete)
{
    for (ObjectList::iterator
             iter = m_objectList.begin(),
             end = m_objectList.end();
         iter != end; ++iter)
    {
        if (iter->object == o)
        {
            o->setHidden(true);
            iter->autoDelete = autoDelete;
            break;
        }
    }
}

//----------------------------------------------------------------------------
void ObjectRepository::PImpl::removeObjectsToRemove()
{
    do
    {
        ObjectList::iterator iter = m_objectList.begin();
        ObjectList::iterator end = m_objectList.end();
        while (iter != end)
        {
            if (iter->object->isHidden())
            {
                if (iter->autoDelete)
                {
                    m_id2Object.erase(iter->object->getId());
                    delete iter->object;
                }
                iter = m_objectList.erase(iter);
            }
            else
            {
                ++iter;
            }
        }
    }
    while (0);

    do
    {
        ParticleList::iterator iter = m_particleList.begin();
        ParticleList::iterator end = m_particleList.end();
        while (iter != end)
        {
            if ((*iter)->isHidden())
            {
                m_id2Object.erase((*iter)->getId());
                delete *iter;
                iter = m_particleList.erase(iter);
            }
            else
            {
                ++iter;
            }
        }
    }
    while (0);
}

//----------------------------------------------------------------------------
void ObjectRepository::PImpl::acceptParticles(ObjectVisitor &v)
{
    for (ParticleList::iterator
             iter = m_particleList.begin(),
             end = m_particleList.end();
         iter != end; ++iter)
    {
        // Only visit particles, that are not to be removed!
        if (!(*iter)->isHidden())
        {
            (*iter)->accept(v);
        }
    }
}

//----------------------------------------------------------------------------
void ObjectRepository::PImpl::acceptParticles(ObjectConstVisitor &v) const
{
    for (ParticleList::const_iterator
             iter = m_particleList.begin(),
             end = m_particleList.end();
         iter != end; ++iter)
    {
        // Only visit particles, that are not to be removed!
        if (!(*iter)->isHidden())
        {
            (*iter)->accept(v);
        }
    }
}

//----------------------------------------------------------------------------
void ObjectRepository::PImpl::acceptObjects(ObjectVisitor &v)
{
    for (ObjectList::iterator
             iter = m_objectList.begin(),
             end = m_objectList.end();
         iter != end && v.continueForLoopInVisitObjectList(); ++iter)
    {
        // Only visit objects, that are not to be removed!
        if (!iter->object->isHidden())
        {
            iter->object->accept(v);
        }
    }
}

//----------------------------------------------------------------------------
void ObjectRepository::PImpl::acceptObjects(ObjectConstVisitor &v) const
{
    for (ObjectList::const_iterator
             iter = m_objectList.begin(),
             end = m_objectList.end();
         iter != end && v.continueForLoopInVisitObjectList(); ++iter)
    {
        // Only visit objects, that are not to be removed!
        if (!iter->object->isHidden())
        {
            iter->object->accept(v);
        }
    }
}



//----------------------------------------------------------------------------
ObjectRepository::ObjectRepository()
{
    m_pImpl = new PImpl();
}

//----------------------------------------------------------------------------
ObjectRepository::~ObjectRepository()
{
    ZAP_POINTER(m_pImpl);
}

//----------------------------------------------------------------------------
size_t ObjectRepository::size() const
{
    return m_pImpl->size();
}

//----------------------------------------------------------------------------
void ObjectRepository::clear()
{
    m_pImpl->clear();
}

//----------------------------------------------------------------------------
bool ObjectRepository::doesExist(const unsigned id) const
{
    return m_pImpl->doesExist(id);
}

//----------------------------------------------------------------------------
unsigned ObjectRepository::getDynamicObjectId() const
{
    return m_pImpl->getDynamicObjectId();
}

//----------------------------------------------------------------------------
void ObjectRepository::addObject(ObjectBase *o)
{
    m_pImpl->addObject(o);
}

//----------------------------------------------------------------------------
void ObjectRepository::showObject(ObjectBase *o)
{
    m_pImpl->showObject(o);
}

//----------------------------------------------------------------------------
void ObjectRepository::showObjectId(const unsigned id)
{
    m_pImpl->showObjectId(id);
}

//----------------------------------------------------------------------------
void ObjectRepository::markObjectToRemove(ObjectBase *o, bool autoDelete)
{
    m_pImpl->markObjectToRemove(o, autoDelete);
}

//----------------------------------------------------------------------------
void ObjectRepository::removeObjectsToRemove()
{
    m_pImpl->removeObjectsToRemove();
}

//----------------------------------------------------------------------------
ObjectBase *ObjectRepository::getObject(const unsigned id)
{
    return m_pImpl->getObject(id);
}

//----------------------------------------------------------------------------
const ObjectBase *ObjectRepository::getObject(const unsigned id) const
{
    return m_pImpl->getObject(id);
}

//----------------------------------------------------------------------------
void ObjectRepository::acceptParticles(ObjectVisitor &v)
{
    m_pImpl->acceptParticles(v);
}

//----------------------------------------------------------------------------
void ObjectRepository::acceptParticles(ObjectConstVisitor &v) const
{
    m_pImpl->acceptParticles(v);
}

//----------------------------------------------------------------------------
void ObjectRepository::acceptObjects(ObjectVisitor &v)
{
    m_pImpl->acceptObjects(v);
}

//----------------------------------------------------------------------------
void ObjectRepository::acceptObjects(ObjectConstVisitor &v) const
{
    m_pImpl->acceptObjects(v);
}
