#include <unistd.h>
#include <iostream>
#include <memory>
#include <list>

#include "Matrix.h"
#include "File.h"


//============================================================================
// This tool is used to modify particular tile indices in map files.
// There is no documentation except for the source itself. ;-)
//============================================================================


//----------------------------------------------------------------------------
namespace CL_OPT
{
    int categoryId = -1;
    int tileId = -1;
    int newTileId = -1;
    char **mapFiles = NULL;
}


//----------------------------------------------------------------------------
class Map
{
    //------------------------------------------------------------------------
    typedef Matrix<uint8_t> Tiles;
    typedef std::pair<uint8_t, Tiles> Layer;
    typedef std::list<Layer> Layers;

public:
    //------------------------------------------------------------------------
    Map(const char *file);
    ~Map() {}

    //------------------------------------------------------------------------
    inline bool isModified() const { return m_modified; }

    //------------------------------------------------------------------------
    void writeTo(const char *file) const;

    //------------------------------------------------------------------------
    void modifyTile(uint8_t categoryId, uint8_t tileId, uint8_t newTileId);

private:
    //------------------------------------------------------------------------
    uint8_t m_version;
    uint16_t m_xSize;
    uint16_t m_ySize;
    uint8_t m_numLayers;

    Layers m_layers;

    bool m_modified;
};

//----------------------------------------------------------------------------
Map::Map(const char *file)
{
    File f(file, "r");

    m_version = f.readUint8();
    m_xSize = f.readUint16();
    m_ySize = f.readUint16();
    m_numLayers = f.readUint8();

    for (unsigned i=0; i<m_numLayers; i++)
    {
        Layer layer;
        layer.first = f.readUint8();
        layer.second.resize(m_xSize, m_ySize);

        for (unsigned y=0; y<m_ySize; y++)
        {
            for (unsigned x=0; x<m_xSize; x++)
            {
                layer.second.set(x, y, f.readUint8());
            }
        }

        m_layers.push_back(layer);
    }

    m_modified = false;
}

//----------------------------------------------------------------------------
void Map::writeTo(const char *file) const
{
    File f(file, "w");

    f.writeUint8(m_version);
    f.writeUint16(m_xSize);
    f.writeUint16(m_ySize);
    f.writeUint8(m_numLayers);

    for (Layers::const_iterator iter = m_layers.begin();
         iter != m_layers.end(); ++iter)
    {
        f.writeUint8(iter->first);
        for (unsigned y=0; y<m_ySize; y++)
        {
            for (unsigned x=0; x<m_xSize; x++)
            {
                f.writeUint8(iter->second.get(x, y));
            }
        }
    }
}

//----------------------------------------------------------------------------
void Map::modifyTile(uint8_t categoryId, uint8_t tileId, uint8_t newTileId)
{
    // For all layers with the category index 'categoryId'
    // all tiles with the id 'tileId' will be changed to 'newTileId'.

    for (Layers::iterator iter = m_layers.begin();
         iter != m_layers.end(); ++iter)
    {
        if (iter->first != categoryId)  continue;

        Tiles &t = iter->second;
        for (unsigned y=0; y<m_ySize; y++)
        {
            for (unsigned x=0; x<m_xSize; x++)
            {
                if (t.get(x, y) == tileId)
                {
                    t.set(x, y, newTileId);
                    m_modified = true;
                }
            }
        }
    }
}


//----------------------------------------------------------------------------
void usage()
{
    std::cout << "Usage: modifymap [options ...] file ...\n\n"
              << "Where options include:\n"
              << "  -h       This help text.\n"
              << "  -c id    The category index (hex value) to modify.\n"
              << "  -t id    The tile index (hex value) to modify.\n"
              << "  -s id    The new tile index to set for these tiles.\n"
              << "  file     The map file(s) to modify."
              << std::endl;
}

//----------------------------------------------------------------------------
int hex2int(const char *hex)
{
    unsigned int x = -1;
    sscanf(hex, "%x", &x);
    return x;
}

//----------------------------------------------------------------------------
void readCommandLine(int argc, char **argv)
{
    int ch;
    char *argString = "hc:t:s:";

    while ((ch = getopt(argc, argv, argString)) != EOF)
    {
        switch (ch)
        {
        case 'h':
            usage();
            exit(0);
            break;

        case 'c':
            CL_OPT::categoryId = hex2int(optarg);
            break;

        case 't':
            CL_OPT::tileId = hex2int(optarg);
            break;

        case 's':
            CL_OPT::newTileId = hex2int(optarg);
            break;

        default:
            usage();
            exit(-1);
        }
    }

    if (optind < argc)
    {
        CL_OPT::mapFiles = &argv[optind];
    }

    if (CL_OPT::categoryId == -1 ||
        CL_OPT::tileId == -1 ||
        CL_OPT::newTileId == -1 ||
        CL_OPT::mapFiles == NULL)
    {
        usage();
        exit(-1);
    }
}

//----------------------------------------------------------------------------
void processMap(const char *file)
{
    std::cout << "Processing '" << file << "' ... " << std::flush;

    Map m(file);
    m.modifyTile(CL_OPT::categoryId, CL_OPT::tileId, CL_OPT::newTileId);
    if (m.isModified())
    {
        std::cout << "writing ... " << std::flush;
        m.writeTo(file);
    }

    std::cout << "done." << std::endl;
}


//----------------------------------------------------------------------------
int main(int argc, char **argv)
{
    readCommandLine(argc, argv);

    for (unsigned i = 0; CL_OPT::mapFiles[i] != NULL; i++)
    {
        processMap(CL_OPT::mapFiles[i]);
    }

    return 0;
}
