/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * Handles command line switches.  Instantiates the
 * Border object for an IP interface and runs it.
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *         
 *
 * $Id$
 *  
 */

#include <BoBorder.h>
#include <UtMain.h>
#include <UtString.h>
#include <UtReport.h>

static Border* border = 0;
  
static void Version() {
  char *version = "mmborder - Mobile Mesh Border Discovery Protocol v1.0\n";
  Report::Info(version);
}

static void Usage() {
  char *usage = "Usage: mmborder -a addr [-d file] [-f file] [-g file] [-h] -i ifname [-vz] \n"\
    " -a addr   , dotted decimal unique private IP address for tunnel interfaces \n"\
    " -d file   , use flags 'file' to enable debug output \n"\
    " -f file   , use config 'file' instead of mmborder.conf \n"\
    " -g file   , use config 'file' instead of mmdiscover-border.conf when launching mmdiscover \n"\
    " -h        , display this help message \n"\
    " -i ifname , the name of the fixed network interface for tunneling to other borders\n"\
    " -v        , display version information \n"\
    " -z        , run in the foreground instead of running as a daemon \n";  
  Report::Error(usage);
}
		
int main(int argc, char* argv[]) {
  String ident("mmborder");
  String configFile  = "/etc/mobilemesh/mmborder.conf";
  String discoverConfigFile  = "/etc/mobilemesh/mmdiscover-border.conf";
  String debugFile   = "";
  String ifname = "";
  String addr   = "";
  int    flags       = Report::fError | Report::fWarn | Report::fDebug;
  bool isDaemon = true;

  Report::UseOstream(&cerr, flags);

  bool done = false;
  while(!done) {
    int c = getopt(argc,argv,"a:d:f:g:hi:vz");
    switch(c) {
    case 'a':
      addr = optarg;
      break;
    case 'd':
      debugFile = optarg;
      break;
    case 'f':
      configFile = optarg;
      break;
    case 'g':
      discoverConfigFile = optarg;
      break;
    case 'h':
      Usage();
      break;
    case 'i':
      ifname = optarg;
      break;
    case 'v':
      Version();
      exit(0);
      break;
    case 'z':
      isDaemon = false;
      break;
    case EOF:
      done = true;
      break;
    default:
      Usage();
    }
  }
  if(optind < argc) Usage();
  
  if(addr == "") {
    Report::Error("Must specify unique private tunnel address with '-a' option");
  }
  if(ifname == "") {
    Report::Error("Must specify interface name with '-i' option");
  }

  if(isDaemon) {
    Daemonize();           
    Report::UseOstream(0); // Don't send anything to cerr

    // Use syslog to record output
    Report::UseSyslog(ident,flags);
  }

  // Convert any relative file names to absolute paths so we can access them after
  // changing from the current directory
  debugFile  = AbsoluteFilename(debugFile);
  configFile = AbsoluteFilename(configFile);
  discoverConfigFile = AbsoluteFilename(discoverConfigFile);

  // We must run in the directory where the Unix sockets exists because
  // their names are limited to something like 14 characters
  Chdir("/var/run/mobilemesh");
    
  // In case a signal occurs, register for garbage collection
  GarbageCollection<Border>::Register(&border);

  border = new Border(addr,ifname,configFile,discoverConfigFile,debugFile);
  border->Run();
  delete(border);
  border = 0;

  return(0);
}












