/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


/* libc */
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
/* other libs */ 
#include <apr.h>
#include <apr_pools.h>
#include <apr_strings.h>
/* local */
#include <libbttracker.h>

/* infohash, stylesheet, infotable, trs */
static const char* btt_cxn_details_html =
 "<HTML>\n"
 " <HEAD>\n"
 "  <TITLE>BitTorrent Download Info - mod_bt/%s - %s</TITLE>\n"
 "  <LINK REL=\"stylesheet\" TYPE=\"text/css\" HREF=\"%s\">\n"
 " </HEAD>\n"
 " <BODY>\n"
 "  <H1>BitTorrent Download Info</H1>\n"
 "  <TABLE CLASS=\"info\">\n"
 "   %s\n"
 "  </TABLE>\n"
 "  <BR/>\n"
 "  <TABLE CLASS=\"listing\">\n"
 "   <CAPTION>Peer Listing:</CAPTION>\n"
 "   <TR CLASS=\"heading\"><TH>Peer ID</TH><TH>UserAgent</TH><TH CLASS=\"numeric\">Uploaded</TH><TH CLASS=\"numeric\">Downloaded</TH><TH CLASS=\"numeric\">Left</TH><TH CLASS=\"numeric\">Start Time</TH><TH CLASS=\"numeric\">Last Announce</TH><TH CLASS=\"numeric\">Announce Bandwidth</TH><TH CLASS=\"numeric\">Last Serve</TH><TH>Flags</TH></TR>\n"
 "%s"
 "  </TABLE>\n"
 "  <BR/>\n"
 "  <HR/>\n"
 " </BODY>\n"
 "</HTML>\n"
 ;

int btt_cxn_details(btt_tracker* tracker, apr_pool_t* p, DB_TXN* ptxn, const char* args, struct sockaddr_in in_address, char** content, int* content_length)
{
 btt_txn_iterator iterator[] =
 {
  { btt_iter_make_peerlist_string, NULL },
  { NULL, NULL }
 };

 DBT hash_key;
 DBT hash_val;
 DBC* hash_cur = NULL;
 DB_TXN* txn = NULL;
 btt_infohash *hash;
 btt_peer in_peer = web_btt_peer;
 btt_iter_content i_content;
 int ret = 0;
 int rv = HTTP_OK;
 char* rc = NULL;
 /* fake peer_id, this bit of silliness here saves us dozens of lines of code since we can reuse the announce iterator */
 char* rargs = apr_pstrcat(p, args, "&peer_id=%00%00%00%00%00%00%00%00%00%00%00%00%00%00%00%00%00%00%00%01", NULL);
 char* info = NULL;
 
 tracker->s->server_time = time(NULL);

 *content = NULL;
 *content_length = 0;

 if((ret = btt_txn_start(tracker, ptxn, &txn, 0)) != 0)
  return HTTP_SERVER_ERROR;

 if(!btt_peer_req2struct(p, rargs, &in_peer))
 {
  rv = HTTP_BAD_REQUEST;
  goto err;
 }

 bzero(&hash_key, sizeof(hash_key));
 bzero(&hash_val, sizeof(hash_val));
 
 hash_key.data = apr_palloc(p, BT_INFOHASH_LEN);
 hash_key.size = hash_key.ulen = BT_INFOHASH_LEN;
 hash_key.flags = DB_DBT_USERMEM;
 
 hash_val.data = apr_palloc(p, sizeof(btt_infohash));
 hash_val.ulen = sizeof(btt_infohash);
 /* hash_val.size handled by bzero */
 hash_val.flags = DB_DBT_USERMEM;

 memcpy(hash_key.data, in_peer.infohash, BT_INFOHASH_LEN);

 if((ret = btt_txn_load_hashcursor(tracker, p, txn, &hash_key, &hash_val, &hash_cur, BTT_WRITE_CURSOR(tracker), DB_RMW, 0)) != 0)
 {
  tracker->db.hashes->err(tracker->db.hashes, ret, "bt_cxn_details(): first bt_txn_load_hashcursor()");

  if(ret == DB_NOTFOUND)
  {
   rv = HTTP_NOT_FOUND;
   goto end;
  }
  else
  {
   rv = HTTP_SERVER_ERROR;
   goto err;
  }
 }
 
 hash = (btt_infohash*)(hash_val.data);
 
 bzero(&i_content, sizeof(content));
 apr_pool_create(&i_content.pool, p);
 
 /* somewhat arbitrary */
 i_content.buffer_length = (BT_SHORT_STRING * hash->peers) + 4096;
 i_content.content = malloc(i_content.buffer_length);
 i_content.content[0] = 0;
 i_content.content_length = 0;
 i_content.hashandpeer.infohash = hash;
 i_content.hashandpeer.peer = &in_peer;
 i_content.hashandpeer.tracker = tracker;
 i_content.hashandpeer.oldhash = apr_palloc(i_content.pool, sizeof(btt_infohash));
 i_content.hashandpeer.updating = 0;
 memcpy(i_content.hashandpeer.oldhash, hash, sizeof(btt_infohash));

 iterator[0].data = &i_content;

 /* technically this modifies and writes but there's some weirdness with locking if i let it ... */
 if((ret = btt_txn_iterate_peerlist(tracker, i_content.pool, hash, txn, 0, DB_RMW, iterator)) != 0)
 {
  tracker->s->bad_announces++;
  tracker->db.peers->err(tracker->db.peers, ret, "bt_cxn_details(): bt_txn_iterate_peerlist()");
  rv = HTTP_SERVER_ERROR;
  goto err;
 }

 btt_infohash2table(i_content.pool, hash, &info);

 rc = apr_psprintf
 (
  p,
  btt_cxn_details_html,
  libbtt_version,
  bt_str_infohash(i_content.pool, hash->infohash),
  tracker->c->stylesheet,
  info,
  i_content.content
 );

 free(i_content.content);
 i_content.content = NULL;
 apr_pool_destroy(i_content.pool);

 
 hash_cur->c_close(hash_cur);
 hash_cur = NULL;

 *content = rc;
 *content_length = strlen(rc);

 end:

 if(hash_cur)
  hash_cur->c_close(hash_cur);

 if(txn)
  if((ret = txn->commit(txn, 0)) != 0)
  {
   tracker->db.env->err(tracker->db.env, ret, "bt_cxn_details(): txn->commit()");
   goto err;
  }

 return rv;
 
 err:
 
 if(i_content.content)
  free(i_content.content);
  
 if(hash_cur)
  hash_cur->c_close(hash_cur);
 
 if(txn)
  txn->abort(txn);
 
 return rv;
}
