#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "mlocale.h"
#include "mhash.h"
#include "misc.h"

#include "datatypes/count/datatype.h"

#include "mail.h"
#include "plugin_config.h"
#include "pictures.h"
#include "template.h"
#include "generate.h"

#define M_REPORT_MAIL_SENDER      "mail_sender"
#define M_REPORT_MAIL_SENDER_TRAFFIC      "mail_sender_traffic"
#define M_REPORT_MAIL_RECEIPIENT  "mail_receipient"
#define M_REPORT_MAIL_RECEIPIENT_TRAFFIC  "mail_receipient_traffic"
#define M_REPORT_MAIL_DOM_IN      "mail_incoming_domains"
#define M_REPORT_MAIL_DOM_IN_TRAFFIC      "mail_incoming_domains_traffic"
#define M_REPORT_MAIL_DOM_OUT     "mail_outgoing_domains"
#define M_REPORT_MAIL_DOM_OUT_TRAFFIC     "mail_outgoing_domains_traffic"
#define M_REPORT_MAIL_DAILY       "mail_daily"
#define	M_REPORT_MAIL_HOURLY      "mail_hourly"
#define	M_REPORT_MAIL_QUEUE_POLLUTION "mail_qmail_queue_pollution"
#define M_REPORT_MAIL_VIRUS       "mail_virus"
#define	M_REPORT_MAIL_SUBJECT     "mail_subject"
#define	M_REPORT_MAIL_SCANNER     "mail_scanner"

#define HIGHLIGHT	1
#define GROUPING	2
#define VISITS		4
#define INDEX		8
#define BROKEN_LINK	16
#define PERCENT		32
#define RESOLVE_TLD     64
#define VISITS_TRAFFIC  128
#define SORT_BY_KEY     256
#define VIEW_DURATION   512
#define TIME            1024
#define SORT_BY_VCOUNT	2048
#define SORT_BY_QUOTIENT	4096

#define BOTTOM_THRESHOLD	16

int show_mhash_mail (mconfig *ext_conf, tmpl_main *tmpl, mhash *h, int count, int opt) {
	int i;
	long sum = 0;
	double sum_vcount = 0;
	mdata **md;

	if (!h) return 0;

	sum = mhash_sumup(h);
	if (opt & VISITS && opt & PERCENT)
		sum_vcount = mhash_sumup_vcount(h);

	if (opt & SORT_BY_KEY) {
		md = mhash_sorted_to_marray(h, M_SORTBY_KEY, M_SORTDIR_ASC);
	} else if (opt & SORT_BY_VCOUNT) {
		md = mhash_sorted_to_marray(h, M_SORTBY_VCOUNT, M_SORTDIR_DESC);
	} else if (opt & SORT_BY_QUOTIENT) {
		md = mhash_sorted_to_marray(h, M_SORTBY_QUOTIENT, M_SORTDIR_DESC);
	} else {
		md = mhash_sorted_to_marray(h, M_SORTBY_COUNT, M_SORTDIR_DESC);
	}

	for (i = 0; md[i] && (i < count); i++) {
		mdata *data = md[i];

		if (data) {
			unsigned int c = 0;
			char buf[255];

			if (opt & INDEX) {
				sprintf(buf,"%d", i+1);

				tmpl_set_current_block(tmpl, "table_cell");
				tmpl_set_var(tmpl, "CELL_ALIGN", "right");
				tmpl_set_var(tmpl, "CELL_CONTENT", buf);
				tmpl_parse_current_block(tmpl);
			}

			c = mdata_get_count(data);

			tmpl_set_current_block(tmpl, "table_cell");
			tmpl_set_var(tmpl, "CELL_ALIGN", "right");
			if (opt & TIME) {
				/*tmpl_set_var(tmpl, "CELL_CONTENT", get_duration_string(c));*/
			} else {
				sprintf(buf,"%d", c);
				tmpl_set_var(tmpl, "CELL_CONTENT", buf);
			}
			tmpl_parse_current_block(tmpl);

			if (opt & PERCENT && sum) {
				tmpl_set_current_block(tmpl, "table_cell");
				tmpl_set_var(tmpl, "CELL_ALIGN", "right");
				sprintf(buf,"%.2f", c * 100.0 / sum);
				tmpl_set_var(tmpl, "CELL_CONTENT", buf);
				tmpl_parse_current_block(tmpl);
			}

			if (opt & VISITS && data->type == M_DATA_TYPE_VISITED ) {
				tmpl_set_current_block(tmpl, "table_cell");
				tmpl_set_var(tmpl, "CELL_ALIGN", "right");
				if (opt & VISITS_TRAFFIC) {
					tmpl_set_var(tmpl, "CELL_CONTENT",
						     bytes_to_string(mdata_get_vcount(data)));
				} else {
					sprintf(buf,"%.0f", mdata_get_vcount(data));
					tmpl_set_var(tmpl, "CELL_CONTENT", buf);
				}
				tmpl_parse_current_block(tmpl);
				if (opt & PERCENT && sum) {
					tmpl_set_current_block(tmpl, "table_cell");
					tmpl_set_var(tmpl, "CELL_ALIGN", "right");
					sprintf(buf,"%.2f", mdata_get_vcount(data) * 100.0 / sum_vcount);
					tmpl_set_var(tmpl, "CELL_CONTENT", buf);
					tmpl_parse_current_block(tmpl);
				}
			}
#if 0
			if (opt & VIEW_DURATION && data->type == M_DATA_TYPE_VISITED ) {
				/* use count as duration and vcount as hits per page view */

				tmpl_set_current_block(tmpl, "table_cell");
				tmpl_set_var(tmpl, "CELL_ALIGN", "right");
				sprintf(buf, "%.0f", data->data.visited.vcount);
				tmpl_set_var(tmpl, "CELL_CONTENT", buf);
				tmpl_parse_current_block(tmpl);

				tmpl_set_current_block(tmpl, "table_cell");
				tmpl_set_var(tmpl, "CELL_ALIGN", "right");
				tmpl_set_var(tmpl, "CELL_CONTENT", data->data.visited.vcount ?
					     get_duration_string(data->data.visited.count / data->data.visited.vcount) :
					     "--");
				tmpl_parse_current_block(tmpl);
			}
#endif
			if ((opt & GROUPING) && mdata_is_grouped(data)) {
				tmpl_set_current_block(tmpl, "table_cell");
				tmpl_set_var(tmpl, "CELL_ALIGN", "left");
				tmpl_set_var(tmpl, "CELL_CLASS", "grouping");
				tmpl_set_var(tmpl, "CELL_CONTENT", data->key);
				tmpl_parse_current_block(tmpl);

				tmpl_clear_var(tmpl, "CELL_CLASS");
			} else {
				if (opt & HIGHLIGHT) {
					tmpl_set_current_block(tmpl, "table_cell");
					tmpl_set_var(tmpl, "CELL_ALIGN", "left");
					sprintf(buf,"<a href=\"mailto:%s\">%s</a>",
						data->key,
						data->key);
					tmpl_set_var(tmpl, "CELL_CONTENT", buf);
					tmpl_parse_current_block(tmpl);
				} else {
					/* added option to resolve tlds on the fly */
					if (opt & RESOLVE_TLD) {
						tmpl_set_current_block(tmpl, "table_cell");
						tmpl_set_var(tmpl, "CELL_ALIGN", "left");
						tmpl_set_var(tmpl, "CELL_CONTENT", misoname(data->key));
						tmpl_parse_current_block(tmpl);
					} else {
						tmpl_set_current_block(tmpl, "table_cell");
						tmpl_set_var(tmpl, "CELL_ALIGN", "left");
						tmpl_set_var(tmpl, "CELL_CONTENT", data->key);
						tmpl_parse_current_block(tmpl);
					}
				}
			}

			if (opt & BROKEN_LINK && data->type == M_DATA_TYPE_BROKENLINK) {
				struct tm *_tm;
				char timebuf[32] = "";

				if (data->data.brokenlink.referrer &&
				    (0 != strcmp(data->data.brokenlink.referrer, "-"))) {
					tmpl_set_current_block(tmpl, "table_cell");
					tmpl_set_var(tmpl, "CELL_ALIGN", "left");
					tmpl_set_var(tmpl, "CELL_CONTENT", "<a href=\"");
					tmpl_append_var(tmpl, "CELL_CONTENT", data->data.brokenlink.referrer);
					tmpl_append_var(tmpl, "CELL_CONTENT", "\">");
					tmpl_append_var(tmpl, "CELL_CONTENT", data->data.brokenlink.referrer);
					tmpl_append_var(tmpl, "CELL_CONTENT", "</a>");
					tmpl_parse_current_block(tmpl);
				} else {
					tmpl_set_current_block(tmpl, "table_cell");
					tmpl_set_var(tmpl, "CELL_ALIGN", "left");
					tmpl_set_var(tmpl, "CELL_CONTENT", "-");
					tmpl_parse_current_block(tmpl);
				}

				_tm = localtime(&(data->data.brokenlink.timestamp));
				if (strftime(timebuf, sizeof(timebuf)-1, "%x", _tm) == 0) {
					fprintf(stderr, "output::modlogan.show_mhash: strftime failed\n");
				}
				tmpl_set_current_block(tmpl, "table_cell");
				tmpl_set_var(tmpl, "CELL_ALIGN", "left");
				tmpl_set_var(tmpl, "CELL_CONTENT", timebuf);
				tmpl_parse_current_block(tmpl);
			}

			tmpl_set_current_block(tmpl, "table_row");
			tmpl_parse_current_block(tmpl);
			tmpl_clear_block(tmpl, "table_cell");
		}
	}

	free(md);

	return 0;
}

reports_def *get_reports_mail (mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf;
	
	reports_def reports[] = {
		/* 0 */
		  /* id             data  title */
		{ M_REPORT_MAIL_SENDER, _("Mails by Sender"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ _("Mail-Address"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 1 */
		  /* id             data  title */
		{ M_REPORT_MAIL_SENDER_TRAFFIC, _("Mails by Sender Traffic"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC | SORT_BY_VCOUNT,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ _("Mail-Address"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 2 */
		{ M_REPORT_MAIL_RECEIPIENT, _("Mails by Receipient"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ _("Mail-Address"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},

		/* 3 */
		{ M_REPORT_MAIL_RECEIPIENT_TRAFFIC, _("Mails by Receipient Traffic"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC | SORT_BY_VCOUNT,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ _("Mail-Address"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},

		/* 4 */
		{ M_REPORT_MAIL_DOM_IN, _("Domains by Destination"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ _("Domain"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},

		/* 5 */
		{ M_REPORT_MAIL_DOM_IN_TRAFFIC, _("Domains by Destination Traffic"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC | SORT_BY_VCOUNT,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ _("Domain"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},

		/* 6 */
		{ M_REPORT_MAIL_DOM_OUT, _("Domains by Source"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ _("Domain"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},

		/* 7 */
		{ M_REPORT_MAIL_DOM_OUT_TRAFFIC, _("Domains by Source Traffic"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC | SORT_BY_VCOUNT,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ _("Domain"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},

		/* 8 */
		{ M_REPORT_MAIL_VIRUS, _("Virus"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Mail-Address"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},

		/* 9 */
		{ M_REPORT_MAIL_SUBJECT, _("Subject used by a Virus"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Domain"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},

		/* 10 */
		{ M_REPORT_MAIL_SCANNER, _("Virus Scanner"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Domain"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},

		{0, NULL, 0, 0, NULL,
			{
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		}
	};

	if (conf->mail_reports == NULL) {
		conf->mail_reports = malloc(sizeof(reports));
		memcpy(conf->mail_reports, reports, sizeof(reports));
	}

	return conf->mail_reports;
}

void generate_mail_cleanup(mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf;
	
	if (conf->mail_reports) free(conf->mail_reports);
	conf->mail_reports = NULL;
}

char * generate_mail_daily(mconfig * ext_conf, mstate * state, const char *current, int max) {
	config_output *conf = ext_conf->plugin_conf;
	int i;
	tmpl_main *tmpl;
	char *s, buf[255], *fn;
	mstate_mail *stamail = NULL;
	marray_mail sum;

	if (state == NULL)
		return NULL;

	if (state->ext == NULL)
		return NULL;

	if (state->ext_type != M_STATE_TYPE_MAIL)
		return NULL;

	stamail = state->ext;

	sum.incoming_mails = 0;
	sum.outgoing_mails = 0;
	sum.incoming_bytes = 0;
	sum.outgoing_bytes = 0;

	tmpl = tmpl_init();
	assert(tmpl);

	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		fprintf(stderr, "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}

	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		fprintf(stderr, "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);

	/* header */

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Day"));
	tmpl_set_var(tmpl, "CELL_CLASS", "none");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Mail - incoming"));
	tmpl_set_var(tmpl, "CELL_CLASS", "hits");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Mail - outgoing"));
	tmpl_set_var(tmpl, "CELL_CLASS", "hits");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Traffic - incoming"));
	tmpl_set_var(tmpl, "CELL_CLASS", "traffic");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Traffic - outgoing"));
	tmpl_set_var(tmpl, "CELL_CLASS", "traffic");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_row");
	tmpl_parse_current_block(tmpl);

	tmpl_clear_block(tmpl, "table_cell");

	for ( i = 0; i < 31; i++) {
		tmpl_set_current_block(tmpl, "table_cell");
		sprintf(buf, "%d", i);
		tmpl_set_var(tmpl, "CELL_CONTENT", buf);
		tmpl_set_var(tmpl, "CELL_CLASS", "none");
		tmpl_set_var(tmpl, "CELL_ALIGN", "right");
		tmpl_parse_current_block(tmpl);

		tmpl_set_current_block(tmpl, "table_cell");
		sprintf(buf, "%ld", stamail->days[i].incoming_mails);
		tmpl_set_var(tmpl, "CELL_CONTENT", buf);
		tmpl_set_var(tmpl, "CELL_CLASS", "none");
		tmpl_set_var(tmpl, "CELL_ALIGN", "right");
		tmpl_parse_current_block(tmpl);

		tmpl_set_current_block(tmpl, "table_cell");
		sprintf(buf, "%ld", stamail->days[i].outgoing_mails);
		tmpl_set_var(tmpl, "CELL_CONTENT", buf);
		tmpl_set_var(tmpl, "CELL_CLASS", "none");
		tmpl_set_var(tmpl, "CELL_ALIGN", "right");
		tmpl_parse_current_block(tmpl);

		tmpl_set_current_block(tmpl, "table_cell");
		tmpl_set_var(tmpl, "CELL_CONTENT",
			     bytes_to_string(stamail->days[i].incoming_bytes));
		tmpl_set_var(tmpl, "CELL_CLASS", "none");
		tmpl_set_var(tmpl, "CELL_ALIGN", "right");
		tmpl_parse_current_block(tmpl);

		tmpl_set_current_block(tmpl, "table_cell");
		tmpl_set_var(tmpl, "CELL_CONTENT",
			     bytes_to_string(stamail->days[i].outgoing_bytes));
		tmpl_set_var(tmpl, "CELL_CLASS", "none");
		tmpl_set_var(tmpl, "CELL_ALIGN", "right");
		tmpl_parse_current_block(tmpl);

		tmpl_set_current_block(tmpl, "table_row");
		tmpl_parse_current_block(tmpl);

		tmpl_clear_block(tmpl, "table_cell");

		sum.incoming_mails += stamail->days[i].incoming_mails;
		sum.outgoing_mails += stamail->days[i].outgoing_mails;
		sum.incoming_bytes += stamail->days[i].incoming_bytes;
		sum.outgoing_bytes += stamail->days[i].outgoing_bytes;
	}

	sprintf(buf, "%d", 6);
	tmpl_set_var(tmpl, "TABLE_TITLE", _("Mails and Traffic per hour"));
	tmpl_set_var(tmpl, "TABLE_COL_SPAN", buf);
	
	if (tmpl_replace(tmpl, conf->tmp_buf)) {
		tmpl_free(tmpl);
		
		return NULL;
	}

	tmpl_free(tmpl);
	
	s = strdup(conf->tmp_buf->ptr);

	return s;
}

char * generate_mail_hourly(mconfig * ext_conf, mstate * state, const char *current, int max) {
	config_output *conf = ext_conf->plugin_conf;
	int i;
	tmpl_main *tmpl;
	char *s, buf[255], *fn;
	mstate_mail *stamail = NULL;
	marray_mail sum;

	if (state == NULL)
		return NULL;

	if (state->ext == NULL)
		return NULL;

	if (state->ext_type != M_STATE_TYPE_MAIL)
		return NULL;

	stamail = state->ext;

	sum.incoming_mails = 0;
	sum.outgoing_mails = 0;
	sum.incoming_bytes = 0;
	sum.outgoing_bytes = 0;

	tmpl = tmpl_init();
	assert(tmpl);

	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		fprintf(stderr, "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}

	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		fprintf(stderr, "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);

	/* header */

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Hour"));
	tmpl_set_var(tmpl, "CELL_CLASS", "none");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Mail - incoming"));
	tmpl_set_var(tmpl, "CELL_CLASS", "hits");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Mail - outgoing"));
	tmpl_set_var(tmpl, "CELL_CLASS", "hits");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Traffic - incoming"));
	tmpl_set_var(tmpl, "CELL_CLASS", "traffic");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Traffic - outgoing"));
	tmpl_set_var(tmpl, "CELL_CLASS", "traffic");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_row");
	tmpl_parse_current_block(tmpl);

	tmpl_clear_block(tmpl, "table_cell");

	for ( i = 0; i < 24; i++) {
		tmpl_set_current_block(tmpl, "table_cell");
		sprintf(buf, "%d", i);
		tmpl_set_var(tmpl, "CELL_CONTENT", buf);
		tmpl_set_var(tmpl, "CELL_CLASS", "none");
		tmpl_set_var(tmpl, "CELL_ALIGN", "right");
		tmpl_parse_current_block(tmpl);

		tmpl_set_current_block(tmpl, "table_cell");
		sprintf(buf, "%ld", stamail->hours[i].incoming_mails);
		tmpl_set_var(tmpl, "CELL_CONTENT", buf);
		tmpl_set_var(tmpl, "CELL_CLASS", "none");
		tmpl_set_var(tmpl, "CELL_ALIGN", "right");
		tmpl_parse_current_block(tmpl);

		tmpl_set_current_block(tmpl, "table_cell");
		sprintf(buf, "%ld", stamail->hours[i].outgoing_mails);
		tmpl_set_var(tmpl, "CELL_CONTENT", buf);
		tmpl_set_var(tmpl, "CELL_CLASS", "none");
		tmpl_set_var(tmpl, "CELL_ALIGN", "right");
		tmpl_parse_current_block(tmpl);

		tmpl_set_current_block(tmpl, "table_cell");
		tmpl_set_var(tmpl, "CELL_CONTENT",
			     bytes_to_string(stamail->hours[i].incoming_bytes));
		tmpl_set_var(tmpl, "CELL_CLASS", "none");
		tmpl_set_var(tmpl, "CELL_ALIGN", "right");
		tmpl_parse_current_block(tmpl);

		tmpl_set_current_block(tmpl, "table_cell");
		tmpl_set_var(tmpl, "CELL_CONTENT",
			     bytes_to_string(stamail->hours[i].outgoing_bytes));
		tmpl_set_var(tmpl, "CELL_CLASS", "none");
		tmpl_set_var(tmpl, "CELL_ALIGN", "right");
		tmpl_parse_current_block(tmpl);

		tmpl_set_current_block(tmpl, "table_row");
		tmpl_parse_current_block(tmpl);

		tmpl_clear_block(tmpl, "table_cell");

		sum.incoming_mails += stamail->hours[i].incoming_mails;
		sum.outgoing_mails += stamail->hours[i].outgoing_mails;
		sum.incoming_bytes += stamail->hours[i].incoming_bytes;
		sum.outgoing_bytes += stamail->hours[i].outgoing_bytes;
	}

	sprintf(buf, "%d", 6);
	tmpl_set_var(tmpl, "TABLE_TITLE", _("Mails and Traffic per hour"));
	tmpl_set_var(tmpl, "TABLE_COL_SPAN", buf);
	
	if (tmpl_replace(tmpl, conf->tmp_buf)) {
		tmpl_free(tmpl);
		
		return NULL;
	}

	tmpl_free(tmpl);
	
	s = strdup(conf->tmp_buf->ptr);


	return s;
}

char * generate_mail_qmail_queue(mconfig * ext_conf, mstate * state, const char *current, int max) {
	config_output *conf = ext_conf->plugin_conf;
	int i;
	tmpl_main *tmpl;
	char *s, buf[255], *fn;
	mstate_mail *stamail = NULL;

	if (state == NULL)
		return NULL;

	if (state->ext == NULL)
		return NULL;

	if (state->ext_type != M_STATE_TYPE_MAIL)
		return NULL;

	stamail = state->ext;

	tmpl = tmpl_init();
	assert(tmpl);

	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		fprintf(stderr, "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}

	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		fprintf(stderr, "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);

	/* header */

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Day"));
	tmpl_set_var(tmpl, "CELL_CLASS", "none");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Hour"));
	tmpl_set_var(tmpl, "CELL_CLASS", "none");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Local - cur"));
	tmpl_set_var(tmpl, "CELL_CLASS", "hits");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Local - max"));
	tmpl_set_var(tmpl, "CELL_CLASS", "hits");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Remote - cur"));
	tmpl_set_var(tmpl, "CELL_CLASS", "hits");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Remote - max"));
	tmpl_set_var(tmpl, "CELL_CLASS", "hits");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Delivery - cur"));
	tmpl_set_var(tmpl, "CELL_CLASS", "hits");
	tmpl_parse_current_block(tmpl);

	tmpl_set_current_block(tmpl, "table_cell");
	tmpl_set_var(tmpl, "CELL_CONTENT", _("Queue - cur"));
	tmpl_set_var(tmpl, "CELL_CLASS", "hits");
	tmpl_parse_current_block(tmpl);


	tmpl_set_current_block(tmpl, "table_row");
	tmpl_parse_current_block(tmpl);

	tmpl_clear_block(tmpl, "table_cell");

	for (i = 0; i < 31; i++) {
		int j;
		for (j = 0; j < 24; j++) {
			if (stamail->qstat[i][j].count) {
				tmpl_set_current_block(tmpl, "table_cell");
				sprintf(buf, "%d", i + 1);
				tmpl_set_var(tmpl, "CELL_CONTENT", buf);
				tmpl_set_var(tmpl, "CELL_CLASS", "none");
				tmpl_set_var(tmpl, "CELL_ALIGN", "right");
				tmpl_parse_current_block(tmpl);

				tmpl_set_current_block(tmpl, "table_cell");
				sprintf(buf, "%d", j);
				tmpl_set_var(tmpl, "CELL_CONTENT", buf);
				tmpl_set_var(tmpl, "CELL_CLASS", "none");
				tmpl_set_var(tmpl, "CELL_ALIGN", "right");
				tmpl_parse_current_block(tmpl);

				tmpl_set_current_block(tmpl, "table_cell");
				sprintf(buf, "%.0f", stamail->qstat[i][j].local_cur / stamail->qstat[i][j].count);
				tmpl_set_var(tmpl, "CELL_CONTENT", buf);
				tmpl_set_var(tmpl, "CELL_CLASS", "none");
				tmpl_set_var(tmpl, "CELL_ALIGN", "right");
				tmpl_parse_current_block(tmpl);

				tmpl_set_current_block(tmpl, "table_cell");
				sprintf(buf, "%.0f", stamail->qstat[i][j].local_max / stamail->qstat[i][j].count);
				tmpl_set_var(tmpl, "CELL_CONTENT", buf);
				tmpl_set_var(tmpl, "CELL_CLASS", "none");
				tmpl_set_var(tmpl, "CELL_ALIGN", "right");
				tmpl_parse_current_block(tmpl);

				tmpl_set_current_block(tmpl, "table_cell");
				sprintf(buf, "%.0f", stamail->qstat[i][j].remote_cur / stamail->qstat[i][j].count);
				tmpl_set_var(tmpl, "CELL_CONTENT", buf);
				tmpl_set_var(tmpl, "CELL_CLASS", "none");
				tmpl_set_var(tmpl, "CELL_ALIGN", "right");
				tmpl_parse_current_block(tmpl);

				tmpl_set_current_block(tmpl, "table_cell");
				sprintf(buf, "%.0f", stamail->qstat[i][j].remote_max / stamail->qstat[i][j].count);
				tmpl_set_var(tmpl, "CELL_CONTENT", buf);
				tmpl_set_var(tmpl, "CELL_CLASS", "none");
				tmpl_set_var(tmpl, "CELL_ALIGN", "right");
				tmpl_parse_current_block(tmpl);

				tmpl_set_current_block(tmpl, "table_cell");
				sprintf(buf, "%.0f", stamail->qstat[i][j].deliver_cur / stamail->qstat[i][j].count);
				tmpl_set_var(tmpl, "CELL_CONTENT", buf);
				tmpl_set_var(tmpl, "CELL_CLASS", "none");
				tmpl_set_var(tmpl, "CELL_ALIGN", "right");
				tmpl_parse_current_block(tmpl);

				tmpl_set_current_block(tmpl, "table_cell");
				sprintf(buf, "%.0f", stamail->qstat[i][j].queue_cur / stamail->qstat[i][j].count);
				tmpl_set_var(tmpl, "CELL_CONTENT", buf);
				tmpl_set_var(tmpl, "CELL_CLASS", "none");
				tmpl_set_var(tmpl, "CELL_ALIGN", "right");
				tmpl_parse_current_block(tmpl);

				tmpl_set_current_block(tmpl, "table_row");
				tmpl_parse_current_block(tmpl);

				tmpl_clear_block(tmpl, "table_cell");
			}
		}
	}

	sprintf(buf, "%d", 8);
	tmpl_set_var(tmpl, "TABLE_TITLE", _("Qmail Queue Pollution"));
	tmpl_set_var(tmpl, "TABLE_COL_SPAN", buf);

	if (tmpl_replace(tmpl, conf->tmp_buf)) {
		tmpl_free(tmpl);
		
		return NULL;
	}

	tmpl_free(tmpl);
	
	s = strdup(conf->tmp_buf->ptr);

	return s;
}



int register_reports_mail (mconfig *ext_conf, tmpl_reports *r) {
	int i, j;

	reports_def *reports = get_reports_mail(ext_conf);
	const char *bla[] = {
		M_REPORT_MAIL_DAILY,
			M_REPORT_MAIL_HOURLY,
			M_REPORT_MAIL_QUEUE_POLLUTION,
			NULL };

	for (i = 0; i < M_TMPL_MAX_REPORTS && r[i].key != NULL; i++);

	if (i != M_TMPL_MAX_REPORTS) {
		for (j = 0; reports[j].key && i < M_TMPL_MAX_REPORTS; j++, i++) {
			r[i].key = reports[j].key;
			r[i].func = generate_mail;
			r[i].title = reports[j].title;
		}
	}

	if (i < M_TMPL_MAX_REPORTS) {
		r[i].key = bla[0];
		r[i].func = generate_mail_hourly;
		r[i].title = _("Hourly Statistics");
	}

	if (++i < M_TMPL_MAX_REPORTS) {
		r[i].key = bla[1];
		r[i].func = generate_mail_daily;
		r[i].title = _("Daily Statistics");
	}

	if (++i < M_TMPL_MAX_REPORTS) {
		r[i].key = bla[2];
		r[i].func = generate_mail_qmail_queue;
		r[i].title = _("Qmail Queue Stats");
	}


	return 0;
}

char * generate_mail(mconfig * ext_conf, mstate * state, const char *current, int max) {
	config_output *conf = ext_conf->plugin_conf;
	mstate_mail *stamail = NULL;
	int i,j;
	int w = 0;
	char *s = NULL;
	char buf[255];
	tmpl_main *tmpl;
	reports_def *reports;
	char *fn;
	mhash *data;

	if (state == NULL) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			"state = NULL\n");
		return NULL;
	}

	if (state->ext == NULL) {
		M_DEBUG3(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			"state->ext = NULL, (%d, %d, %d)\n",
			 state->year,
			 state->month,
			 state->ext_type
			 );
		return NULL;
	}

	if (state->ext_type != M_STATE_TYPE_MAIL) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			"state extension != web\n");
		return NULL;
	}

	stamail = state->ext;

	reports = get_reports_mail(ext_conf);

	for (i = 0; reports[i].key; i++) {
		if (0 == strcmp(reports[i].key, current))
			break;
	}

	/* unknown report */
	if (reports[i].key == NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "report '%s' no found here\n", current);
		return NULL;
	}

	switch(i) {
	case 0: data = stamail->sender; break;
	case 1: data = stamail->sender; break;
	case 2: data = stamail->receipient; break;
	case 3: data = stamail->receipient; break;
	case 4: data = stamail->recp_domain; break;
	case 5: data = stamail->recp_domain; break;
	case 6: data = stamail->send_domain; break;
	case 7: data = stamail->send_domain; break;
	case 8: data = stamail->virus; break;
	case 9: data = stamail->subject; break;
	case 10: data = stamail->scanner; break;
	default:
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "report '%s' no found here - what's up ??\n", current);
		return NULL;
	}

	tmpl = tmpl_init();
	assert(tmpl);

	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}

	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);
	
	/* only create the report if we have data */
	if (mhash_count(data)) {
		if (reports[i].show_graph && reports[i].draw_graph) {
			char *ref = reports[i].draw_graph(ext_conf, state);

			if (ref && strlen(ref)) {
				tmpl_set_var(tmpl, "IMAGE", ref);
			}
		}

		/* calculate the number columns */
		for (w = 0; reports[i].fields[w].name != NULL; w++);
		if (reports[i].options & INDEX) w++;
		if (reports[i].options & PERCENT) w++;
		if (reports[i].options & PERCENT && reports[i].options & VISITS) w++;

		if (reports[i].options & INDEX) {
			tmpl_set_current_block(tmpl, "table_cell");
			tmpl_set_var(tmpl, "CELL_CONTENT", "#");
			tmpl_parse_current_block(tmpl);
		}

		for (j = 0; reports[i].fields[j].name != NULL; j++) {
			tmpl_set_current_block(tmpl, "table_cell");

			if (reports[i].fields[j].class) {
				tmpl_set_var(tmpl, "CELL_CONTENT", reports[i].fields[j].name);
				tmpl_set_var(tmpl, "CELL_CLASS", reports[i].fields[j].class);
			} else {
				tmpl_set_var(tmpl, "CELL_CONTENT", reports[i].fields[j].name);
				tmpl_set_var(tmpl, "CELL_CLASS", "none");
			}
			tmpl_parse_current_block(tmpl);

			if (j == 0 && reports[i].options & PERCENT) {
				tmpl_set_current_block(tmpl, "table_cell");
				tmpl_set_var(tmpl, "CELL_CONTENT", "%");
				tmpl_parse_current_block(tmpl);
			}
			if (j == 1 && reports[i].options & VISITS && reports[i].options & PERCENT) {
				tmpl_set_current_block(tmpl, "table_cell");
				tmpl_set_var(tmpl, "CELL_CONTENT", "%");
				tmpl_parse_current_block(tmpl);
			}
		}

		tmpl_set_current_block(tmpl, "table_row");
		tmpl_parse_current_block(tmpl);

		tmpl_clear_block(tmpl, "table_cell");

#if 0
		fprintf(stderr, "%s.%d: %d (%s - %d)\n", __FILE__, __LINE__, i, reports[i].title, max);
#endif
		if (show_mhash_mail(ext_conf, tmpl, data, max, reports[i].options)) {
			fprintf(stderr, "show mhash web failed for '%s'\n", current);
		}

		tmpl_clear_var(tmpl, "CELL_ALIGN");
		if (max > BOTTOM_THRESHOLD) {
			if (reports[i].options & INDEX) {
				tmpl_set_current_block(tmpl, "table_cell");
				tmpl_set_var(tmpl, "CELL_CONTENT", "#");
				tmpl_parse_current_block(tmpl);
			}

			for (j = 0; reports[i].fields[j].name != NULL; j++) {
				tmpl_set_current_block(tmpl, "table_cell");

				if (reports[i].fields[j].class) {
					tmpl_set_var(tmpl, "CELL_CONTENT", reports[i].fields[j].name);
					tmpl_set_var(tmpl, "CELL_CLASS", reports[i].fields[j].class);
				} else {
					tmpl_set_var(tmpl, "CELL_CONTENT", reports[i].fields[j].name);
					tmpl_set_var(tmpl, "CELL_CLASS", "none");
				}
				tmpl_parse_current_block(tmpl);

				if (j == 0 && reports[i].options & PERCENT) {
					tmpl_set_current_block(tmpl, "table_cell");
					tmpl_set_var(tmpl, "CELL_CONTENT", "%");
					tmpl_parse_current_block(tmpl);
				}
				if (j == 1 && reports[i].options & VISITS && reports[i].options & PERCENT) {
					tmpl_set_current_block(tmpl, "table_cell");
					tmpl_set_var(tmpl, "CELL_CONTENT", "%");
					tmpl_parse_current_block(tmpl);
				}
			}

			tmpl_set_current_block(tmpl, "table_row");
			tmpl_parse_current_block(tmpl);
			tmpl_clear_block(tmpl, "table_cell");
		}
	} else {
		/* generate a dummy page if we have no data to process */
		w = 1;

		tmpl_set_current_block(tmpl, "table_cell");
		tmpl_set_var(tmpl, "CELL_CONTENT", _("Sorry, no data to display"));
		tmpl_parse_current_block(tmpl);

		tmpl_set_current_block(tmpl, "table_row");
		tmpl_parse_current_block(tmpl);
		tmpl_clear_block(tmpl, "table_cell");
	}


	sprintf(buf, "%d", w);
	tmpl_set_var(tmpl, "TABLE_TITLE", reports[i].title);
	tmpl_set_var(tmpl, "TABLE_COL_SPAN", buf);
	
	if (tmpl_replace(tmpl, conf->tmp_buf)) {
		s = NULL;
	} else {
		s = strdup(conf->tmp_buf->ptr);
	}

	tmpl_free(tmpl);

#ifdef MTIMER_ENABLED
	MTIMER_STOP(timer);
	MTIMER_CALC(timer);

	fprintf(stderr, "timer %s: %ld msec\n",
		__FUNCTION__, timer.span );
#endif
	return s;
}

static int set_line(tmpl_main *tmpl, const char *desc,
	     long incoming_mails,
	     long outgoing_mails,
	     long incoming_bytes,
	     long outgoing_bytes,
	     int days_passed
	     ) {

	char buf[255];

	tmpl_set_current_block(tmpl, "row");

	tmpl_set_var(tmpl, "DESC", desc);

	sprintf(buf, "%ld", incoming_mails / days_passed);
	tmpl_set_var(tmpl, "AVG_MAILS_OUT", buf);
	sprintf(buf, "%ld", outgoing_mails / days_passed);
	tmpl_set_var(tmpl, "AVG_MAILS_IN", buf);
	tmpl_set_var(tmpl, "AVG_BYTES_OUT",
		     bytes_to_string(incoming_bytes / days_passed));

	tmpl_set_var(tmpl, "AVG_BYTES_IN",
		     bytes_to_string(outgoing_bytes / days_passed));

	sprintf(buf, "%ld", incoming_mails);
	tmpl_set_var(tmpl, "TOT_MAILS_OUT", buf);
	sprintf(buf, "%ld", outgoing_mails);
	tmpl_set_var(tmpl, "TOT_MAILS_IN", buf);
	tmpl_set_var(tmpl, "TOT_BYTES_OUT",
		     bytes_to_string(incoming_bytes));
	tmpl_set_var(tmpl, "TOT_BYTES_IN",
		     bytes_to_string(outgoing_bytes));

	tmpl_parse_current_block(tmpl);

	return 0;
}


int mplugins_output_generate_history_output_mail(mconfig *ext_conf, mlist *history, tmpl_main *tmpl) {
	mlist *l = history;
	data_History hist, yearly;
	char  *first_report;
	config_output *conf = ext_conf->plugin_conf;
	char buf[255];

	/* header vars */

#define HIST(x) \
	hist.data.mail.x = 0;

	HIST(incoming_mails);
	HIST(outgoing_mails);
	HIST(incoming_bytes);
	HIST(outgoing_bytes);
#undef HIST
	hist.days_passed = 0;

#define HIST(x) \
	yearly.data.mail.x = 0;

	HIST(incoming_mails);
	HIST(outgoing_mails);
	HIST(incoming_bytes);
	HIST(outgoing_bytes);
#undef HIST
	yearly.days_passed = 0;
	yearly.year = 0;
	
	/* set the correct link */
	if (conf->menu && conf->menu->data && conf->menu->data->key) {
		first_report = conf->menu->data->key;
	} else {
		first_report = conf->reports->data->key;
	}

	/* go to the last element */
	while (l->next) l = l->next;

	while (l) {
		mdata *data = l->data;

		if (!data) break;

		if (data->data.hist->days_passed != 0) {
			char *lnk;
			if (yearly.year > data->data.hist->year) {
				sprintf(buf, "%04d",
					yearly.year
					);

				set_line(tmpl,
					 buf,
					 yearly.data.mail.incoming_mails,
					 yearly.data.mail.outgoing_mails,
					 yearly.data.mail.incoming_bytes,
					 yearly.data.mail.outgoing_bytes,
					 yearly.days_passed
					 );
			}

			lnk = generate_output_link(ext_conf,
						   data->data.hist->year,
						   data->data.hist->month,
						   first_report);

			sprintf(buf, "<a href=\"%s\">%s&nbsp;%04d</a>",
				lnk,
				get_month_string(data->data.hist->month, 1),
				data->data.hist->year
				);
			free(lnk);

			set_line(tmpl,
				 buf,
				 data->data.hist->data.mail.incoming_mails,
				 data->data.hist->data.mail.outgoing_mails,
				 data->data.hist->data.mail.incoming_bytes,
				 data->data.hist->data.mail.outgoing_bytes,
				 data->data.hist->days_passed
				 );

			if (yearly.year > data->data.hist->year) {
				yearly.year = data->data.hist->year;
				yearly.days_passed = data->data.hist->days_passed;

#define HIST(x) \
	yearly.data.mail.x = data->data.hist->data.mail.x;
				HIST(incoming_mails);
				HIST(outgoing_mails);
				HIST(incoming_bytes);
				HIST(outgoing_bytes);
#undef HIST

			} else {
#define HIST(x) \
	yearly.data.mail.x += data->data.hist->data.mail.x
				yearly.year = data->data.hist->year;
				yearly.days_passed += data->data.hist->days_passed; /* CHECKME */

				HIST(incoming_mails);
				HIST(outgoing_mails);
				HIST(incoming_bytes);
				HIST(outgoing_bytes);
#undef HIST
			}
#define HIST(x) \
	hist.data.mail.x += data->data.hist->data.mail.x

			HIST(incoming_mails);
			HIST(outgoing_mails);
			HIST(incoming_bytes);
			HIST(outgoing_bytes);

			hist.days_passed += data->data.hist->days_passed;
#undef HIST
		} else {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_OUTPUT, M_DEBUG_LEVEL_WARNINGS,
				 "count == 0, is this ok ?? splitby for '%s' without an entry ??\n",
				 data->key);
		}
		l = l->prev;
	}

	if (yearly.year && yearly.days_passed) {
		sprintf(buf, "%04d",
			yearly.year
			);

		set_line(tmpl,
			 buf,
			 yearly.data.mail.incoming_mails,
			 yearly.data.mail.outgoing_mails,
			 yearly.data.mail.incoming_bytes,
			 yearly.data.mail.outgoing_bytes,
			 yearly.days_passed
			 );
	}

	if (hist.days_passed) {
		set_line(tmpl,
			 _("totals"),
			 hist.data.mail.incoming_mails,
			 hist.data.mail.outgoing_mails,
			 hist.data.mail.incoming_bytes,
			 hist.data.mail.outgoing_bytes,
			 hist.days_passed
			 );
	}

	return 0;
}
