#include <stdio.h>
#include <stdarg.h>
#include <errno.h>
#include <string.h>
#include <endian.h>

//#include <usb.h>

#include "font.h"
#include  "f60lib.h"

char f60_strerror[512] = "";
int f60_error = F60_ERROR_NONE;

unsigned char response[F60_MAX_RESPONSE];
int response_size;
int debug = 0;

int ep_addr_write = F60_EPA_1;
int ep_addr_read = F60_EPA_2;

int selected_memory = FLASH_MEMORY;

struct usb_device *f60_dev = NULL;

const char F60_CmdTerminator = 0x04;
const char F60_GetFirmwareVersion[] = {0x05,0x30};
const char F60_GetDirectory[] = {0x05,0x10,0x43};
const char F60_DownloadFile[] = {0x05,0x50};
const char F60_DownloadFileInfo[] = {0x05,0x14,0x43};
const char F60_UploadFile[] = {0x05,0x16,0x43};
const char F60_DeleteFile[] = {0x05,0x18,0x43};
const char F60_MoveFiles[] = {0x05,0x1a,0x43};
const char F60_FormatMedia[] = {0x05,0x12,0x43};

const char F60_FormatMediaParam[] = {0x4e}; 

/* Smartmedia support thanks to Tholom Kiely */
const char SMF60_GetDirectory[] = {0x05,0x10,0x41};
//***Same as above, seems to work fine if do SMF60_DownloadFileInfo 1st! 
const char SMF60_DownloadFile[] = {0x05,0x50};
const char SMF60_DownloadFileInfo[] = {0x05,0x14,0x41};
const char SMF60_UploadFile[] = {0x05,0x16,0x41};
const char SMF60_DeleteFile[] = {0x05,0x18,0x41};
const char SMF60_MoveFiles[] = {0x05,0x1a,0x41};
const char SMF60_FormatMedia[] = {0x05,0x12,0x41};


char* f60_strip_path(char* filename) {
     char* c;

     c = filename + strlen(filename);
     while (c != filename) {
	  c--;
	  if (*c == '/' || *c == '\\') {
	       c++;
	       break;
	  }
     }
     return c;
}

#ifndef __BYTE_ORDER
#error  "__BYTE_ORDER not defined"
#endif

// byteswap should only be done on little endian machines. should rename functions
void f60_big2little_endian(int* from,int* to) {
 #if __BYTE_ORDER == LITTLE_ENDIAN
     char* f;
     char* t;

     f = (char*) from;
     t = (char*) to;

     t[3] = f[0];
     t[2] = f[1];
     t[1] = f[2];
     t[0] = f[3];
#else
	*to = *from;
#endif

}

void f60_little2big_endian(int* from,int* to) {
 #if __BYTE_ORDER == LITTLE_ENDIAN
     char* f;
     char* t;

     f = (char*) from;
     t = (char*) to;

     t[0] = f[3];
     t[1] = f[2];
     t[2] = f[1];
     t[3] = f[0];
#else 
	*to = *from;
#endif

}

void f60_set_error(int error, char *fmt, ...) {
     va_list argptr;
     char msg[512];

     va_start(argptr, fmt);
     vsprintf(msg, fmt, argptr);
     va_end(argptr);

     switch (error) {
     case F60_ERROR_USB:
	  snprintf(f60_strerror, 512, "%s : %s", msg,usb_strerror()); 
	  break;
     case F60_ERROR_OTHER:
	  snprintf(f60_strerror, 512, "%s : %s", msg,strerror(errno)); 
	  break;
     default: 
	  snprintf(f60_strerror, 512, "%s",msg); 
     }


     f60_error = error;
     if (debug)
	  printf("%s\n",f60_strerror);
}

void f60_print_data(char* buffer,int len) {
    int i;

    if (len > debug)
	 len = debug;
    
    for(i=0;i<len;i++) {
	 printf("%02x ",(unsigned char) buffer[i]);
	 if (i % 16 == 15)
	      printf("\n");
    }
    printf("\n");
    
}

void f60_set_debug(int level) {
     debug = level;
}

int f60_write_font_string(char* buffer,char* string) {
  const unsigned char* font = mpman_font;
  unsigned char *c;
  unsigned char *output;

  for(c=string, output=buffer;*c != '\0';c++,output = output + 16)
    memcpy(output,font + (*c * 16),16);

  return 1;
}


struct usb_device* f60_get_device_by_id(short vendor,short product) {
    struct usb_bus *bus;
    struct usb_device *dev;

    for (bus = usb_busses; bus; bus = bus->next)
      for(dev = bus->devices; dev; dev = dev->next) 
	if (dev->descriptor.idVendor == vendor)
	  if (dev->descriptor.idProduct == product)
	    return dev;
    
    
    return NULL;
    
} 

/*
int f60_check_for_device() {
    struct usb_device *dev;

    dev = f60_get_device_by_id(F60_VENDOR_ID,F60_PRODUCT_ID);

    if (dev == NULL) {
	  f60_set_error(F60_ERROR_NO_DEVICE,"cannot find device");
	  return 0;
    }


    return 1;
}
*/

struct usb_dev_handle* f60_init() {
     struct usb_dev_handle* ret;
 
     if (debug) printf("Initializing\n");

     f60_dev = f60_get_device_by_id(F60_VENDOR_ID,F60_PRODUCT_ID);
	if (debug && f60_dev) printf("Found MP-Man F60\n");

     // Check for F50 product ID
     if (!f60_dev) {
	  f60_dev = f60_get_device_by_id(F50_VENDOR_ID,F50_PRODUCT_ID);
	  if (debug && f60_dev) printf("Found MP-Man F50\n");
     }
	
	// Check for F55 product ID  
	if (!f60_dev) {
		f60_dev = f60_get_device_by_id(F55_VENDOR_ID, F55_PRODUCT_ID);
		if (debug && f60_dev) printf("Found MP-Man F55\n");	
	}
	
     if (!f60_dev) {
	  f60_set_error(F60_ERROR_NO_DEVICE,"cannot find device");
	  return NULL;
     }
     ret = usb_open(f60_dev);
     if (!ret) {
	  f60_set_error(F60_ERROR_NO_DEVICE,"cannot open device");
	  return NULL;
     }
     usb_set_configuration(ret,f60_dev->config->bConfigurationValue);    
     usb_claim_interface(ret,f60_dev->config->interface->altsetting->bInterfaceNumber);
     usb_set_altinterface(ret,f60_dev->config->interface->altsetting->bAlternateSetting);

     return ret;
}

struct usb_dev_handle* f60_init_with_id(unsigned short vendor,
                                        unsigned short product) {

     struct usb_dev_handle* ret;
 
     if (debug) printf("Initializing\nUsing Vendor ID: %x , Product ID: %x\n",vendor,product);
     
     f60_dev = f60_get_device_by_id(vendor,product);

     if (!f60_dev) {
         f60_set_error(F60_ERROR_NO_DEVICE,"cannot find device (with vendor id: %x , product id: %x)",vendor,product);
         return NULL;
     }
     else {
         if (debug) printf("Found device with vendor id %x , product id %x\n",vendor,product); 
     }
     
     ret = usb_open(f60_dev);
     if (!ret) {
	  f60_set_error(F60_ERROR_NO_DEVICE,"cannot open device");
	  return NULL;
     }
     usb_set_configuration(ret,f60_dev->config->bConfigurationValue);    
     usb_claim_interface(ret,f60_dev->config->interface->altsetting->bInterfaceNumber);
     usb_set_altinterface(ret,f60_dev->config->interface->altsetting->bAlternateSetting);

     return ret;
}

void f60_close(struct usb_dev_handle* usb_hand) {
     if (!usb_hand)
	  return;
     if (f60_dev)
	  usb_release_interface(usb_hand,f60_dev->config->interface->altsetting->bInterfaceNumber);
     usb_close(usb_hand);    
     f60_dev=NULL;
}

int progress_size = 0;

float f60_get_rw_progress() {
	if (progress_size == 0)
			return 0.0f;
	if (progress_size == -1)
			return 1.0f;	
#ifdef USE_SYSTEM_LIBUSB
	return 1;
#else
	return (float) ((float)usb_rw_progress() / (float)progress_size);  	
#endif
}

int f60_write(usb_dev_handle* dev, char* buffer, int size, int timeout) {
  int ret;

  progress_size = size;
  if (debug)
       printf("writing\n");
  if (debug)
       f60_print_data((char*) buffer,size);
  ret = usb_bulk_write(dev,ep_addr_write,buffer,size,timeout);
  progress_size = 0;
  if (ret < 0)
    goto error;
  return 1;

 error:
  f60_set_error(F60_ERROR_USB,"F60_Write failed");
  return 0;
}

int f60_read(usb_dev_handle* dev,  char* buffer, int size, int timeout) {
  int ret;

  progress_size = size;
  if (debug)
       printf("reading\n");

  ret = usb_bulk_read(dev,ep_addr_read,buffer,size,timeout);
  progress_size = 0;
  if (ret < 0)
    goto error;

  if (debug)
       f60_print_data((char*) buffer,size);
  return 1;

 error:
  f60_set_error(F60_ERROR_USB,"F60_Read failed");
  return 0;
}

int f60_check_response(const char* cmd) {
     if ((cmd[0] + 1 != response[0]) || (cmd[1] != response[1])) {
	  f60_set_error(F60_ERROR_CMD,"command was not successful");
	  return 0;
     }
     return 1;

}

int mysizeof(const char* cmd) {

     
     if (cmd == F60_GetFirmwareVersion) return sizeof(F60_GetFirmwareVersion); 
     if (cmd == F60_GetDirectory) return sizeof(F60_GetDirectory);
     if (cmd == F60_DownloadFile) return sizeof(F60_DownloadFile);
     if (cmd == F60_DownloadFileInfo) return sizeof(F60_DownloadFileInfo);
     if (cmd == F60_UploadFile) return sizeof(F60_UploadFile);
     if (cmd == F60_DeleteFile) return sizeof(F60_DeleteFile);
     if (cmd == F60_MoveFiles) return sizeof(F60_MoveFiles);
	 if (cmd == F60_FormatMedia) return sizeof(F60_FormatMedia);
	 
     if (cmd == SMF60_GetDirectory) return sizeof(SMF60_GetDirectory);
     if (cmd == SMF60_DownloadFile) return sizeof(SMF60_DownloadFile);
     if (cmd == SMF60_DownloadFileInfo) return sizeof(SMF60_DownloadFileInfo);
     if (cmd == SMF60_UploadFile) return sizeof(SMF60_UploadFile);
     if (cmd == SMF60_DeleteFile) return sizeof(SMF60_DeleteFile);
     if (cmd == SMF60_MoveFiles) return sizeof(SMF60_MoveFiles);

     return 0;
}

int f60_send_command(usb_dev_handle* dev,const char* cmd,char* param,
		     int param_size,int resp_size) {
     char command[F60_MAX_FILES + 5];
     char* c;

     bzero(response,F60_MAX_RESPONSE);
     
     if (resp_size > F60_MAX_RESPONSE) {
	  f60_set_error(F60_ERROR_CMD,"command response size to big");
	  return 0;
     }

     if (debug)
	  printf("sending command\n");
     
     c = command;
     memcpy(c,cmd,mysizeof(cmd));
     
     // if this is a 3 byte command and we are using smartmedia, change command
     if ( (mysizeof(cmd) == 3) && (selected_memory == SMARTMEDIA_MEMORY)) {
	  if (debug)
	       printf("SMARTMEDIA command\n");
	  c[2] = 0x41;  
     }
     
     c = c + mysizeof(cmd);
     if (param != NULL) {
	  memcpy(c,param,param_size);
	  c = c +  param_size;
     }
     *c = F60_CmdTerminator;
     c++;
     if (! f60_write(dev,command,c - command,2000))
	  return 0;

     if (!f60_read(dev, response,resp_size,30000))
	  return 0;
     response_size = resp_size;
     if (!f60_check_response(cmd))
	  return 0;
     return 1;
}

int f60_get_firmware_version(usb_dev_handle* dev) {
     int ret;
     if (debug)
	  printf("getting firmware version\n");
      ret = f60_send_command(dev,F60_GetFirmwareVersion,NULL,0,7);
      if (!ret)
	   return 0;
      ret = response[4] * 256;
      ret += response[5];
      return ret;
}


struct f60_dir* f60_get_dir(usb_dev_handle *dev) {
     int ret;
     static struct f60_dir dir;

     if (debug)
	  printf("getting directory\n");

     ret = f60_send_command(dev,F60_GetDirectory,NULL,0,12);
     if (!ret)
	  return NULL;

     dir.size = response[10];
     f60_big2little_endian((int*)&response[2],&dir.total_memory);
     f60_big2little_endian((int*)&response[6],&dir.free_memory);
     if (debug)
	  printf("dir size = %d total mem = %d free mem = %d\n",
		 dir.size,dir.total_memory,dir.free_memory);

     if (dir.size == 0) 
	  ret = 1;
     else
	  ret = f60_read(dev,dir.contents,dir.size * 256,3000);

     if (!ret)
	  return NULL;

     return &dir;
}

int f60_send_file(usb_dev_handle *dev,char* filename,char* text) {
     int ret;
     FILE* fp = NULL;
     char* filemap = NULL;
     int filesize;
     int bytesread;
     char fileinfo[1540];
     int fileinfo_size;
     static struct f60_dir* dir;

     if (debug) {
	  printf("sending file %s\n",filename);
	  if (selected_memory == SMARTMEDIA_MEMORY)
	       printf("SMARTMEDIA command\n");
     }


     dir = f60_get_dir(dev);


     fp = fopen(filename,"r");
     if (!fp) {
	  f60_set_error(F60_ERROR_OTHER,"cant open file %s",filename);	  
	  goto error;
     }
     fseek(fp,0,SEEK_END);
     filesize = ftell(fp);
     if (filesize > dir->free_memory) {
	  f60_set_error(F60_ERROR_DEVICE_FULL,"Not enough memory to write file");
	  goto error;
     }
     fseek(fp,0,SEEK_SET);
     filemap = (char*) malloc(filesize);
     if (!filemap) {
	  f60_set_error(F60_ERROR_OTHER,"cant malloc %d",filesize);
	  goto error;
     }

     bytesread = fread(filemap,1,filesize,fp);
     fclose(fp); fp = NULL;
   
     if (bytesread != filesize) {
	  f60_set_error(F60_ERROR_OTHER,"read error file size = %d read =%d",filesize,bytesread);
	  goto error;
     }

     
     ret = f60_send_command(dev,F60_DownloadFile,NULL,0,19);
     if (!ret) goto error;

     if (strlen(text) > 48) text[48] = '\0';
     fileinfo_size = 256 +  strlen(text) * 16;
     fileinfo_size = (256 - (fileinfo_size % 256) ) + fileinfo_size + 4;
     if (debug) printf("fileinfo size = %d\n",fileinfo_size);
     bzero(fileinfo,fileinfo_size);
     if (selected_memory == SMARTMEDIA_MEMORY)
	  memcpy(fileinfo,(char*) SMF60_DownloadFileInfo,sizeof(SMF60_DownloadFileInfo));
     else 
	  memcpy(fileinfo,(char*) F60_DownloadFileInfo,sizeof(F60_DownloadFileInfo));

     f60_little2big_endian(&filesize,(int*)&fileinfo[3]);
     fileinfo[7] = (fileinfo_size / 256) - 1;
     fileinfo[8] = 0x00;
     fileinfo[9] = 0x1a;
     fileinfo[10] = 0x1a; 
     memcpy(&fileinfo[11],"12345-678-",10);
     snprintf(&fileinfo[21],F60_MAX_FILENAME,"C:\\%s",f60_strip_path(filename));
     ret = f60_write_font_string(&fileinfo[259],text);
     if (!ret) goto error;
     fileinfo[fileinfo_size - 1] = 0x04;
     
     ret = f60_write(dev,fileinfo,fileinfo_size,3000);
     if (!ret) goto error;
     ret = f60_read(dev,response,3,1000);
     if (!ret) goto error;
     ret = f60_check_response(F60_DownloadFileInfo);
     if (!ret) goto error;

     ret = f60_write(dev,filemap,filesize,30000);
     if (!ret) goto error;
     ret = f60_read(dev,response,3,1000);
     if (!ret) goto error;
     ret = f60_check_response(F60_DownloadFileInfo);
     if (!ret) goto error;

     free(filemap);
     return 1;
error:
     if (fp)
	  fclose(fp);
     if (filemap)
	  free(filemap);
     return 0;
}

char* f60_get_file(usb_dev_handle *dev,unsigned char index,char* path) {
     FILE* fp = NULL;
     char* filemap = NULL;
     int filesize;
     int nbytes;
     int ret;
     char* filename;
     static char fullpath[256];

     if (debug) printf("getting file number %d to path %s",index,path);

     ret = f60_send_command(dev,F60_UploadFile,&index,1,3);
     if (!ret) goto error;

     ret = f60_read(dev,response,256,1000);
     if (!ret) goto error;

     f60_big2little_endian((int*) response,&filesize);

     filemap = malloc (filesize);
     if (!filemap) {
	  f60_set_error(F60_ERROR_OTHER,"could not malloc %d",filesize);
	  goto error;
     }
     ret = f60_read(dev,filemap,filesize,20000);
     if (!ret) goto error;

     filename = f60_strip_path(&response[21]);

     if (path[strlen(path) - 1] == '/')
	  snprintf(fullpath,256,"%s%s",path,filename);
     else
	  snprintf(fullpath,256,"%s/%s",path,filename);

     fp = fopen(fullpath,"w");
     if (!fp) {
	  f60_set_error(F60_ERROR_OTHER,"could not create %s",fullpath);
	  goto error;
     }

     nbytes = fwrite(filemap,1,filesize,fp);
     if (nbytes != filesize) {
	  f60_set_error(F60_ERROR_OTHER,"write error");
	  goto error;
     }
     fclose(fp); fp = NULL;
     free(filemap);
     
     return fullpath;
error:
     if (fp)
	  fclose(fp);
     if (filemap)
	  free(filemap);
     return NULL;
}

int f60_delete_file(usb_dev_handle *dev,unsigned char index) {
     int ret;

     if (debug) printf("deleteing file no %d\n",index);

     ret = f60_send_command(dev,F60_DownloadFile,NULL,0,19);
     if (!ret) return 0;

     ret = f60_send_command(dev,F60_DeleteFile,&index,1,3);
     if (!ret) return 0;

     return 1;
}
// first array element must contain number(num) of files so command size is (num + 1) bytes 
int f60_move_file(usb_dev_handle *dev,char* order,unsigned char num) {
     if (debug) printf("moving files");
     return f60_send_command(dev,F60_MoveFiles,order,num + 1,3);
}

int f60_format_media(usb_dev_handle *dev, char x1) {
	char x2;
	if (debug) printf("formatting media");
	//printf("formatting media -- x=%d",(int)x1);
        x2=x1;
	return f60_send_command(dev,F60_FormatMedia,(char*)&x2,
         /*F60_FormatMediaParam*/
           1,	3);	
}


int f60_encode_file(char* source,char* dest) {
     FILE* src = NULL;
     FILE* dst = NULL;
     int i;
     int ii;
     int byteswritten;
     unsigned char buffer[512];
     char inchar;

     if (debug) printf("encoding file\n");

     src = fopen(source,"r");
     if (!src) {
	  f60_set_error(F60_ERROR_OTHER,"cant open encode input file");	
	  goto error;
     }
     dst = fopen(dest,"w");
     if (!dst) {
	  f60_set_error(F60_ERROR_OTHER,"cant open encode output file");
	  goto error;
     }
     i=0; ii=0;
     while (!feof(src)) {
	  inchar = fgetc(src);
	  if (feof(src)) break;
	  buffer[(ii * 32) +i] = inchar ^ 1; 
	  if (++ii == 16) {
	       ii = 0;
	       if (++i == 32) {
		    i = 0;
		    byteswritten = fwrite(buffer,512,1,dst);
		    if (!byteswritten) {
			 f60_set_error(F60_ERROR_OTHER,"write error to encode output file");
			 goto error;
		    }
	       }    
	  }
     }     
     
     if (i || ii) {
	  byteswritten = fwrite(buffer, 1,ii + (i * 16),dst);
	  if (!byteswritten) {
	       f60_set_error(F60_ERROR_OTHER,"write error to encode output file");
	       goto error;
	  }
     }


  for(i=0;i<3;i++) {
    fputc(0x00,dst);
    fputc(0x5b,dst);
    for(ii=0;ii<10;ii++)
      fputc(0x5a,dst);
    fputc(0x4d,dst);
    fputc(0x4b,dst);
    fputc(0x4c,dst);
    fputc(0x02,dst);   
  }

  fputc(0x00,dst);
  fputc(0x5b,dst);
  for(i=0;i<29;i++)
    fputc(0x5a,dst);
  fputc(0x5b,dst);

  
     fclose(src);
     fclose(dst);
     return 1;
error:
     if (src)
	  fclose(src);
     if (dst)
	  fclose(dst);
     return 0;
}

int f60_decode_file(char* source,char* dest) {
     FILE* src = NULL;
     FILE* dst = NULL;
     int i;
     int ii;
     int bytesread;
     unsigned char buffer[512];
     int srcsize;

     if (debug) printf("decoding file\n");

    src = fopen(source,"r");
     if (!src) {
	  f60_set_error(F60_ERROR_OTHER,"cant open decode input file");	
	  goto error;
     }
     fseek(src,0,SEEK_END);
     srcsize = ftell(src);
     fclose(src);
     if ( truncate(source,srcsize - (5 * 16)) == -1) {
	  f60_set_error(F60_ERROR_OTHER,"cant truncate decode input file");	
	  goto error;
     }
     src = fopen(source,"r");

     dst = fopen(dest,"w");
     if (!dst) {
	  f60_set_error(F60_ERROR_OTHER,"cant open decode output file");
	  goto error;
     }

     while ( (bytesread = fread(buffer,1,512,src) == 512) )
	  for(i=0;i<32;i++)
	       for (ii=0;ii<16;ii++) 
		    fputc(buffer[(ii *32) + i] ^ 1,dst);
     
     if (bytesread) 
	  for(i=0;i<32;i++)
	       for (ii=0;ii<16;ii++) {
		    if ( (ii*32) + i >= bytesread) continue;
		    fputc(buffer[(ii *32) + i] ^ 1,dst);
	       }
     
  
  fclose(src);
  fclose(dst);

     return 1;
error:
     if (src)
	  fclose(src);
     if (dst)
	  fclose(dst);
     return 0;
}

void f60_select_memory(int a) {

     if (a == SMARTMEDIA_MEMORY)
	  selected_memory = SMARTMEDIA_MEMORY;
     else
	  selected_memory = FLASH_MEMORY;
}
