/*--------------------------------*-C-*--------------------------------------*
 * File:      screen.c
 *---------------------------------------------------------------------------*
 * Copyright (C) 2001 (cursor color) Jordan DeLong <fracture at allusion dot net>
 * Copyright (C) 2000 Sasha Vasko <sashav at sprintmail dot com>
 * Copyright (C) 1997,1998 Geoff Wing <gcw at pobox dot com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*
 * Originally written:
 *    1997,1998 Geoff Wing <mason@primenet.com.au>
 *              - The routine names and calling structure was based upon
 *                previous work by Robert Nation.  Most things have gone
 *                through major change.
 *
 *    Michael Bruun Petersen" <mbp@image.dk>
 *		- tinting fix
 *    2002      Alexis <materm@tele2.fr>
 *              - modifications for multi-terms support
 *--------------------------------------------------------------------------*/
/*
 * We handle _all_ screen updates and selections
 *
 * $Id: screen.c,v 1.16 2004/08/15 16:10:27 alexis Exp $
 */

#include "../config.h"

#ifdef DEBUG
#define DEBUG_SCR_ALEXIS 0
#define DEBUG_SEL_ALEXIS 0
#else
#define DEBUG_SCR_ALEXIS 0
#define DEBUG_SEL_ALEXIS 0
#endif

#if DEBUG_SCR_ALEXIS
#define DSCR_ALEXIS(d,x) if(d <= DEBUG_SCR_ALEXIS) fprintf x
#else
#define DSCR_ALEXIS(d,x)
#endif

#if DEBUG_SEL_ALEXIS
#define DSEL_ALEXIS(d,x) if(d <= DEBUG_SEL_ALEXIS) fprintf x
#else
#define DSEL_ALEXIS(d,x)
#endif



#define INTERN_SCREEN
#define LIBAFTERSTEP_HEADER_INCLUDED

#include "rxvt.h"		/* NECESSARY */

#include <X11/Xatom.h>
#include <X11/Xmd.h>		/* get the typedef for CARD32 */

static screen_t screen[MAX_PAGES];

int             colors_changed = 0;	/* to indicate that foreground and background colors has changed */

/* ------------------------------------------------------------------------- */
#ifdef MULTICHAR_SET
short           multi_byte;	/* set ==> currently using 2 bytes per glyph */
short           lost_multi;	/* set ==> we only got half a glyph */
enum
{
	SBYTE, WBYTE
}
chstat;

# define RESET_CHSTAT			\
    if (chstat == WBYTE)		\
	chstat = SBYTE, lost_multi = 1
#else
# define RESET_CHSTAT
#endif

typedef enum
{
	EUCJ, SJIS,		/* KANJI methods */
	BIG5, CNS		/* BIG5 methods: CNS not implemented */
}
ENC_METHOD;

#if defined(KANJI) || defined(ZH)
ENC_METHOD      encoding_method;
#endif

/* ------------------------------------------------------------------------- */
#define PROP_SIZE		4096
#define TABSIZE			8	/* default tab size */

#define DEBUG_SCREEN 1
#ifdef DEBUG_SCREEN
# define D_SCREEN(x)		fprintf x ; fputc('\n', stderr)
#else
# define D_SCREEN(x)
#endif
#ifdef DEBUG_SELECT
# define D_SELECT(x)		fprintf x ; fputc('\n', stderr)
#else
# define D_SELECT(x)
#endif

/* ------------------------------------------------------------------------- *
 *             GENERAL SCREEN AND SELECTION UPDATE ROUTINES                  *
 * ------------------------------------------------------------------------- */

#define f_ZERO_SCROLLBACK(page)					\
  if ((Options & Opt_scrollTtyOutputInh) == 0)	\
    TermWin.vts[page].view_start = 0

#define CHECK_SELECTION(pagepage,x)				\
    if (selection.op)					\
	f_selection_check(page,x)
#define CLEAR_SELECTION					\
    selection.beg.row = selection.beg.col		\
	= selection.end.row = selection.end.col = 0
#define CLEAR_ALL_SELECTION				\
    selection.beg.row = selection.beg.col		\
	= selection.mark.row = selection.mark.col	\
	= selection.end.row = selection.end.col = 0

#define ROW_AND_COL_IS_AFTER(A, B, C, D)				\
    (((A) > (C)) || (((A) == (C)) && ((B) > (D))))
#define ROW_AND_COL_IS_BEFORE(A, B, C, D)				\
    (((A) < (C)) || (((A) == (C)) && ((B) < (D))))
#define ROW_AND_COL_IN_ROW_AFTER(A, B, C, D)				\
    (((A) == (C)) && ((B) > (D)))
#define ROW_AND_COL_IN_ROW_ON_OR_AFTER(A, B, C, D)			\
    (((A) == (C)) && ((B) >= (D)))
#define ROW_AND_COL_IN_ROW_BEFORE(A, B, C, D)				\
    (((A) == (C)) && ((B) < (D)))
#define ROW_AND_COL_IN_ROW_ON_OR_BEFORE(A, B, C, D)			\
    (((A) == (C)) && ((B) <= (D)))

/* these must be row_col_t */
#define ROWCOL_IS_AFTER(X, Y)						\
    ROW_AND_COL_IS_AFTER((X).row, (X).col, (Y).row, (Y).col)
#define ROWCOL_IS_BEFORE(X, Y)						\
    ROW_AND_COL_IS_BEFORE((X).row, (X).col, (Y).row, (Y).col)
#define ROWCOL_IN_ROW_AFTER(X, Y)					\
    ROW_AND_COL_IN_ROW_AFTER((X).row, (X).col, (Y).row, (Y).col)
#define ROWCOL_IN_ROW_BEFORE(X, Y)					\
    ROW_AND_COL_IN_ROW_BEFORE((X).row, (X).col, (Y).row, (Y).col)
#define ROWCOL_IN_ROW_ON_OR_AFTER(X, Y)					\
    ROW_AND_COL_IN_ROW_ON_OR_AFTER((X).row, (X).col, (Y).row, (Y).col)
#define ROWCOL_IN_ROW_ON_OR_BEFORE(X, Y)				\
    ROW_AND_COL_IN_ROW_ON_OR_BEFORE((X).row, (X).col, (Y).row, (Y).col)

void print_info_screen() {
  fprintf(stderr,"screen[%d] : tscroll: %d, bscroll: %d, lscroll: %d, rscroll: %d\n",TermWin.active_page,
    screen[TermWin.active_page].tscroll,screen[TermWin.active_page].bscroll,
    screen[TermWin.active_page].lscroll,screen[TermWin.active_page].rscroll
  );
}

/*
 * CLEAR_ROWS : clear <num> rows starting from row <row>
 * CLEAR_CHARS: clear <num> chars starting from pixel position <x,y>
 * ERASE_ROWS : set <num> rows starting from row <row> to the foreground colour
 */

# define drawBufferPage(x)	(TermWin.vts[x].vt)
void
f_CLEAR_ROWS(int page,unsigned row, unsigned num)
{
	unsigned        line_h = Height2Pixel(1) - 1;
	register unsigned hy = (line_h + 1) * num, y = Row2Pixel(row);

  DSCR_ALEXIS(2,(stderr,"vt[%d] is %smapped\n",page,TermWin.vts[page].bMapped ? "" : "Un"));
	if (!TermWin.vts[page].bMapped)
		return;

	if (y < TermWin_internalBorder + line_h)
	{
		y = 0;
		hy += TermWin_internalBorder;
	}
	if (y + hy > (TermWin.height + TermWin_internalBorder) - line_h)
		hy = (TermWin.height - y) + TermWin_internalBorders;
  
	XClearArea(Xdisplay, drawBufferPage(page), 0, y,
		   TermWin.width + TermWin_internalBorders, hy, 0);
#if defined(BACKGROUND_IMAGE) || defined(TRANSPARENT)
	if (TermWin.vts[page].tintGC)
	{
		XFillRectangle(Xdisplay, drawBufferPage(page), TermWin.vts[page].tintGC,
      0, y, TermWin.width + TermWin_internalBorders,hy);
	}
#endif
}

void
f_CLEAR_CHARS(int page,unsigned x, unsigned y, unsigned num)
{
	unsigned char_w = Width2Pixel(1) - 1;
  unsigned line_h = Height2Pixel(1) - 1;
	register unsigned hx = (char_w + 1) * num, hy = line_h + 1;

	if (!TermWin.vts[page].bMapped)
		return;

	if (x < TermWin_internalBorder + char_w)
	{
		x = 0;
		hx += TermWin_internalBorder;
	}
	if (y < TermWin_internalBorder + line_h)
	{
		y = 0;
		hy += TermWin_internalBorder;
	}

	if (x + hx > (TermWin.width + TermWin_internalBorder) - char_w)
		hx = (TermWin.width + TermWin_internalBorders + 1) - x;
	if (y + hy > (TermWin.height + TermWin_internalBorder) - line_h)
		hy = (TermWin.height + TermWin_internalBorders + 1) - y;
	XClearArea(Xdisplay, drawBufferPage(page), x, y, hx, hy, 0);
#if defined(BACKGROUND_IMAGE) || defined(TRANSPARENT)
	if (TermWin.vts[page].tintGC)
	{
		XFillRectangle(Xdisplay, drawBufferPage(page), TermWin.vts[page].tintGC, x, y, hx,hy);
	}
#endif
}


#define f_ERASE_ROWS(page,row, num) 						\
    XFillRectangle(Xdisplay, drawBufferPage(page), TermWin.vts[page].gc,			\
		   Col2Pixel(0), Row2Pixel(row),			\
		   TermWin.width, Height2Pixel(num))

/* ------------------------------------------------------------------------- *
 *                        SCREEN `COMMON' ROUTINES                           *
 * ------------------------------------------------------------------------- */
/* Fill part/all of a line with blanks. */
/* PROTO */
void
blank_line(text_t * et, rend_t * er, int width, rend_t efs)
{
	MEMSET(et, ' ', width);
	for (; width--;)
		*er++ = efs;
}

/* ------------------------------------------------------------------------- */
/* allocate memory for this screen line and blank it appropriately */
/* PROTO */
void
f_make_screen_mem(int page,text_t ** tp, rend_t ** rp, int row, rend_t rstyle)
{
	tp[row] = MALLOC(sizeof(text_t) * TermWin.vts[page].bcol);
	rp[row] = MALLOC(sizeof(rend_t) * TermWin.vts[page].bcol);
	blank_line(tp[row], rp[row], TermWin.vts[page].bcol, rstyle);
}

/* ------------------------------------------------------------------------- *
 *                          SCREEN INITIALISATION                            *
 * ------------------------------------------------------------------------- */
static int prev_nrows[MAX_PAGES];
static int prev_bcols[MAX_PAGES];

void scr_init() {
  int i;
  
  for(i=0; i < MAX_PAGES; i++) {
    prev_nrows[i] = -1;
    prev_bcols[i] = -1;
  }
}


#ifdef DEBUG_SCR_ALEXIS
  
void put_screen(int page) {
  int i,j;
  fprintf(stderr,"tscroll:%d, nscroll:%d, nscrolled:%d\n",screen[page].tscroll,
    screen[page].bscroll,TermWin.vts[page].nscrolled);
  for(i=0; i < TermWin.vts[page].saveLines; i++) {
    if(screen[page].text[i]) {
      fprintf(stderr,"saveLines.text[%d] = ",i);
      for(j=0; j < screen[page].tlen[i]; j++) {
        fputc(screen[page].text[i][j],stderr);
      }
      fputc('\n',stderr);
    } else {
      fprintf(stderr,"saveLines.text[%d] = NULL\n",i);
    }
  }
  for(; i < TermWin.vts[page].saveLines+TermWin.nrow; i++) {
    if(screen[page].text[i]) {
      fprintf(stderr,"screen.text[%d] = ",i);
      for(j=0; j < screen[page].tlen[i]; j++) {
        fputc(screen[page].text[i][j],stderr);
      }
      fputc('\n',stderr);
    } else {
      fprintf(stderr,"screen.text[%d] = NULL\n",i);
    }
  }
}

void put_drawn(int page) {
  int i,j;
  for(i=0; i < TermWin.nrow; i++) {
    if(drawn_text[i]) {
      fprintf(stderr,"drawn_text[%d] = ",i);
      for(j=0; j < screen[page].tlen[i]; j++) {
        fputc(drawn_text[i][j],stderr);
      }
      fputc('\n',stderr);
    } else {
      fprintf(stderr,"drawn_text[%d] = NULL",i);
    }
  }
}


#define display_screen(page) put_screen(page)
#define display_drawn(page) put_drawn(page)
#else
#define display_screen(page)
#define display_drawn(page)
#endif  /* DEBUG_SCR_ALEXIS */


void f_scr_reset(int page)
{
  int        i, j, k, total_rows, prev_total_rows;
  rend_t     setrstyle;
  static int prev_nrow = -1;
  static int prev_ncol = -1;
  static int scr_mem_done = 0;
  static int resize_col_done = 0;
  
  DSCR_ALEXIS(1,(stderr, "scr_reset(%d)\n",page));
  
  TermWin.vts[page].view_start = 0;
  RESET_CHSTAT;
  
  if (TermWin.vts[page].bcol == prev_bcols[page] && TermWin.nrow == prev_nrows[page]) {
    DSCR_ALEXIS(1,(stderr,"scr_reset : no reset\n"));
    return;
  }

#ifdef DEBUG_STRICT
  assert(TermWin.vts[page].saveLines >= 0);
#else				/* drive with your eyes closed */
  MAX_IT(TermWin.vts[page].saveLines, 0);
#endif

  total_rows = TermWin.nrow + TermWin.vts[page].saveLines;
  prev_total_rows = prev_nrows[page] + TermWin.vts[page].saveLines;
  
  screen[page].tscroll = 0;
  screen[page].bscroll = (TermWin.nrow - 1);
  
  /* *INDENT-OFF* */
  if (prev_nrows[page] == -1) {
/*
 * A: first time called so just malloc everything : don't rely on realloc
 *    Note: this is still needed so that all the scrollback lines are NULL
 */	
    DSCR_ALEXIS(2,(stderr, "scr_reset(): first call\n",page));
    screen[page].text	= CALLOC(text_t * , total_rows);
    screen[page].tlen	= CALLOC(R_int16_t, total_rows);
    screen[page].rend = CALLOC(rend_t *, total_rows  );
    
    /* - on fait attention de n'allouer buf_*, swap_*, et
     *   drawn_* qu'une seule fois si +sieurs appel de scr_reset
     * - idem pour selection
     */
    if( prev_nrow == -1 ) {
      DSCR_ALEXIS(2,(stderr, "scr_reset(): buf, drawn et swap initialised\n"));
      buf_text	= CALLOC(text_t * , total_rows);
      buf_tlen	= CALLOC(R_int16_t, total_rows);
      buf_rend  = CALLOC(rend_t *, total_rows);
      
      drawn_text	= CALLOC(text_t * , TermWin.nrow);
      drawn_rend  = CALLOC(rend_t *, TermWin.nrow);
      for(i=0; i < TermWin.nrow; i++) {
        
        f_make_screen_mem(page,drawn_text, drawn_rend, i, DEFAULT_RSTYLE );
      }
      
      swap.text	= CALLOC(text_t * , TermWin.nrow);
      swap.tlen	= CALLOC(R_int16_t, TermWin.nrow);
      swap.rend = CALLOC(rend_t *, TermWin.nrow);
      for(i=0; i < TermWin.nrow; i++) {
        f_make_screen_mem(page,swap.text, swap.rend, i, DEFAULT_RSTYLE);
        swap.tlen[i] = 0;
      }
      swap.flags = Screen_DefaultFlags;
      
      selection.text = NULL;
      selection.op = SELECTION_CLEAR;
      selection.screen = PRIMARY;
      selection.clicks = 0;
      CLEAR_ALL_SELECTION;
      
      tabs = MALLOC(TermWin.vts[page].bcol * sizeof(char));
      for(i=0; i < TermWin.vts[page].bcol; i++) {
        tabs[i] = (i % TABSIZE == 0) ? 1 : 0;
      }
    }
    
    for (i = 0; i < TermWin.nrow; i++) {
      j = i + TermWin.vts[page].saveLines;
      f_make_screen_mem(page,screen[page].text, screen[page].rend, j, DEFAULT_RSTYLE);
      screen[page].tlen[j] = 0;
    }
    
    TermWin.vts[page].nscrolled = 0;	/* no saved lines */
    screen[page].flags = Screen_DefaultFlags;
    screen[page].cur.row = screen[page].cur.col = 0;
    save[page].cur.row = save[page].cur.col = 0;
    save[page].charset = 0;
    save[page].charset_char = 'B';
    rstyle = save[page].rstyle = DEFAULT_RSTYLE;
    MEMSET(charsets, 'B', sizeof(charsets));
    current_screen = PRIMARY;
    rvideo = 0;
#ifdef MULTICHAR_SET
    multi_byte = 0;
    lost_multi = 0;
    chstat = SBYTE;
# ifdef KANJI
    encoding_method = EUCJ;
# else
    encoding_method = BIG5;
# endif
#endif

  } else {
/*
 * B1: add or delete rows as appropriate
 */
    setrstyle = DEFAULT_RSTYLE | (rvideo ? RS_RVid : 0);
    DSCR_ALEXIS(2,(stderr, "scr_reset(): general case\n",page));

    if (TermWin.nrow < prev_nrows[page]) {
    /* delete rows */
      DSCR_ALEXIS(2,(stderr,"removing rows: %d->%d\n",
        prev_nrows[page],TermWin.nrow));
      k = min(TermWin.vts[page].nscrolled, prev_nrows[page] - TermWin.nrow);
      
      // il faut scroller vers le haut
      // du nombre de ligne qu'on a perdu
      f_scroll_text(page,0, prev_nrows[page] - 1, k, 1);
      
      for (i = TermWin.nrow; i < prev_nrows[page]; i++) {
        j = i + TermWin.vts[page].saveLines;
        if (screen[page].text[j])
            FREE(screen[page].text[j]);
        
        if (screen[page].rend[j])
            FREE(screen[page].rend[j]);
      }
      
      if( prev_nrow != TermWin.nrow ) {
        for(i = TermWin.nrow; i < prev_nrows[page]; i++) {
          if (swap.text[i]) {
            FREE(swap.text[i]);
          }
          
          if (swap.rend[i]) {
            FREE(swap.rend[i]);
          }
          
          FREE(drawn_text[i]);
          FREE(drawn_rend[i]);
          
        }
      }
    /* we have fewer rows so fix up number of scrolled lines */
      MIN_IT(screen[page].cur.row, TermWin.nrow - 1);
      MIN_IT(swap.cur.row, TermWin.nrow - 1);
    }
    
    if(TermWin.nrow != prev_nrows[page]) {
      screen[page].text = REALLOC(screen[page].text, total_rows  *sizeof(text_t*));
      screen[page].tlen = REALLOC(screen[page].tlen, total_rows  *sizeof(R_int16_t));
      if( prev_nrow != TermWin.nrow ) {
        DSCR_ALEXIS(2,(stderr,"reallocating swap, buf et drawn\n"));
        buf_text  	= REALLOC(buf_text   , total_rows  *sizeof(text_t*));
        buf_tlen  	= REALLOC(buf_tlen   , total_rows  *sizeof(R_int16_t));
        buf_rend  	= REALLOC(buf_rend   , total_rows  *sizeof(rend_t*));
        
        drawn_text  = REALLOC(drawn_text , TermWin.nrow*sizeof(text_t*));
        drawn_rend  = REALLOC(drawn_rend , TermWin.nrow*sizeof(rend_t*));
        

        swap.text  	= REALLOC(swap.text  , TermWin.nrow*sizeof(text_t*));
        swap.tlen   = REALLOC(swap.tlen  , total_rows  *sizeof(R_int16_t));
        swap.rend  	= REALLOC(swap.rend  , TermWin.nrow*sizeof(rend_t*));
      }
      screen[page].rend = REALLOC(screen[page].rend, total_rows  *sizeof(rend_t*));
    }
    
    if (TermWin.nrow > prev_nrows[page]) {
      /* add rows */
      DSCR_ALEXIS(2,(stderr,"adding rows: %d->%d\n",
        prev_nrows[page],TermWin.nrow));
      /* k correspond au nbr de lignes qui vont passer de la partie
       * nscrolled a la partie [tscroll,bscroll] => il faut allouer
       * k nouvelles lgn
       */
      k = min(TermWin.vts[page].nscrolled, TermWin.nrow - prev_nrows[page]);
      
      /* on fait l'allocation des nouvelles lgn a la fin
       */
      for (i = prev_total_rows; i < total_rows - k; i++) {
        f_make_screen_mem(page,screen[page].text, screen[page].rend, i, setrstyle);
        screen[page].tlen[i] = 0;
      }
      for (i = total_rows - k; i < total_rows; i++) {
        screen[page].text[i] = NULL;
        screen[page].rend[i] = NULL;
        screen[page].tlen[i] = 0;
      }
      /* on alloue les nouvelles lgn de la partie
       * qui n'est pas scrollee seulement pour swap et drawn_*
       */
      if( prev_nrow != TermWin.nrow ) {
        
        for (i = prev_nrows[page]; i < TermWin.nrow; i++) {
          f_make_screen_mem(page,swap.text, swap.rend, i, setrstyle);
          f_make_screen_mem(page,drawn_text, drawn_rend, i, setrstyle );
          swap.tlen[i] = 0;
        }
        
      }
      
      if (k > 0) {
        f_scroll_text(page,0, TermWin.nrow - 1, -k, 1);
        screen[page].cur.row += k;
        TermWin.vts[page].nscrolled -= k;
        for (i = TermWin.vts[page].saveLines - TermWin.vts[page].nscrolled; k--; i--)
          if (screen[page].text[i] == NULL) {
            f_make_screen_mem(page,screen[page].text, screen[page].rend, i,setrstyle);
            screen[page].tlen[i] = 0;
          }
      }
#ifdef DEBUG_STRICT
      assert(screen[page].cur.row < TermWin.nrow);
      assert(swap.cur.row < TermWin.nrow);
#else				/* drive with your eyes closed */
      MIN_IT(screen[page].cur.row, TermWin.nrow - 1);
      MIN_IT(swap.cur.row, TermWin.nrow - 1);
#endif
    }

/* B2: resize columns */
    if (TermWin.vts[page].bcol != prev_bcols[page]) {
      for (i = 0; i < total_rows; i++) {
        if (screen[page].text[i]) {
          screen[page].text[i] = REALLOC(screen[page].text[i],
          TermWin.vts[page].bcol * sizeof(text_t));
          screen[page].rend[i] = REALLOC(screen[page].rend[i],
          TermWin.vts[page].bcol * sizeof(rend_t));
          MIN_IT(screen[page].tlen[i], TermWin.vts[page].bcol);
          if (TermWin.vts[page].bcol > prev_bcols[page])
            blank_line(&(screen[page].text[i][prev_bcols[page]]),&(screen[page].rend[i][prev_bcols[page]]),
              TermWin.vts[page].bcol - prev_bcols[page], setrstyle);
        }
      }
      
      /* resize swap and drawn only if we see that ncol has changed
       * after the last call to scr_reset
       */
      if( prev_ncol !=  TermWin.vts[page].bcol ) {
        DSCR_ALEXIS(2,(stderr,"resizing col\n"));
        for (i = 0; i < TermWin.nrow; i++) {
          drawn_text[i] = REALLOC(drawn_text[i],TermWin.vts[page].bcol * sizeof(text_t));
          drawn_rend[i] = REALLOC(drawn_rend[i],TermWin.vts[page].bcol * sizeof(rend_t));
          if (swap.text[i]) {
            swap.text[i] = REALLOC(swap.text[i],TermWin.vts[page].bcol * sizeof(text_t));
            swap.rend[i] = REALLOC(swap.rend[i],TermWin.vts[page].bcol * sizeof(rend_t));
            MIN_IT(swap.tlen[i], TermWin.vts[page].bcol);
            if (TermWin.vts[page].bcol > prev_bcols[page])
              blank_line(&(swap.text[i][prev_bcols[page]]),&(swap.rend[i][prev_bcols[page]]),
                TermWin.vts[page].bcol - prev_bcols[page], setrstyle);
          }
          if (TermWin.vts[page].bcol > prev_bcols[page])
            blank_line(&(drawn_text[i][prev_bcols[page]]),&(drawn_rend[i][prev_bcols[page]]),
              TermWin.vts[page].bcol - prev_bcols[page], setrstyle);
        }
        if (tabs)
          FREE(tabs);
        tabs = MALLOC(TermWin.vts[page].bcol * sizeof(char));
      
        for (i = 0; i < TermWin.vts[page].bcol; i++)
          tabs[i] = (i % TABSIZE == 0) ? 1 : 0;
      }
      
      MIN_IT(screen[page].cur.col, TermWin.vts[page].bcol - 1);
      MIN_IT(swap.cur.col, TermWin.vts[page].bcol - 1);
    } //end of -> if (TermWin.vts[page].bcol != prev_bcols[page])
  }

  prev_nrows[page] = TermWin.nrow;
  prev_nrow = TermWin.nrow;
  
  prev_bcols[page] = TermWin.vts[page].bcol;
  prev_ncol = TermWin.vts[page].bcol;

  /* don't do that here because the fd that page refer to is
   * is created ...
   * do it in f_run_command
   */
  /* f_tt_resize(page); */

}

/* ------------------------------------------------------------------------- */
/*
 * Free everything.  That way malloc debugging can find leakage.
 */
// on ne libere drawn que si c'est le dernier screen qui reste
/* PROTO */
void
f_scr_release(int page,int release_drawn)
{
	int             i, total_rows;

  DSCR_ALEXIS(1,(stderr,"f_scr_release(%d,%d)\n",page,release_drawn));
	total_rows = TermWin.nrow + TermWin.vts[page].saveLines;
	for (i = 0; i < total_rows; i++)
	{
		if (screen[page].text[i])
		{		/* then so is screen.rend[i] */
			FREE(screen[page].text[i]);
			FREE(screen[page].rend[i]);
		}
	}
  if(release_drawn) {
    for (i = 0; i < TermWin.nrow; i++)
    {
      FREE(drawn_text[i]);
      FREE(drawn_rend[i]);
      FREE(swap.text[i]);
      FREE(swap.rend[i]);
    }
	}
	FREE(screen[page].text);
	FREE(screen[page].tlen);
	FREE(screen[page].rend);
  if( release_drawn) {
    FREE(drawn_text);
    FREE(drawn_rend);
    FREE(buf_text);
    FREE(buf_tlen);
    FREE(buf_rend);
   	FREE(swap.text);
    FREE(swap.tlen);
    FREE(swap.rend);
    FREE(tabs);
  }
	

/* NULL these so if anything tries to use them, we'll know about it */
  if(release_drawn) {
    drawn_text = swap.text = NULL;
    drawn_rend = swap.rend = NULL;
    buf_text = NULL;
    buf_rend = NULL;
    buf_tlen = NULL;
    tabs = NULL;
  }
	screen[page].text = NULL;
	screen[page].rend = NULL;
	screen[page].tlen = NULL;
	
}

/* ------------------------------------------------------------------------- */

void pat_scr_remove(int page,int all) {
  int i;
  
  f_scr_release(page,all);
  for(i=page; i < TermWin.last_page; i++) {
    screen[i] = screen[i+1];
    prev_nrows[i] = prev_nrows[i];
    prev_bcols[i] = prev_bcols[i];
  }
  // il faut reinitialiser prev_nrows et prev_bcols
  prev_nrows[i] = prev_bcols[i] = -1;
}

/* ------------------------------------------------------------------------- */

void f_scr_poweron(int page)
{
	DSCR_ALEXIS(1,(stderr, "scr_poweron()\n"));

	MEMSET(charsets, 'B', sizeof(charsets));
	rvideo = 0;
	swap.tscroll = 0;
	swap.bscroll = TermWin.nrow - 1;
	screen[page].cur.row = screen[page].cur.col = swap.cur.row = swap.cur.col = 0;
	screen[page].charset = swap.charset = 0;
	screen[page].flags = swap.flags = Screen_DefaultFlags;

	f_scr_cursor(page,SAVE);

	f_scr_release(page,0);
	prev_nrows[page] = -1;
	prev_bcols[page] = -1;
	f_scr_reset(page);

	f_scr_clear(page);
	f_scr_refresh(page,SLOW_REFRESH);

}

/* ------------------------------------------------------------------------- *
 *                         PROCESS SCREEN COMMANDS                           *
 * ------------------------------------------------------------------------- */
/*
 * Save and Restore cursor
 * XTERM_SEQ: Save cursor   : ESC 7
 * XTERM_SEQ: Restore cursor: ESC 8
 */



/* PROTO */
void
f_scr_cursor(int page,int mode)
{
	DSCR_ALEXIS(1,(stderr, "scr_cursor(%d,%d)\n", page,mode));

	switch (mode)
	{
	case SAVE:
		save[page].cur.row = screen[page].cur.row;
		save[page].cur.col = screen[page].cur.col;
		save[page].rstyle = rstyle;
		save[page].charset = screen[page].charset;
		save[page].charset_char = charsets[screen[page].charset];
		break;
	case RESTORE:
		screen[page].cur.row = save[page].cur.row;
		screen[page].cur.col = save[page].cur.col;
		rstyle = save[page].rstyle;
		screen[page].charset = save[page].charset;
		charsets[screen[page].charset] = save[page].charset_char;
		f_set_font_style(page);
		break;
	}
/* boundary check in case screen size changed between SAVE and RESTORE */
	MIN_IT(screen[page].cur.row, TermWin.nrow - 1);
	MIN_IT(screen[page].cur.col, TermWin.vts[page].bcol - 1);
#ifdef DEBUG_STRICT
	assert(screen[page].cur.row >= 0);
	assert(screen[page].cur.col >= 0);
#else				/* drive with your eyes closed */
	MAX_IT(screen[page].cur.row, 0);
	MAX_IT(screen[page].cur.col, 0);
#endif
}

/* ------------------------------------------------------------------------- */
/*
 * Swap between primary and secondary screens
 * XTERM_SEQ: Primary screen  : ESC [ ? 4 7 h
 * XTERM_SEQ: Secondary screen: ESC [ ? 4 7 l
 */


/* PROTO */
int
f_scr_change_screen(int page,int scrn)
{
	int             i, offset, tmp;
	text_t         *t0;
	rend_t         *r0;
	R_int16_t       l0;

	DSCR_ALEXIS(1,(stderr, "scr_change_screen(%d)\n", scrn));

	TermWin.vts[page].view_start = 0;
	RESET_CHSTAT;

	if (current_screen == scrn)
		return current_screen;

  DSCR_ALEXIS(2,(stderr,"Gloups, continuing in change screen\n"));
	CHECK_SELECTION(pagepage,2);	/* check for boundary cross */

	SWAP_IT(current_screen, scrn, tmp);
#if NSCREENS
	offset = TermWin.vts[page].saveLines;
	for (i = TermWin.nrow; i--;)
	{
		SWAP_IT(screen[page].text[i + offset], swap.text[i], t0);
		SWAP_IT(screen[page].tlen[i + offset], swap.tlen[i], l0);
		SWAP_IT(screen[page].rend[i + offset], swap.rend[i], r0);
	}
	SWAP_IT(screen[page].cur.row, swap.cur.row, l0);
	SWAP_IT(screen[page].cur.col, swap.cur.col, l0);
# ifdef DEBUG_STRICT
	assert(screen[page].cur.row >= 0);
	assert(screen[page].cur.col >= 0);
	assert(screen[page].cur.row < TermWin.nrow);
	assert(screen[page].cur.col < TermWin.vts[page].bcol);
# else				/* drive with your eyes closed */
	MAX_IT(screen[page].cur.row, 0);
	MAX_IT(screen[page].cur.col, 0);
	MIN_IT(screen[page].cur.row, TermWin.nrow - 1);
	MIN_IT(screen[page].cur.col, TermWin.vts[page].bcol - 1);
# endif
	SWAP_IT(screen[page].charset, swap.charset, l0);
	SWAP_IT(screen[page].flags, swap.flags, tmp);
	screen[page].flags |= Screen_VisibleCursor;
	swap.flags |= Screen_VisibleCursor;

#else
# ifndef DONT_SCROLL_ME
	if (current_screen == PRIMARY)
	{

		f_scroll_text(page,0, (TermWin.nrow - 1), TermWin.nrow, 0);
		for (i = TermWin.vts[page].saveLines;
		     i < TermWin.nrow + TermWin.vts[page].saveLines; i++)
			if (screen[page].text[i] == NULL)
			{
				f_make_screen_mem(page,screen[page].text, screen[page].rend, i,DEFAULT_RSTYLE);
				screen[page].tlen[i] = 0;
			}
	}
# endif
#endif
	return scrn;
}

/* ------------------------------------------------------------------------- */
/*
 * Change the colour for following text
 */
/* PROTO */
void
scr_color(unsigned int color, unsigned int Intensity)
{
	if (color == restoreFG)
		color = Color_fg;
	else if (color == restoreBG)
		color = Color_bg;
	else
	{
		if (Xdepth <= 2)
		{		/* Monochrome - ignore colour changes */
			switch (Intensity)
			{
			case RS_Bold:
				color = Color_fg;
				break;
			case RS_Blink:
				color = Color_bg;
				break;
			}
		} else
		{
#ifndef NO_BRIGHTCOLOR
			if ((rstyle & Intensity) && color >= minCOLOR
			    && color <= maxCOLOR)
				color += (minBrightCOLOR - minCOLOR);
			else if (color >= minBrightCOLOR
				 && color <= maxBrightCOLOR)
			{
				if (rstyle & Intensity)
					return;
				color -= (minBrightCOLOR - minCOLOR);
			}
#endif
		}
	}
	switch (Intensity)
	{
	case RS_Bold:
		rstyle = SET_FGCOLOR(rstyle, color);
		break;
	case RS_Blink:
		rstyle = SET_BGCOLOR(rstyle, color);
		break;
	}
}

/* ------------------------------------------------------------------------- */
/*
 * Change the rendition style for following text
 */
/* PROTO */
void
scr_rendition(int set, int style)
{
	unsigned int    color;
	rend_t          font_attr;

	if (set)
	{
/* A: Set style */
		rstyle |= style;
		switch (style)
		{
		case RS_RVid:
			if (rvideo)
				rstyle &= ~RS_RVid;
			break;
#ifndef NO_BRIGHTCOLOR
		case RS_Bold:
			color = GET_FGCOLOR(rstyle);
			scr_color((color == Color_fg ? GET_FGCOLOR(colorfgbg) : color),RS_Bold);
			break;
		case RS_Blink:
			color = GET_BGCOLOR(rstyle);
			scr_color((color == Color_bg ? GET_BGCOLOR(colorfgbg) : color),RS_Blink);
			break;
#endif
		}
	} else
	{
/* B: Unset style */
		font_attr = rstyle & RS_fontMask;
		rstyle &= ~style;

		switch (style)
		{
		case ~RS_None:	/* default fg/bg colours */
			rstyle = DEFAULT_RSTYLE | font_attr;
		    /* FALLTHROUGH */
		case RS_RVid:
			if (rvideo)
				rstyle |= RS_RVid;
			break;
#ifndef NO_BRIGHTCOLOR
		case RS_Bold:
			color = GET_FGCOLOR(rstyle);
			if (color >= minBrightCOLOR && color <= maxBrightCOLOR)
			{
				scr_color(color, RS_Bold);
				if ((rstyle & RS_fgMask) == (colorfgbg & RS_fgMask))
					scr_color(restoreFG, RS_Bold);
			}
			break;
		case RS_Blink:
			color = GET_BGCOLOR(rstyle);
			if (color >= minBrightCOLOR && color <= maxBrightCOLOR)
			{
				scr_color(color, RS_Blink);
				if ((rstyle & RS_bgMask) == (colorfgbg & RS_bgMask))
					scr_color(restoreBG, RS_Blink);
			}
			break;
#endif
		}
	}
}

/* ------------------------------------------------------------------------- */
/*
 * Scroll text between <row1> and <row2> inclusive, by <count> lines
 * count positive ==> scroll up
 * count negative ==> scroll down
 * spec == 0 for normal routines
 */


/* PROTO */
int
f_scroll_text(int page,int row1, int row2, int count, int spec)
{
	int             i, j;

  DSCR_ALEXIS(1,(stderr, "f_scroll_text(%d,%d,%d,%d,%d): %s\n", page,row1, row2, count,
    spec,(current_screen == PRIMARY) ? "Primary" : "Secondary"));

  if (count == 0 || (row1 > row2))
    return 0;

  if ((count > 0) && (row1 == 0) && (current_screen == PRIMARY)) {
    DSCR_ALEXIS(2,(stderr,"TermWin.nscrolled = %d, TermWin.saveLines = %d\n",
      TermWin.vts[page].nscrolled,TermWin.vts[page].saveLines));
    TermWin.vts[page].nscrolled += count;
    MIN_IT(TermWin.vts[page].nscrolled, TermWin.vts[page].saveLines);
    
  } else if (!spec)
    row1 += TermWin.vts[page].saveLines;
  row2 += TermWin.vts[page].saveLines;
  
  DSCR_ALEXIS(2,(stderr, "scroll_text(...) -> %d,%d\n",row1,row2));
  
  if (selection.op && current_screen == selection.screen && page == selection.vt)
  {
    i = selection.beg.row + TermWin.vts[page].saveLines;
    j = selection.end.row + TermWin.vts[page].saveLines;
    if ((i < row1 && j > row1)
        || (i < row2 && j > row2)
        || (i - count < row1 && i >= row1)
        || (i - count > row2 && i <= row2)
        || (j - count < row1 && j >= row1)
        || (j - count > row2 && j <= row2))
    {
      CLEAR_ALL_SELECTION;
      selection.op = SELECTION_CLEAR;	/* XXX: too aggressive? */
    } else if (j >= row1 && j <= row2)
    {
        /* move selected region too */
      selection.beg.row -= count;
      selection.end.row -= count;
      selection.mark.row -= count;
    }
  }
  CHECK_SELECTION(pagepage,0);	/* _after_ TermWin.nscrolled update */

  if (count > 0)
  {
/* A: scroll up */

    // on prend le min entre count et le nbr de lignes entre row1 et row2
		MIN_IT(count, row2 - row1 + 1);
/* A1: Copy lines that will get clobbered by the rotation */
    for (i = 0, j = row1; i < count; i++, j++)
    {
      buf_text[i] = screen[page].text[j];
      buf_tlen[i] = screen[page].tlen[j];
      buf_rend[i] = screen[page].rend[j];
    }
/* A2: Rotate lines */
    for (j = row1; (j + count) <= row2; j++)
    {
      screen[page].text[j] = screen[page].text[j + count];
      screen[page].tlen[j] = screen[page].tlen[j + count];
      screen[page].rend[j] = screen[page].rend[j + count];
    }
/* A3: Resurrect lines */
		for (i = 0; i < count; i++, j++)
		{
			screen[page].text[j] = buf_text[i];
			screen[page].tlen[j] = buf_tlen[i];
			screen[page].rend[j] = buf_rend[i];
		}
	} else if (count < 0)
	{
/* B: scroll down */

		count = min(-count, row2 - row1 + 1);
/* B1: Copy lines that will get clobbered by the rotation */
		for (i = 0, j = row2; i < count; i++, j--)
		{
			buf_text[i] = screen[page].text[j];
			buf_tlen[i] = screen[page].tlen[j];
			buf_rend[i] = screen[page].rend[j];
		}
/* B2: Rotate lines */
		for (j = row2; (j - count) >= row1; j--)
		{
			screen[page].text[j] = screen[page].text[j - count];
			screen[page].tlen[j] = screen[page].tlen[j - count];
			screen[page].rend[j] = screen[page].rend[j - count];
		}
/* B3: Resurrect lines */
		for (i = 0, j = row1; i < count; i++, j++)
		{
			screen[page].text[j] = buf_text[i];
			screen[page].tlen[j] = buf_tlen[i];
			screen[page].rend[j] = buf_rend[i];
		}
		count = -count;
	}
	return count;
}

/* ------------------------------------------------------------------------- */
/*
 * Add text given in <str> of length <len> to screen struct
 */


/* PROTO */
void
f_scr_add_lines(int page,const unsigned char *str, int nlines, int len)
{
	char            c;
	int             i, j, row, last_col, wherecursor, wrotespecial;
	text_t         *stp;
	rend_t         *srp;

	if (len <= 0)		/* sanity */
		return;

	last_col = TermWin.vts[page].bcol;

	DSCR_ALEXIS(2,(stderr, "scr_add_lines(%d,%d,%d), screen.bscroll:%d,bcol:%d\n", 
    page,nlines, len,screen[page].bscroll,last_col));
  DSCR_ALEXIS(2,(stderr, "cursor(%d,%d)\n",screen[page].cur.row,screen[page].cur.col));
	f_ZERO_SCROLLBACK(page);
	if (nlines > 0)
	{
    // nombre de lignes entre le curseur et bscroll
		nlines += (screen[page].cur.row - screen[page].bscroll);
		if ((nlines > 0) && (screen[page].tscroll == 0) && (screen[page].bscroll == (TermWin.nrow - 1)))
		{
		    /* _at least_ this many lines need to be scrolled */
			f_scroll_text(page,screen[page].tscroll, screen[page].bscroll, nlines, 0);
			for (i = nlines, j = screen[page].bscroll + TermWin.vts[page].saveLines; i--; j--)
			{
				if (screen[page].text[j] == NULL)
					f_make_screen_mem(page,screen[page].text,screen[page].rend, j,rstyle);
				else
					blank_line(screen[page].text[j], screen[page].rend[j], TermWin.vts[page].bcol, rstyle);
				screen[page].tlen[j] = 0;
			}
			screen[page].cur.row -= nlines;
		}
	}
#ifdef DEBUG_STRICT
	assert(screen[page].cur.col < last_col);
	assert(screen[page].cur.row < TermWin.nrow);
	assert(screen[page].cur.row >= -TermWin.vts[page].nscrolled);
#else				/* drive with your eyes closed */
	MIN_IT(screen[page].cur.col, last_col - 1);
	MIN_IT(screen[page].cur.row, TermWin.nrow - 1);
	MAX_IT(screen[page].cur.row, -TermWin.vts[page].nscrolled);
#endif
	row = screen[page].cur.row + TermWin.vts[page].saveLines;

	if (ROW_AND_COL_IS_BEFORE(screen[page].cur.row, screen[page].cur.col,
				  selection.beg.row, selection.beg.col))
		wherecursor = -1;
	else if (ROW_AND_COL_IS_BEFORE(screen[page].cur.row, screen[page].cur.col,
				       selection.end.row, selection.end.col))
		wherecursor = 0;
	else
		wherecursor = 1;

	stp = screen[page].text[row];
	srp = screen[page].rend[row];

#ifdef MULTICHAR_SET
	if (lost_multi && screen[page].cur.col > 0
	    && ((srp[screen[page].cur.col - 1] & RS_multiMask) == RS_multi1)
	    && *str != '\n' && *str != '\r' && *str != '\t')
		chstat = WBYTE;
#endif

	for (wrotespecial = len, i = 0; i < len;)
	{
		c = str[i++];
#ifdef MULTICHAR_SET
		if (chstat == WBYTE)
		{
			rstyle |= RS_multiMask;	/* multibyte 2nd byte */
			chstat = SBYTE;
			if (encoding_method == EUCJ)
				c |= 0x80;	/* maybe overkill, but makes it selectable */
		} else if (chstat == SBYTE)
			if (multi_byte || (c & 0x80))
			{	/* multibyte 1st byte */
				rstyle &= ~RS_multiMask;
				rstyle |= RS_multi1;
				chstat = WBYTE;
				if (encoding_method == EUCJ)
					c |= 0x80;	/* maybe overkill, but makes it selectable */
			} else
#endif				/* MULTICHAR_SET */
				switch (c)
				{
				case 127:
					wrotespecial--;
					continue;	/* yummmm..... */
				case '\t':
					wrotespecial--;
					f_scr_tab(page,1);
					continue;
				case '\n':
					wrotespecial--;
					if (screen[page].tlen[row] != -1)	/* XXX: think about this */
						MAX_IT(screen[page].tlen[row],screen[page].cur.col);
					screen[page].flags &= ~Screen_WrapNext;
					if (screen[page].cur.row == screen[page].bscroll)
					{
						f_scroll_text(page,screen[page].tscroll, screen[page].bscroll, 1, 0);
						j = screen[page].bscroll + TermWin.vts[page].saveLines;
						if (screen[page].text[j] == NULL)
							f_make_screen_mem(page,screen[page].text,screen[page].rend,j,rstyle);
						else
							blank_line(screen[page].text[j],screen[page].rend[j],TermWin.vts[page].bcol,rstyle);
              screen[page].tlen[j] = 0;
					} else if (screen[page].cur.row < (TermWin.nrow - 1))
						row = (++screen[page].cur.row) + TermWin.vts[page].saveLines;
					stp = screen[page].text[row];	/* _must_ refresh */
					srp = screen[page].rend[row];	/* _must_ refresh */
					continue;
				case '\r':
					wrotespecial--;
					if (screen[page].tlen[row] != -1)	/* XXX: think about this */
						MAX_IT(screen[page].tlen[row], screen[page].cur.col);
					screen[page].flags &= ~Screen_WrapNext;
					screen[page].cur.col = 0;
					continue;
				default:
#ifdef MULTICHAR_SET
					rstyle &= ~RS_multiMask;
#endif
					break;
				}
		if (screen[page].flags & Screen_WrapNext)
		{
			screen[page].tlen[row] = -1;
			if (screen[page].cur.row == screen[page].bscroll)
			{
				f_scroll_text(page,screen[page].tscroll, screen[page].bscroll, 1,0);
				j = screen[page].bscroll + TermWin.vts[page].saveLines;
				if (screen[page].text[j] == NULL)
					f_make_screen_mem(page,screen[page].text,screen[page].rend, j,rstyle);
				else
					blank_line(screen[page].text[j],screen[page].rend[j],TermWin.vts[page].bcol, rstyle);
				screen[page].tlen[j] = 0;
			} else if (screen[page].cur.row < (TermWin.nrow - 1))
				row = (++screen[page].cur.row) + TermWin.vts[page].saveLines;
			stp = screen[page].text[row];	/* _must_ refresh */
			srp = screen[page].rend[row];	/* _must_ refresh */
			screen[page].cur.col = 0;
			screen[page].flags &= ~Screen_WrapNext;
		}
		if (screen[page].flags & Screen_Insert)
			f_scr_insdel_chars(page,1, INSERT);
		stp[screen[page].cur.col] = c;
		srp[screen[page].cur.col] = rstyle;
	    /*
	     * TODO: This might end up wrapping the tone sign in Thai
	     */
		if (screen[page].cur.col < (last_col - 1))
			screen[page].cur.col++;
		else
		{
			screen[page].tlen[row] = last_col;
			if (screen[page].flags & Screen_Autowrap)
				screen[page].flags |= Screen_WrapNext;
			else
				screen[page].flags &= ~Screen_WrapNext;
		}
	}
	if (screen[page].tlen[row] != -1)	/* XXX: think about this */
		MAX_IT(screen[page].tlen[row], screen[page].cur.col);

/*
 * If we wrote anywhere in the selected area, kill the selection
 * XXX: should we kill the mark too?  Possibly, but maybe that
 *      should be a similar check.
 */
	if (ROW_AND_COL_IS_BEFORE(screen[page].cur.row, screen[page].cur.col,
				  selection.beg.row, selection.beg.col))
		i = -1;
	else if (ROW_AND_COL_IS_BEFORE(screen[page].cur.row, screen[page].cur.col,
				       selection.end.row, selection.end.col))
		i = 0;
	else
		i = 1;
	if (selection.op && current_screen == selection.screen
	    && wrotespecial != 0 && (i != wherecursor || i == 0))
		CLEAR_SELECTION;

#ifdef DEBUG_STRICT
	assert(screen[page].cur.row >= 0);
#else				/* drive with your eyes closed */
	MAX_IT(screen[page].cur.row, 0);
#endif
}




/* ------------------------------------------------------------------------- */
/*
 * Process Backspace.  Move back the cursor back a position, wrap if have to
 * XTERM_SEQ: CTRL-H
 */


void f_scr_backspace(int page)
{
	RESET_CHSTAT;
  DSCR_ALEXIS(2,(stderr,"f_scr_backspace(%d)\n",page));
	if (screen[page].cur.col == 0 && screen[page].cur.row > 0)
	{
		screen[page].cur.col = TermWin.vts[page].bcol - 1;
		screen[page].cur.row--;
	} else if (screen[page].flags & Screen_WrapNext)
	{
		screen[page].flags &= ~Screen_WrapNext;
	} else
		f_scr_gotorc(page,0, -1, RELATIVE);
}

/* ------------------------------------------------------------------------- */
/*
 * Process Horizontal Tab
 * count: +ve = forward; -ve = backwards
 * XTERM_SEQ: CTRL-I
 */

void f_scr_tab(int page,int count)
{
	int             i, x;

	RESET_CHSTAT;
  DSCR_ALEXIS(2,(stderr,"f_scr_tab(...)\n"));
	x = screen[page].cur.col;
	if (count == 0)
		return;
	else if (count > 0)
	{
		for (i = x + 1; i < TermWin.vts[page].bcol; i++)
		{
			if (tabs[i])
			{
				x = i;
				if (!--count)
					break;
			}
		}
	} else if (count < 0)
	{
		for (i = x - 1; i >= 0; i--)
		{
			if (tabs[i])
			{
				x = i;
				if (!++count)
					break;
			}
		}
	}
	if (x != screen[page].cur.col)
		f_scr_gotorc(page,0, x, R_RELATIVE);
}

/* ------------------------------------------------------------------------- */
/*
 * Goto Row/Column
 */


/* PROTO */
void
f_scr_gotorc(int page,int row, int col, int relative)
{
	f_ZERO_SCROLLBACK(page);
	RESET_CHSTAT;
	DSCR_ALEXIS(1,(stderr, "scr_gotorc(r:%d,c:%d,%d): from (r:%d,c:%d), bcol:%d\n", row,
    col, relative, screen[page].cur.row, screen[page].cur.col,TermWin.vts[page].bcol));

	screen[page].cur.col = ((relative & C_RELATIVE) ? (screen[page].cur.col + col) : col);
	MAX_IT(screen[page].cur.col, 0); // 0 <= screen[0].cur.col <= col
	MIN_IT(screen[page].cur.col, TermWin.vts[page].bcol - 1);

	if (screen[page].flags & Screen_WrapNext)
	{
		screen[page].flags &= ~Screen_WrapNext;
	}
	if (relative & R_RELATIVE)
	{
		if (row > 0)
		{
			if (screen[page].cur.row <= screen[page].bscroll && 
        (screen[page].cur.row + row) > screen[page].bscroll)
				screen[page].cur.row = screen[page].bscroll;
			else
				screen[page].cur.row += row;
		} else if (row < 0)
		{
			if (screen[page].cur.row >= screen[page].tscroll  && 
        (screen[page].cur.row + row) < screen[page].tscroll)
				screen[page].cur.row = screen[page].tscroll;
			else
				screen[page].cur.row += row;
		}
	} else
	{
		if (screen[page].flags & Screen_Relative)
		{		/* relative origin mode */
			screen[page].cur.row = row + screen[page].tscroll;
			MIN_IT(screen[page].cur.row, screen[page].bscroll);
		} else
			screen[page].cur.row = row;
	}
	MAX_IT(screen[page].cur.row, 0);
	MIN_IT(screen[page].cur.row, TermWin.nrow - 1);
}

/* ------------------------------------------------------------------------- */
/*
 * direction  should be UP or DN
 */


/* PROTO */
void
f_scr_index(int page,int direction)
{
	int dirn;

	dirn = ((direction == UP) ? 1 : -1);
	DSCR_ALEXIS(1,(stderr, "scr_index(%d)\n", dirn));

	f_ZERO_SCROLLBACK(page);
	RESET_CHSTAT;
	if (screen[page].flags & Screen_WrapNext)
	{
		screen[page].flags &= ~Screen_WrapNext;
	}
	if ((screen[page].cur.row == screen[page].bscroll && direction == UP)
	    || (screen[page].cur.row == screen[page].tscroll && direction == DN))
	{
		f_scroll_text(page,screen[page].tscroll, screen[page].bscroll, dirn, 0);
		if (direction == UP)
			dirn = screen[page].bscroll + TermWin.vts[page].saveLines;
		else
			dirn = screen[page].tscroll + TermWin.vts[page].saveLines;
		if (screen[page].text[dirn] == NULL)	/* then so is screen[0].rend[dirn] */
			f_make_screen_mem(page,screen[page].text, screen[page].rend, dirn,rstyle);
		else
			blank_line(screen[page].text[dirn], screen[page].rend[dirn],TermWin.vts[page].bcol, rstyle);
		screen[page].tlen[dirn] = 0;
	} else
		screen[page].cur.row += dirn;
  
	MAX_IT(screen[page].cur.row, 0);
	MIN_IT(screen[page].cur.row, TermWin.nrow - 1);
	CHECK_SELECTION(pagepage,0);
}

/* ------------------------------------------------------------------------- */
/*
 * Erase part or whole of a line
 * XTERM_SEQ: Clear line to right: ESC [ 0 K
 * XTERM_SEQ: Clear line to left : ESC [ 1 K
 * XTERM_SEQ: Clear whole line   : ESC [ 2 K
 */


/* PROTO */
void
f_scr_erase_line(int page,int mode)
{
	int             row, col, num;

	DSCR_ALEXIS(2,(stderr, "scr_erase_line(%d,%d) at screen row: %d\n", page,mode,
		  screen[page].cur.row));
	f_ZERO_SCROLLBACK(page);
	RESET_CHSTAT;
	CHECK_SELECTION(pagepage,1);

	if (screen[page].flags & Screen_WrapNext)
		screen[page].flags &= ~Screen_WrapNext;

	row = TermWin.vts[page].saveLines + screen[page].cur.row;
	switch (mode)
	{
	case 0:		/* erase to end of line */
		col = screen[page].cur.col;
		num = TermWin.vts[page].bcol - col;
		MIN_IT(screen[page].tlen[row], col);
    // pour voir si la ligne qu'on va effacer contient la selection ou pas
		if (ROWCOL_IN_ROW_ON_OR_AFTER(selection.beg, screen[page].cur)
		    || ROWCOL_IN_ROW_ON_OR_AFTER(selection.end, screen[page].cur))
			CLEAR_SELECTION;
		break;
	case 1:		/* erase to beginning of line */
		col = 0;
		num = screen[page].cur.col + 1;
		if (ROWCOL_IN_ROW_ON_OR_BEFORE(selection.beg, screen[page].cur)
		    || ROWCOL_IN_ROW_ON_OR_BEFORE(selection.end, screen[page].cur))
			CLEAR_SELECTION;
		break;
	case 2:		/* erase whole line */
		col = 0;
		num = TermWin.vts[page].bcol;
		screen[page].tlen[row] = 0;
		if (selection.beg.row <= screen[page].cur.row
		    || selection.end.row >= screen[page].cur.row)
			CLEAR_SELECTION;
		break;
	default:
		return;
	}
	blank_line(&(screen[page].text[row][col]), &(screen[page].rend[row][col]), num,
		   rstyle & ~RS_Uline);
}

/* ------------------------------------------------------------------------- */
/*
 * Erase part of whole of the screen
 * XTERM_SEQ: Clear screen after cursor : ESC [ 0 J
 * XTERM_SEQ: Clear screen before cursor: ESC [ 1 J
 * XTERM_SEQ: Clear whole screen        : ESC [ 2 J
 */


/* PROTO */
void
f_scr_erase_screen(int page,int mode)
{
	int             row, num, row_offset;
	rend_t          ren;
	long            gcmask;
	XGCValues       gcvalue;

	DSCR_ALEXIS(1,(stderr, "scr_erase_screen(%d) at screen row: %d\n", 
    mode,screen[page].cur.row));
	f_ZERO_SCROLLBACK(page);
	RESET_CHSTAT;
//	row_offset = TermWin.vts[TermWin.active_page].saveLines;
	row_offset = TermWin.vts[page].saveLines;

	switch (mode)
	{
	case 0:		/* erase to end of screen */
		CHECK_SELECTION(pagepage,1);
		f_scr_erase_line(page,0);
		row = screen[page].cur.row + 1;	/* possible OOB */
		num = TermWin.nrow - row;
		break;
	case 1:		/* erase to beginning of screen */
		CHECK_SELECTION(pagepage,3);
		f_scr_erase_line(page,1);
		row = 0;	/* possible OOB */
		num = screen[page].cur.row;
		break;
	case 2:		/* erase whole screen */
		CHECK_SELECTION(pagepage,3);
		row = 0;
		num = TermWin.nrow;
		break;
	default:
		return;
	}
	if (selection.op && current_screen == selection.screen
    && selection.vt == page
	    && ((selection.beg.row >= row
		 && selection.beg.row <= row + num)
		|| (selection.end.row >= row
		    && selection.end.row <= row + num)))
		CLEAR_SELECTION;
	if (row >= 0 && row < TermWin.nrow)
	{			/* check OOB */
		MIN_IT(num, (TermWin.nrow - row));
		if (rstyle & (RS_RVid | RS_Uline))
			ren = (rend_t) ~ RS_None;
		else if (GET_BGCOLOR(rstyle) == Color_bg)
		{
			ren = DEFAULT_RSTYLE;
			f_CLEAR_ROWS(page,row, num);
		} else
		{
			ren = (rstyle & (RS_fgMask | RS_bgMask));
			gcvalue.foreground = PixColors[GET_BGCOLOR(ren)];
			gcmask = GCForeground;
			XChangeGC(Xdisplay, TermWin.vts[page].gc, gcmask, &gcvalue);
			f_ERASE_ROWS(page,row, num);
			gcvalue.foreground = PixColors[Color_fg];
			XChangeGC(Xdisplay, TermWin.vts[page].gc, gcmask, &gcvalue);
		}
		for (; num--; row++)
		{
			blank_line(screen[page].text[row + row_offset],
        screen[page].rend[row + row_offset], TermWin.vts[page].bcol,
        rstyle & ~RS_Uline);
			screen[page].tlen[row + row_offset] = 0;
			blank_line(drawn_text[row], drawn_rend[row],TermWin.vts[page].bcol, ren);
		}
	}
}

/* ------------------------------------------------------------------------- */
/*
 * Fill the screen with `E's
 * XTERM_SEQ: Screen Alignment Test: ESC # 8
 */

/* PROTO */
void
f_scr_E(int page)
{
	int             i, j;
	text_t         *t;
	rend_t         *r, fs;

	f_ZERO_SCROLLBACK(page);
	RESET_CHSTAT;
	CHECK_SELECTION(pagepage,3);

	fs = rstyle;
	for (i = TermWin.vts[page].saveLines; i < TermWin.nrow + TermWin.vts[page].saveLines; i++)
	{
		t = screen[page].text[i];
		r = screen[page].rend[i];
		for (j = 0; j < TermWin.vts[page].bcol; j++)
		{
			*t++ = 'E';
			*r++ = fs;
		}
		screen[page].tlen[i] = TermWin.vts[page].bcol;	/* make the `E's selectable */
	}
}

/* ------------------------------------------------------------------------- */
/*
 * Insert/Delete <count> lines
 */

/* PROTO */
void
f_scr_insdel_lines(int page,int count, int insdel)
{
	int             end;

	f_ZERO_SCROLLBACK(page);
	RESET_CHSTAT;
	CHECK_SELECTION(pagepage,1);

	if (screen[page].cur.row > screen[page].bscroll)
		return;

	end = screen[page].bscroll - screen[page].cur.row + 1;
	if (count > end)
	{
		if (insdel == DELETE)
			return;
		else if (insdel == INSERT)
			count = end;
	}
	if (screen[page].flags & Screen_WrapNext)
	{
		screen[page].flags &= ~Screen_WrapNext;
	}
	f_scroll_text(page,screen[page].cur.row, screen[page].bscroll, insdel * count, 0);

/* fill the inserted or new lines with rstyle. TODO: correct for delete? */
	if (insdel == DELETE)
	{
		end = screen[page].bscroll + TermWin.vts[page].saveLines;
	} else if (insdel == INSERT)
	{
		end = screen[page].cur.row + count - 1 + TermWin.vts[page].saveLines;
	}
	for (; count--; end--)
	{
		if (screen[page].text[end] == NULL)	/* then so is screen[0].rend[end] */
			f_make_screen_mem(page,screen[page].text, screen[page].rend, end, rstyle);
		else
			blank_line(screen[page].text[end], screen[page].rend[end],
				   TermWin.vts[page].bcol, rstyle);
		screen[page].tlen[end] = 0;
	}
}


/* ------------------------------------------------------------------------- */
/*
 * Insert/Delete <count> characters from the current position
 */


/* PROTO */
void
f_scr_insdel_chars(int page,int count, int insdel)
{
	int             col, row;
	rend_t          tr;

	f_ZERO_SCROLLBACK(page);
	RESET_CHSTAT;
	if (count <= 0)
		return;
	CHECK_SELECTION(pagepage,1);
	MIN_IT(count, (TermWin.vts[page].bcol - screen[page].cur.col));

	row = screen[page].cur.row + TermWin.vts[page].saveLines;
	screen[page].flags &= ~Screen_WrapNext;

	switch (insdel)
	{
	case INSERT:
		for (col = TermWin.vts[page].bcol - 1; (col - count) >= screen[page].cur.col;
		     col--)
		{
			screen[page].text[row][col] = screen[page].text[row][col - count];
			screen[page].rend[row][col] = screen[page].rend[row][col - count];
		}
		if (screen[page].tlen[row] != -1)
		{
			screen[page].tlen[row] += count;
			MIN_IT(screen[page].tlen[row], TermWin.vts[page].bcol);
		}
		if (selection.op && current_screen == selection.screen
		    && ROWCOL_IN_ROW_ON_OR_AFTER(selection.beg, screen[page].cur))
		{
			if (selection.end.row != screen[page].cur.row
			    || (selection.end.col + count >= TermWin.vts[page].bcol))
				CLEAR_SELECTION;
			else
			{	/* shift selection */
				selection.beg.col += count;
				selection.mark.col += count;	/* XXX: yes? */
				selection.end.col += count;
			}
		}
		blank_line(&(screen[page].text[row][screen[page].cur.col]),
			   &(screen[page].rend[row][screen[page].cur.col]), count, rstyle);
		break;
	case ERASE:
		screen[page].cur.col += count;	/* don't worry if > TermWin.bcol */
		CHECK_SELECTION(page,1);
		screen[page].cur.col -= count;
		blank_line(&(screen[page].text[row][screen[page].cur.col]),
			   &(screen[page].rend[row][screen[page].cur.col]), count, rstyle);
		break;
	case DELETE:
		tr = screen[page].rend[row][TermWin.vts[page].bcol - 1]
		    & (RS_fgMask | RS_bgMask | RS_baseattrMask);
		for (col = screen[page].cur.col; (col + count) < TermWin.vts[page].bcol; col++)
		{
			screen[page].text[row][col] = screen[page].text[row][col + count];
			screen[page].rend[row][col] = screen[page].rend[row][col + count];
		}
		blank_line(&(screen[page].text[row][TermWin.vts[page].bcol - count]),
			   &(screen[page].rend[row][TermWin.vts[page].bcol - count]),
			   count, tr);
		if (screen[page].tlen[row] == -1)	/* break line continuation */
			screen[page].tlen[row] = TermWin.vts[page].bcol;
		screen[page].tlen[row] -= count;
		MAX_IT(screen[page].tlen[row], 0);
		if (selection.op && current_screen == selection.screen
		    && ROWCOL_IN_ROW_ON_OR_AFTER(selection.beg, screen[page].cur))
		{
			if (selection.end.row != screen[page].cur.row
			    || (screen[page].cur.col >= selection.beg.col - count)
			    || selection.end.col >= TermWin.vts[page].bcol)
				CLEAR_SELECTION;
			else
			{
			    /* shift selection */
				selection.beg.col -= count;
				selection.mark.col -= count;	/* XXX: yes? */
				selection.end.col -= count;
			}
		}
		break;
	}
#ifdef MULTICHAR_SET
	if ((screen[page].rend[row][page] & RS_multiMask) == RS_multi2)
	{
		screen[page].rend[row][page] &= ~RS_multiMask;
		screen[page].text[row][page] = ' ';
	}
	if ((screen[page].rend[row][TermWin.vts[page].bcol - 1] & RS_multiMask) == RS_multi1)
	{
		screen[page].rend[row][TermWin.vts[page].bcol - 1] &= ~RS_multiMask;
		screen[page].text[row][TermWin.vts[page].bcol - 1] = ' ';
	}
#endif
}

/* ------------------------------------------------------------------------- */
/*
 * Set the scrolling region
 * XTERM_SEQ: Set region <top> - <bot> inclusive: ESC [ <top> ; <bot> r
 */


/* PROTO */
void
f_scr_scroll_region(int page,int top, int bot)
{
  DSCR_ALEXIS(2,(stderr,"f_scr_scroll_region(%d,%d,%d)\n",page,top,bot));
	MAX_IT(top, 0);
	MIN_IT(bot, TermWin.nrow - 1);
	if (top > bot)
		return;
	screen[page].tscroll = top;
	screen[page].bscroll = bot;
	f_scr_gotorc(page,0, 0, 0);
}

/* ------------------------------------------------------------------------- */
/*
 * Make the cursor visible/invisible
 * XTERM_SEQ: Make cursor visible  : ESC [ ? 25 h
 * XTERM_SEQ: Make cursor invisible: ESC [ ? 25 l
 */


/* PROTO */
void
f_scr_cursor_visible(int page,int mode)
{
	if (mode)
		screen[page].flags |= Screen_VisibleCursor;
	else
		screen[page].flags &= ~Screen_VisibleCursor;
}

/* ------------------------------------------------------------------------- */
/*
 * Set/unset automatic wrapping
 * XTERM_SEQ: Set Wraparound  : ESC [ ? 7 h
 * XTERM_SEQ: Unset Wraparound: ESC [ ? 7 l
 */


/* PROTO */
void
f_scr_autowrap(int page,int mode)
{
	if (mode)
		screen[page].flags |= Screen_Autowrap;
	else
		screen[page].flags &= ~Screen_Autowrap;
}

/* ------------------------------------------------------------------------- */
/*
 * Set/unset margin origin mode
 * Absolute mode: line numbers are counted relative to top margin of screen
 *      and the cursor can be moved outside the scrolling region.
 * Relative mode: line numbers are relative to top margin of scrolling region
 *      and the cursor cannot be moved outside.
 * XTERM_SEQ: Set Absolute: ESC [ ? 6 h
 * XTERM_SEQ: Set Relative: ESC [ ? 6 l
 */


/* PROTO */
void
f_scr_relative_origin(int page,int mode)
{
  DSCR_ALEXIS(2,(stderr,"f_scr_relative_origin(..)\n"));
	if (mode)
		screen[page].flags |= Screen_Relative;
	else
		screen[page].flags &= ~Screen_Relative;
	f_scr_gotorc(page,0, 0, 0);
}

/* ------------------------------------------------------------------------- */
/*
 * Set insert/replace mode
 * XTERM_SEQ: Set Insert mode : ESC [ ? 4 h
 * XTERM_SEQ: Set Replace mode: ESC [ ? 4 l
 */


/* PROTO */
void
f_scr_insert_mode(int page,int mode)
{
	if (mode)
		screen[page].flags |= Screen_Insert;
	else
		screen[page].flags &= ~Screen_Insert;
}

/* ------------------------------------------------------------------------- */
/*
 * Set/Unset tabs
 * XTERM_SEQ: Set tab at current column  : ESC H
 * XTERM_SEQ: Clear tab at current column: ESC [ 0 g
 * XTERM_SEQ: Clear all tabs             : ESC [ 3 g
 */


/* PROTO */
void
f_scr_set_tab(int page,int mode)
{
	if (mode < 0)
		MEMSET(tabs, 0, TermWin.vts[page].bcol * sizeof(char));

	else if (screen[page].cur.col < TermWin.vts[page].bcol)
		tabs[screen[page].cur.col] = (mode ? 1 : 0);
}

/* ------------------------------------------------------------------------- */
/*
 * Set reverse/normal video
 * XTERM_SEQ: Reverse video: ESC [ ? 5 h
 * XTERM_SEQ: Normal video : ESC [ ? 5 l
 */
/* PROTO */
void
f_scr_rvideo_mode(int page,int mode)
{
	int i, j, maxlines;

  if (rvideo != mode) {
		rvideo = mode;
		rstyle ^= RS_RVid;
    
		maxlines = TermWin.vts[page].saveLines + TermWin.nrow;
		for (i = TermWin.vts[page].saveLines; i < maxlines; i++)
			for (j = 0; j < TermWin.vts[page].bcol + 1; j++)
				screen[page].rend[i][j] ^= RS_RVid;
		f_scr_refresh(page,SLOW_REFRESH);
	}
}

/* ------------------------------------------------------------------------- */
/*
 * Report current cursor position
 * XTERM_SEQ: Report position: ESC [ 6 n
 */


void f_scr_report_position(int page)
{
  tt_printf((unsigned char *)"\033[%d;%dR",
    screen[page].cur.row + 1, screen[page].cur.col + 1);
}

/* ------------------------------------------------------------------------- *
 *                                  FONTS                                    *
 * ------------------------------------------------------------------------- */

/*
 * Set font style
 */



void f_set_font_style(int page)
{
	rstyle &= ~RS_fontMask;
	switch (charsets[screen[page].charset])
	{
	case '0':		/* DEC Special Character & Line Drawing Set */
		rstyle |= RS_acsFont;
		break;
	case 'A':		/* United Kingdom (UK) */
		rstyle |= RS_ukFont;
		break;
	case 'B':		/* United States (USASCII) */
		break;
	case '<':		/* Multinational character set */
		break;
	case '5':		/* Finnish character set */
		break;
	case 'C':		/* Finnish character set */
		break;
	case 'K':		/* German character set */
		break;
	}
}

/* ------------------------------------------------------------------------- */
/*
 * Choose a font
 * XTERM_SEQ: Invoke G0 character set: CTRL-O
 * XTERM_SEQ: Invoke G1 character set: CTRL-N
 * XTERM_SEQ: Invoke G2 character set: ESC N
 * XTERM_SEQ: Invoke G3 character set: ESC O
 */

void f_scr_charset_choose(int page,int set)
{
	screen[page].charset = set;
	f_set_font_style(page);
}


/* ------------------------------------------------------------------------- */
/*
 * Set a font
 * XTERM_SEQ: Set G0 character set: ESC ( <C>
 * XTERM_SEQ: Set G1 character set: ESC ) <C>
 * XTERM_SEQ: Set G2 character set: ESC * <C>
 * XTERM_SEQ: Set G3 character set: ESC + <C>
 * See set_font_style for possible values for <C>
 */


void f_scr_charset_set(int page,int set, unsigned int ch)
{
#ifdef MULTICHAR_SET
	multi_byte = (set < 0);
	set = abs(set);
#endif
	charsets[set] = (unsigned char)ch;
	f_set_font_style(page);
}

/* ------------------------------------------------------------------------- *
 *          MULTIPLE-CHARACTER FONT SET MANIPULATION FUNCTIONS               *
 * ------------------------------------------------------------------------- */
#ifdef MULTICHAR_SET
# ifdef KANJI
static void     (*multichar_decode) (unsigned char *str, int len) = eucj2jis;
# else				/* then we must be BIG5 to get in here */
static void     (*multichar_decode) (unsigned char *str, int len) = big5dummy;
# endif

void eucj2jis(unsigned char *str, int len)
{
	register int    i;

	for (i = 0; i < len; i++)
		str[i] &= 0x7F;
}

/* ------------------------------------------------------------------------- */
void sjis2jis(unsigned char *str, int len)
{
	register int    i;
	unsigned char  *high, *low;

	for (i = 0; i < len; i += 2, str += 2)
	{
		high = str;
		low = str + 1;
		(*high) -= (*high > 0x9F ? 0xB1 : 0x71);
		*high = (*high) * 2 + 1;
		if (*low > 0x9E)
		{
			*low -= 0x7E;
			(*high)++;
		} else
		{
			if (*low > 0x7E)
				(*low)--;
			*low -= 0x1F;
		}
	}
}

void big5dummy(unsigned char *str, int len)
{
}

void set_multichar_encoding(const char *str)
{
	if (str && *str)
	{
		if (!strcmp(str, "sjis"))
		{
			encoding_method = SJIS;	/* Kanji SJIS */
			multichar_decode = sjis2jis;
		} else if (!strcmp(str, "eucj"))
		{
			encoding_method = EUCJ;	/* Kanji EUCJ */
			multichar_decode = eucj2jis;
		}
	}
}
#endif				/* MULTICHAR_SET */

/* ------------------------------------------------------------------------- *
 *                           GRAPHICS COLOURS                                *
 * ------------------------------------------------------------------------- */

#ifdef RXVT_GRAPHICS
int scr_get_fgcolor(void)
{
	return GET_FGCOLOR(rstyle);
}

/* ------------------------------------------------------------------------- */
int scr_get_bgcolor(void)
{
	return GET_BGCOLOR(rstyle);
}
#endif

/* ------------------------------------------------------------------------- *
 *                        MAJOR SCREEN MANIPULATION                          *
 * ------------------------------------------------------------------------- */

/*
 * Refresh an area
 */

void f_scr_expose(int page,int x, int y, int width, int height)
{
	int             i;

#if 0
	text_t         *t;
#endif
	row_col_t       full_beg, full_end, part_beg, part_end;

	if (drawn_text == NULL)	/* sanity check */
		return;

/* round down */
	part_beg.col = Pixel2Col(x);
	part_beg.row = Pixel2Row(y);
	full_end.col = Pixel2Width(x + width);
	full_end.row = Pixel2Row(y + height);

/* round up */
	part_end.col = Pixel2Width(x + width + TermWin.fwidth - 1);
	part_end.row = Pixel2Row(y + height + TermWin.fheight - 1);
	full_beg.col = Pixel2Col(x + TermWin.fwidth - 1);
	full_beg.row = Pixel2Row(y + TermWin.fheight - 1);

/* sanity checks */
	MAX_IT(part_beg.col, 0);
	MAX_IT(full_beg.col, 0);
	MAX_IT(part_end.col, 0);
	MAX_IT(full_end.col, 0);
	MAX_IT(part_beg.row, 0);
	MAX_IT(full_beg.row, 0);
	MAX_IT(part_end.row, 0);
	MAX_IT(full_end.row, 0);
	MIN_IT(part_beg.col, TermWin.vts[page].bcol - 1);
	MIN_IT(full_beg.col, TermWin.vts[page].bcol - 1);
	MIN_IT(part_end.col, TermWin.vts[page].bcol - 1);
	MIN_IT(full_end.col, TermWin.vts[page].bcol - 1);
	MIN_IT(part_beg.row, TermWin.nrow - 1);
	MIN_IT(full_beg.row, TermWin.nrow - 1);
	MIN_IT(part_end.row, TermWin.nrow - 1);
	MIN_IT(full_end.row, TermWin.nrow - 1);

	DSCR_ALEXIS(1,(stderr,
		  "scr_expose(x:%d, y:%d, w:%d, h:%d) area (c:%d,r:%d)-(c:%d,r:%d)\n",
		  x, y, width, height, part_beg.col, part_beg.row,
		  part_end.col, part_end.row));

#if 1				/* XXX: check for backing store/save unders? */
	for (i = part_beg.row; i <= part_end.row; i++)
		MEMSET(&(drawn_text[i][part_beg.col]), 0,part_end.col - part_beg.col + 1);
#else
	if (full_end.col >= full_beg.col)
	    /* set DEFAULT_RSTYLE for totally exposed characters */
		for (i = full_beg.row; i <= full_end.row; i++)
			blank_line(&(drawn_text[i][full_beg.col]),&(drawn_rend[i][full_beg.col]),
        full_end.col - full_beg.col + 1,DEFAULT_RSTYLE);
/* force an update for partially exposed characters */
	if (part_beg.row != full_beg.row)
	{
		t = &(drawn_text[part_beg.row][part_beg.col]);
		for (i = part_end.col - part_beg.col + 1; i--;)
			*t++ = 0;
	}
	if (part_end.row != full_end.row)
	{
		t = &(drawn_text[part_end.row][part_beg.col]);
		for (i = part_end.col - part_beg.col + 1; i--;)
			*t++ = 0;
	}
	if (part_beg.col != full_beg.col)
		for (i = full_beg.row; i <= full_end.row; i++)
			drawn_text[i][part_beg.col] = 0;
	if (part_end.col != full_end.col)
		for (i = full_beg.row; i <= full_end.row; i++)
			drawn_text[i][part_end.col] = 0;
#endif
	f_scr_refresh(page,SLOW_REFRESH);
}

/* ------------------------------------------------------------------------- */
/*
 * Refresh the entire screen
 */

void f_scr_touch(int page)
{
	f_scr_expose(page,0, 0, TermWin.width, TermWin.height);
}

/* ------------------------------------------------------------------------- */
/*
 * Move the display so that the line represented by scrollbar value Y is at
 * the top of the screen
 */
int f_scr_move_to(int page,int y, int len)
{
	int             start;

	start = TermWin.vts[page].view_start;
	TermWin.vts[page].view_start = ((len - y) * (TermWin.nrow - 1 + TermWin.vts[page].nscrolled) / (len)) -
    (TermWin.nrow - 1);
	DSCR_ALEXIS(1,(stderr, "scr_move_to(%d, %d) view_start:%d\n", y, len,TermWin.vts[page].view_start));

	MAX_IT(TermWin.vts[page].view_start, 0);
	MIN_IT(TermWin.vts[page].view_start, TermWin.vts[page].nscrolled);

	return (TermWin.vts[page].view_start - start);
}

/* ------------------------------------------------------------------------- */
/*
 * Page the screen up/down nlines
 * direction  should be UP or DN
 */
int f_scr_page(int page,int direction, int nlines)
{
	int             start, dirn;

	DSCR_ALEXIS(1,(stderr, "scr_page(%d,%s,%d) view_start:%d, nscrolled:%d\n",
    page,((direction == UP) ? "UP" : "DN"), nlines,
    TermWin.vts[page].view_start,TermWin.vts[page].nscrolled));
  DSCR_ALEXIS(1,(stderr,"scr_page(...) tscroll:%d, bscroll:%d\n",
    screen[page].tscroll,screen[page].bscroll));
  
	dirn = (direction == UP) ? 1 : -1;
	start = TermWin.vts[page].view_start;
  // on ne fait pas un deplacement inferieur a 1 ligne
	MAX_IT(nlines, 1);
  // on ne fait pas un deplacement de plus de nrow
	MIN_IT(nlines, TermWin.nrow);
	TermWin.vts[page].view_start += (nlines * dirn);
	MAX_IT(TermWin.vts[page].view_start, 0);
	MIN_IT(TermWin.vts[page].view_start, TermWin.vts[page].nscrolled);

	return (TermWin.vts[page].view_start - start);
}

/* ------------------------------------------------------------------------- */

void scr_bell(void)
{
#ifndef NO_MAPALERT
# ifdef MAPALERT_OPTION
	if (Options & Opt_mapAlert)
# endif
		XMapWindow(Xdisplay, TermWin.parent);
#endif
	if (Options & Opt_visualBell)
	{
		f_scr_rvideo_mode(0,!rvideo);	/* scr_refresh() also done */
		f_scr_rvideo_mode(0,!rvideo);	/* scr_refresh() also done */
	} else
		XBell(Xdisplay, 0);
}

/* ------------------------------------------------------------------------- */
/* ARGSUSED */
void f_scr_printscreen(int page,int fullhist)
{
#ifdef PRINTPIPE
	int             i, r, nrows, row_offset;
	text_t         *t;
	FILE           *fd;

	if ((fd = popen_printer()) == NULL)
		return;
	nrows = TermWin.nrow;
	if (fullhist)
		nrows += TermWin.vts[page].view_start;

	row_offset = TermWin.vts[page].saveLines - TermWin.vts[page].view_start;
	for (r = 0; r < nrows; r++)
	{
		t = screen[page].text[r + row_offset];
		for (i = TermWin.vts[page].bcol - 1; i >= 0; i--)
			if (!isspace(t[i]))
				break;
		fprintf(fd, "%.*s\n", (i + 1), t);
	}
	pclose_printer(fd);
#endif
}

/* ------------------------------------------------------------------------- */
/*
 * Refresh the screen
 * drawn_text/drawn_rend contain the screen information before the update.
 * screen[0].text/screen[0].rend contain what the screen will change to.
 */

/*#define DRAW_STRING(Func, x, y, str, len)				\
    Func(Xdisplay, drawBufferPage(0), TermWin.vts[0].gc, (x), (y), (str), (len))*/
#define f_DRAW_STRING(page,Func, x, y, str, len)				\
    Func(Xdisplay, drawBufferPage(page), TermWin.vts[page].gc, (x), (y), (str), (len))

#if defined (NO_BRIGHTCOLOR) || defined (VERYBOLD)
# define MONO_BOLD(x)	((x) & (RS_Bold|RS_Blink))
#else
# define MONO_BOLD(x)	(((x) & RS_Bold) && fore == Color_fg)
#endif


void f_scr_refresh(int page,int type)
{
  int i, j,	/* tmp                                       */
      col, row,	/* column/row we're processing               */
      scrrow,	/* screen row offset                         */
      row_offset,	/* basic offset in screen structure          */
      currow,	/* cursor row at appropriate offset          */
      boldlast,	/* last character in some row was bold       */
      len, wlen,	/* text length screen/buffer                 */
      fprop,	/* proportional font used                    */
      is_cursor,	/* cursor this position                      */
      rvid,	/* reverse video this position               */
      rend,	/* rendition                                 */
      fore, back,	/* desired foreground/background             */
      wbyte,	/* we're in multibyte                        */
      morecur = 0, xpixel,	/* x offset for start of drawing (font)      */
      ypixel;	/* y offset for start of drawing (font)      */

  static int      focus = -1;	/* screen in focus?                          */
  long            gcmask;	/* Graphics Context mask                     */
  unsigned long   ltmp;
  rend_t          rt1, rt2;	/* tmp rend values                           */

#ifndef NO_CURSORCOLOR
	rend_t          ccol1,	/* Cursor colour                             */
	                ccol2,	/* Cursor colour2                            */
	                cc1;	/* store colours at cursor position(s)       */

# ifdef MULTICHAR_SET
	rend_t          cc2;	/* store colours at cursor position(s)       */
# endif
#endif
	rend_t         *drp, *srp;	/* drawn-rend-pointer, screen-rend-pointer   */
	text_t         *dtp, *stp;	/* drawn-text-pointer, screen-text-pointer   */
	XGCValues       gcvalue;	/* Graphics Context values                   */
	XFontStruct    *wf;	/* which font are we in                      */
	static char    *buffer = NULL;
	static int      currmaxcol = 0;

#ifdef MULTICHAR_SET
	static int      oldcursormulti = 0;
#endif
	static row_col_t oldcursor = { -1, -1 };

    /* is there an old outline cursor on screen? */

#ifdef THAI			/* variables to clean un-touched space at the end of screen */
	int             thaicol_stp_offset = 0;
	int             thaicol_dtp_offset = 0;
#endif

#ifndef NO_BOLDFONT
	int             bfont = 0;	/* we've changed font to bold font           */
#endif
	int             (*draw_string) (), (*draw_image_string) ();

	if (type == NO_REFRESH)
		return;

  DSCR_ALEXIS(2,(stderr,"f_scr_refresh(%d,...)\n",page));
/*
 * A: set up vars
 */
	if (currmaxcol < TermWin.ncol)
	{
		currmaxcol = TermWin.ncol;
		if (buffer)
			buffer =  REALLOC(buffer, (sizeof(char) * (currmaxcol + 1)));
		else
			buffer = MALLOC((sizeof(char) * (currmaxcol + 1)));
	}
	row_offset = TermWin.vts[page].saveLines - TermWin.vts[page].view_start;
	fprop = TermWin.fprop;
	is_cursor = 0;
	gcvalue.foreground = PixColors[Color_fg];
	gcvalue.background = PixColors[Color_bg];

/*
 * always go back to the base font - it's much safer
 */
	wbyte = 0;
  //assert(TermWin.vts[page].gc != None);
	XSetFont(Xdisplay, TermWin.vts[page].gc, TermWin.font->fid);
	draw_string = XDrawString;
	draw_image_string = XDrawImageString;
	boldlast = 0;

#ifndef NO_BOLDOVERSTRIKE
/*
 * B: Bold Overstrike pixel dropping avoidance.  Do this FIRST.
 *    Do a pass across each line at the start, require a refresh of anything
 *    that will need to be refreshed, due to pixels being dropped into our
 *    area by a previous character which has now been changed.
 */
	for (row = 0; row < TermWin.nrow; row++)
	{
		scrrow = row + row_offset;
		stp = screen[page].text[scrrow]; // on accede a la ligne
		srp = screen[page].rend[scrrow]; // on accede au rendu de la ligne
		dtp = drawn_text[row]; // on accede a la ligne qui sera affichee
		drp = drawn_rend[row]; // on accede au rendu de la ligne a afficher
# ifndef NO_BOLDFONT
		if (TermWin.boldFont == NULL)
		{
# endif
			wf = TermWin.font;
			j = wbyte;
			for (col = TermWin.ncol - 2; col >= 0; col--)
			{
				int             do_stuff;

# if ! defined (NO_BRIGHTCOLOR) && ! defined (VERYBOLD)
				fore = GET_FGCOLOR(drp[col]);
# endif
				if (!MONO_BOLD(drp[col])) 
					continue;
				if (dtp[col] == stp[col] && drp[col] == srp[col])
					continue;
				if (wbyte)
				{
					;	/* TODO: handle multibyte */
					continue;	/* don't go past here */
				}
				if (dtp[col] == ' ')
				{	/* TODO: check character set? */
					continue;
				}

				if (wf->per_char == NULL)
					do_stuff = 1;
				else if (dtp[col] >= wf->min_char_or_byte2 && dtp[col] < wf->max_char_or_byte2)
				{
					int char_num = dtp[col] - wf->min_char_or_byte2;
					do_stuff = (wf->per_char[char_num].width == wf->per_char[char_num].rbearing);
				} else
					do_stuff = 0;

				if (do_stuff)
				{
#ifndef THAI			/* this mess up Thai chars, we have to implement ourselves */
					dtp[col + 1] = 0;
#endif
# if defined(MULTICHAR_SET) && ! defined(NO_BOLDOVERSTRIKE_MULTI)
					if ((srp[col] & RS_multiMask) == RS_multi2)
					{
						col--;
						wbyte = 1;
						continue;
					}
# endif
				}
			}
# if ! defined (NO_BRIGHTCOLOR) && ! defined (VERYBOLD)
			fore = GET_FGCOLOR(srp[TermWin.ncol - 1]);
# endif
			if (MONO_BOLD(srp[TermWin.ncol - 1]))
				boldlast = 1;
			wbyte = j;
# ifndef NO_BOLDFONT
		}
# endif
	}
#endif				/* ! NO_BOLDOVERSTRIKE */

/*
 * C: reverse any characters which are selected
 */
	f_scr_reverse_selection(page);

/*
 * D: set the cursor character(s)
 */
	currow = screen[page].cur.row + TermWin.vts[page].saveLines;
	if (focus != TermWin.focus)
		focus = TermWin.focus;
	if (screen[page].flags & Screen_VisibleCursor && focus)
	{
		srp = &(screen[page].rend[currow][screen[page].cur.col]);
#ifdef THAI
		if (CharWidth(wf, screen[page].text[currow][screen[page].cur.col]) <= 0)
			*srp ^= RS_Bold;
#endif
#ifndef NO_CURSORCOLOR
		cc1 = *srp & (RS_fgMask | RS_bgMask);
		if (Xdepth <= 2 || !rs_color[Color_cursor])
			ccol1 = Color_fg;
		else
			ccol1 = Color_cursor;
		if (Xdepth <= 2 || !rs_color[Color_cursor2])
			ccol2 = Color_bg;
		else
			ccol2 = Color_cursor2;
	    /*  changed by rev vide cursor patch */
		*srp = SET_BGCOLOR(*srp, ccol1);
		*srp = SET_FGCOLOR(*srp, ccol2);
#endif
#ifdef MULTICHAR_SET
		rt1 = *srp & RS_multiMask;
		if (rt1 == RS_multi1)
		{
			if (screen[page].cur.col < TermWin.vts[page].bcol - 2 && ((srp[1] & RS_multiMask) == RS_multi2))
				morecur = 1;
		} else if (rt1 == RS_multi2)
		{
			if (screen[page].cur.col > 0 && ((srp[-1] & RS_multiMask) == RS_multi1))
				morecur = -1;
		}
		if (morecur)
		{
			srp += morecur;
			*srp ^= RS_RVid;
		}
# ifndef NO_CURSORCOLOR
		if (morecur)
		{
			cc2 = *srp & (RS_fgMask | RS_bgMask);
			*srp = SET_BGCOLOR(*srp, ccol1);
			*srp = SET_FGCOLOR(*srp, ccol2);
		}
# endif
#endif
	}
	i = 0;
	if (oldcursor.row != -1)
	{
	    /* make sure no outline cursor is left around */
		if (screen[page].cur.row != oldcursor.row || screen[page].cur.col != oldcursor.col)
		{
			if (oldcursor.row < TermWin.nrow && oldcursor.col < TermWin.vts[page].bcol)
			{
				drawn_text[oldcursor.row][oldcursor.col] = 0;
#ifdef MULTICHAR_SET
				if (oldcursormulti)
				{
					col = oldcursor.col + oldcursormulti;
					if (col < TermWin.ncol)
						drawn_text[oldcursor.row][col] = 0;
				}
#endif
			}
			if (focus || !(screen[page].flags & Screen_VisibleCursor))
				oldcursor.row = -1;
			else
				i = 1;
		}
	} else if (!focus)
		i = 1;
	if (i)
	{
		oldcursor.row = screen[page].cur.row;
		oldcursor.col = screen[page].cur.col;
#ifdef MULTICHAR_SET
		oldcursormulti = morecur;
#endif
	}

/*
 * E: OK, now the real pass
 */
	for (row = 0; row < TermWin.nrow; row++)
	{
#ifdef THAI
		char            thai_update[TermWin.ncol];	/* update map */
		int             stpx, dtpx;

	    /* compare drawn_text and screen[page].text and check which to update */
		ThaiUpdateMap(screen[page].text[row + row_offset], drawn_text[row],
			      screen[page].rend[row + row_offset], drawn_rend[row],
			      thai_update, TermWin.ncol);

/*
 * TODO: Change this algo to scalefont compatible
 */
		thaicol_stp_offset = 0;	/* records how many column deficit */
		thaicol_dtp_offset = TermWin.ncol - Thai_ColMaxPaint(dtp, TermWin.ncol);
	    /* clean strings before redraw */
		for (col = 0; col < TermWin.ncol; col++)
		{
			if (!ThaiIsMiddleLineCh(stp[col]))
				thaicol_stp_offset++;
		}

		if (thaicol_stp_offset > thaicol_dtp_offset)
		{
			f_CLEAR_CHARS(page,Col2Pixel(TermWin.ncol - thaicol_stp_offset),
				    ypixel - TermWin.font->ascent,
				    thaicol_stp_offset);

		}
#endif
		scrrow = row + row_offset;
		stp = screen[page].text[scrrow];
		srp = screen[page].rend[scrrow];
		dtp = drawn_text[row];
		drp = drawn_rend[row];
		for (col = 0; col < TermWin.ncol; col++)
		{
		    /* compare new text with old - if exactly the same then continue */
			rt1 = srp[col];	/* screen rendition */
			rt2 = drp[col];	/* drawn rendition  */
#ifdef THAI			/* updating conditional */
			if (!thai_update[col])
			{
#else
			if ((stp[col] == dtp[col])	/* must match characters to skip */
			    && ((rt1 == rt2)	/* either rendition the same or  */
				|| ((stp[col] == ' ')	/* space w/ no bg change */
				    && (GET_BGATTR(rt1) == GET_BGATTR(rt2)))))
			{
#endif
#ifdef MULTICHAR_SET
			    /* if first byte is Kanji then compare second bytes */
				if ((rt1 & RS_multiMask) != RS_multi1)
					continue;
				else if (stp[col + 1] == dtp[col + 1])
				{
				    /* assume no corrupt characters on the screen */
					col++;
					continue;
				}
#else
				continue;
#endif
			}
		    /* redraw one or more characters */
#ifndef THAI			/* can not set dtp as we go because it causes
				 * unpredictable results (bug)
				 */
			dtp[col] = stp[col];
#endif
			rend = drp[col] = srp[col];

			len = 0;
			buffer[len++] = stp[col];
			ypixel = TermWin.font->ascent + Row2Pixel(row);
#ifdef THAI			/* compute x coordinate */
			xpixel = ThaiCol2Pixel(col, screen[page].text[scrrow]);
#else
			xpixel = Col2Pixel(col);
#endif				/* THAI */
			wlen = 1;

/*
 * Find out the longest string we can write out at once
 */
			if (fprop == 0)
			{	/* Fixed width font */
#ifdef MULTICHAR_SET
				if (((rend & RS_multiMask) == RS_multi1)
				    && col < TermWin.ncol - 1
				    && ((srp[col + 1]) & RS_multiMask) ==  RS_multi2)
				{
					if (!wbyte)
					{
						wbyte = 1;
						XSetFont(Xdisplay, TermWin.vts[page].gc,TermWin.mfont->fid);
						draw_string = XDrawString16;
						draw_image_string = XDrawImageString16;
					}
				    /* double stepping - we're in Kanji mode */
					for (; ++col < TermWin.vts[page].bcol;)
					{
					    /* XXX: could check sanity on 2nd byte */
						dtp[col] = stp[col];
						drp[col] = srp[col];
						buffer[len++] = stp[col];
						col++;
						if ((col == TermWin.vts[page].bcol)
						    || (srp[col] != rend))
							break;
						if ((stp[col] == dtp[col])
						    && (srp[col] == drp[col])
						    && (stp[col + 1] ==	dtp[col + 1]))
							break;
						if (len == currmaxcol)
							break;
						dtp[col] = stp[col];
						drp[col] = srp[col];
						buffer[len++] = stp[col];
					}
					col--;
					if (buffer[0] & 0x80)
						multichar_decode(buffer, len);
					wlen = len / 2;
				} else
				{
					if ((rend & RS_multiMask) == RS_multi1)
					{
					    /* XXX : maybe do the same thing for RS_multi2 */
					    /* corrupt character - you're outta there */
						rend &= ~RS_multiMask;
						drp[col] = rend;	/* TODO check: may also want */
						dtp[col] = ' ';	/* to poke into stp/srp      */
						buffer[0] = ' ';
					}
					if (wbyte)
					{
						wbyte = 0;
						XSetFont(Xdisplay, TermWin.vts[page].gc,TermWin.font->fid);
						draw_string = XDrawString;
						draw_image_string = XDrawImageString;
					}
#endif
				    /* single stepping - `normal' mode */
					for (; ++col < TermWin.ncol - 1;)
					{
						if (rend != srp[col])
							break;
						if ((stp[col] == dtp[col]) && (srp[col] == drp[col]))
							break;
						if (len == currmaxcol)
							break;
						dtp[col] = stp[col];
						drp[col] = srp[col];
						buffer[len++] = stp[col];
					}
					col--;
					wlen = len;
#ifdef MULTICHAR_SET
				}
#endif
			}
			buffer[len] = '\0';

/*
 * Determine the attributes for the string
 */
			fore = GET_FGCOLOR(rend);
			back = GET_BGCOLOR(rend);
			rend = GET_ATTR(rend);
			gcmask = 0;
			rvid = (rend & RS_RVid) ? 1 : 0;

			switch (rend & RS_fontMask)
			{
			case RS_acsFont:
				for (i = 0; i < len; i++)
					if (buffer[i] == 0x5f)
						buffer[i] = 0x7f;
					else if (buffer[i] > 0x5f
						 && buffer[i] < 0x7f)
						buffer[i] -= 0x5f;
				break;
			case RS_ukFont:
				for (i = 0; i < len; i++)
					if (buffer[i] == '#')
						buffer[i] = 0x1e;
				break;
			}
			if (rvid)
				SWAP_IT(fore, back, i);

			if (colors_changed)
			{
				gcvalue.background = PixColors[back];
				gcmask |= GCBackground;
				gcvalue.foreground = PixColors[fore];
				gcmask |= GCForeground;
				colors_changed = 0;
			} else if (back != Color_bg)
			{
				gcvalue.background = PixColors[back];
				gcmask |= GCBackground;
			}
			if (fore != Color_fg)
			{
				gcvalue.foreground = PixColors[fore];
				gcmask |= GCForeground;
			}
#ifndef NO_BOLDUNDERLINE
			else if (rend & RS_Bold)
			{
				if (Xdepth > 2 && rs_color[Color_BD]
				    && PixColors[fore] != PixColors[Color_BD]
				    && PixColors[back] != PixColors[Color_BD])
				{
					gcvalue.foreground = PixColors[Color_BD];
					gcmask |= GCForeground;
# ifndef VERYBOLD
					rend &= ~RS_Bold;	/* we've taken care of it */
# endif
				}
			} else if (rend & RS_Uline)
			{
				if (Xdepth > 2 && rs_color[Color_UL]
				    && PixColors[fore] != PixColors[Color_UL]
				    && PixColors[back] != PixColors[Color_UL])
				{
					gcvalue.foreground = PixColors[Color_UL];
					gcmask |= GCForeground;
					rend &= ~RS_Uline;	/* we've taken care of it */
				}
			}
#endif

			if (gcmask)
				XChangeGC(Xdisplay, TermWin.vts[page].gc, gcmask,&gcvalue);

#ifndef NO_BOLDFONT
			if (!wbyte && MONO_BOLD(rend)
			    && TermWin.boldFont != NULL)
			{
				bfont = 1;
				XSetFont(Xdisplay, TermWin.vts[page].gc,TermWin.boldFont->fid);
				rend &= ~RS_Bold;	/* we've taken care of it */
			} else if (bfont)
			{
				bfont = 0;
				XSetFont(Xdisplay, TermWin.vts[page].gc,TermWin.font->fid);
			}
#endif
/*
 * Actually do the drawing of the string here
 */
			if (fprop)
			{
#ifdef THAI
				if (rvid && CharWidth(wf, stp[col]) > 0)
				{
#else
				if (rvid)
				{
#endif
					SWAP_IT(gcvalue.foreground,gcvalue.background, ltmp);
					gcmask |= (GCForeground | GCBackground);
					XChangeGC(Xdisplay, TermWin.vts[page].gc, gcmask,&gcvalue);
					XFillRectangle(Xdisplay, drawBufferPage(page),
						       TermWin.vts[page].gc, xpixel,
						       ypixel - TermWin.font->ascent,
						       Width2Pixel(1),
						       Height2Pixel(1));
					SWAP_IT(gcvalue.foreground,gcvalue.background, ltmp);
					XChangeGC(Xdisplay, TermWin.vts[page].gc, gcmask, &gcvalue);
				} else
#ifdef THAI
				if (CharWidth(wf, stp[col]) > 0)
#endif
        f_CLEAR_CHARS(page,xpixel, ypixel - TermWin.font->ascent, 1);
				f_DRAW_STRING(page,draw_string, xpixel, ypixel,buffer, 1);
#ifdef THAI
/*
 * Trap error
 */
				if (wlen > 1)
					DSCR_ALEXIS(2,(stderr, "wlen > 1: %d\n",wlen));
#endif
#ifndef NO_BOLDOVERSTRIKE
				if (MONO_BOLD(rend))
					f_DRAW_STRING(page,draw_string, xpixel + 1, ypixel, buffer, 1);
#endif
			} else
#ifdef TRANSPARENT
			if ((Options & Opt_transparent) && back == Color_bg)
      //if((Options & TermWin.vts[page].bg.transparent) && back == Color_bg)
			{
#ifdef THAI
				if (CharWidth(wf, stp[col]) > 0)
#endif
					f_CLEAR_CHARS(page,xpixel,ypixel - TermWin.font->ascent, len);
				f_DRAW_STRING(page,draw_string, xpixel, ypixel, buffer, wlen);
			} else
#endif
#if defined(BACKGROUND_IMAGE) || defined(_MYSTYLE_)
			if (TermWin.vts[page].bg.trgPixmapSet && back == Color_bg)
			{
				f_CLEAR_CHARS(page,xpixel,ypixel - TermWin.font->ascent,len);
				f_DRAW_STRING(page,draw_string, xpixel, ypixel,buffer, wlen);
			} else
#endif
				f_DRAW_STRING(page,draw_image_string, xpixel, ypixel,buffer, wlen);

#ifndef NO_BOLDOVERSTRIKE
# ifdef NO_BOLDOVERSTRIKE_MULTI
			if (!wbyte)
# endif
				if (MONO_BOLD(rend))
					f_DRAW_STRING(page,draw_string, xpixel + 1,ypixel, buffer, wlen);
#endif
			if ((rend & RS_Uline) && (TermWin.font->descent > 1))
				XDrawLine(Xdisplay, drawBufferPage(page), TermWin.vts[page].gc,
					  xpixel, ypixel + 1,
					  xpixel + Width2Pixel(len) - 1,
					  ypixel + 1);
			if (gcmask)
			{	/* restore normal colours */
				gcvalue.foreground = PixColors[Color_fg];
				gcvalue.background = PixColors[Color_bg];
				XChangeGC(Xdisplay, TermWin.vts[page].gc, gcmask,&gcvalue);
			}
		}
	}
#ifdef THAI
/*
 * Hack: update drawn_text
 */
	for (row = 0; row < TermWin.nrow; row++)
	{
		scrrow = row + row_offset;
		stp = screen[page].text[scrrow];
		srp = screen[page].rend[scrrow];
		dtp = drawn_text[row];
		drp = drawn_rend[row];

		for (col = 0; col < TermWin.ncol; col++)
		{
			dtp[col] = stp[col];
			drp[col] = srp[col];
		}
	}
#endif

/*
 * F: cleanup cursor and display outline cursor in necessary
 */
	if (screen[page].flags & Screen_VisibleCursor)
	{
		if (focus)
		{
			srp = &(screen[page].rend[currow][screen[page].cur.col]);
#ifdef THAI
			if (CharWidth(wf, screen[page].text[currow][screen[page].cur.col]) <= 0)
				*srp ^= RS_Bold;
#endif
#ifndef NO_CURSORCOLOR
			*srp = (*srp & ~(RS_fgMask | RS_bgMask)) | cc1;
#endif
			if (morecur)
			{
				srp += morecur;
#if defined(MULTICHAR_SET) && ! defined(NO_CURSORCOLOR)
				*srp = (*srp & ~(RS_fgMask | RS_bgMask)) | cc2;
#endif
			}
		} else
		{
			currow = screen[page].cur.row - TermWin.vts[page].view_start;
			col = screen[page].cur.col + morecur;
			wbyte = morecur ? 1 : 0;
			if (currow >= 0 && currow < TermWin.nrow)
			{
#ifndef NO_CURSORCOLOR
				gcmask = 0;
				if (Xdepth > 2 && rs_color[Color_cursor])
				{
					gcvalue.foreground = PixColors[Color_cursor];
					gcmask = GCForeground;
					XChangeGC(Xdisplay, TermWin.vts[page].gc, gcmask, &gcvalue);
					gcvalue.foreground = PixColors[Color_fg];
				}
#endif
#ifdef THAI
        XDrawRectangle(Xdisplay, drawBufferPage(page),
          TermWin.vts[page].gc, 
          ThaiCol2Pixel(col,screen[page].text[screen[page].cur.row + TermWin.vts[page].saveLines]),
          Row2Pixel(currow),
          Width2Pixel(1 + wbyte) - 1,
          Height2Pixel(1) - 1);

#else

				XDrawRectangle(Xdisplay, drawBufferPage(page),
          TermWin.vts[page].gc, Col2Pixel(col),
          Row2Pixel(currow),
          Width2Pixel(1 + wbyte) - 1,
          Height2Pixel(1) - 1);
#endif
#ifndef NO_CURSORCOLOR
				if (gcmask)	/* restore normal colours */
					XChangeGC(Xdisplay, TermWin.vts[page].gc, gcmask,&gcvalue);
#endif
			}
		}
	}

/*
 * G: cleanup selection
 */
	f_scr_reverse_selection(page);

/*
 * H: other general cleanup
 */
 DSCR_ALEXIS(2,(stderr,"vt[%d] is %s\n",page,TermWin.vts[page].bMapped ? "mapped" : "unmapped"));
	if (boldlast && TermWin.vts[page].bMapped)	/* clear the whole screen height */
		XClearArea(Xdisplay, TermWin.vts[page].vt, TermWin_TotalWidth() - 2, 0,
			   1, TermWin_TotalHeight() - 1, 0);
	if (type & SMOOTH_REFRESH)
		XSync(Xdisplay, False);
}



void f_scr_clear_tint(int page,int bWithTinting)
{
	if (!TermWin.vts[page].bMapped)
		return;
  DSCR_ALEXIS(2,(stderr,"f_scr_clear_tint(%d,%d)\n",page,bWithTinting));
#ifdef TRANSPARENT
	if (Options & Opt_transparent)
  //if (Options & TermWin.vts[page].bg.transparent)
	{
/*	int n ;
	fprintf( stderr, "\n Clearing Parents ...");
	for( n = 0 ; n < ParentWinNum ; n++ )
*/
		XClearWindow(Xdisplay, TermWin.parent);
/*	XFlush( Xdisplay );
	sleep(5);
	fprintf( stderr, "Done.");
*/ }
#endif
	XClearWindow(Xdisplay, TermWin.vts[page].vt);

#if defined(BACKGROUND_IMAGE) || defined(TRANSPARENT)
	if (TermWin.vts[page].tintGC && bWithTinting)
	{
		XFillRectangle(Xdisplay, TermWin.vts[page].vt, TermWin.vts[page].tintGC, 0, 0,
			       TermWin.width + TermWin_internalBorders,
			       TermWin.height + TermWin_internalBorders);
	}
#endif
}

/* ------------------------------------------------------------------------- */
/* colors has changed so we have to repaint self			     */
void f_on_colors_changed(int page,int idx)
{
	colors_changed = 1;
	if (idx == Color_bg && !(Options & Opt_transparent))
  //if (idx == Color_bg && ! (Options & TermWin.vts[page].bg.transparent))
	{
		XSetWindowBackground(Xdisplay, TermWin.vts[page].vt,PixColors[Color_bg]);
	}

/* handle Color_BD, scrollbar background, etc. */

	set_colorfgbg();
	set_cursor_color();
/* the only reasonable way to enforce a clean update : */
	f_scr_clear(page);
	f_scr_touch(page);
}


/* ------------------------------------------------------------------------- */
/* PROTO */
void
f_scr_reverse_selection(int page)
{
	int             i, col, row, end_row;
	rend_t         *srp;

  DSEL_ALEXIS(2,(stderr,"f_scr_reverse_selection(%d)\n",page));
	end_row = TermWin.vts[page].saveLines - TermWin.vts[page].view_start;
	if (selection.op /*&& current_screen == selection.screen*/ && selection.vt == page)
	{
		i = selection.beg.row + TermWin.vts[page].saveLines;
		row = selection.end.row + TermWin.vts[page].saveLines;
		if (i >= end_row)
			col = selection.beg.col;
		else
		{
			col = 0;
			i = end_row;
		}
		end_row += TermWin.nrow;
		for (; i < row && i < end_row; i++, col = 0)
			for (srp = screen[page].rend[i]; col < TermWin.vts[page].bcol; col++)
				srp[col] ^= RS_RVid;
		if (i == row && i < end_row)
			for (srp = screen[page].rend[i]; col < selection.end.col; col++)
				srp[col] ^= RS_RVid;
	}
  
}

/* ------------------------------------------------------------------------- *
 *                           CHARACTER SELECTION                             *
 * ------------------------------------------------------------------------- */

/*
 * -TermWin.nscrolled <= (selection row) <= TermWin.nrow - 1
 */
/* PROTO */
void
f_selection_check(int page,int check_more)
{
	row_col_t       pos;

  DSEL_ALEXIS(2,(stderr,"f_selection_check(%d,%d)\n",page,check_more));
	if ((selection.beg.row < -TermWin.vts[page].nscrolled)
	    || (selection.beg.row >= TermWin.nrow)
	    || (selection.mark.row < -TermWin.vts[page].nscrolled)
	    || (selection.mark.row >= TermWin.nrow)
	    || (selection.end.row < -TermWin.vts[page].nscrolled)
	    || (selection.end.row >= TermWin.nrow))
		CLEAR_ALL_SELECTION;

	if (check_more == 1 && current_screen == selection.screen)
	{
	    /* check for cursor position */
		pos.row = screen[page].cur.row;
		pos.col = screen[page].cur.col;
		if (!ROWCOL_IS_BEFORE(pos, selection.beg)
		    && ROWCOL_IS_BEFORE(pos, selection.end))
			CLEAR_SELECTION;
	} else if (check_more == 2)
	{
		pos.row = 0;
		pos.col = 0;
		if (ROWCOL_IS_BEFORE(selection.beg, pos)
		    && ROWCOL_IS_AFTER(selection.end, pos))
			CLEAR_SELECTION;
	} else if (check_more == 3)
	{
		pos.row = 0;
		pos.col = 0;
		if (ROWCOL_IS_AFTER(selection.end, pos))
			CLEAR_SELECTION;
	}
}

/* ------------------------------------------------------------------------- */
/*
 * Paste a selection direct to the command
 */
/* PROTO */
void
f_PasteIt(int page,unsigned char *data, unsigned int nitems)
{
	int             num;
	unsigned char  *p, cr;

  DSEL_ALEXIS(2,(stderr,"f_PasteIt(%d)\n",page));
  
	cr = '\r';
	for (p = data, num = 0; nitems--; p++)
		if (*p != '\n')
			num++;
		else
		{
			f_tt_write(page,data, num);
			f_tt_write(page,&cr, 1);
			data += (num + 1);
			num = 0;
		}
	if (num)
		f_tt_write(page,data, num);
}

/* ------------------------------------------------------------------------- */
/*
 * Respond to a notification that a primary selection has been sent
 * EXT: SelectionNotify
 */
/* PROTO */
void
selection_paste(Window win, unsigned prop, int Delete)
{
	long            nread;
	unsigned long   bytes_after, nitems;
	unsigned char  *data;
	Atom            actual_type;
	int             actual_fmt;

  DSEL_ALEXIS(2,(stderr,"selection_paste(...)\n"));
	if (prop == None)
		return;
	for (nread = 0, bytes_after = 1; bytes_after > 0; nread += nitems)
	{
		if ((XGetWindowProperty
		     (Xdisplay, win, prop, (nread / 4), PROP_SIZE, Delete,
		      AnyPropertyType, &actual_type, &actual_fmt, &nitems,
		      &bytes_after, &data) != Success))
		{
			XFree(data);
			return;
		}
		f_PasteIt(TermWin.active_page,data, nitems);
		XFree(data);
	}
}

/* ------------------------------------------------------------------------- */
/*
 * Request the current primary selection
 * EXT: button 2 release
 */
void f_selection_request(int page,Time tm, int x, int y)
{
	Atom            prop;
  DSEL_ALEXIS(2,(stderr,"f_selection_request(%d)\n",page));

	if (x < 0 || x >= TermWin.width || y < 0 || y >= TermWin.height)
		return;		/* outside window */

	if (selection.text != NULL)
	{
		f_PasteIt(TermWin.active_page,selection.text, selection.len);	/* internal selection */
	} else if (XGetSelectionOwner(Xdisplay, XA_PRIMARY) == None)
	{
		selection_paste(Xroot, XA_CUT_BUFFER0, False);
	} else
	{
		prop = XInternAtom(Xdisplay, "VT_SELECTION", False);
		XConvertSelection(Xdisplay, XA_PRIMARY, XA_STRING, prop,
				  TermWin.vts[page].vt, tm);
	}
}

/* ------------------------------------------------------------------------- */
/*
 * Clear all selected text
 * EXT: SelectionClear
 */
void selection_clear(void)
{
	//D_SELECT((stderr, "selection_clear()"));

  DSEL_ALEXIS(2,(stderr,"selection_clear()\n"));
  
	if (selection.text)
		FREE(selection.text);
	selection.text = NULL;
	selection.len = 0;
	CLEAR_SELECTION;
  selection.vt = -1;
}

/* ------------------------------------------------------------------------- */
/*
 * Copy a selection into the cut buffer
 * EXT: button 1 or 3 release
 */

void f_selection_make(int page,Time tm)
{
	int             i, col, end_col, row, end_row;
	unsigned char  *new_selection_text;
	char           *str;
	text_t         *t;

	DSEL_ALEXIS(2,(stderr,
		  "selection_make(): selection.op=%d, selection.clicks=%d\n",
		  selection.op, selection.clicks));
	switch (selection.op)
	{
	case SELECTION_CONT:
		break;
	case SELECTION_INIT:
		CLEAR_SELECTION;
	    /* FALLTHROUGH */
	case SELECTION_BEGIN:
		selection.op = SELECTION_DONE;
	    /* FALLTHROUGH */
	default:
		return;
	}
	selection.op = SELECTION_DONE;
  selection.vt = page;

	if (selection.clicks == 4)
		return;		/* nothing selected, go away */

	i = (selection.end.row - selection.beg.row + 1) * (TermWin.vts[page].bcol + 1) +
	    1;
	str = MALLOC(i * sizeof(char));
	new_selection_text = (unsigned char *)str;

	col = max(selection.beg.col, 0);
	row = selection.beg.row + TermWin.vts[page].saveLines;
	end_row = selection.end.row + TermWin.vts[page].saveLines;
/*
 * A: rows before end row
 */
	for (; row < end_row; row++)
	{
		t = &(screen[page].text[row][col]);
		if ((end_col = screen[page].tlen[row]) == -1)
			end_col = TermWin.vts[page].bcol;
		for (; col < end_col; col++)
			*str++ = *t++;
		col = 0;
		if (screen[page].tlen[row] != -1)
			*str++ = '\n';
	}
/*
 * B: end row
 */
	t = &(screen[page].text[row][col]);
	end_col = screen[page].tlen[row];
	if (end_col == -1 || selection.end.col <= end_col)
		end_col = selection.end.col;
	MIN_IT(end_col, TermWin.vts[page].bcol);	/* CHANGE */
	for (; col < end_col; col++)
		*str++ = *t++;
	if (end_col != selection.end.col)
		*str++ = '\n';
	*str = '\0';
	if ((i = strlen((char *)new_selection_text)) == 0)
	{
		FREE(new_selection_text);
		return;
	}
	selection.len = i;
  
	if (selection.text)
		FREE(selection.text);
  
	selection.text = new_selection_text;

	XSetSelectionOwner(Xdisplay, XA_PRIMARY, TermWin.vts[page].vt, tm);
  
	if (XGetSelectionOwner(Xdisplay, XA_PRIMARY) != TermWin.vts[page].vt)
		print_error("can't get primary selection");
  
	XChangeProperty(Xdisplay, Xroot, XA_CUT_BUFFER0, XA_STRING, 8,
			PropModeReplace, selection.text, selection.len);
	DSEL_ALEXIS(2,(stderr, "selection_make(): selection.len=%d\n", selection.len));
}

/* ------------------------------------------------------------------------- */
/*
 * Mark or select text based upon number of clicks: 1, 2, or 3
 * EXT: button 1 press
 */
/* PROTO */
void
selection_click(int clicks, int x, int y)
{
/*    int             r, c;
 *   row_col_t             ext_beg, ext_end;
 */

	DSEL_ALEXIS(2,(stderr, "selection_click(%d, %d, %d)\n", clicks, x, y));
  selection.vt = TermWin.active_page;

	clicks = ((clicks - 1) % 3) + 1;
	selection.clicks = clicks;	/* save clicks so extend will work */
#ifdef THAI
	selection_start_colrow(ThaiPixel2Col(x, y), Pixel2Row(y));
#else
	selection_start_colrow(Pixel2Col(x), Pixel2Row(y));
#endif
	if (clicks == 2 || clicks == 3)
		f_selection_extend_colrow(TermWin.active_page,selection.mark.col, 
      selection.mark.row + TermWin.vts[TermWin.active_page].view_start, 
      0,	/* button 3     */
			1,	/* button press */
			0);	/* click change */
}

/* ------------------------------------------------------------------------- */
/*
 * Mark a selection at the specified col/row
 */

void selection_start_colrow(int col, int row)
{
  DSEL_ALEXIS(2,(stderr, "selection_start_colrow(%d, %d)\n",col,row));
  selection.vt = TermWin.active_page;
	selection.mark.col = col;
	selection.mark.row = row - TermWin.vts[TermWin.active_page].view_start;
	MAX_IT(selection.mark.row, -TermWin.vts[TermWin.active_page].nscrolled);
	MIN_IT(selection.mark.row, TermWin.nrow - 1);
	MAX_IT(selection.mark.col, 0);
	MIN_IT(selection.mark.col, TermWin.vts[TermWin.active_page].bcol - 1);

	if (selection.op)
	{			/* clear the old selection */
		selection.beg.row = selection.end.row = selection.mark.row;
		selection.beg.col = selection.end.col = selection.mark.col;
	}
	selection.op = SELECTION_INIT;
	selection.screen = current_screen;
}

/* ------------------------------------------------------------------------- */
/*
 * Word select: select text for 2 clicks
 * We now only find out the boundary in one direction
 */

/* what do we want: spaces/tabs are delimiters or cutchars or non-cutchars */
#define DELIMIT_TEXT(x) \
    (((x) == ' ' || (x) == '\t') ? 2 : (strchr(rs_cutchars, (x)) != NULL))
#ifdef MULTICHAR_SET
# define DELIMIT_REND(x)	(((x) & RS_multiMask) ? 1 : 0)
#else
# define DELIMIT_REND(x)	1
#endif

void f_selection_delimit_word(int page,int dirn, row_col_t * mark, row_col_t * ret)
{
	int             col, row, dirnadd, tcol, trow, w1, w2;
	row_col_t       bound;
	text_t         *stp;
	rend_t         *srp;

  DSEL_ALEXIS(2,(stderr, "f_selection_delimit_word(%d, %d)\n",page,dirn));
  selection.vt = TermWin.active_page;
	if (selection.clicks != 2)
		return;		/* Go away: we only handle double clicks */

	if (dirn == UP)
	{
		bound.row = TermWin.vts[page].saveLines - TermWin.vts[page].nscrolled - 1;
		bound.col = 0;
		dirnadd = -1;
	} else
	{
		bound.row = TermWin.vts[page].saveLines + TermWin.nrow;
		bound.col = TermWin.vts[page].bcol - 1;
		dirnadd = 1;
	}
	row = mark->row + TermWin.vts[page].saveLines;
	col = mark->col;
	MAX_IT(col, 0);
/* find the edge of a word */
	stp = &(screen[page].text[row][col]);
	w1 = DELIMIT_TEXT(*stp);
#ifdef OLD_WORD_SELECTION
	if (w1 == 1)
	{
		stp += dirnadd;
		if (DELIMIT_TEXT(*stp) == 1)
			goto Old_Word_Selection_You_Die;
		col += dirnadd;
		srp += dirnadd;
	}
	w1 = 0;
#endif
	srp = (&screen[page].rend[row][col]);
	w2 = DELIMIT_REND(*srp);

	for (;;)
	{
		for (; col != bound.col; col += dirnadd)
		{
			stp += dirnadd;
			if (DELIMIT_TEXT(*stp) != w1)
				break;
			srp += dirnadd;
			if (DELIMIT_REND(*srp) != w2)
				break;
		}
		if ((col == bound.col) && (row != bound.row))
		{
			if (screen[page].tlen[(row - (dirn == UP))] == -1)
			{
				trow = row + dirnadd;
				tcol = (dirn == UP) ? (TermWin.vts[page].bcol - 1) : 0;
				if (screen[page].text[trow] == NULL)
					break;
				stp = &(screen[page].text[trow][tcol]);
				srp = &(screen[page].rend[trow][tcol]);
				if (DELIMIT_TEXT(*stp) != w1 || DELIMIT_REND(*srp) != w2)
					break;
				row = trow;
				col = tcol;
				continue;
			}
		}
		break;
	}
#ifdef OLD_WORD_SELECTION
      Old_Word_Selection_You_Die:
#endif
	DSEL_ALEXIS(2,(stderr,
		  "selection_delimit_word(%s,...) @ (r:%3d, c:%3d) has boundary (r:%3d, c:%3d)\n",
		  (dirn == UP ? "up	" : "down"), mark->row, mark->col,
		  row - TermWin.vts[page].saveLines, col));

	if (dirn == DN)
		col++;		/* put us on one past the end */

/* Poke the values back in */
	ret->row = row - TermWin.vts[page].saveLines;
	ret->col = col;
}

/* ------------------------------------------------------------------------- */
/*
 * Extend the selection to the specified x/y pixel location
 * EXT: button 3 press; button 1 or 3 drag
 * flag == 0 ==> button 1
 * flag == 1 ==> button 3 press
 * flag == 2 ==> button 3 motion
 */
void selection_extend(int x, int y, int flag)
{
	int             col, row;

  DSEL_ALEXIS(2,(stderr, "selection_extend(%d, %d, %d)\n",x,y,flag));
#ifdef THAI
	col = ThaiPixel2Col(x, y);
#else
	col = Pixel2Col(x);
#endif
	row = Pixel2Row(y);
	MAX_IT(row, 0);
	MIN_IT(row, TermWin.nrow - 1);
	MAX_IT(col, 0);
	MIN_IT(col, TermWin.vts[TermWin.active_page].bcol);
#ifndef OLD_SELECTION
/*
 * If we're selecting characters (single click) then we must check first
 * if we are at the same place as the original mark.  If we are then
 * select nothing.  Otherwise, if we're to the right of the mark, you have to
 * be _past_ a character for it to be selected.
 */
	if (((selection.clicks % 3) == 1) && !flag
	    && (col == selection.mark.col
		&& (row == selection.mark.row + TermWin.vts[TermWin.active_page].view_start)))
	{
	    /* select nothing */
		selection.beg.row = selection.end.row = 0;
		selection.beg.col = selection.end.col = 0;
		selection.clicks = 4;
		D_SELECT((stderr, "selection_extend() selection.clicks = 4"));
		return;
	}
#endif
	if (selection.clicks == 4)
		selection.clicks = 1;
	f_selection_extend_colrow(TermWin.active_page,col, row, !!flag,	/* ? button 3      */
				flag == 1 ? 1 : 0,	/* ? button press  */
				0);	/* no click change */
}

/* ------------------------------------------------------------------------- */
/*
 * Extend the selection to the specified col/row
 */
void f_selection_extend_colrow(int page,int col, int row, int button3, int buttonpress,int clickchange)
{
	int             end_col;
	row_col_t       pos;
	enum
	{
		LEFT, RIGHT
	}
	closeto = RIGHT;

#ifdef MULTICHAR_SET
	int             c, r;
#endif

	DSEL_ALEXIS(2,(stderr,
		  "selection_extend_colrow(c:%d, r:%d, %d, %d) clicks:%d\n", col,
		  row, button3, buttonpress, selection.clicks));
	DSEL_ALEXIS(2,(stderr,
		  "selection_extend_colrow() ENT  b:(r:%d,c:%d) m:(r:%d,c:%d), e:(r:%d,c:%d)\n",
		  selection.beg.row, selection.beg.col, selection.mark.row,
		  selection.mark.col, selection.end.row, selection.end.col));

	switch (selection.op)
	{
	case SELECTION_INIT:
		CLEAR_SELECTION;
		selection.op = SELECTION_BEGIN;
	    /* FALLTHROUGH */
	case SELECTION_BEGIN:
		if (row != selection.mark.row || col != selection.mark.col
		    || (!button3 && buttonpress))
			selection.op = SELECTION_CONT;
		break;
	case SELECTION_DONE:
		selection.op = SELECTION_CONT;
	    /* FALLTHROUGH */
	case SELECTION_CONT:
		break;
	case SELECTION_CLEAR:
		selection_start_colrow(col, row);
	    /* FALLTHROUGH */
	default:
		return;
	}

	pos.col = col;
	pos.row = row;

	pos.row -= TermWin.vts[page].view_start;	/* adjust for scroll */

#ifdef OLD_SELECTION
/*
 * This mimics some of the selection behaviour of version 2.20 and before.
 * There are no ``selection modes'', button3 is always character extension.
 * Note: button3 drag is always available, c.f. v2.20
 * Selection always terminates (left or right as appropriate) at the mark.
 */
	{
		static int      hate_those_clicks = 0;	/* a.k.a. keep mark position */

		if (selection.clicks == 1 || button3)
		{
			if (hate_those_clicks)
			{
				selection.mark.row = selection.beg.row;
				selection.mark.col = selection.beg.col;
				hate_those_clicks = 0;
			}
			if (ROWCOL_IS_BEFORE(pos, selection.beg))
			{
				selection.end.row = selection.beg.row;
				selection.end.col = selection.beg.col + 1;
				selection.beg.row = pos.row;
				selection.beg.col = pos.col;
			} else
			{
				selection.beg.row = selection.mark.row;
				selection.beg.col = selection.mark.col;
				selection.end.row = pos.row;
				selection.end.col = pos.col + 1;
			}
		} else if (selection.clicks == 2)
		{
			f_selection_delimit_word(page,UP, &(selection.mark), &(selection.beg));
			f_selection_delimit_word(page,DN, &(selection.mark), &(selection.end));
			hate_those_clicks = 1;
		} else if (selection.clicks == 3)
		{
			selection.beg.row = selection.end.row = selection.mark.row;
			selection.beg.col = 0;
			selection.end.col = TermWin.vts[page].bcol;
			hate_those_clicks = 1;
		}
	}
#else				/* ! OLD_SELECTION */
/*
 * This is mainly xterm style selection with a couple of differences, mainly
 * in the way button3 drag extension works.
 * We're either doing: button1 drag; button3 press; or button3 drag
 *  a) button1 drag : select around a midpoint/word/line - that point/word/line
 *     is always at the left/right edge of the selection.
 *  b) button3 press: extend/contract character/word/line at whichever edge of
 *     the selection we are closest to.
 *  c) button3 drag : extend/contract character/word/line - we select around
 *     a point/word/line which is either the start or end of the selection
 *     and it was decided by whichever point/word/line was `fixed' at the
 *     time of the most recent button3 press
 */
	if (button3 && buttonpress)
	{			/* button3 press */
	    /*
	     * first determine which edge of the selection we are closest to
	     */
		if (ROWCOL_IS_BEFORE(pos, selection.beg)
		    || (!ROWCOL_IS_AFTER(pos, selection.end)
			&& (((pos.col - selection.beg.col)
			     + ((pos.row - selection.beg.row) * TermWin.vts[page].bcol))
			    < ((selection.end.col - pos.col)
			       +
			       ((selection.end.row -pos.row) * TermWin.vts[page].bcol)))))
			closeto = LEFT;
		if (closeto == LEFT)
		{
			selection.beg.row = pos.row;
			selection.beg.col = pos.col;
			selection.mark.row = selection.end.row;
			selection.mark.col =
			    selection.end.col - (selection.clicks == 2);
		} else
		{
			selection.end.row = pos.row;
			selection.end.col = pos.col;
			selection.mark.row = selection.beg.row;
			selection.mark.col = selection.beg.col;
		}
	} else
	{			/* button1 drag or button3 drag */
		if (ROWCOL_IS_AFTER(selection.mark, pos))
		{
			if ((selection.mark.row == selection.end.row)
			    && (selection.mark.col == selection.end.col)
			    && clickchange && selection.clicks == 2)
				selection.mark.col--;
			selection.beg.row = pos.row;
			selection.beg.col = pos.col;
			selection.end.row = selection.mark.row;
			selection.end.col =
			    selection.mark.col + (selection.clicks == 2);
		} else
		{
			selection.beg.row = selection.mark.row;
			selection.beg.col = selection.mark.col;
			selection.end.row = pos.row;
			selection.end.col = pos.col;
		}
	}

	if (selection.clicks == 1)
	{
		end_col = screen[page].tlen[selection.beg.row + TermWin.vts[page].saveLines];
		if (end_col != -1 && selection.beg.col > end_col)
		{
#if 1
			selection.beg.col = TermWin.vts[page].bcol;
#else
			if (selection.beg.row != selection.end.row)
				selection.beg.col = TermWin.vts[page].bcol;
			else
				selection.beg.col = selection.mark.col;
#endif
		}
		end_col = screen[page].tlen[selection.end.row + TermWin.vts[page].saveLines];
		if (end_col != -1 && selection.end.col > end_col)
			selection.end.col = TermWin.vts[page].bcol;

# ifdef MULTICHAR_SET
		if (selection.beg.col > 0)
		{
			r = selection.beg.row + TermWin.vts[page].saveLines;
			c = selection.beg.col;
			if (((screen[page].rend[r][c] & RS_multiMask) == RS_multi2)
			    && ((screen[page].rend[r][c - 1] & RS_multiMask) ==
				RS_multi1))
				selection.beg.col--;
		}
		if (selection.end.col < TermWin.vts[page].bcol)
		{
			r = selection.end.row + TermWin.vts[page].saveLines;
			c = selection.end.col;
			if (((screen[page].rend[r][c - 1] & RS_multiMask) == RS_multi1)
			    && ((screen[page].rend[r][c] & RS_multiMask) == RS_multi2))
				selection.end.col++;
		}
# endif				/* MULTICHAR_SET */
	} else if (selection.clicks == 2)
	{
		if (ROWCOL_IS_AFTER(selection.end, selection.beg))
			selection.end.col--;
		f_selection_delimit_word(page,UP, &(selection.beg), &(selection.beg));
		f_selection_delimit_word(page,DN, &(selection.end), &(selection.end));
	} else if (selection.clicks == 3)
	{
		if (ROWCOL_IS_AFTER(selection.mark, selection.beg))
			selection.mark.col++;
		selection.beg.col = 0;
		selection.end.col = TermWin.vts[page].bcol;
	}
	if (button3 && buttonpress)
	{			/* mark may need to be changed */
		if (closeto == LEFT)
		{
			selection.mark.row = selection.end.row;
			selection.mark.col = selection.end.col - (selection.clicks == 2);
		} else
		{
			selection.mark.row = selection.beg.row;
			selection.mark.col = selection.beg.col;
		}
	}
#endif				/* ! OLD_SELECTION */
	DSEL_ALEXIS(2,(stderr,
		  "selection_extend_colrow() EXIT b:(r:%d,c:%d) m:(r:%d,c:%d), e:(r:%d,c:%d)\n",
		  selection.beg.row, selection.beg.col, selection.mark.row,
		  selection.mark.col, selection.end.row, selection.end.col));
}

/* ------------------------------------------------------------------------- */
/*
 * Double click on button 3 when already selected
 * EXT: button 3 double click
 */
/* PROTO */
void
selection_rotate(int x, int y)
{
  DSEL_ALEXIS(2,(stderr, "selection_rotate(%d, %d)\n",x,y));
	selection.clicks = selection.clicks % 3 + 1;
#ifdef THAI
	f_selection_extend_colrow(TermWin.active_page,ThaiPixel2Col(x, y), Pixel2Row(y), 1, 0, 1);
#else
	f_selection_extend_colrow(TermWin.active_page,Pixel2Col(x), Pixel2Row(y), 1, 0, 1);
#endif
}

/* ------------------------------------------------------------------------- */
/*
 * On some systems, the Atom typedef is 64 bits wide.  We need to have a type
 * that is exactly 32 bits wide, because a format of 64 is not allowed by
 * the X11 protocol.
 */
typedef CARD32  Atom32;

/* ------------------------------------------------------------------------- */
/*
 * Respond to a request for our current selection
 * EXT: SelectionRequest
 */

void selection_send(XSelectionRequestEvent * rq)
{
	XEvent          ev;
	Atom32          target_list[2];
	static Atom     xa_targets = None;

	if (xa_targets == None)
		xa_targets = XInternAtom(Xdisplay, "TARGETS", False);

	ev.xselection.type = SelectionNotify;
	ev.xselection.property = None;
	ev.xselection.display = rq->display;
	ev.xselection.requestor = rq->requestor;
	ev.xselection.selection = rq->selection;
	ev.xselection.target = rq->target;
	ev.xselection.time = rq->time;

	if (rq->target == xa_targets)
	{
		target_list[0] = (Atom32) xa_targets;
		target_list[1] = (Atom32) XA_STRING;
		XChangeProperty(Xdisplay, rq->requestor, rq->property,
				rq->target, (8 * sizeof(target_list[0])),
				PropModeReplace, (unsigned char *)target_list,
				(sizeof(target_list) /
				 sizeof(target_list[0])));
		ev.xselection.property = rq->property;
	} else if (rq->target == XA_STRING)
	{
		XChangeProperty(Xdisplay, rq->requestor, rq->property,
				rq->target, 8, PropModeReplace, selection.text,
				selection.len);
		ev.xselection.property = rq->property;
	}
	XSendEvent(Xdisplay, rq->requestor, False, 0, &ev);
}

/* ------------------------------------------------------------------------- *
 *                              MOUSE ROUTINES                               *
 * ------------------------------------------------------------------------- */

/*
 * return col/row values corresponding to x/y pixel values
 */
/* PROTO */
void
pixel_position(int *x, int *y)
{
#ifdef THAI
	*x = ThaiPixel2Col(*x, *y);
#else
	*x = Pixel2Col(*x);
#endif
    /* MAX_IT(*x, 0); MIN_IT(*x, TermWin.bcol - 1); */
	*y = Pixel2Row(*y);
    /* MAX_IT(*y, 0); MIN_IT(*y, TermWin.nrow - 1); */
}

/* ------------------------------------------------------------------------- */
/* ARGSUSED */
/* PROTO */
void
mouse_tracking(int report, int x, int y, int firstrow, int lastrow)
{
/* TODO */
}

/* ------------------------------------------------------------------------- *
 *                              DEBUG ROUTINES                               *
 * ------------------------------------------------------------------------- */
/* ARGSUSED */
/* PROTO */
void
debug_PasteIt(unsigned char *data, int nitems)
{
/* TODO */
}

/* ------------------------------------------------------------------------- */
/* PROTO */
void
debug_colors(void)
{
	int             color;
	char           *name[] = { "fg", "bg",
		"black", "red", "green", "yellow", "blue", "magenta", "cyan",
		    "white"
	};

	fprintf(stderr, "Color ( ");
	if (rstyle & RS_RVid)
		fprintf(stderr, "rvid ");
	if (rstyle & RS_Bold)
		fprintf(stderr, "bold ");
	if (rstyle & RS_Blink)
		fprintf(stderr, "blink ");
	if (rstyle & RS_Uline)
		fprintf(stderr, "uline ");
	fprintf(stderr, "): ");

	color = GET_FGCOLOR(rstyle);
#ifndef NO_BRIGHTCOLOR
	if (color >= minBrightCOLOR && color <= maxBrightCOLOR)
	{
		color -= (minBrightCOLOR - minCOLOR);
		fprintf(stderr, "bright ");
	}
#endif
	fprintf(stderr, "%s on ", name[color]);

	color = GET_BGCOLOR(rstyle);
#ifndef NO_BRIGHTCOLOR
	if (color >= minBrightCOLOR && color <= maxBrightCOLOR)
	{
		color -= (minBrightCOLOR - minCOLOR);
		fprintf(stderr, "bright ");
	}
#endif
	fprintf(stderr, "%s\n", name[color]);
}
