/* Copyright (C) 2005 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "MWLayerTree.h"
#include "myg_gtkutils.h"

MWLayerTree::MWLayerTree(GtkTreeView *tree)
  : Gtk::TreeView(tree)
{
  Gtk::TreeViewColumn *column= new Gtk::TreeView::Column("");
  column->pack_start(_columns.icon, false);
  column->pack_start(_columns.text);
  append_column(*Gtk::manage(column));
  std::vector<Gtk::CellRenderer*> crend= column->get_cell_renderers();
  column->add_attribute(crend[0]->property_cell_background(),
                        _columns.color);
  column->add_attribute(crend[0]->property_cell_background_set(),
                        _columns.is_layer);
  
  column->add_attribute(crend[1]->property_cell_background(),
                        _columns.color);
  column->add_attribute(crend[1]->property_cell_background_set(),
                        _columns.is_layer);
  
  _layer_icon= PIXCACHE->load("db.workbench.Layer.16x16.png");
  _routine_icon= PIXCACHE->load("db.workbench.RoutinesElement.16x16.png");
  _table_icon= PIXCACHE->load("db.workbench.TableElement.16x16.png");
  _view_icon= PIXCACHE->load("db.workbench.ViewElement.16x16.png");
  
  _store= create_model();
  _root_layer= 0;
}



void MWLayerTree::set_grt(MGRT *grt)
{
  _grt= grt;
}

bool MWLayerTree::compare_object(const Gtk::TreeIter &iter, Gtk::TreeIter *found,
                                MYX_GRT_VALUE *obj)
{
  Gtk::TreeRow row= *iter;
  MYX_GRT_VALUE *value(row[_columns.grt]);

  if (value == obj)
  {
    *found= iter;
    return true;
  }
  return false;
}


MYX_GRT_VALUE *MWLayerTree::find_new_parent_layer(MYX_GRT_VALUE *layer, MYX_GRT_VALUE *root)
{
  unsigned int i, c;
  MYX_GRT_VALUE *subLayers;
  
  if (!root)
    root= _root_layer;
    
  subLayers= myx_grt_dict_item_get_value(root, "subLayers");
  if (!subLayers)
    return NULL;

  c= myx_grt_bridge_list_item_count(subLayers);
  for (i= 0; i < c; i++)
  {
    MYX_GRT_VALUE *subLayer= myx_grt_bridge_list_item_get(subLayers, i, 0);
    
    subLayer= myx_grt_reference_cache_lookup(_grt->grt(), 
                                             myx_grt_value_as_string(subLayer));
    if (subLayer == layer)
      return root;

    subLayer= find_new_parent_layer(layer, subLayer);
    if (subLayer)
      return subLayer;
  }

  return NULL;
}


Gtk::TreeIter MWLayerTree::get_parent_layer(MYX_GRT_VALUE *layer)
{
  if (_value_rows.find(layer) != _value_rows.end())
  {
    Gtk::TreeRowReference ref= _value_rows[layer];
    if (ref.is_valid())
    {
      Gtk::TreeRow row= *_store->get_iter(ref.get_path());
      
      return row.parent();
    }
  }
  return Gtk::TreeIter();
}


void MWLayerTree::update_element(Gtk::TreeIter &iter, MYX_GRT_VALUE *elem)
{
  Gtk::TreeRow row= *iter;
  Glib::ustring text;
  MYX_GRT_VALUE *object, *value= 0;
  
  if (strcmp(myx_grt_dict_struct_get_name(elem), "db.workbench.TableElement")==0)
  {
    object= myx_grt_bridge_dict_item_get_value(elem, "table", 0);
    if (object)
      row[_columns.object]= value= myx_grt_reference_cache_lookup(_grt->grt(), myx_grt_value_as_string(object));
    row[_columns.icon]= _table_icon;
  }
  else if (strcmp(myx_grt_dict_struct_get_name(elem), "db.workbench.ViewElement")==0)
  {
    object= myx_grt_bridge_dict_item_get_value(elem, "view", 0);
    if (object)
      row[_columns.object]= value= myx_grt_reference_cache_lookup(_grt->grt(), myx_grt_value_as_string(object));
    row[_columns.icon]= _view_icon;
  }
  else if (strcmp(myx_grt_dict_struct_get_name(elem), "db.workbench.RoutinesElement")==0)
  {
    object= myx_grt_bridge_dict_item_get_value(elem, "routineGroup", 0);
    if (object)
      row[_columns.object]= value= myx_grt_reference_cache_lookup(_grt->grt(), myx_grt_value_as_string(object));
    row[_columns.icon]= _routine_icon;
  }

  row[_columns.text]= myx_grt_bridge_value_as_string(myx_grt_bridge_dict_item_get_value(value?:elem, "name", 0));
  row[_columns.grt]= elem;
  row[_columns.is_layer]= false;
}


void MWLayerTree::change(MYX_GRT_VALUE *root_layer, Glib::RefPtr<Gtk::TreeStore> model)
{
  _root_layer= root_layer;
  if (!model)
    _store= create_model();
  else
    _store= model;
  set_model(_store);
}


Glib::RefPtr<Gtk::TreeStore> MWLayerTree::create_model()
{
  return Gtk::TreeStore::create(_columns);
}


void MWLayerTree::handle_element_add(MYX_GRT_VALUE *elem)
{
  MYX_GRT_VALUE *layer= myx_grt_reference_cache_lookup(_grt->grt(), myx_grt_dict_item_get_as_string(elem, "layer"));
  Gtk::TreeRowReference ref= _value_rows[layer];
  if (ref)
  {
    Gtk::TreeIter layer_iter= _store->get_iter(ref.get_path());
    Gtk::TreeRow layer_row= *layer_iter;
    Gtk::TreeIter iter;

    iter= _store->append(layer_row.children());
    
    update_element(iter, elem);
    
    expand_row(Gtk::TreePath(layer_iter), true);
  }
  else
  {
    Gtk::TreeIter iter;
    
    iter= _store->append();
    
    update_element(iter, elem);
  }
}


void MWLayerTree::handle_element_change(MYX_GRT_VALUE *elem)
{
  Gtk::TreeIter iter;

  _store->foreach_iter(sigc::bind<Gtk::TreeIter*,MYX_GRT_VALUE*>(sigc::mem_fun(*this, &MWLayerTree::compare_object),&iter,elem));

  if (iter)
    update_element(iter, elem);
}


void MWLayerTree::handle_element_delete(MYX_GRT_VALUE *elem)
{
  Gtk::TreeIter iter;

  _store->foreach_iter(sigc::bind<Gtk::TreeIter*,MYX_GRT_VALUE*>(sigc::mem_fun(*this, &MWLayerTree::compare_object),&iter,elem));
  if (iter)
    _store->erase(iter);
}


Gtk::TreeIter MWLayerTree::handle_layer_add(MYX_GRT_VALUE *layer)
{
  Gtk::TreeIter parent;
  Gtk::TreeIter iter;

  if (strcmp(myx_grt_dict_item_get_as_string(layer, "name"), "root")==0)
  {
    _root_layer= layer;
    return Gtk::TreeIter();
  }

  if (_root_layer == layer)
    return Gtk::TreeIter();

  // check whether it's already in the tree
  if (_value_rows.find(layer) != _value_rows.end())
    return Gtk::TreeIter();

  if (_root_layer)
    parent= get_parent_layer(layer);
  
  Gtk::TreeRow row;
  if (!parent)
  {
    iter= _store->append();
    row= *iter;
  }
  else
  {
    Gtk::TreeRow prow= *parent;
    iter= _store->append(prow.children());
    row= *iter;
    expand_row(Gtk::TreePath(prow), true);
  }
  row[_columns.icon]= _layer_icon;
  row[_columns.text]= myx_grt_dict_item_get_as_string(layer,"name");
  row[_columns.grt]= layer;
  row[_columns.is_layer]= true;
  Glib::ustring color= myx_grt_dict_item_get_as_string(layer, "color");
  if (_color_value.find(color) != _color_value.end())
    row[_columns.color]= _color_value[color];
  else
    row[_columns.color]= "#ffffff";
    
  _value_rows[layer]= Gtk::TreeRowReference(_store, Gtk::TreePath(iter));
  
  return iter;
}


void MWLayerTree::handle_layer_change(MYX_GRT_VALUE *layer)
{
  Gtk::TreeIter iter;
  Gtk::TreeRow row;
  MYX_GRT_VALUE *parent;

  if (_value_rows.find(layer) == _value_rows.end())
    return;
  Gtk::TreeRowReference ref= _value_rows[layer];

  if (!ref || !(iter= _store->get_iter(ref.get_path())))
    return;
  row= *iter;
  parent= find_new_parent_layer(layer);
  if (parent)
  {
    MYX_GRT_VALUE *old_parent;
    Gtk::TreeIter piter;
    Gtk::TreeRow prow;

    // check if parent changed
    piter= get_parent_layer(layer);
    if (piter)
    {
      prow= *piter;
      old_parent= prow[_columns.grt];
    }
    else
      old_parent= 0;
    if (old_parent != parent)
    {
      Gtk::TreeIter new_iter;
      Gtk::TreeRow new_row;
      Gtk::TreeIter new_piter;
      Gtk::TreeRow new_prow;
      Gtk::TreePath parent_path;
      if (parent && parent != _root_layer)
      {
        new_piter= _store->get_iter(_value_rows[parent].get_path());
        new_prow= *new_piter;
        new_iter= _store->append(new_prow.children());
        new_row= *new_iter;
        parent_path= Gtk::TreePath(new_piter);
      }
      else
      {
        new_iter= _store->append();
        new_row= *new_iter;
      }

      iter= _store->get_iter(ref.get_path());
      row= *iter;
      new_row[_columns.icon]= (Glib::RefPtr<Gdk::Pixbuf>)row[_columns.icon];
      new_row[_columns.text]= (Glib::ustring)row[_columns.text];
      new_row[_columns.grt]= (MYX_GRT_VALUE*)row[_columns.grt];
      new_row[_columns.object]= (MYX_GRT_VALUE*)row[_columns.object];
      new_row[_columns.is_layer]= (bool)row[_columns.is_layer];
      new_row[_columns.color]= (Glib::ustring)row[_columns.color];

      _store->erase(iter);
      iter= new_iter;
      row= *iter;

      if (!parent_path.empty())
        expand_row(parent_path, false);

      _value_rows[layer]= Gtk::TreeRowReference(_store, Gtk::TreePath(new_iter));

      rescan_children(layer, iter);

      expand_row(Gtk::TreePath(new_iter), false);
    }
  }

  // update name
  row[_columns.text]= myx_grt_bridge_value_as_string(myx_grt_bridge_dict_item_get_value(layer,"name",0));
}


void MWLayerTree::handle_layer_delete(MYX_GRT_VALUE *layer)
{
  Gtk::TreeIter iter;

  _store->foreach_iter(sigc::bind<Gtk::TreeIter*,MYX_GRT_VALUE*>(sigc::mem_fun(*this, &MWLayerTree::compare_object),&iter,layer));
  if (iter)
    _store->erase(iter);
}


void MWLayerTree::handle_layer_move(MYX_GRT_VALUE *layer)
{
}



void MWLayerTree::rescan_children(const MGRTValue &parent, Gtk::TreeIter &node)
{
  Gtk::TreeRow row= *node;

  for (unsigned int i= 0; i < parent["subLayers"].count(); i++)
  {
    MGRTValue child;

    child= parent["subLayers"][i];
    if (child.type() == MYX_STRING_VALUE)
      child= MGRTValue::refObject(_grt->grt(), child.asString());

    if (child.isValid())
    {
      Gtk::TreeIter ni= _store->append(row.children());
      Gtk::TreeRow nr= *ni;

      nr[_columns.icon]= _layer_icon;
      nr[_columns.text]= child["name"].asString();
      nr[_columns.grt]= child;
      nr[_columns.is_layer]= true;

      _value_rows[child.grtValue()]= Gtk::TreeRowReference(_store, Gtk::TreePath(ni));

      rescan_children(child, ni);
      
      expand_row(Gtk::TreePath(ni), true);
    }
  }

  for (unsigned int i= 0; i < parent["elements"].count(); i++)
  {
    Gtk::TreeIter ni= _store->append(row.children());

    update_element(ni, MGRTValue::refObject(_grt->grt(), parent["elements"][i]));
  }
}



void MWLayerTree::rescan(MGRTValue new_root)
{
  MGRTValue node;
  Gtk::TreeIter iter;
  Gtk::TreeRow row;

  if (!new_root.isValid())
    return;
  
  _store->clear();
  _value_rows.clear();
  
  node= new_root;
  _root_layer= new_root;

  rescan_children(node, iter);
}



std::list<MYX_GRT_VALUE*> MWLayerTree::get_selected_objects()
{
  std::list<MYX_GRT_VALUE*> objects;
  std::list<Gtk::TreePath> selected= get_selection()->get_selected_rows();
  
  for (std::list<Gtk::TreePath>::iterator iter= selected.begin();
       iter != selected.end(); ++iter)
  {
    Gtk::TreeIter rowIter= get_model()->get_iter(*iter);
    Gtk::TreeRow row= *rowIter;

    objects.push_back(row[_columns.grt]);
  }
  return objects;
}


void MWLayerTree::select_items_in_list(Gtk::TreeIter iter, MYX_GRT_VALUE **value, int count)
{
  do
  {
    Gtk::TreeRow row= *iter;
    MYX_GRT_VALUE *obj= row[_columns.grt];
    
    for (int i= 0; i < count; i++)
    {
      if (value[i] == obj)
      {
        get_selection()->select(iter);
        break;
      }
    }

    if (row.children().begin() != row.children().end())
      select_items_in_list(*row.children().begin(), value, count);
  } while (++iter);
}


void MWLayerTree::select_objects(MYX_GRT_VALUE **value, int count)
{
  Gtk::TreeIter iter;
  Gtk::TreeRow row;

  get_selection()->unselect_all();

  if (_store->children().begin() != _store->children().end())
    select_items_in_list(*_store->children().begin(), value, count);
}


void MWLayerTree::set_color_values(const std::map<Glib::ustring,Glib::ustring> &color_values)
{
  _color_value= color_values;
}
