# $Id: ScannedHost.pm,v 1.22 2001/11/24 22:28:15 levine Exp $
#
# A record holding port scanning info on a per-host basis, and
# useful accessor and manipulation methods.
#
#
# Copyright (C) 2000  James D. Levine (jdl@vinecorp.com)
#
#
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your option) any later version.
# 
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
# 
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
#   02111-1307, USA.
#
####################################################################


#
# Encapsulates the information about a scanned host.
#


package PortScan::ScannedHost;
use strict;

use vars qw(@ISA);

@ISA = qw( Exporter );

use PortScan::SetOps;
use PortScan::PortSpec;


sub new
{
    my $type = shift;
    my $self = 
    {
	'scan_set' => undef,	# the ScanSet this belongs to, if any
	'port_specs' => {},	# all explicitly noted ports
	'addr' => '',		# IP addr in dotted-quad text
	'default_state' => 'closed', # perhaps 'unkown' is correct
    };

    bless $self, $type;

}


# 
# outputs the host in human-readable text
#
sub to_text
  {
    my $self = shift;
    return
      "addr: " . $self->{addr} . "\n"
      . "default_state: " . $self->{default_state} . "\n"
      . "ports: " . hash_to_text($self->{port_specs}) . "\n"
	;
  }

#
# outputs a hash as key -> value, one per line
#
sub hash_to_text
  {
    my $h = shift;
    my $t;
    while (my ($k, $v) = each %$h)
      {
	$t .= "`${k}' -> `${v}'\n";
      }
    $t;
  }


# generic accessor method
sub set_or_get
{
    my $field = shift;
    my $self = shift;
    my $val = shift;

    $self->{$field} = $val if defined $val;

    return $self->{$field};
}


#
# returns a hashref of all scanned ports from the ScanSet to which the
# hosts belongs.  (TODO: document how the hash is keyed.)
#

sub all_scanned_ports
{
    my $self = shift;

    return (defined $self->scan_set()) ? $self->scan_set()->all_scanned_ports() :{};
}

#
# returns a hashref of all scanned ports.  (TODO:  confusing.  This
# method lists all ports inserted into this host; the above method
# returns to the master set owned by the ScanSet. )

sub all_ports
{
    my ($self) = @_;

    my $ports = $self->all_scanned_ports();
    my $result_hash = {};

    @$result_hash{keys %$ports} = (values %$ports);

    foreach my $p (values %$result_hash)
    {
	$p->state($self->default_state());
    }
    # now overlay with noteworthy ports

    $ports = $self->port_specs();
    foreach my $p (values %$ports )
    {
	$result_hash->{$p->key_for()} = $p->clone();
#print "cloned $p with state " , $p->state(), "\n";
#print "overlaying ", $p->key_for(), " state ", $p->state(), "\n";
    }

    $result_hash;
}


# returns/sets the default state of this host (for all unspecified ports)
sub default_state() {set_or_get('default_state', @_);} 

# address in dotted-quad text
sub addr {set_or_get('addr', @_);}

# returns the ScanSet to which this host belongs
sub scan_set{set_or_get('scan_set', @_);}

#
# (TODO: is this redundant?)
#

sub scanned_ports_hash
{
    my $self = shift;
    return (defined $self->scan_set()) ? $self->scan_set()->all_scanned_ports()
	: {};
}

#
# Inserts a scanned port (PortSpec).
#

sub add_port
{				# clobbers an existing port of same number
    my( $self, $pspec ) = @_;

    $self->port_specs()->{$pspec->key_for()} = $pspec;
    if ( $self->scan_set() )
    {
	$self->scan_set()->add_scanned_port( $pspec );

    }
}


#
# Inserts a new scanned port, first constructing the PortSpec from
# the supplied properties.
#

sub set_port
{
    my( $self, $number, $state, $proto, $u1, $service, $u2, $u3 ) = @_;
#    print "set_port: $self, $number, $state, $proto, $u1, $service, $u2, $u3 \n";

    $self->add_port( new PortScan::PortSpec( $number, $state, $proto, $u1, $service, $u2, $u3 ) );
}

#
# Deletes the key specified by pkey (generated by PortSpec::key_for() )
#

sub remove_port
{
    my( $self, $pkey ) = @_;

    delete $self->port_specs()->{$pkey};
}

#
# Returns the hashref of all specified ports for this host.
#

sub port_specs
{
    set_or_get( 'port_specs', @_ );
}

#
# Returns the ports as a sorted list of PortSpecs.
#

sub port_specs_sorted_list
{
    my $self= shift;
    return PortScan::PortSpec::sorted_list values %{$self->port_specs()};
}


# I wonder if this is used anywhere
#  sub port_spec
#  {
#      my( $self, $number ) = @_;
#
#      return $self->port_specs()->{$number}; # port_spec or undef
#  }


#
# Retrieves the port based on the supplied ( proto, port )
# proto= tcp | udp, port is an integer
#

sub get_port
{
    my( $self, $proto, $port ) = @_;

#    print "get_port:$self - $proto - $port\n";

    my $k = PortScan::PortSpec::make_key( $port, $proto );

#    print "get_port: $k \n";

    my $r =  $self->port_specs()->{ $k }; # port_spec or undef

#    print "get_port: $r " . $r->key_for() . "\n";

    return $r;
}


#
# Returns the state of the supplied (proto, port)
# proto= tcp | udp, port is an integer
# states are: open | closed | filtered | unfiltered | unknown

sub get_state
{
    my( $self, $proto, $port ) = @_;

#    print "get_state:entry\n";
#    print "get_state:$self - $proto - $port\n";

    my $port = $self->get_port( $proto, $port );

#    print "get_state: about to exit\n";

    return "unknown" if ! defined $port;

    return $port->state;
}

#
# Outputs self in human readable.
#

sub out_pretty
{
    my $self = shift;

    printf "%s:\n", $self->addr();

    my $ports = $self->port_specs();

    foreach my $port (values %$ports)
    {
	printf $port->nmap_default_human_format(). "\n";
    }
}


#
# private. Sorts a list of ip addresses.
#
sub sorted_list
{
    return sort compare @_;
}

#
# private.  This appears to be redundant. 
#
sub sorted_ip_scalar_list
{
    return sort compare_ip_scalar @_; ;
}

# private comparator.
sub compare
{
    return compare_ip_scalar( $a->addr(), $b->addr() )
}

#
# Private sort comparator.
#

sub compare_ip_scalar
{
    my( $a1, $a2, $a3, $a4 ) = split (/\./, $a);
    my( $b1, $b2, $b3, $b4 ) = split (/\./, $b);

#    print "a $a1 $a2 $a3 $a4 \n";
#    print "b $b1 $b2 $b3 $b4 \n";
    return -1 if ($a1 < $b1);
    return 1  if ($a1 > $b1);

    return -1 if ($a2 < $b2);
    return 1  if ($a2 > $b2);

    return -1 if ($a3 < $b3);
    return 1  if ($a3 > $b3);

    return -1 if ($a4 < $b4);
    return 1  if ($a4 > $b4);

    0;
}


#
# Private (and redundant? )
#

sub addrs_sorted_list
{
    return sort compare_addrs @_;
}

#
# Redundant?
#
sub compare_addrs
{
    my ($a1, $a2, $a3, $a4) = split (/\./, $a);
    my ($b1, $b2, $b3, $b4) = split (/\./, $b);

    return -1 if ($a1 < $b1);
    return 1  if ($a1 > $b1);

    return -1 if ($a2 < $b2);
    return 1  if ($a2 > $b2);

    return -1 if ($a3 < $b3);
    return 1  if ($a3 > $b3);

    return -1 if ($a4 < $b4);
    return 1  if ($a4 > $b4);

    0;
}


# generic accessor;
sub set_or_get
{
    my $field = shift;
    my $self = shift;
    my $val = shift;

    $self->{$field} = $val if defined $val;

    return $self->{$field};
}





1;






















