
// Bridge pattern -- Real World example
namespace RealWorldExample {
  using System;
  using Nemerle.Collections;

  // "Abstraction"

  class BusinessObject
  {
    // Fields
    [DesignPatterns.Bridge ((Next, NextRecord),
                            (Prior, PriorRecord),
                            (New (x : string), NewRecord),
                            (Delete (x : string), DeleteRecord),
                            (Show, ShowRecord))]
    private mutable dataObject : DataObject;

    protected group : string;

    // Constructors
    public this( group : string )
    {
      this.group = group;
    }

    // Properties
    public DataObject : DataObject
    {
      set{ dataObject = value; }
      get{ dataObject; }
    }

    // Methods
    virtual public ShowAll() : void
    {
      Console.WriteLine( "Customer Group: {0}", group );
      dataObject.ShowAllRecords();
    }
  }

  // "RefinedAbstraction"

  class CustomersBusinessObject : BusinessObject
  {
    // Constructors
    public this ( group :  string) {
        base( group )
    }

    // Methods
    override public ShowAll() : void
    {
      // Add separator lines
      Console.WriteLine();
      Console.WriteLine( "------------------------" );
      base.ShowAll();
      Console.WriteLine( "------------------------" );
    }
  }

  // "Implementor"

  abstract class DataObject
  {
    // Methods
    abstract public NextRecord() : void;
    abstract public PriorRecord() : void;
    abstract public NewRecord( name : string ) : void;
    abstract public DeleteRecord( name : string ) : void;    
    abstract public ShowRecord() : void;
    abstract public ShowAllRecords() : void;
  }

  // "ConcreteImplementor"

  class CustomersDataObject : DataObject
  {
    // Fields
    private customers : Vector [string] = Vector (10);
    private mutable current : int = 0;

    // Constructors
    public this()
    {
      // Loaded from a database
      NewRecord ( "Jim Jones" );
      NewRecord ( "Samual Jackson" );
      NewRecord ( "Allen Good" );
      NewRecord ( "Ann Stills" );
      NewRecord ( "Lisa Giolani" );
    }

    // Methods
    public override NextRecord() : void
    {
      when ( current <= customers.Count - 1 )
        current++;
    }

    public override PriorRecord() : void
    {
      when ( current > 0 )
        current--;
    }

    public override NewRecord( name : string ) : void
    {
      customers.Add( name );
    }

    public override DeleteRecord( _name : string ) : void
    {
      // customers.Remove( name );
    }
    
    public override ShowRecord() : void
    {
      Console.WriteLine( customers[ current ] );
    }

    public override ShowAllRecords() : void
    {
      foreach( name in customers )
        Console.WriteLine( " " + name );
    }
  }

  /// <summary>
  /// Client test
  /// </summary>
  public class BusinessApp
  {
    public static Main( ) : void
    {
      // Create RefinedAbstraction
      def customers = CustomersBusinessObject(" Chicago ");

      // Set ConcreteImplementor
      customers.DataObject = CustomersDataObject();

      // Exercise the bridge
      customers.Show();
      customers.Next();
      customers.Show();
      customers.Next();
      customers.Show();
      customers.New( "Henry Velasquez" );

      customers.ShowAll();
    }
  }
}


// REFERENCE: bridge-m.dll

/*
BEGIN-OUTPUT
Jim Jones
Samual Jackson
Allen Good

------------------------
Customer Group:  Chicago
 Jim Jones
 Samual Jackson
 Allen Good
 Ann Stills
 Lisa Giolani
 Henry Velasquez
------------------------
END-OUTPUT
*/
