/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2007 Sun Microsystems, Inc.
 */
package org.netbeans.modules.editor.settings.storage.codetemplates;

import java.awt.event.KeyEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.lang.ref.WeakReference;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import javax.swing.KeyStroke;
import org.netbeans.api.editor.mimelookup.MimePath;
import org.netbeans.api.editor.settings.CodeTemplateDescription;
import org.netbeans.api.editor.settings.CodeTemplateSettings;
import org.netbeans.modules.editor.settings.storage.EditorSettingsImpl;
import org.netbeans.modules.editor.settings.storage.api.CodeTemplateSettingsFactory;
import org.openide.util.Lookup;
import org.openide.util.Utilities;

/**
 *
 * @author vita
 */
public final class CodeTemplateSettingsImpl extends CodeTemplateSettingsFactory {

    public static final String PROP_CODE_TEMPLATES = "CodeTemplateSettingsImpl.PROP_CODE_TEMPLATES"; //NOI18N
    public static final String PROP_EXPANSION_KEY = "CodeTemplateSettingsImpl.PROP_EXPANSION_KEY"; //NOI18N
    
    public static synchronized CodeTemplateSettingsImpl get(MimePath mimePath) {
        WeakReference<CodeTemplateSettingsImpl> reference = INSTANCES.get(mimePath);
        CodeTemplateSettingsImpl result = reference == null ? null : reference.get();
        
        if (result == null) {
            result = new CodeTemplateSettingsImpl(mimePath);
            INSTANCES.put(mimePath, new WeakReference<CodeTemplateSettingsImpl>(result));
        }
        
        return result;
    }
    
    public Map<String, CodeTemplateDescription> getCodeTemplates() {
        synchronized(this) {
            if (codeTamplatesMap == null) {
                codeTamplatesMap = CodeTemplatesStorage.load(mimePath, false);
            }
            return codeTamplatesMap;
        }
    }

    public void setCodeTemplates(Map<String, CodeTemplateDescription> map) {
        synchronized(this) {
            if (map == null) {
                CodeTemplatesStorage.delete(mimePath, false);
                map = null;
            } else {
                Map<String, CodeTemplateDescription> defaultCTMap = CodeTemplatesStorage.load(mimePath, true);
                List<String> removed = new ArrayList<String>();

                map = Collections.unmodifiableMap(new HashMap<String, CodeTemplateDescription>(map));

                // Compute removed items
                removed.addAll(defaultCTMap.keySet());
                removed.removeAll(map.keySet());

                CodeTemplatesStorage.save(mimePath, false, diff(defaultCTMap, map, CTC).values(), removed);

                this.codeTamplatesMap = map;
            }
        }
        
        pcs.firePropertyChange(PROP_CODE_TEMPLATES, null, null);
    }

    public KeyStroke getExpandKey() {
        // XXX: use SimpleValueSettings or whatever other appropriate way
        return BaseOptions_getCodeTemplateExpandKey();
    }

    public void setExpandKey(KeyStroke expansionKey) {
        // XXX: use SimpleValueSettings or whatever other appropriate way
        BaseOptions_setCodeTemplateExpandKey(expansionKey);
        try {
            Thread.sleep(100);
        } catch (InterruptedException ex) {
            // ignore
        }
        EditorSettingsImpl.getInstance().notifyExpansionKeyChange();
    }

    public Object createInstanceForLookup() {
        Map<String, CodeTemplateDescription> map = getCodeTemplates();
        return new Immutable(
            Collections.unmodifiableList(new ArrayList<CodeTemplateDescription>(map.values())), 
            getExpandKey()
        );
    }
    
    public void addPropertyChangeListener(PropertyChangeListener l) {
        pcs.addPropertyChangeListener(l);
    }
    
    public void removePropertyChangeListener(PropertyChangeListener l) {
        pcs.removePropertyChangeListener(l);
    }
    
    // ---------------------------------------------
    // Private implementation
    // ---------------------------------------------

    private static final KeyStroke DEFAULT_EXPANSION_KEY = KeyStroke.getKeyStroke(KeyEvent.VK_TAB, 0);

    private static final Map<MimePath, WeakReference<CodeTemplateSettingsImpl>> INSTANCES =
        new WeakHashMap<MimePath, WeakReference<CodeTemplateSettingsImpl>>();
    
    private static final CodeTemplateDescriptionComparator CTC = new CodeTemplateDescriptionComparator();
    
    private final MimePath mimePath;
    private final PropertyChangeSupport pcs = new PropertyChangeSupport(this);
    
    private Map<String, CodeTemplateDescription> codeTamplatesMap = null;
    
    private CodeTemplateSettingsImpl(MimePath mimePath) {
        this.mimePath = mimePath;
    }

    private static KeyStroke BaseOptions_getCodeTemplateExpandKey() {
        try {
            ClassLoader cl = Lookup.getDefault().lookup(ClassLoader.class);
            Class clazz = cl.loadClass("org.netbeans.modules.editor.options.BaseOptions"); //NOI18N
            Method m = clazz.getDeclaredMethod("getCodeTemplateExpandKey"); //NOI18N
            return (KeyStroke) m.invoke(null);
        } catch (Exception e) {
            return DEFAULT_EXPANSION_KEY;
        }
    }

    private static void BaseOptions_setCodeTemplateExpandKey(KeyStroke keyStroke) {
        try {
            ClassLoader cl = Lookup.getDefault().lookup(ClassLoader.class);
            Class clazz = cl.loadClass("org.netbeans.modules.editor.options.BaseOptions"); //NOI18N
            Method m = clazz.getDeclaredMethod("setCodeTemplateExpandKey", KeyStroke.class); //NOI18N
            m.invoke(null, keyStroke);
        } catch (Exception e) {
            // ignore
        }
    }
    
    private static final class Immutable extends CodeTemplateSettings {
        
        private final List<CodeTemplateDescription> codeTemplates;
        private final KeyStroke expansionKey;
        
        public Immutable(List<CodeTemplateDescription> codeTemplates, KeyStroke expansionKey) {
            this.codeTemplates = codeTemplates;
            this.expansionKey = expansionKey;
        }
        
        public List<CodeTemplateDescription> getCodeTemplateDescriptions() {
            return codeTemplates;
        }

        public KeyStroke getExpandKey() {
            return expansionKey;
        }
    } // End of Immutable class
    
    private static final class CodeTemplateDescriptionComparator implements Comparator<CodeTemplateDescription> {
        public int compare(CodeTemplateDescription t1, CodeTemplateDescription t2) {
            if (t1.getAbbreviation().equals(t2.getAbbreviation()) &&
                compareTexts(t1.getDescription(), t2.getDescription()) &&
                compareTexts(t1.getParametrizedText(), t2.getParametrizedText()) &&
                Utilities.compareObjects(t1.getContexts(), t2.getContexts())
            ) {
                return 0;
            } else {
                return -1;
            }
        }
    } //NOI18N
    
    private static boolean compareTexts(String t1, String t2) {
        if (t1 == null || t1.length() == 0) {
            t1 = null;
        }
        if (t2 == null || t2.length() == 0) {
            t2 = null;
        }
        if (t1 != null && t2 != null) {
            return t1.equals(t2);
        } else {
            return t1 == null && t2 == null;
        }
    }
    
    private static <K, V> Map<K, V> diff(Map<K, V> orig, Map<K, V> updated, Comparator<V> comparator) {
        Map<K, V> map = new HashMap<K, V>();
        
        for(K key : updated.keySet()) {
            V value = updated.get(key);
            V origValue = orig.get(key);
            
            if (origValue != null && 0 == comparator.compare(origValue, value)) {
                // identical
                continue;
            }
            
            map.put(key, value);
        }
        
        return map;
    }
}
