/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.scripting.php.dbginterface.models;

import java.beans.Customizer;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.netbeans.modules.scripting.php.dbginterface.api.VariableNode;
import org.netbeans.spi.viewmodel.ModelEvent;
import org.openide.util.NbBundle;

/**
 *
 * @author marcow
 */
public class Variable implements VariableNode, PropertyChangeListener, Customizer, Cloneable {
    public static final String PROP_VALUE = "value"; // NOI18N
    public static enum Type {
        UNDEFINED,
        LONG,
        DOUBLE,
        STRING,
        ARRAY,
        OBJECT,
        BOOLEAN,
        RESOURCE,
        REFERENCE,
        SOFT_REFERENCE };
        
    private static final Map<Type, String> typeStringMap;
    static {
        typeStringMap = new HashMap<Type, String>();
        typeStringMap.put(Type.UNDEFINED, NbBundle.getBundle(Variable.class).getString("TYPE_undefined")); // NOI18N
        typeStringMap.put(Type.LONG, NbBundle.getBundle(Variable.class).getString("TYPE_long")); // NOI18N
        typeStringMap.put(Type.DOUBLE, NbBundle.getBundle(Variable.class).getString("TYPE_double")); // NOI18N
        typeStringMap.put(Type.STRING, NbBundle.getBundle(Variable.class).getString("TYPE_string")); // NOI18N
        typeStringMap.put(Type.ARRAY, NbBundle.getBundle(Variable.class).getString("TYPE_array")); // NOI18N
        typeStringMap.put(Type.OBJECT, NbBundle.getBundle(Variable.class).getString("TYPE_object")); // NOI18N
        typeStringMap.put(Type.BOOLEAN, NbBundle.getBundle(Variable.class).getString("TYPE_boolean")); // NOI18N
        typeStringMap.put(Type.RESOURCE, NbBundle.getBundle(Variable.class).getString("TYPE_resource")); // NOI18N
        typeStringMap.put(Type.REFERENCE, NbBundle.getBundle(Variable.class).getString("TYPE_reference")); // NOI18N
        typeStringMap.put(Type.SOFT_REFERENCE, NbBundle.getBundle(Variable.class).getString("TYPE_soft_reference")); // NOI18N
    }
    
    // private DbgDebugger debugger; // Something like that, need to listen on state changes and needed to set values!
    private String name;
    private String longName;        // including possible parents
    private Variable parent;
    private Type type;
    private String typeString;
    private String value;           // String value
    private Variable ref;           // XXX Reference for now?
    private List<Variable> children;
    private Set<PropertyChangeListener> listeners = new HashSet<PropertyChangeListener>();
    
    /** Creates a new instance of Variable */
    public Variable(Variable parent, String name, Type type, String typeString) {
        this.parent = parent;
        this.type = type;
        this.typeString = typeString;
        setName(name);
        
        if (typeString == null || typeString.length() == 0) {
            this.typeString = typeStringMap.get(type);
        }
        
        if (parent != null) {
            addPropertyChangeListener(parent);
        }
    }

    public Variable(Variable parent, String name, Type type, String typeString, String value) {
        this(parent, name, type, typeString);
        
        this.value = value;
    }
    
    public String getValue() {
        return value;
    }
    
    public void setValue(String str) {
        String v;
        
        if (type == Type.STRING) {
            v = longName + "=\"" + str + "\"";
        }
        else {
            v = longName + "=" + str;
        }

        List<Variable> ret = new ArrayList<Variable>();
        // ret = debugger.eval(evalString);
        
        for (Variable va : ret) {
            if (va.longName.equals(longName)) {
                value = va.value;
                type = va.type;
                fireValueChanged(this);
                
                break;
            }
        }
    }
    
    public void setChildren(List<Variable> cList) {
        this.children = cList;
   }

    public VariableNode[] getChildren(int from, int to) {
        Variable[] ret = new Variable[to - from];
        
        for (int i = from; i < to; i++) {
            ret[i - from] = children.get(i);
        }
        
        return ret;
    }

    public boolean isLeaf() {
        return children == null || children.size() == 0;
    }

    public int getChildrenCount() {
        if (children == null) {
            return 0;
        }
        
        return children.size();
    }

    public String getTypeName() {
        return typeString;
    }

    public void setRef(Variable ref) {
        this.ref = ref;
    }
    
    public void setName(String name) {
        if (parent == null || parent.name == null || parent.name.length() == 0) {
            this.name = name;
            this.longName = name;
        }
        else {
            switch (parent.type) {
            case ARRAY:
                this.name = "[" + name + "]";
                this.longName = parent.longName + this.name;
                break;
            case OBJECT:
                this.name = name;
                this.longName = parent.longName + "->" + this.name;
                break;
            default:
                this.name = name;
                this.longName = name;
            }
        }
    }
    
    public void setObject(Object bean) {
        if (bean instanceof String) {
            setValue((String)bean);
        }
        else {
            throw new IllegalArgumentException("" + bean);
        }
    }

    public String toString() {
        return longName + " = " + getTooltipValue();
    }
    
    
    public void addPropertyChangeListener(PropertyChangeListener l) {
        synchronized (listeners) {
            listeners.add(l);
        }
    }

    public void removePropertyChangeListener(PropertyChangeListener l) {
        synchronized (listeners) {
            listeners.remove(l);
        }
    }

    public void propertyChange(PropertyChangeEvent e) {
        if (children.contains(e.getSource())) {
            fireValueChanged(e.getSource());
            
            return;
        }
        throw new UnsupportedOperationException("Not supported yet.");
    }

    private void fireValueChanged(Object value) {
        PropertyChangeEvent evt = new PropertyChangeEvent(this, PROP_VALUE, null, value);
        PropertyChangeListener[] ls;
        
        synchronized (listeners) {
            ls = listeners.toArray(new PropertyChangeListener[listeners.size()]);
        }
        
        for (int i = 0; i < ls.length; i++) {
            ls[i].propertyChange(evt);
        }
    }

    public String getName() {
        return name;
    }

    public String getDisplayName() {
        return longName;
    }

    public String getShortDescription() {
        return null;
    }

    public String getIconBase() {
        // XXX for now
        return LOCAL_VARIABLE_ICON;
    }
    

    public boolean isReadOnly() {
       return type != Type.BOOLEAN && type != Type.DOUBLE && type != Type.LONG && type != Type.STRING ;
    }

    public String getTooltipValue() {
        String ret = "";
        
        if (children != null) {
            boolean first = true;
            
            ret += "[ ";
            
            for (Variable child : children) {
                if (!first) {
                    ret += ", ";
                }
                ret += child.toString();
                first = false;
            }
            
            ret += " ]";
        }
        else if (ref != null) {
            ret += " -> " + ref.longName;
        }
        else {
            ret += value;
        }
        
        return ret;
    }

    public void collectUpdates(Object source,
                               Collection<ModelEvent> events,
                               VariableNode n) {
        Variable newVar = (Variable)n;
        boolean hasChanged = false;
        
        assert name.equals(newVar.getName());

        if (type != newVar.type) {
            type = newVar.type;
            hasChanged = true;
        }
        
        if (!typeString.equals(newVar.typeString)) {
            typeString = newVar.typeString;
            hasChanged = true;
        }
        
        if (value == null && newVar.value != null ||
                newVar.value != null && !newVar.value.equals(value)) {
            value = newVar.value;
            hasChanged = true;
        }
        
        if (ref == null && newVar.ref != null ||
                newVar.ref != null && !newVar.ref.longName.equals(ref.longName)) {
            ref = newVar.ref;
            hasChanged = true;
        }
        
        if (children == null && newVar.children != null) {
            children = newVar.children;
            hasChanged = true;
        }
        else if (children != null) {
            Iterator<Variable> it = children.iterator();
            
            while (it.hasNext()) {
                Variable child = it.next();
                Variable newChild = newVar.findChild(child.name);
                
                if (newChild != null) {
                    newVar.removeChild(newChild);
                    child.collectUpdates(source, events, newChild);
                }
                else {
                    it.remove();
                    hasChanged = true;
                }
            }
            
            if (newVar.children != null && newVar.children.size() > 0) {
                it = newVar.children.iterator();
                
                while (it.hasNext()) {
                    Variable newChild = it.next();
                    
                    it.remove();
                    addChild(newChild);
                    hasChanged = true;
                }
            }
        }

        if (hasChanged) {
            events.add(new ModelEvent.NodeChanged(source, this));
        }
    }
    
    private Variable findChild(String n) {
        if (children == null || n == null || n.length() == 0) {
            return null;
        }
        
        for (Variable child : children) {
            if (child.name.equals(n)) {
                return child;
            }
        }
        
        return null;
    }
    
    private void removeChild(Variable child) {
        if (child == null || children == null) {
            return;
        }
        
        children.remove(child);
        child.parent = null;
        // XXX Change events?
    }
    
    private void addChild(Variable child) {
        if (child == null) {
            return;
        }
        
        if (children == null) {
            children = new ArrayList<Variable>();
        }
        
        children.add(child);
        child.parent = this;
        // XXX Change events?
    }
    
    public int getType() {
        return TYPE_LOCAL_FIELD;
    }
}
