/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.xml.core.cookies;

import java.awt.Dialog;
import java.io.Serializable;
import java.util.Vector;
import javax.swing.DefaultComboBoxModel;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;

import org.netbeans.modules.xml.api.scenario.Scenario;
import org.netbeans.modules.xml.core.scenario.ScenarioPanel;
import org.openide.DialogDescriptor;
import org.openide.NotifyDescriptor;
import org.openide.DialogDisplayer;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.util.HelpCtx;
import org.netbeans.modules.xml.api.cookies.ScenarioCookie;

/**
 * Implements the ScenarioCookie.
 * This class manages the scenarios for a particular DataObject.
 * The Scenarios are stored persistently in attribute of the primary FileObject
 * of this data object.
 *
 * @author  asgeir@dimonsoftware.com
 */
public class ScenarioSupport implements ScenarioCookie {
    
//     public static final String OLD_SCENARIO_DATA_ATTRIBUTE = "org.netbeans.modules.xsl.scenario.ScenarioSupport.Data";
    public static final String SCENARIO_DATA_ATTRIBUTE = "org.netbeans.modules.xml.core.cookies.ScenarioData";
    
    /** Owning dataObject */
    private DataObject dataObject;
    
    /** The ComboBoxModel model which describes the available scenarios and the active scenario */
    private DefaultComboBoxModel scenarioModel;
    
    /** The ScenarioData which will be stored persistently
     * @link aggregation*/
    private ScenarioData data;
    
    /** @link dependency */
    /*# ScenarioPanel lnkScenarioPanel; */
    
    /** @link dependency */
    /*# Scenario lnkScenario; */
    
    /** Creates a new instance of ScenarioSupport */
    public ScenarioSupport(DataObject dataObject) {
        this.dataObject = dataObject;
    }
    
    /**
     * Displays the Customize scenarios dialog
     * @return true if the Customize dialog was pressed with the OK button,
     *         but false if it was closed with the Cancel button.
     */
    public boolean customizeScenarios() {
        
        DefaultComboBoxModel model = getModel();
        ScenarioPanel scenarioPanel = new ScenarioPanel(dataObject, model);
        DialogDescriptor scenariosDD = new DialogDescriptor
        (scenarioPanel,
        Util.THIS.getString("NAME_customize_scenarios_title"), true,
        DialogDescriptor.OK_CANCEL_OPTION, DialogDescriptor.OK_OPTION,
        DialogDescriptor.BOTTOM_ALIGN,
        new HelpCtx(ScenarioSupport.class), null);
        scenariosDD.setClosingOptions(new Object[] { DialogDescriptor.OK_OPTION, DialogDescriptor.CANCEL_OPTION });
        
        Dialog dialog = DialogDisplayer.getDefault().createDialog(scenariosDD);
        
        // inlined ScenarioPanel.addNotify() content
        if (model.getSize() == 0) {
            // Displays the add-scenario dialog is no scenarios are found.
            if (!scenarioPanel.createScenario()) {
                return false;
            }
        }
        
        dialog.setVisible(true);
        
        if (scenariosDD.getValue() == DialogDescriptor.OK_OPTION) {
            for (int ind = 0; ind < model.getSize(); ind++) {
                Scenario scenario = (Scenario)model.getElementAt(ind);
                scenario.saveChanges();
            }
            saveData();
            return true;
        } else {
            // cancel all changes made
            scenarioPanel.rollBackAddedAndRemovedScenarios();
        }
        
        return false;
    }
    
    /**
     * Returns the ComboBoxModel that represents the a list of scenarios.
     * The active scenario is the selected item in that ComboBoxModel
     * @return The ComboBoxModel that represents the a list of scenarios.
     *         The active scenario is the selected item in that ComboBoxModel.
     */
    public DefaultComboBoxModel getModel() {
        if (scenarioModel == null) {
            // Create a new DefaultComboBoxModel which listens to selection changes
            // and saves the data when selection is changed.
            scenarioModel = new DefaultComboBoxModel(getData().getScenarios()) {
                public void setSelectedItem(Object anItem) {
                    super.setSelectedItem(anItem);
                    saveData();
                }
            };
            
            // Set the initial selected item
            scenarioModel.setSelectedItem(scenarioModel.getElementAt(getData().getActiveScenarioIndex()));
            
            // Listener to change events in the scenarioModel and save the
            // data when the model changes.
            scenarioModel.addListDataListener(new ListDataListener() {
                public void contentsChanged(ListDataEvent e) {
                    saveData();
                }
                public void intervalAdded(ListDataEvent e) {
                    saveData();
                }
                public void intervalRemoved(ListDataEvent e) {
                    saveData();
                }
            });
        }
        return scenarioModel;
    }
    
    /**
     * Executes an object using the active scenario.
     * If no scenario is active, the Customize scenarios dialog is displayed
     * before execution.
     */
    public void executeActiveScenario() {
        DefaultComboBoxModel model = getModel();
        Scenario activeScenario = (Scenario)model.getSelectedItem();
        
        if (activeScenario == null) {
            // No scenario active -> let's display the customize dialog
            if (customizeScenarios() == true) {
                activeScenario = (Scenario)model.getSelectedItem();
            }
        }
        
        if (activeScenario != null) {
            activeScenario.execute(dataObject);
        }
    }
    
    /**
     * Serialize the scenario model to the an attribute of the primary FileObject
     * of the DataObject.
     */
    private void saveData() {
        if (scenarioModel == null || data == null) {
            return; // Nothing to save
        }
        
        try {
            data.setActiveScenarioIndex(scenarioModel.getIndexOf(scenarioModel.getSelectedItem()));
            
            FileObject fileObject = dataObject.getPrimaryFile();
            if (fileObject != null) {
                fileObject.setAttribute(SCENARIO_DATA_ATTRIBUTE, data);
            }
        } catch(java.io.IOException e) {
            String message = Util.THIS.getString("MSG_Problems_saving_scenarios_persistently") + "\n" + e.getMessage();
            NotifyDescriptor nd = new NotifyDescriptor.Message(message, NotifyDescriptor.WARNING_MESSAGE);
            DialogDisplayer.getDefault().notify(nd);
        }
    }
    
    /**
     * Retuns the ScenarioData object which describes the scenarios for this
     * DataObject.  If it has not been loaded yet, it is loaded from an
     * attribute of the primary FileObject.
     * @return the ScenarioData object, never null.
     */
    private ScenarioData getData() {
        if (data == null) {
            FileObject fileObject = dataObject.getPrimaryFile();
            if (fileObject != null) {
                Object objData = fileObject.getAttribute(SCENARIO_DATA_ATTRIBUTE);
                if (objData instanceof ScenarioData) {
                    data = (ScenarioData) fileObject.getAttribute(SCENARIO_DATA_ATTRIBUTE);
                }
            }
            if (data == null ) {
                data = new ScenarioData();
            }
        }
        return data;
    }
}
