/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <errno.h>

#include <gtk/gtk.h>

#include <netdude/nd_debug.h>
#include <netdude/nd_prefs.h>
#include <netdude/nd_recent.h>
#include <netdude/nd_macros.h>
#include <netdude/nd_misc.h>
#include <netdude/nd_trace_registry.h>
#include <netdude/nd_tcpdump.h>
#include <netdude/nd_gui.h>
#include "interface.h"
#include "support.h"

#define ND_CONFIG_FILE "preferences"

typedef struct nd_prefs_user_data
{
  FILE        *f;
  GHashTable  *table;
  char        *domain;
} ND_PrefsUserData;

typedef struct nd_prefs_domain
{
  /* The name of this domain */
  char                   *name;

  /* The widget from which the various gui items can be requested */
  GtkWidget              *container;

  /* The widget that actually displays the config items */
  GtkWidget              *gui;

  /* The preferences description for this domain */
  ND_PrefsEntry          *entries;
  int                     num_entries;

  /* The hashtable that stores the settings */
  GHashTable             *table;

  /* The callback that gets called when the user applies the
   * preferences.
   */
  ND_PrefsApplyCallback   apply_cb;

} ND_PrefsDomain;


static ND_PrefsEntry prefs_entries_netdude[] = {
  { "trace_win_size",            ND_PREFS_INT, 1000,      ND_UNUSED, ND_UNUSED  },
  { "show_timestamps",           ND_PREFS_INT, TRUE,      ND_UNUSED, ND_UNUSED  },
  { "show_timestamps_absolute",  ND_PREFS_INT, FALSE,     ND_UNUSED, ND_UNUSED  }, 
  { "timestamps_delay",          ND_PREFS_FLT, ND_UNUSED, 0.5,       ND_UNUSED  },
  { "show_splash",               ND_PREFS_INT, TRUE,      ND_UNUSED, ND_UNUSED  },
  { "show_full_path",            ND_PREFS_INT, TRUE,      ND_UNUSED, ND_UNUSED  },
  { "tcpdump_path",              ND_PREFS_STR, ND_UNUSED, ND_UNUSED, TCPDUMP },
  { "tcpdump_resolve",           ND_PREFS_INT, FALSE,     ND_UNUSED, ND_UNUSED  },
  { "tcpdump_print_domains",     ND_PREFS_INT, FALSE,     ND_UNUSED, ND_UNUSED  },
  { "tcpdump_quick",             ND_PREFS_INT, FALSE,     ND_UNUSED, ND_UNUSED  },
  { "tcpdump_print_link",        ND_PREFS_INT, FALSE,     ND_UNUSED, ND_UNUSED  },
  { "tcpdump_print_timestamp",   ND_PREFS_INT, FALSE,     ND_UNUSED, ND_UNUSED  },
  { "font_mono",                 ND_PREFS_STR, ND_UNUSED, ND_UNUSED, "-*-courier-medium-r-*-*-*-100-*-*-*-*-*-*" },
  { "recent_0",                  ND_PREFS_STR, ND_UNUSED, ND_UNUSED, "" },
  { "recent_1",                  ND_PREFS_STR, ND_UNUSED, ND_UNUSED, "" },
  { "recent_2",                  ND_PREFS_STR, ND_UNUSED, ND_UNUSED, "" },
  { "recent_3",                  ND_PREFS_STR, ND_UNUSED, ND_UNUSED, "" },
  { "recent_4",                  ND_PREFS_STR, ND_UNUSED, ND_UNUSED, "" },
};

static GtkWidget   *prefs_dialog = NULL;
static GList       *global_domains = NULL;

static mode_t       mode_755 = (S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP |
				S_IXGRP | S_IROTH | S_IXOTH);

static gboolean     prefs_assert_dirs(void);
static void         prefs_table_free(GHashTable *table);
static void         prefs_table_clear(GHashTable *table);
static GList       *prefs_read_config_file(GList *domains);
static gboolean     prefs_write_config_file(GList *domains);
static void         prefs_sync_settings_to_dialog(void);
static void         prefs_sync_app_to_settings(void);
static void         prefs_apply_cb(GtkWidget *container,
				   ND_PrefsEntry *entries,
				   int num_entries);

static ND_PrefsDomain *prefs_domain_new(GList *domains,
					const char *dom_name,
					GtkWidget *container,
					GtkWidget *gui,
					ND_PrefsEntry *entries,
					int num_entries,
					ND_PrefsApplyCallback apply_cb);

static ND_PrefsDomain *prefs_domain_find(GList *domains, const char *dom_name);
static GList       *prefs_domain_add(GList *domains, ND_PrefsDomain *domain);
static void         prefs_domains_free(GList *domains);
static void         prefs_domain_free(ND_PrefsDomain *domain);

static void         prefs_set_item_direct(GHashTable *table, const char *key,
					  ND_PrefsType type, const void *data);
static void         prefs_set_item(GList *domains,
				   const char *dom_name, const char *key,
				   ND_PrefsType type, const void *data);

static void        *prefs_get_item_direct(GHashTable *table, const char *key);
static void        *prefs_get_item(GList *domains, const char *dom_name, const char *key);

static void         prefs_del_item_direct(GHashTable *prefs_table, const char *key);
static void         prefs_del_item(GList *domains, const char *dom_name, const char *key);

static ND_PrefsType prefs_get_item_type(GHashTable *table, const char *key);


const char *
nd_prefs_get_netdude_dir(void)
{
  static char cfg_file[MAXPATHLEN] = "\0";

  if (cfg_file[0] != 0)
    return cfg_file;

  g_snprintf(cfg_file, MAXPATHLEN, "%s/.nd", getenv("HOME"));

  return cfg_file;
}


const char *
nd_prefs_get_config_file(void)
{
  static char cfg_file[MAXPATHLEN] = "\0";

  if (cfg_file[0] != 0)
    return cfg_file;

  g_snprintf(cfg_file, MAXPATHLEN, "%s/%s", nd_prefs_get_netdude_dir(), ND_CONFIG_FILE);

  return cfg_file;
}


const char    *
nd_prefs_get_plugin_dir_global(void)
{
  static char dir[MAXPATHLEN] = "\0";

  if (dir[0] != 0)
    return dir;

  g_snprintf(dir, MAXPATHLEN, "%s/plugins", PACKAGE_DATA_DIR);

  return dir;
}


const char    *
nd_prefs_get_plugin_dir_user(void)
{
  static char dir[MAXPATHLEN] = "\0";

  if (dir[0] != 0)
    return dir;

  g_snprintf(dir, MAXPATHLEN, "%s/plugins", nd_prefs_get_netdude_dir());

  return dir;
}


const char    *
nd_prefs_get_proto_dir_global(void)
{
  static char dir[MAXPATHLEN] = "\0";

  if (dir[0] != 0)
    return dir;

  g_snprintf(dir, MAXPATHLEN, "%s/protocols", PACKAGE_DATA_DIR);

  return dir;
}


const char    *
nd_prefs_get_proto_dir_user(void)
{
  static char dir[MAXPATHLEN] = "\0";

  if (dir[0] != 0)
    return dir;

  g_snprintf(dir, MAXPATHLEN, "%s/protocols", nd_prefs_get_netdude_dir());

  return dir;
}


static gboolean
prefs_table_clear_cb(gpointer key, gpointer value, gpointer user_data)
{
  char *key_str = (char *) key;

  if (!key || !value)
    return FALSE;

  if (key_str[0] != '#')
    g_free(value);

  g_free(key);
  return TRUE;
  TOUCH(user_data);
}

static void
prefs_table_free(GHashTable *table)
{
  if (!table)
    return;

  prefs_table_clear(table);
  g_hash_table_destroy(table);
}

static void
prefs_table_clear(GHashTable *table)
{
  if (!table)
    return;

  g_hash_table_foreach_remove(table, prefs_table_clear_cb, NULL);
}


static ND_PrefsDomain *
prefs_domain_new(GList *domains,
		 const char *dom_name,
		 GtkWidget *container,
		 GtkWidget *gui,
		 ND_PrefsEntry *entries,
		 int num_entries,
		 ND_PrefsApplyCallback apply_cb)
{
  ND_PrefsDomain *domain;
  int i;

  if (!dom_name)
    return NULL;

  if ( (domain = prefs_domain_find(domains, dom_name)) == NULL)
    {
      domain = g_new0(ND_PrefsDomain, 1);
      domain->table = g_hash_table_new(g_str_hash, g_str_equal);
    }

  g_free(domain->name);
  domain->name = g_strdup(dom_name);

  if (container)
    domain->container = container;

  if (gui)
    domain->gui = gui;

  if (entries)
    {
      domain->entries = entries;
      domain->num_entries = num_entries;
    }

  if (apply_cb)
    domain->apply_cb = apply_cb;

  D(("Domain %s status: %i entries, container: %p, gui %p\n",
     domain->name, domain->num_entries,
     domain->container, domain->gui));

  for (i = 0; i < num_entries; i++)
    {
      switch (entries[i].type)
	{
	case ND_PREFS_INT:
	  prefs_set_item_direct(domain->table, entries[i].key, ND_PREFS_INT, &entries[i].int_val);
	  break;
	  
	case ND_PREFS_STR:
	  prefs_set_item_direct(domain->table, entries[i].key, ND_PREFS_STR, entries[i].str_val);
	  break;
	  
	case ND_PREFS_FLT:
	  prefs_set_item_direct(domain->table, entries[i].key, ND_PREFS_FLT, &entries[i].flt_val);
	  break;

	default:
	  D(("Unknown preferences data type %i\n", entries[i].type));
	}      
    }

  return domain;
}


static ND_PrefsDomain *
prefs_domain_find(GList *domains, const char *dom_name)
{
  GList *l;
  
  for (l = domains; l; l = g_list_next(l))
    {
      ND_PrefsDomain *domain = (ND_PrefsDomain *) l->data;

      if (strcmp(domain->name, dom_name) == 0)
	return domain;
    }
  
  return NULL;
}


static GList *
prefs_domain_add(GList *domains, ND_PrefsDomain *domain)
{
  if (!domain)
    return NULL;

  if (prefs_domain_find(domains, domain->name))
    {
      D(("Domain %s alread found, not adding\n", domain->name));
      return domains;
    }

  D(("Adding domain %s to list, now %i domains\n",
     domain->name, g_list_length(domains) + 1));

  return g_list_prepend(domains, domain);
}


static void
prefs_domain_free(ND_PrefsDomain *domain)
{
  g_free(domain->name);
  prefs_table_free(domain->table);
  g_free(domain);
}


static void         
prefs_domains_free(GList *domains)
{
  GList *l;

  if (!domains)
    return;

  for (l = domains; l; l = g_list_next(l))
    {
      ND_PrefsDomain *domain = (ND_PrefsDomain *) l->data;
      prefs_domain_free(domain);
      l->data = NULL;
    }

  g_list_free(domains);
}

/*
static ND_PrefsType
prefs_get_data_type_from_entries(const char *key, ND_PrefsEntry *entries, int num_entries)
{
  int i;

  for (i = 0; i < num_entries; i++)
    {
      if (strcmp(entries[i].key, key) == 0)
	return entries[i].type;
    }

  return ND_PREFS_UNK;
}
*/

static GList *
prefs_read_config_file(GList *domains)
{
  ND_PrefsDomain *domain;
  int     int_data;
  float   flt_data;
  char    str_data[MAXPATHLEN];
  char    dom_name[MAXPATHLEN];
  char   *key;
  const char *cfg_file = NULL;
  FILE   *f;
  ND_PrefsType type;

  if ( (cfg_file = nd_prefs_get_config_file()) == NULL)
    return NULL;

  if ((f = fopen(cfg_file, "r")) == NULL)
    return NULL;

  for ( ; ; )
    {
      if (fscanf(f, "%s %u", dom_name, (int*) &type) == EOF)
	break;
      
      if ( (key = strchr(dom_name, '/')) == NULL)
	{
	  D(("Domain/key error in key %s\n", dom_name));
	  continue;
	}

      *key = '\0';
      key++;

      domain = prefs_domain_find(domains, dom_name);
      
      if (!domain)
	{
	  D(("Domain %s not found when reading file %s\n",
	     dom_name, cfg_file));
	  domain = prefs_domain_new(domains, dom_name, NULL, NULL, NULL, 0, NULL);
	  domains = prefs_domain_add(domains, domain);
	}
      
      switch (type)
	{
	case ND_PREFS_INT:
	  fscanf(f, "%i\n", &int_data);
	  prefs_set_item_direct(domain->table, key, ND_PREFS_INT, &int_data);
	  break;

	case ND_PREFS_FLT:
	  fscanf(f, "%f\n", &flt_data);
	  prefs_set_item_direct(domain->table, key, ND_PREFS_FLT, &flt_data);
	  break;

	case ND_PREFS_STR:
	  fscanf(f, "%s\n", str_data);
	  /* Store only if it's not our "unset" marker */
	  if (strcmp(str_data, "---"))
	    prefs_set_item_direct(domain->table, key, ND_PREFS_STR, str_data);
	  break;

	default:
	  D(("Unknown preference entry type for item '%s'.\n", key));
	}
    }
  
  fclose(f);
  return domains;
}


static void 
prefs_write_config_entry(gpointer key_ptr, gpointer value, gpointer user_data)
{
  char              key[MAXPATHLEN];
  char             *key_str  = (char *) key_ptr;
  ND_PrefsUserData *data = (ND_PrefsUserData *) user_data;
  ND_PrefsType      type;

  if (key_str[0] == '#')
    return;

  g_snprintf(key, MAXPATHLEN, "%s/%s", data->domain, key_str);
  type = prefs_get_item_type(data->table, key_str);

  /* D(("Writing config item '%s'\n", (char*) key)); */

  switch (type)
    {
    case ND_PREFS_INT:
      fprintf(data->f, "%-40s \t %i %i\n",
	      key, type, *((int *) value));
      break;
      
    case ND_PREFS_FLT:
      fprintf(data->f, "%-40s \t %i %f\n",
	      key, type, *((float *) value));
      break;
      
    case ND_PREFS_STR:
      {
	char *s = (char *) value;
	
	/* Use "unset" marker if undefined */
	if (!s || *s == '\0')
	  s = "---";
	
	fprintf(data->f, "%-40s \t %i %s\n",
		key, type, s);
      }
      break;
  
    case ND_PREFS_UNK:
      break;

    default:
      D(("Unknown preference entry type %i.\n", type));
    }
}


static gboolean
prefs_assert_dirs(void)
{
  char dir[MAXPATHLEN];

  if (!nd_misc_exists(nd_prefs_get_netdude_dir()))
    {
      if (mkdir(nd_prefs_get_netdude_dir(), mode_755) < 0)
	{
	  if (errno != EEXIST)
	    return FALSE;
	}

      g_snprintf(dir, MAXPATHLEN, "%s/plugins", nd_prefs_get_netdude_dir());

      if (mkdir(dir, mode_755) < 0)
	{
	  if (errno != EEXIST)
	    return FALSE;
	}

      g_snprintf(dir, MAXPATHLEN, "%s/protocols", nd_prefs_get_netdude_dir());
      
      if (mkdir(dir, mode_755) < 0)
	{
	  if (errno != EEXIST)
	    return FALSE;
	}
    }

  return TRUE;
}


static gboolean
prefs_write_config_file(GList *domains)
{
  GList *l;
  ND_PrefsUserData data;

  if (!domains)
    return FALSE;

  if (!prefs_assert_dirs())
    return FALSE;

  if ((data.f = fopen(nd_prefs_get_config_file(), "w")) == NULL)
    return FALSE;
  
  for (l = domains; l; l = g_list_next(l))
    {
      ND_PrefsDomain *domain = (ND_PrefsDomain *) l->data;

      D(("Saving domain %s\n", domain->name));

      data.table  = domain->table;
      data.domain = domain->name;
      g_hash_table_foreach(domain->table, prefs_write_config_entry, &data);
    }

  fclose(data.f);
  return TRUE;
}


static void  
prefs_sync_settings_to_dialog(void)
{
  int        i;
  GtkWidget *w;
  GList     *l;

  D_ASSERT_PTR(prefs_dialog);

  for (l = global_domains; l; l = g_list_next(l))
    {
      ND_PrefsDomain *domain = (ND_PrefsDomain *) l->data;

      for (i = 0; i < domain->num_entries; i++)
	{
	  if (!domain->container)
	    continue;
	  
	  if ( (w = gtk_object_get_data(GTK_OBJECT(domain->container),
					domain->entries[i].key)) == NULL)
	    {
	      D(("Widget lookup failed for %s/%s\n",
		 domain->name, domain->entries[i].key));
	      continue;
	    }
      
	  if (GTK_IS_TOGGLE_BUTTON(w))
	    {
	      nd_prefs_set_int_item(domain->name, domain->entries[i].key,
				    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)));
	    }
	  else if (GTK_IS_SPIN_BUTTON(w))
	    {
	      switch (domain->entries[i].type)
		{
		case ND_PREFS_FLT:
		  nd_prefs_set_flt_item(domain->name, domain->entries[i].key,
					gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(w)));
		  break;
		  
		case ND_PREFS_INT:
		  nd_prefs_set_int_item(domain->name, domain->entries[i].key,
					gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w)));
		  break;
		  
		default:
		  D(("WARNING -- unexpected preference data type.\n"));					
		}
	    }
	  else if (GTK_IS_ENTRY(w))
	    {
	      nd_prefs_set_str_item(domain->name, domain->entries[i].key,
				    gtk_entry_get_text(GTK_ENTRY(w)));
	    }
	  else if (GTK_IS_LABEL(w))
	    {
	      char *value;
	      
	      gtk_label_get(GTK_LABEL(w), &value);
	      nd_prefs_set_str_item(domain->name, domain->entries[i].key, value);
	    }
	  else
	    {
	      D(("Unhandled widget type.\n"));
	    }
	}
    }
}


void
nd_prefs_dialog_sync(void)
{
  int          i;
  int          int_val;
  float        flt_val;
  char        *str_val;
  GtkWidget   *w;
  GList       *l;

  D_ASSERT_PTR(prefs_dialog);
  if (!prefs_dialog)
    return;

  for (l = global_domains; l; l = g_list_next(l))
    {
      ND_PrefsDomain *domain = (ND_PrefsDomain *) l->data;
      
      if (!domain) /* Shouldn't ever happen */
	continue;

      D(("Syncing gui for prefs domain %s with %i entries\n",
	 domain->name, domain->num_entries));
      for (i = 0; i < domain->num_entries; i++)
	{
	  if (!domain->container)
	    {
	      D(("  No container\n"));
	      continue;
	    }
	  
	  if ( (w = gtk_object_get_data(GTK_OBJECT(domain->container),
					domain->entries[i].key)) == NULL)
	    {
	      D(("  Widget lookup failed for %s/%s\n",
		 domain->name, domain->entries[i].key));
	      continue;
	    }
	        
	  if (GTK_IS_TOGGLE_BUTTON(w))
	    {
	      nd_prefs_get_int_item(domain->name, domain->entries[i].key, &int_val);
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), int_val);
	      D(("  Setting toggle button %s to %i\n", domain->entries[i].key, int_val));
	    }
	  else if (GTK_IS_SPIN_BUTTON(w))
	    {
	      switch (domain->entries[i].type)
		{
		case ND_PREFS_FLT:
		  nd_prefs_get_flt_item(domain->name, domain->entries[i].key, &flt_val);
		  gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), flt_val); 
		  D(("  Setting spin button %s to %f\n", domain->entries[i].key, flt_val));
		  break;
		  
		case ND_PREFS_INT:
		  nd_prefs_get_int_item(domain->name, domain->entries[i].key, &int_val);
		  gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), (float) int_val); 
		  D(("  Setting spin button %s to %f\n", domain->entries[i].key, (float) int_val));
		  break;
		  
		default:
		  D(("  WARNING -- unexpected preference data type.\n"));					
		}
	    }
	  else if (GTK_IS_ENTRY(w))
	    {
	      nd_prefs_get_str_item(domain->name, domain->entries[i].key, &str_val);
	      gtk_entry_set_text(GTK_ENTRY(w), str_val);
	      D(("  Setting entry field %s to %s\n", domain->entries[i].key, str_val));
	    }
	  else if (GTK_IS_LABEL(w))
	    {
	      nd_prefs_get_str_item(domain->name, domain->entries[i].key, &str_val);
	      nd_gui_add_monowidth_widget(w);
	      gtk_label_set_text(GTK_LABEL(w), str_val);	  
	    }
	  else
	    D(("  Unhandled widget type.\n"));
	}
    }
}


static void         
prefs_sync_app_to_settings(void)
{
  GList *l;
  
  for (l = global_domains; l; l = g_list_next(l))
    {
      ND_PrefsDomain *domain = (ND_PrefsDomain *) l->data;
      
      if (domain->apply_cb)
	domain->apply_cb(domain->container,
			 domain->entries,
			 domain->num_entries);
    }
}


static void
prefs_apply_cb(GtkWidget *container,
	       ND_PrefsEntry *entries,
	       int num_entries)
{
  ND_Trace *trace;
  
  if ( (trace = nd_trace_registry_get_current()))
    nd_gui_windowtitle_set(nd_trace_get_name(trace));

  nd_gui_update_monowidth_widgets();

  return;
  TOUCH(container);
  TOUCH(entries);
  TOUCH(num_entries);
}

	       
void
nd_prefs_init(void)
{
  if (!prefs_dialog)
    prefs_dialog = create_pref_dialog();
}


void
nd_prefs_load(void)
{
  ND_PrefsDomain *domain;

  domain = prefs_domain_new(global_domains, ND_DOM_NETDUDE,
			    prefs_dialog, prefs_dialog,
			    prefs_entries_netdude,
			    sizeof (prefs_entries_netdude) / sizeof(ND_PrefsEntry),
			    prefs_apply_cb);

  global_domains = prefs_domain_add(global_domains, domain);
  
  if (nd_misc_exists(nd_prefs_get_config_file()))
    global_domains = prefs_read_config_file(global_domains);
  else /* We don't have a config file yet -- create one. */
    prefs_write_config_file(global_domains);

  nd_prefs_dialog_sync();
  prefs_sync_app_to_settings();
}


void           
nd_prefs_add_domain(const char *dom_name,
		    GtkWidget *container,
		    GtkWidget *gui,
		    ND_PrefsEntry *entries,
		    int num_entries,
		    ND_PrefsApplyCallback apply_cb)
{
  GtkNotebook    *notebook;
  ND_PrefsDomain *domain;

  domain = prefs_domain_new(global_domains, dom_name, container, gui,
			    entries, num_entries, apply_cb);
  global_domains = prefs_domain_add(global_domains, domain);

  notebook = GTK_NOTEBOOK(gtk_object_get_data(GTK_OBJECT(prefs_dialog),
					      "prefs_notebook"));

  if (gui)
    {
      gtk_widget_show(gui);
      gtk_notebook_append_page(notebook, gui, gtk_label_new(dom_name));
    }
}


int
nd_prefs_save(void)
{
  return prefs_write_config_file(global_domains);
}


void
nd_prefs_save_general(void)
{
  ND_PrefsDomain *domain;
  char  recent[MAXPATHLEN];
  char *s;
  int   i;
  GList *tmp_domains = NULL;

  /* Load the preferences as currently on disk into
     new hash table, then overwrite the user-independent
     settings and write back to disk.
  */
  
  tmp_domains = prefs_read_config_file(NULL);
  domain = prefs_domain_find(global_domains, ND_DOM_NETDUDE);
  
  s = prefs_get_item_direct(domain->table, "load_dir");
  prefs_set_item(tmp_domains, ND_DOM_NETDUDE, "load_dir", ND_PREFS_STR, s);

  s = prefs_get_item_direct(domain->table, "save_dir");
  prefs_set_item(tmp_domains, ND_DOM_NETDUDE, "save_dir", ND_PREFS_STR, s);

  for (i = 0; i < NUM_RECENT; i++)
    {
      g_snprintf(recent, MAXPATHLEN, "recent_%i", i);
      s = prefs_get_item_direct(domain->table, recent);
      prefs_set_item(tmp_domains, ND_DOM_NETDUDE, recent, ND_PREFS_STR, s);      
    }

  prefs_write_config_file(tmp_domains);
  prefs_domains_free(tmp_domains);
}


static void *
prefs_get_item_direct(GHashTable *table, const char *key)
{
  return g_hash_table_lookup(table, key);
}


static void *
prefs_get_item(GList *domains, const char *dom_name, const char *key)
{
  ND_PrefsDomain *domain;

  if (!domains || !dom_name || !key || key[0] == '\0')
    return NULL;
  
  if ( (domain = prefs_domain_find(domains, dom_name)) == NULL)
    return NULL;

  D_ASSERT_PTR(domain->table);
  if (!domain->table)
    {
      D(("No hash table in preferences domain! ARGH!\n"));
      return NULL;
    }
  
  return g_hash_table_lookup(domain->table, key);
}


gboolean
nd_prefs_get_str_item(const char *domain, const char *key, char **result)
{
  void *data = prefs_get_item(global_domains, domain, key);

  if (!data || !result)
    return FALSE;

  *result = (char *) data;
  return TRUE;
}


gboolean 
nd_prefs_get_int_item(const char *domain, const char *key, int *result)
{
  void *data = prefs_get_item(global_domains, domain, key);

  if (!data || !result)
    return FALSE;

  *result = *((int *) data);
  return TRUE;
}


gboolean
nd_prefs_get_flt_item(const char *domain, const char *key, float *result)
{
  void *data = prefs_get_item(global_domains, domain, key);

  if (!data || !result)
    return FALSE;

  *result = *((float *) data);
  return TRUE;
}


static ND_PrefsType
prefs_get_item_type(GHashTable *table, const char *key)
{
  void *data;
  char  data_type_key[MAXPATHLEN];

  D_ASSERT_PTR(table);

  if (!table)
    return ND_PREFS_UNK;

  g_snprintf(data_type_key, MAXPATHLEN, "#%s", key);
  data = g_hash_table_lookup(table, data_type_key);

  if (!data)
    {
      D(("Returning type UNK for '%s' in %p\n", data_type_key, table));
      return ND_PREFS_UNK;
    }

  return GPOINTER_TO_INT(data);
}


static void
prefs_del_item_direct(GHashTable *prefs_table, const char *key)
{
  char   data_type_key[MAXPATHLEN];

  if (!prefs_table || !key)
    return;
  
  /* FIXME -- how do I properly clean up the key and value? */

  g_hash_table_remove(prefs_table, key);
  g_snprintf(data_type_key, MAXPATHLEN, "#%s", key);
  g_hash_table_remove(prefs_table, data_type_key);
}


static void
prefs_del_item(GList *domains,
	       const char *dom_name, const char *key)
{
  ND_PrefsDomain *domain;

  if (!domains || !dom_name || !key || key[0] == '\0' || key[0] == '#')
    return;

  if ( (domain = prefs_domain_find(domains, dom_name)) == NULL)
    return;
  
  prefs_del_item_direct(domain->table, key);
}


static void
prefs_set_item_direct(GHashTable *table, const char *key,
		      ND_PrefsType type, const void *data)     
{
  char   data_type_key[MAXPATHLEN];
  void  *data_copy, *data_old;

  if (!table || !key || !data)
    return;

  switch (type)    
    {
    case ND_PREFS_INT:
      data_copy = g_new0(int, 1);
      memcpy(data_copy, data, sizeof(int));
      break;
      
    case ND_PREFS_STR:
      data_copy = g_strdup((char *)data);
      break;
      
    case ND_PREFS_FLT:
      data_copy = g_new0(float, 1);
      memcpy(data_copy, data, sizeof(float));
      break;
      
    default:
      data_copy = NULL;
      D(("Unknown preferences data type %i\n", type));
      return;
    }

  if ( (data_old = g_hash_table_lookup(table, key)))
    {
      /* D(("Updating prefs item '%s'.\n", key)); */
      g_hash_table_remove(table, key);
      g_hash_table_insert(table, g_strdup(key), data_copy);
      g_snprintf(data_type_key, MAXPATHLEN, "#%s", key);
      g_hash_table_insert(table, g_strdup(data_type_key), GINT_TO_POINTER(type));
    }
  else
    {
      /* D(("Inserting new prefs item '%s' into %p.\n", key, table)); */
      g_hash_table_insert(table, strdup(key), data_copy);
      g_snprintf(data_type_key, MAXPATHLEN, "#%s", key);
      g_hash_table_insert(table, g_strdup(data_type_key), GINT_TO_POINTER(type));
    }
}


static void
prefs_set_item(GList *domains,
	       const char *dom_name, const char *key,
	       ND_PrefsType type, const void *data)
{
  ND_PrefsDomain *domain;

  if (!domains || !dom_name || !key || key[0] == '\0' || key[0] == '#')
    return;

  if ( (domain = prefs_domain_find(domains, dom_name)) == NULL)
    return;
  
  prefs_set_item_direct(domain->table, key, type, data);
}


void
nd_prefs_set_str_item(const char *domain, const char *key, const char *data)
{
  prefs_set_item(global_domains, domain, key, ND_PREFS_STR, data);
}


void
nd_prefs_set_flt_item(const char *domain, const char *key, float data)
{
  prefs_set_item(global_domains, domain, key, ND_PREFS_FLT, &data);
}


void
nd_prefs_set_int_item(const char *domain, const char *key, int data)
{
  prefs_set_item(global_domains, domain, key, ND_PREFS_INT, &data);
}


void
nd_prefs_del_item(const char *domain, const char *key)
{
  prefs_del_item(global_domains, domain, key);
}


void     
nd_prefs_dialog_show(void)
{
  if (!prefs_dialog)
    prefs_dialog = create_pref_dialog();
  
  nd_prefs_dialog_sync();
  gtk_widget_show(prefs_dialog);
}


void     
nd_prefs_dialog_ok(void)
{
  nd_prefs_dialog_apply();

  if (!prefs_dialog)
    return;

  gtk_widget_hide(prefs_dialog);
}


void     
nd_prefs_dialog_apply(void)
{
  prefs_sync_settings_to_dialog();
  prefs_sync_app_to_settings();
  prefs_write_config_file(global_domains);
}


void     
nd_prefs_dialog_cancel(void)
{
  if (!prefs_dialog)
    return;

  gtk_widget_hide(prefs_dialog);
}


static void
prefs_select_tcpdump_cb(const char *filename,
			void *user_data)			
{
  if (nd_misc_can_exec(filename))
    {
      nd_dialog_filesel_close();
      nd_prefs_set_str_item(ND_DOM_NETDUDE, "tcpdump_path", filename);
      nd_prefs_dialog_sync();
    }

  return;
  TOUCH(user_data);
}


void           
nd_prefs_select_tcpdump(void)
{
  char *tcpdump_path;
  
  nd_prefs_get_str_item(ND_DOM_NETDUDE, "tcpdump_path", &tcpdump_path);
  nd_dialog_filesel(_("Select Tcpdump Executable"),
		    tcpdump_path,
		    prefs_select_tcpdump_cb,
		    NULL);
}


static void
prefs_fontsel_cb(const char *font_name,
		 void *user_data)
{
  GtkStyle     *style;
  GtkWidget    *w;

  if (!prefs_dialog)
    return;

  w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "font_mono");
  gtk_label_set_text(GTK_LABEL(w), font_name);

  style = gtk_style_copy(gtk_widget_get_style(w));
  gdk_font_unref(style->font);
  style->font = gdk_font_load (font_name);
  gtk_widget_set_style(w, style);

  return;
  TOUCH(user_data);
}


void           
nd_prefs_fontsel_show(void)
{
  GtkWidget *w;
  char *font_name;

  if (!prefs_dialog)
    return;

  w = gtk_object_get_data(GTK_OBJECT(prefs_dialog), "font_mono");
  gtk_label_get(GTK_LABEL(w), &font_name);
  nd_dialog_fontsel(_("Select Monowidth Font"), font_name,
		    prefs_fontsel_cb, NULL); 
}
