/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// DlgDisplayLogs.cpp: implementation of the DlgDisplayLogs class.
//
//////////////////////////////////////////////////////////////////////

#include "DlgShowLog.h"
#include "DlgMessage.h"
#include "DlgSetLogFilters.h"
#include "DlgProgress.h"

#include "DlgDisplayLogs.h"
#include "dlgs_wdr.h"
#include "clintl.h"


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


BEGIN_EVENT_TABLE(DlgDisplayLogsListCtrl, wxListCtrl)
    EVT_LIST_ITEM_RIGHT_CLICK(IDC_LIST_LOGS, DlgDisplayLogsListCtrl::OnItemRightClick)
    EVT_LIST_ITEM_ACTIVATED(IDC_LIST_LOGS, DlgDisplayLogsListCtrl::OnItemClick)
END_EVENT_TABLE()



DlgDisplayLogs * wOpParent;

void DlgDisplayLogsListCtrl::OnItemRightClick(wxListEvent& event)
{
	wOpParent->OnItemRightClick(event);
}

void DlgDisplayLogsListCtrl::OnItemClick(wxListEvent& event)
{
	wOpParent->OnItemClick(event);
}


BEGIN_EVENT_TABLE(DlgDisplayLogs, wxDialog)
    EVT_BUTTON(IDC_REFRESH, DlgDisplayLogs::OnRefreshEntries)
    EVT_BUTTON(IDC_SET_LOG_FILTERS, DlgDisplayLogs::OnSetLogFilters)
    EVT_BUTTON(IDC_CHECK_LOGS_INTEGRITY, DlgDisplayLogs::OnCheckLogsIntegrity)
    EVT_BUTTON(IDC_EXPORT_LOG, DlgDisplayLogs::OnExportLogs)
END_EVENT_TABLE()

DlgDisplayLogs::DlgDisplayLogs(wxWindow * wParent, PkiClient * ClientPki, const PKI_CERT & SigningCert):wxDialog(wParent, (wxWindowID)-1, _("Display logs"), wxDefaultPosition, wxDefaultSize, wxSYSTEM_MENU | wxMINIMIZE_BOX | wxCAPTION | wxMAXIMIZE_BOX | wxRESIZE_BORDER), m_SigningCert(SigningCert)
{
	DlgDisplayLogs_SetWindow(this);

	m_listCtrl = (DlgDisplayLogsListCtrl *)FindWindow(IDC_LIST_LOGS);
	m_ClientPki = ClientPki;

	m_imageListSmall = new wxImageList(16, 16, TRUE);

	wxIcon ico;
	ico.CopyFromBitmap(DlgDisplayLogs_GetBitmap(0));
	LogIconId = m_imageListSmall->Add( ico );

	m_listCtrl->SetImageList(m_imageListSmall, wxIMAGE_LIST_SMALL);

	wOpParent = this;
	
	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("Id"), wxLIST_ALIGN_DEFAULT, 80);
	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("Date"), wxLIST_ALIGN_DEFAULT, 150);
	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("Status"), wxLIST_ALIGN_DEFAULT, 150);
	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("Type"), wxLIST_ALIGN_DEFAULT, 150);
	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("User"), wxLIST_ALIGN_DEFAULT, 150);
	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("Object"), wxLIST_ALIGN_DEFAULT, 150);
	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("Signature"), wxLIST_ALIGN_DEFAULT, 150);
	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("Error"), wxLIST_ALIGN_DEFAULT, 150);

	DisplayLOG(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());

	CenterOnScreen();
	ShowModal();
}

DlgDisplayLogs::~DlgDisplayLogs()
{
	if(m_listCtrl)
	{
		m_listCtrl->ClearAll();
		m_listCtrl->SetImageList(NULL, wxIMAGE_LIST_SMALL);
	}
	if(m_imageListSmall) delete m_imageListSmall;
}

bool DlgDisplayLogs::DisplayLOG(int NumEntries)
{
	size_t i;
	mString object;

	DlgMessage Msg(this);
	logs.clear();
	Types.clear();

	m_LogFilters.set_index((((wxSpinCtrl*)FindWindow(IDC_PAGE))->GetValue()-1) * NumEntries);
	m_LogFilters.set_max(NumEntries);

	Msg.wShow(_("Loading Log(s) List..."));
	if(!m_ClientPki->EnumLOG(m_LogFilters, logs))
	{
		Msg.wHide();
		HandleError(m_ClientPki->GetError(), this);
		return false;
	}
	//Get local entity log type
	if(!m_ClientPki->GetLogsType(Types))
	{
		Msg.wHide();
		HandleError(m_ClientPki->GetError(), this);
	}
	Msg.wHide();


	m_listCtrl->DeleteAllItems();

	for(i=0; i<logs.size(); i++)
	{
		object.sprintf("%ld", logs[i].get_logId());
		m_listCtrl->InsertItem(i , object.c_str(), LogIconId);
		m_listCtrl->SetItemData(i, (long)i);

		m_listCtrl->SetItem(i, 1, FormatTime(logs[i].get_body().get_logDate()));

		if((LOG_MESSAGE_STATUS)logs[i].get_body().get_logStatus() == LOG_STATUS_TYPE_FAILURE)
		{
			m_listCtrl->SetItemTextColour(i, *wxRED);
		}

		m_listCtrl->SetItem(i, 2, ASN1_logs_get_StatusString((LOG_MESSAGE_STATUS)logs[i].get_body().get_logStatus()));
		m_listCtrl->SetItem(i, 3, ASN1_logs_get_TypeString((LOG_MESSAGE_TYPE)logs[i].get_body().get_logType()));
		m_listCtrl->SetItem(i, 4, logs[i].get_body().get_user().c_str());
		
		object = logs[i].get_body().get_objectName();
		if(!object.size())
			object = _("None");
		m_listCtrl->SetItem(i, 5, object.c_str());

		if(logs[i].get_sig().to_PEM(object))
		{
			m_listCtrl->SetItem(i, 6, object.c_str());
		}
		
		m_listCtrl->SetItem(i, 7, logs[i].get_body().get_error().c_str());
	}

	return true;
}


void DlgDisplayLogs::ShowLog()
{
	const LogEntry & currEntry = GetSelectedLog();
	if(!currEntry) return;

	DlgShowLog Dlg(this, currEntry, m_SigningCert);
}


void DlgDisplayLogs::OnItemRightClick(wxListEvent& event)
{
	long item;
	int state;
	item = m_listCtrl->HitTest(event.GetPoint(), state);
	if(item == -1) return;

	m_listCtrl->SetItemState(item, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
//	OnRightClick(event);
}

void DlgDisplayLogs::OnItemClick(wxListEvent& event)
{
	wxCommandEvent m_event;
	OnShowProperties(m_event);
}

const LogEntry & DlgDisplayLogs::GetSelectedLog()
{
	long SelectedItem;
	size_t index;

	SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(SelectedItem == -1) return LogEntry::EmptyInstance;
	index = (size_t)m_listCtrl->GetItemData(SelectedItem);
	return logs[index];
}


void DlgDisplayLogs::OnRefreshEntries(wxCommandEvent &event)
{
	DisplayLOG(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgDisplayLogs::OnShowProperties(wxCommandEvent &event)
{
	ShowLog();
}


void DlgDisplayLogs::OnSetLogFilters(wxCommandEvent &event)
{
	
	DlgSetLogFilters dlg(this, Types, m_LogFilters);
	if(!dlg.IsOK()) return;

	m_LogFilters = dlg.GetFilters();
	DisplayLOG(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}


void DlgDisplayLogs::OnCheckLogsIntegrity(wxCommandEvent &event)
{
	if(DisplayMessage(this, _("Are you sure to want to check the logs integrity, it might take a very long time ?"), wxYES_NO) == wxNO) return;

	DlgMessage Msg(this);
	Msg.wShow(_("Checking Log(s) Integrity..."));
	if(!m_ClientPki->CheckLogsIntegrity())
	{
		Msg.wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	Msg.wHide();
	DisplayMessage(this, _("The logs integrity has been successfully controlled"), wxOK);
}

bool DlgDisplayLogs::AddNode(FILE * fp, xmlDocPtr xDoc, const LogEntry & currLog)
{
	xmlNodePtr LogNode;
	xmlNodePtr NewNode;
	mString value;
	mString sig_pem;

	LogNode = xmlNewNode(NULL, (const xmlChar*)"LOG_ENTRY");
	if(!LogNode)
	{
		return false;
	}

	// Adding the properties
	value.sprintf("%ld", currLog.get_logId());
	if(!xmlSetProp(LogNode, (const xmlChar*)"log_id", (const xmlChar*)value.c_str()))
	{
		xmlFreeNode(LogNode);
		return false;
	}

	value.sprintf("%ld", currLog.get_body().get_logStatus());
	if(!xmlSetProp(LogNode, (const xmlChar*)"log_status", (const xmlChar*)value.c_str()))
	{
		xmlFreeNode(LogNode);
		return false;
	}

	value.sprintf("%ld", currLog.get_body().get_logType());
	if(!xmlSetProp(LogNode, (const xmlChar*)"log_type", (const xmlChar*)value.c_str()))
	{
		xmlFreeNode(LogNode);
		return false;
	}

	value.sprintf("%ld", currLog.get_body().get_logDate());
	if(!xmlSetProp(LogNode, (const xmlChar*)"log_date", (const xmlChar*)value.c_str()))
	{
		xmlFreeNode(LogNode);
		return false;
	}

	// Adding node USER
	value = currLog.get_body().get_user();
	NewNode = xmlNewTextChild(LogNode, NULL, (const xmlChar*)"USER", (const xmlChar*)value.c_str());
	if(!NewNode)
	{
		xmlFreeNode(LogNode);
		return false;
	}

	// Adding node OBJECT
	value = currLog.get_body().get_objectName();
	NewNode = xmlNewTextChild(LogNode, NULL, (const xmlChar*)"OBJECT", (const xmlChar*)value.c_str());
	if(!NewNode)
	{
		xmlFreeNode(LogNode);
		return false;
	}

	// Adding node ERROR
	if(currLog.get_body().get_logStatus() == LOG_STATUS_TYPE_FAILURE)
	{
		value = currLog.get_body().get_error();
		NewNode = xmlNewTextChild(LogNode, NULL, (const xmlChar*)"ERROR", (const xmlChar*)value.c_str());
		if(!NewNode)
		{
			xmlFreeNode(LogNode);
			return false;
		}
	}

	// Adding node SIGNATURE
	if(currLog.get_sig().to_PEM(sig_pem))
	{
		NewNode = xmlNewTextChild(LogNode, NULL, (const xmlChar*)"SIGNATURE", (const xmlChar*)sig_pem.c_str());
		if(!NewNode)
		{
			xmlFreeNode(LogNode);
			return false;
		}
	}
	xmlElemDump(fp, xDoc, LogNode);
	xmlFreeNode(LogNode);

	return true;
}

void DlgDisplayLogs::OnExportLogs(wxCommandEvent &event)
{
	FILE * fp;
	size_t i;
	mString title;
	unsigned long index;
	unsigned long LogsCount;
	mVector<LogEntry> AllLogs;
	xmlDocPtr xDoc;
	char * strDTD = "<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n" \
"<!DOCTYPE LOG_ENTRIES [\n" \
"	<!ELEMENT LOG_ENTRIES	(LOG_ENTRY*)>\n" \
"	<!ELEMENT LOG_ENTRY		(USER,OBJECT,ERROR?,SIGNATURE?)>\n" \
"	<!ELEMENT USER			(#PCDATA)>\n" \
"	<!ELEMENT OBJECT		(#PCDATA)>\n" \
"	<!ELEMENT ERROR			(#PCDATA)>\n" \
"	<!ELEMENT SIGNATURE		(#PCDATA)>\n" \
"	<!ATTLIST LOG_ENTRY\n" \
"		log_id				CDATA #REQUIRED\n" \
"		log_status			CDATA #REQUIRED\n" \
"		log_type			CDATA #REQUIRED\n" \
"		log_date			CDATA #REQUIRED\n" \
"	>\n" \
"]>\n" \
"<LOG_ENTRIES>\n";


	wxFileDialog dialog(this, _("Export Logs"), "", "", _("XML File (*.xml)|*.xml"), wxSAVE|wxOVERWRITE_PROMPT);
	if (dialog.ShowModal() != wxID_OK)
	{
		return;
	}
	fp = fopen(dialog.GetPath(), "wb");
	if(!fp)
	{
		HandleError(strerror(errno), this);
		return;
	}
	
	fprintf(fp, strDTD);
	
	
	DlgProgress Msg(this);
	Msg.wShow();
	if(!m_ClientPki->GetLogsCount(m_LogFilters, LogsCount))
	{
		Msg.wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}

	Msg.SetMax(100);

	m_LogFilters.set_max(150);
	index = 0;

	xDoc = xmlNewDoc((const xmlChar*)"1.0");
	if(!xDoc)
	{
		Msg.wHide();
		return;
	}
	xDoc->encoding = (xmlChar*)strdup("iso-8859-1");

/*
	xMem = xmlParserInputBufferCreateMem(strDTD, strlen(strDTD), XML_CHAR_ENCODING_NONE);
	if(!xMem)
	{
		xmlFreeDoc(xDoc);
		Msg.wHide();
		return;
	}
	xDtd = xmlIOParseDTD(NULL, xMem, XML_CHAR_ENCODING_NONE);
	if(!xDtd)
	{
		xmlFreeDoc(xDoc);
		Msg.wHide();
		return;
	}


//	free((void*)xDtd->ExternalID);
	xDtd->ExternalID = NULL;
//	free((void*)xDtd->SystemID);
	xDtd->SystemID = NULL;
//	free((void*)xDtd->name);
	xDtd->name = (xmlChar*)strdup("LOG_ENTRIES");
//	xDtd->name = (xmlChar*)"LOG_ENTRIES";

	xDoc->intSubset = xDtd;
	if (xDoc->children == NULL)
         xmlAddChild((xmlNodePtr)xDoc, (xmlNodePtr)xDtd);
     else
         xmlAddPrevSibling(xDoc->children, (xmlNodePtr)xDtd);

	RootNode = xmlNewNode(NULL, (const xmlChar*)"LOG_ENTRIES");
	if(!RootNode)
	{
		Msg.wHide();
		xmlFreeDoc(xDoc);
		return;
	}
	xmlDocSetRootElement(xDoc, RootNode);
*/	
	do
	{
		AllLogs.clear();
		m_LogFilters.set_index(index);

		if(!m_ClientPki->EnumLOG(m_LogFilters, AllLogs))
		{
			Msg.wHide();
			HandleError(m_ClientPki->GetError(), this);
			xmlFreeDoc(xDoc);
			return;
		}

		for(i=0; i<AllLogs.size() && !Msg.IsCanceled(); i++)
		{
			AddNode(fp, xDoc, AllLogs[i]);
			wxYieldIfNeeded();
		}

		index += 150;
		Msg.SetValue((int)((index*100)/LogsCount));
		title.sprintf("%ld/%ld", index, LogsCount);
		Msg.SetTitle(title.c_str());
		wxYieldIfNeeded();
	}
	while(AllLogs.size()  && !Msg.IsCanceled());
	Msg.wHide();

	fprintf(fp, "</LOG_ENTRIES>\n");
	fclose(fp);

	xmlFreeDoc(xDoc);
	return;
}
