/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef ASN1ERR_H
#define ASN1ERR_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <openssl/asn1.h>
#include <openssl/asn1t.h>
#include <mString.h>
#include <PKI_ERR.h>
#include "Asn1Helper.h"


/*!
  This structure represents an error
*/
typedef struct st_ERROR_ENTRY
{
	ASN1_INTEGER *line;			//!< The line
	ASN1_UTF8STRING * file;		//!< The file
	ASN1_INTEGER * lib;			//!< The lib
	ASN1_INTEGER * function;	//!< The function
	ASN1_INTEGER * code;		//!< The error code
	ASN1_UTF8STRING * data;		//!< Some additional datas
}ERROR_ENTRY;
DECLARE_ASN1_ITEM(ERROR_ENTRY)


/*!
  This class is the representation of ERROR_ENTRY
*/
class ErrorEntry : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ErrorEntry();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ErrorEntry(const ErrorEntry & other);

	/*! \brief This is the destructor.
	 */
	~ErrorEntry();


	/*! \brief This function converts a ERROR_ENTRY to a ErrorEntry.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ERROR_ENTRY * Datas);

	/*! \brief This function converts a ErrorEntry to a ERROR_ENTRY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ERROR_ENTRY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ERROR_ENTRY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::code.
	 *  \param c_code [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_code(unsigned long c_code);

	/*! \brief This function returns ASN1_INTEGER::code.
	 *  \return The value of ASN1_INTEGER::code.
	 */
	unsigned long get_code() const;


	/*! \brief This function sets ASN1_UTF8STRING::data.
	 *  \param c_data [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_data(const mString & c_data);

	/*! \brief This function returns ASN1_UTF8STRING::data.
	 *  \return The value of ASN1_UTF8STRING::data.
	 */
	const mString & get_data() const;

	/*! \brief This function returns ASN1_UTF8STRING::data.
	 *  \return The value of ASN1_UTF8STRING::data.
	 */
	mString & get_data();


	/*! \brief This function sets ASN1_UTF8STRING::file.
	 *  \param c_file [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_file(const mString & c_file);

	/*! \brief This function returns ASN1_UTF8STRING::file.
	 *  \return The value of ASN1_UTF8STRING::file.
	 */
	const mString & get_file() const;

	/*! \brief This function returns ASN1_UTF8STRING::file.
	 *  \return The value of ASN1_UTF8STRING::file.
	 */
	mString & get_file();


	/*! \brief This function sets ASN1_INTEGER::function.
	 *  \param c_function [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_function(unsigned long c_function);

	/*! \brief This function returns ASN1_INTEGER::function.
	 *  \return The value of ASN1_INTEGER::function.
	 */
	unsigned long get_function() const;


	/*! \brief This function sets ASN1_INTEGER::lib.
	 *  \param c_lib [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_lib(unsigned long c_lib);

	/*! \brief This function returns ASN1_INTEGER::lib.
	 *  \return The value of ASN1_INTEGER::lib.
	 */
	unsigned long get_lib() const;


	/*! \brief This function sets ASN1_INTEGER::line.
	 *  \param c_line [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_line(unsigned long c_line);

	/*! \brief This function returns ASN1_INTEGER::line.
	 *  \return The value of ASN1_INTEGER::line.
	 */
	unsigned long get_line() const;


	/*! \brief This function copies a ErrorEntry to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ErrorEntry & other);
	/*! \brief This member is an empty instance of ErrorEntry.
	 */
	static ErrorEntry EmptyInstance;

private:
	unsigned long m_code;
	mString m_data;
	mString m_file;
	unsigned long m_function;
	unsigned long m_lib;
	unsigned long m_line;
	void resetAll();
	void freeAll();
protected:
};


/*! \brief This function converts OpenSSL's errors stack into a list of errors.
 *  \param errors [OUT] The list of errors.
 *  \return The list of errors.
 */
void ERR_to_ERROR_ENTRIES(mVector< ErrorEntry > & errors);

/*! \brief This function converts a stack of errors into a readable string
 *  \param errors [IN] The stack of errors.
 *  \param error [OUT] The error.
 *  \param OnlyString [IN] Return only the errors string.
 */
void ERROR_ENTRIES_to_string(const vector<ErrorEntry> & errors, mString & error, bool OnlyString = false);

#endif // #ifndef ASN1ERR_H
