/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef ASN1HASHVAL_H
#define ASN1HASHVAL_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <openssl/asn1.h>
#include <openssl/asn1t.h>
#include "Asn1Helper.h"

/*!
	This structure defines an extension
*/
typedef struct st_EXTENSION_VALUE
{
	ASN1_UTF8STRING * name;		//!< The name
	ASN1_UTF8STRING * value;	//!< The value
}EXTENSION_VALUE;
DECLARE_ASN1_ITEM(EXTENSION_VALUE)

/*!
	This structure defines a DN specifications
*/
typedef struct st_DN_SPECS
{
	ASN1_UTF8STRING * Name;		//!< The name of the DN field
	ASN1_UTF8STRING * Default;	//!< The default value of the DN field
	ASN1_UTF8STRING * Comment;	//!< The comment of the DN field
	ASN1_INTEGER * Min;			//!< The min length of the DN field
	ASN1_INTEGER * Max;			//!< The max length of the DN field
}DN_SPECS;
DECLARE_ASN1_ITEM(DN_SPECS)

/*!
	This enumeration defines the type of DN policy
*/
typedef enum DN_TYPE
{
	DN_TYPE_OPTIONAL = 0,		//!< The DN field is optional
	DN_TYPE_SUPPLIED = 1,		//!< The DN must be supplied
	DN_TYPE_MATCH = 2,			//!< The DN must a value declared in the list
	DN_TYPE_UNKNOWN = 3			//!< Unknown policy type
};

#define STR_OPTIONAL	"optional"	//!< The string representation of DN_TYPE_OPTIONAL
#define STR_SUPPLIED	"supplied"	//!< The string representation of DN_TYPE_SUPPLIED
#define STR_MATCH		"match:"	//!< The string representation of DN_TYPE_MATCH
#define NUM_DN_TYPE 3				//!< The number of policy representation strings

/*!
	This structure defines an policy
*/
typedef struct st_POLICY_VALUE
{
	ASN1_UTF8STRING * name;		//!< The name
	ASN1_UTF8STRING * value;	//!< The value
}POLICY_VALUE;
DECLARE_ASN1_ITEM(POLICY_VALUE)


/*!
	This structure represents a plugin option
*/
typedef struct st_PLUG_OPTION
{
	ASN1_UTF8STRING * name;		//!< The name
	ASN1_UTF8STRING * value;	//!< The value
}PLUG_OPTION;
DECLARE_ASN1_ITEM(PLUG_OPTION)



/*!
  This class is the representation of EXTENSION_VALUE
*/
class ExtensionValue : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ExtensionValue();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ExtensionValue(const ExtensionValue & other);

	/*! \brief This is the destructor.
	 */
	~ExtensionValue();


	/*! \brief This function converts a EXTENSION_VALUE to a ExtensionValue.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const EXTENSION_VALUE * Datas);

	/*! \brief This function converts a ExtensionValue to a EXTENSION_VALUE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(EXTENSION_VALUE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of EXTENSION_VALUE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	mString & get_name();


	/*! \brief This function sets ASN1_UTF8STRING::value.
	 *  \param c_value [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_value(const mString & c_value);

	/*! \brief This function returns ASN1_UTF8STRING::value.
	 *  \return The value of ASN1_UTF8STRING::value.
	 */
	const mString & get_value() const;

	/*! \brief This function returns ASN1_UTF8STRING::value.
	 *  \return The value of ASN1_UTF8STRING::value.
	 */
	mString & get_value();


	/*! \brief This function copies a ExtensionValue to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ExtensionValue & other);
	/*! \brief This member is an empty instance of ExtensionValue.
	 */
	static ExtensionValue EmptyInstance;

private:
	mString m_name;
	mString m_value;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of POLICY_VALUE
*/
class PolicyValue : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	PolicyValue();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	PolicyValue(const PolicyValue & other);

	/*! \brief This is the destructor.
	 */
	~PolicyValue();


	/*! \brief This function converts a POLICY_VALUE to a PolicyValue.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const POLICY_VALUE * Datas);

	/*! \brief This function converts a PolicyValue to a POLICY_VALUE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(POLICY_VALUE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of POLICY_VALUE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	mString & get_name();


	/*! \brief This function sets ASN1_UTF8STRING::value.
	 *  \param c_value [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_value(const mString & c_value);

	/*! \brief This function returns ASN1_UTF8STRING::value.
	 *  \return The value of ASN1_UTF8STRING::value.
	 */
	const mString & get_value() const;

	/*! \brief This function returns ASN1_UTF8STRING::value.
	 *  \return The value of ASN1_UTF8STRING::value.
	 */
	mString & get_value();


	/*! \brief This function copies a PolicyValue to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const PolicyValue & other);
	/*! \brief This member is an empty instance of PolicyValue.
	 */
	static PolicyValue EmptyInstance;

private:
	mString m_name;
	mString m_value;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PLUG_OPTION
*/
class PlugOption : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	PlugOption();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	PlugOption(const PlugOption & other);

	/*! \brief This is the destructor.
	 */
	~PlugOption();


	/*! \brief This function converts a PLUG_OPTION to a PlugOption.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PLUG_OPTION * Datas);

	/*! \brief This function converts a PlugOption to a PLUG_OPTION.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PLUG_OPTION ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PLUG_OPTION.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	mString & get_name();


	/*! \brief This function sets ASN1_UTF8STRING::value.
	 *  \param c_value [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_value(const mString & c_value);

	/*! \brief This function returns ASN1_UTF8STRING::value.
	 *  \return The value of ASN1_UTF8STRING::value.
	 */
	const mString & get_value() const;

	/*! \brief This function returns ASN1_UTF8STRING::value.
	 *  \return The value of ASN1_UTF8STRING::value.
	 */
	mString & get_value();


	/*! \brief This function copies a PlugOption to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const PlugOption & other);
	/*! \brief This member is an empty instance of PlugOption.
	 */
	static PlugOption EmptyInstance;

private:
	mString m_name;
	mString m_value;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of DN_SPECS
*/
class DnSpecs : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	DnSpecs();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	DnSpecs(const DnSpecs & other);

	/*! \brief This is the destructor.
	 */
	~DnSpecs();


	/*! \brief This function converts a DN_SPECS to a DnSpecs.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const DN_SPECS * Datas);

	/*! \brief This function converts a DnSpecs to a DN_SPECS.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(DN_SPECS ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of DN_SPECS.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::Comment.
	 *  \param c_comment [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_comment(const mString & c_comment);

	/*! \brief This function returns ASN1_UTF8STRING::Comment.
	 *  \return The value of ASN1_UTF8STRING::Comment.
	 */
	const mString & get_comment() const;

	/*! \brief This function returns ASN1_UTF8STRING::Comment.
	 *  \return The value of ASN1_UTF8STRING::Comment.
	 */
	mString & get_comment();


	/*! \brief This function sets ASN1_UTF8STRING::Default.
	 *  \param c_default [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_default(const mString & c_default);

	/*! \brief This function returns ASN1_UTF8STRING::Default.
	 *  \return The value of ASN1_UTF8STRING::Default.
	 */
	const mString & get_default() const;

	/*! \brief This function returns ASN1_UTF8STRING::Default.
	 *  \return The value of ASN1_UTF8STRING::Default.
	 */
	mString & get_default();


	/*! \brief This function sets ASN1_INTEGER::Max.
	 *  \param c_max [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_max(unsigned long c_max);

	/*! \brief This function returns ASN1_INTEGER::Max.
	 *  \return The value of ASN1_INTEGER::Max.
	 */
	unsigned long get_max() const;


	/*! \brief This function sets ASN1_INTEGER::Min.
	 *  \param c_min [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_min(unsigned long c_min);

	/*! \brief This function returns ASN1_INTEGER::Min.
	 *  \return The value of ASN1_INTEGER::Min.
	 */
	unsigned long get_min() const;


	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function copies a DnSpecs to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const DnSpecs & other);
	/*! \brief This member is an empty instance of DnSpecs.
	 */
	static DnSpecs EmptyInstance;

private:
	mString m_comment;
	mString m_default;
	unsigned long m_max;
	unsigned long m_min;
	mString m_name;
	void resetAll();
	void freeAll();
protected:
};


#endif // #ifndef ASN1HASHVAL_H
