/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#include "Asn1Resp.h"
#include <PKI_ERR.h>
#include <openssl/asn1t.h>

ASN1_SEQUENCE(INTERNAL_CA_CERT) = {
	ASN1_SIMPLE(INTERNAL_CA_CERT, ca_type, ASN1_INTEGER),
	ASN1_SIMPLE(INTERNAL_CA_CERT, serial, ASN1_INTEGER),
	ASN1_SIMPLE(INTERNAL_CA_CERT, state, ASN1_INTEGER),
	ASN1_SIMPLE(INTERNAL_CA_CERT, dn, ASN1_UTF8STRING),
	ASN1_SIMPLE(INTERNAL_CA_CERT, uid, ASN1_UTF8STRING),
	ASN1_SIMPLE(INTERNAL_CA_CERT, begin_date, ASN1_INTEGER),
	ASN1_SIMPLE(INTERNAL_CA_CERT, end_date, ASN1_INTEGER),
	ASN1_EXP_OPT(INTERNAL_CA_CERT, rev_date, ASN1_INTEGER, 0),
	ASN1_EXP(INTERNAL_CA_CERT, cert, X509, 1),
	ASN1_EXP_OPT(INTERNAL_CA_CERT, susp_date, ASN1_INTEGER, 2),
}ASN1_SEQUENCE_END(INTERNAL_CA_CERT)
ASN1_SEQUENCE(LDAP_OBJECT) = {
	ASN1_SIMPLE(LDAP_OBJECT, name, ASN1_UTF8STRING),
	ASN1_SIMPLE(LDAP_OBJECT, value, ASN1_UTF8STRING),
}ASN1_SEQUENCE_END(LDAP_OBJECT)
ASN1_SEQUENCE(LDAP_RESULT) = {
	ASN1_SIMPLE(LDAP_RESULT, rdn, ASN1_UTF8STRING),
	ASN1_SIMPLE(LDAP_RESULT, uid, ASN1_UTF8STRING),
	ASN1_SEQUENCE_OF(LDAP_RESULT, objects, LDAP_OBJECT),
}ASN1_SEQUENCE_END(LDAP_RESULT)
ASN1_CHOICE(ADMIN_RESPONSE_BODY) = {
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.errors, ERROR_ENTRY, ADMIN_RESP_TYPE_ERRORS),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.other, ASN1_NULL, ADMIN_RESP_TYPE_NONE),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.usertype, ASN1_INTEGER, ADMIN_RESP_TYPE_USER_TYPE),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.users, USER_ENTRY, ADMIN_RESP_TYPE_ENUM_USERS),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.entities, ENTITY_ENTRY_INFO, ADMIN_RESP_TYPE_ENUM_ENTITIES),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.logs, LOG_ENTRY, ADMIN_RESP_TYPE_ENUM_LOGS),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.cre_entity, ENTITY_CREATION_RESP, ADMIN_RESP_TYPE_CREATE_ENTITY),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.sign_entity, ENTITY_SIGNATURE_RESP, ADMIN_RESP_TYPE_SIGN_ENTITY),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.users_cert, USER_ENTRY_CERT, ADMIN_RESP_TYPE_ENUM_USERS_CERT),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.create_pki_user, CREATE_PKI_USER_RESPONSE, ADMIN_RESP_TYPE_CREATE_PKI_USER),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.status, ASN1_INTEGER, ADMIN_RESP_TYPE_STATUS),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.acls_conf, X509_ACL, ADMIN_RESP_TYPE_ACLS),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.my_acl, ASN1_BIT_STRING, ADMIN_RESP_TYPE_MY_ACL),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.conf, EXPORTED_PKI_CONF, ADMIN_RESP_TYPE_CONF),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.my_conf, ENTITY_CONF_CRYPTED, ADMIN_RESP_TYPE_MY_CONF),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.repositories, REP_ENTRY_INFO, ADMIN_RESP_TYPE_GET_REPOSITORIES),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.logs_type, ASN1_INTEGER, ADMIN_RESP_TYPE_LOGS_TYPE),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.entity_conf, ENTITY_CONF_BODY, ADMIN_RESP_TYPE_ENTITY_CONF),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.mail_conf, EMAIL_CONF, ADMIN_RESP_TYPE_MAIL_CONF),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.audits, ENTITY_AUDIT_ENTRY, ADMIN_RESP_TYPE_AUDITS),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.entities_links, ENTITY_LINKS, ADMIN_RESP_TYPE_ENTITIES_LINKS),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.object_reqs, CRYPTED_NEWPKI_REQUESTS, ADMIN_RESP_TYPE_REQS),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.certs, INTERNAL_CA_CERT, ADMIN_RESP_TYPE_CERTS),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.p7b, PKCS7, ADMIN_RESP_TYPE_P7B),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.csr, X509_REQ, ADMIN_RESP_TYPE_CSR),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.crls, X509_CRL, ADMIN_RESP_TYPE_CRLS),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.groups, USERS_GROUP, ADMIN_RESP_TYPE_GROUPS),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.id, ASN1_INTEGER, ADMIN_RESP_TYPE_PROFILE_ID),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.transactionID, ASN1_OCTET_STRING, ADMIN_RESP_TYPE_TRANSACTION_ID),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.object_resps, CRYPTED_NEWPKI_RESPONSES, ADMIN_RESP_TYPE_RESPS),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.profiles, NEWPKI_PROFILE_DATAS, ADMIN_RESP_TYPE_PROFILES),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.local_entity_conf, ENTITY_CONF, ADMIN_RESP_TYPE_LOCAL_ENTITY_CONF),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.known_objects, OBJECTS_LIST_SYNCHRO, ADMIN_RESP_TYPE_KNOWN_OBJECTS),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.id, ASN1_INTEGER, ADMIN_RESP_TYPE_REQ_ID),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.cert, X509, ADMIN_RESP_TYPE_CERT),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.status, ASN1_INTEGER, ADMIN_RESP_TYPE_LOGS_COUNT),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.ldap_results, LDAP_RESULT, ADMIN_RESP_TYPE_LDAP_RESULTS),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.status, ASN1_INTEGER, ADMIN_RESP_TYPE_OBJECT_STATUS),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.dn, X509_NAME, ADMIN_RESP_TYPE_DN_VALIDATION),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.id, ASN1_INTEGER, ADMIN_RESP_TYPE_WEBUSER_ID),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.webuser_certs, NEWPKI_PROFILE_DATAS_CERT, ADMIN_RESP_TYPE_WEBUSER_CERTS),
	ASN1_EXP(ADMIN_RESPONSE_BODY, d.dn, X509_NAME, ADMIN_RESP_TYPE_DN),
	ASN1_EXP_SEQUENCE_OF(ADMIN_RESPONSE_BODY, d.ee_users, NEWPKI_EE_USER, ADMIN_RESP_TYPE_EE_USERS),
} ASN1_CHOICE_END(ADMIN_RESPONSE_BODY)
ASN1_SEQUENCE(ADMIN_RESPONSE) = {
	ASN1_SIMPLE(ADMIN_RESPONSE, body, ADMIN_RESPONSE_BODY),
	ASN1_SIMPLE(ADMIN_RESPONSE, signing_cert, X509),
	ASN1_SIMPLE(ADMIN_RESPONSE, sig_algo, X509_ALGOR),
	ASN1_SIMPLE(ADMIN_RESPONSE, signature, ASN1_BIT_STRING),
}ASN1_SEQUENCE_END(ADMIN_RESPONSE)
InternalCaCert InternalCaCert::EmptyInstance;
bool InternalCaCert::set_beginDate(unsigned long c_beginDate)
{
	m_beginDate = c_beginDate;
	return true;
}

unsigned long InternalCaCert::get_beginDate() const
{
	return m_beginDate;
}

bool InternalCaCert::set_caType(unsigned long c_caType)
{
	m_caType = c_caType;
	return true;
}

unsigned long InternalCaCert::get_caType() const
{
	return m_caType;
}

bool InternalCaCert::set_cert(const PKI_CERT & c_cert)
{
	m_cert = c_cert;
	return true;
}

const PKI_CERT & InternalCaCert::get_cert() const
{
	return m_cert;
}

PKI_CERT & InternalCaCert::get_cert()
{
	return m_cert;
}

bool InternalCaCert::set_dn(const mString & c_dn)
{
	m_dn = c_dn;
	return true;
}

const mString & InternalCaCert::get_dn() const
{
	return m_dn;
}

mString & InternalCaCert::get_dn()
{
	return m_dn;
}

bool InternalCaCert::set_endDate(unsigned long c_endDate)
{
	m_endDate = c_endDate;
	return true;
}

unsigned long InternalCaCert::get_endDate() const
{
	return m_endDate;
}

bool InternalCaCert::set_revDate(unsigned long c_revDate)
{
	m_revDate = c_revDate;
	return true;
}

unsigned long InternalCaCert::get_revDate() const
{
	return m_revDate;
}

bool InternalCaCert::set_serial(unsigned long c_serial)
{
	m_serial = c_serial;
	return true;
}

unsigned long InternalCaCert::get_serial() const
{
	return m_serial;
}

bool InternalCaCert::set_state(unsigned long c_state)
{
	m_state = c_state;
	return true;
}

unsigned long InternalCaCert::get_state() const
{
	return m_state;
}

bool InternalCaCert::set_suspDate(unsigned long c_suspDate)
{
	m_suspDate = c_suspDate;
	return true;
}

unsigned long InternalCaCert::get_suspDate() const
{
	return m_suspDate;
}

bool InternalCaCert::set_uid(const mString & c_uid)
{
	m_uid = c_uid;
	return true;
}

const mString & InternalCaCert::get_uid() const
{
	return m_uid;
}

mString & InternalCaCert::get_uid()
{
	return m_uid;
}

InternalCaCert::InternalCaCert():NewPKIObject()
{
	resetAll();
}

InternalCaCert::InternalCaCert(const InternalCaCert & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

InternalCaCert::~InternalCaCert()
{
	Clear();
}

void InternalCaCert::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void InternalCaCert::freeAll()
{
}

void InternalCaCert::resetAll()
{
	m_beginDate = 0;
	m_caType = 0;
	m_cert.Clear();
	m_dn = "";
	m_endDate = 0;
	m_revDate = 0;
	m_serial = 0;
	m_state = 0;
	m_suspDate = 0;
	m_uid = "";
}

bool InternalCaCert::load_Datas(const INTERNAL_CA_CERT * Datas)
{
	Clear();
	if(Datas->begin_date)
	{
		m_beginDate = ASN1_INTEGER_GET(Datas->begin_date);
	}
	if(Datas->ca_type)
	{
		m_caType = ASN1_INTEGER_GET(Datas->ca_type);
	}
	if(Datas->cert)
	{
		if(!m_cert.load_Datas(Datas->cert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->dn)
	{
		m_dn = Datas->dn;
	}
	if(Datas->end_date)
	{
		m_endDate = ASN1_INTEGER_GET(Datas->end_date);
	}
	if(Datas->rev_date)
	{
		m_revDate = ASN1_INTEGER_GET(Datas->rev_date);
	}
	if(Datas->serial)
	{
		m_serial = ASN1_INTEGER_GET(Datas->serial);
	}
	if(Datas->state)
	{
		m_state = ASN1_INTEGER_GET(Datas->state);
	}
	if(Datas->susp_date)
	{
		m_suspDate = ASN1_INTEGER_GET(Datas->susp_date);
	}
	if(Datas->uid)
	{
		m_uid = Datas->uid;
	}
	m_isOk=true;
	return true;
}

bool InternalCaCert::give_Datas(INTERNAL_CA_CERT ** Datas) const
{
	if(!(*Datas) && !(*Datas = (INTERNAL_CA_CERT*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->begin_date && !((*Datas)->begin_date = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->begin_date, m_beginDate) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->begin_date);
		(*Datas)->begin_date = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->ca_type && !((*Datas)->ca_type = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->ca_type, m_caType) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->ca_type);
		(*Datas)->ca_type = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->cert && !((*Datas)->cert = (X509*)ASN1_item_new(ASN1_ITEM_rptr(X509))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_cert.give_Datas(&(*Datas)->cert))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->cert, ASN1_ITEM_rptr(X509));
		(*Datas)->cert = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->dn && !((*Datas)->dn = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_dn.c_ASN1_UTF8STRING(&(*Datas)->dn))
	{
		ASN1_UTF8STRING_free((*Datas)->dn);
		(*Datas)->dn = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->end_date && !((*Datas)->end_date = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->end_date, m_endDate) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->end_date);
		(*Datas)->end_date = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->rev_date && !((*Datas)->rev_date = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->rev_date, m_revDate) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->rev_date);
		(*Datas)->rev_date = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->serial && !((*Datas)->serial = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->serial, m_serial) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->serial);
		(*Datas)->serial = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->state && !((*Datas)->state = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->state, m_state) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->state);
		(*Datas)->state = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->susp_date && !((*Datas)->susp_date = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->susp_date, m_suspDate) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->susp_date);
		(*Datas)->susp_date = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->uid && !((*Datas)->uid = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_uid.c_ASN1_UTF8STRING(&(*Datas)->uid))
	{
		ASN1_UTF8STRING_free((*Datas)->uid);
		(*Datas)->uid = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool InternalCaCert::operator=(const InternalCaCert & other)
{
	Clear();
	m_beginDate = other.m_beginDate;
	m_caType = other.m_caType;
	m_cert = other.m_cert;
	m_dn = other.m_dn;
	m_endDate = other.m_endDate;
	m_revDate = other.m_revDate;
	m_serial = other.m_serial;
	m_state = other.m_state;
	m_suspDate = other.m_suspDate;
	m_uid = other.m_uid;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * InternalCaCert::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(INTERNAL_CA_CERT);
}
LdapObject LdapObject::EmptyInstance;
bool LdapObject::set_name(const mString & c_name)
{
	m_name = c_name;
	return true;
}

const mString & LdapObject::get_name() const
{
	return m_name;
}

mString & LdapObject::get_name()
{
	return m_name;
}

bool LdapObject::set_value(const mString & c_value)
{
	m_value = c_value;
	return true;
}

const mString & LdapObject::get_value() const
{
	return m_value;
}

mString & LdapObject::get_value()
{
	return m_value;
}

LdapObject::LdapObject():NewPKIObject()
{
	resetAll();
}

LdapObject::LdapObject(const LdapObject & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

LdapObject::~LdapObject()
{
	Clear();
}

void LdapObject::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void LdapObject::freeAll()
{
}

void LdapObject::resetAll()
{
	m_name = "";
	m_value = "";
}

bool LdapObject::load_Datas(const LDAP_OBJECT * Datas)
{
	Clear();
	if(Datas->name)
	{
		m_name = Datas->name;
	}
	if(Datas->value)
	{
		m_value = Datas->value;
	}
	m_isOk=true;
	return true;
}

bool LdapObject::give_Datas(LDAP_OBJECT ** Datas) const
{
	if(!(*Datas) && !(*Datas = (LDAP_OBJECT*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->name && !((*Datas)->name = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_name.c_ASN1_UTF8STRING(&(*Datas)->name))
	{
		ASN1_UTF8STRING_free((*Datas)->name);
		(*Datas)->name = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->value && !((*Datas)->value = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_value.c_ASN1_UTF8STRING(&(*Datas)->value))
	{
		ASN1_UTF8STRING_free((*Datas)->value);
		(*Datas)->value = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool LdapObject::operator=(const LdapObject & other)
{
	Clear();
	m_name = other.m_name;
	m_value = other.m_value;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * LdapObject::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(LDAP_OBJECT);
}
LdapResult LdapResult::EmptyInstance;
bool LdapResult::set_objects(const mVector< LdapObject > & c_objects)
{
	m_objects = c_objects;
	return true;
}

const mVector< LdapObject > & LdapResult::get_objects() const
{
	return m_objects;
}

mVector< LdapObject > & LdapResult::get_objects()
{
	return m_objects;
}

bool LdapResult::set_rdn(const mString & c_rdn)
{
	m_rdn = c_rdn;
	return true;
}

const mString & LdapResult::get_rdn() const
{
	return m_rdn;
}

mString & LdapResult::get_rdn()
{
	return m_rdn;
}

bool LdapResult::set_uid(const mString & c_uid)
{
	m_uid = c_uid;
	return true;
}

const mString & LdapResult::get_uid() const
{
	return m_uid;
}

mString & LdapResult::get_uid()
{
	return m_uid;
}

LdapResult::LdapResult():NewPKIObject()
{
	resetAll();
}

LdapResult::LdapResult(const LdapResult & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

LdapResult::~LdapResult()
{
	Clear();
}

void LdapResult::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void LdapResult::freeAll()
{
}

void LdapResult::resetAll()
{
	m_objects.clear();
	m_rdn = "";
	m_uid = "";
}

bool LdapResult::load_Datas(const LDAP_RESULT * Datas)
{
	Clear();
	int i;
	LDAP_OBJECT * currobjects;
	if(Datas->objects)
	{
		for(i=0; i<SKM_sk_num(LDAP_OBJECT, Datas->objects); i++)
		{
			currobjects = SKM_sk_value(LDAP_OBJECT, Datas->objects, i);
			if(!currobjects)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_objects.insert(m_objects.begin() + i);
			if(!m_objects[i].load_Datas(currobjects))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	if(Datas->rdn)
	{
		m_rdn = Datas->rdn;
	}
	if(Datas->uid)
	{
		m_uid = Datas->uid;
	}
	m_isOk=true;
	return true;
}

bool LdapResult::give_Datas(LDAP_RESULT ** Datas) const
{
	if(!(*Datas) && !(*Datas = (LDAP_RESULT*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	LDAP_OBJECT * currobjects;
	size_t i;
	if(!(*Datas)->objects && !((*Datas)->objects = SKM_sk_new_null(LDAP_OBJECT)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_objects.size() ; i++)
	{
		currobjects = NULL;
		if(!m_objects[i].give_Datas(&currobjects))
		{
			ASN1_item_free((ASN1_VALUE*)currobjects, ASN1_ITEM_rptr(LDAP_OBJECT));
			currobjects = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(LDAP_OBJECT, (*Datas)->objects, currobjects) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currobjects, ASN1_ITEM_rptr(LDAP_OBJECT));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	if(!(*Datas)->rdn && !((*Datas)->rdn = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_rdn.c_ASN1_UTF8STRING(&(*Datas)->rdn))
	{
		ASN1_UTF8STRING_free((*Datas)->rdn);
		(*Datas)->rdn = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->uid && !((*Datas)->uid = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_uid.c_ASN1_UTF8STRING(&(*Datas)->uid))
	{
		ASN1_UTF8STRING_free((*Datas)->uid);
		(*Datas)->uid = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool LdapResult::operator=(const LdapResult & other)
{
	Clear();
	m_objects = other.m_objects;
	m_rdn = other.m_rdn;
	m_uid = other.m_uid;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * LdapResult::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(LDAP_RESULT);
}
AdminResponseBody AdminResponseBody::EmptyInstance;
bool AdminResponseBody::set_type(int c_type)
{
	Clear();
	m_type = c_type;
	if(!malloc_byType(m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

int AdminResponseBody::get_type() const
{
	 return m_type;
}

bool AdminResponseBody::set_aclsConf(const X509Acl & c_aclsConf)
{
	if(m_type != 11)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_aclsConf) = c_aclsConf;
	m_isOk=true;
	return true;
}

const X509Acl & AdminResponseBody::get_aclsConf() const
{
	if((m_type != 11) || !m_aclsConf)
	{
		return X509Acl::EmptyInstance;
	}
	return (*m_aclsConf);
}

X509Acl & AdminResponseBody::get_aclsConf()
{
	if((m_type != 11) || !m_aclsConf)
	{
		return X509Acl::EmptyInstance;
	}
	return (*m_aclsConf);
}

bool AdminResponseBody::set_audits(const mVector< EntityAuditEntry > & c_audits)
{
	if(m_type != 19)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_audits) = c_audits;
	m_isOk=true;
	return true;
}

const mVector< EntityAuditEntry > & AdminResponseBody::get_audits() const
{
	if((m_type != 19) || !m_audits)
	{
		return mVector< EntityAuditEntry >::EmptyInstance;
	}
	return (*m_audits);
}

mVector< EntityAuditEntry > & AdminResponseBody::get_audits()
{
	if((m_type != 19) || !m_audits)
	{
		return mVector< EntityAuditEntry >::EmptyInstance;
	}
	return (*m_audits);
}

bool AdminResponseBody::set_cert(const PKI_CERT & c_cert)
{
	if(m_type != 34)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_cert) = c_cert;
	m_isOk=true;
	return true;
}

const PKI_CERT & AdminResponseBody::get_cert() const
{
	if((m_type != 34) || !m_cert)
	{
		return PKI_CERT::EmptyInstance;
	}
	return (*m_cert);
}

PKI_CERT & AdminResponseBody::get_cert()
{
	if((m_type != 34) || !m_cert)
	{
		return PKI_CERT::EmptyInstance;
	}
	return (*m_cert);
}

bool AdminResponseBody::set_certs(const mVector< InternalCaCert > & c_certs)
{
	if(m_type != 22)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_certs) = c_certs;
	m_isOk=true;
	return true;
}

const mVector< InternalCaCert > & AdminResponseBody::get_certs() const
{
	if((m_type != 22) || !m_certs)
	{
		return mVector< InternalCaCert >::EmptyInstance;
	}
	return (*m_certs);
}

mVector< InternalCaCert > & AdminResponseBody::get_certs()
{
	if((m_type != 22) || !m_certs)
	{
		return mVector< InternalCaCert >::EmptyInstance;
	}
	return (*m_certs);
}

bool AdminResponseBody::set_conf(const ExportedPkiConf & c_conf)
{
	if(m_type != 13)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_conf) = c_conf;
	m_isOk=true;
	return true;
}

const ExportedPkiConf & AdminResponseBody::get_conf() const
{
	if((m_type != 13) || !m_conf)
	{
		return ExportedPkiConf::EmptyInstance;
	}
	return (*m_conf);
}

ExportedPkiConf & AdminResponseBody::get_conf()
{
	if((m_type != 13) || !m_conf)
	{
		return ExportedPkiConf::EmptyInstance;
	}
	return (*m_conf);
}

bool AdminResponseBody::set_creEntity(const EntityCreationResp & c_creEntity)
{
	if(m_type != 6)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_creEntity) = c_creEntity;
	m_isOk=true;
	return true;
}

const EntityCreationResp & AdminResponseBody::get_creEntity() const
{
	if((m_type != 6) || !m_creEntity)
	{
		return EntityCreationResp::EmptyInstance;
	}
	return (*m_creEntity);
}

EntityCreationResp & AdminResponseBody::get_creEntity()
{
	if((m_type != 6) || !m_creEntity)
	{
		return EntityCreationResp::EmptyInstance;
	}
	return (*m_creEntity);
}

bool AdminResponseBody::set_createPkiUser(const CreatePkiUserResponse & c_createPkiUser)
{
	if(m_type != 9)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_createPkiUser) = c_createPkiUser;
	m_isOk=true;
	return true;
}

const CreatePkiUserResponse & AdminResponseBody::get_createPkiUser() const
{
	if((m_type != 9) || !m_createPkiUser)
	{
		return CreatePkiUserResponse::EmptyInstance;
	}
	return (*m_createPkiUser);
}

CreatePkiUserResponse & AdminResponseBody::get_createPkiUser()
{
	if((m_type != 9) || !m_createPkiUser)
	{
		return CreatePkiUserResponse::EmptyInstance;
	}
	return (*m_createPkiUser);
}

bool AdminResponseBody::set_crls(const mVector< PKI_CRL > & c_crls)
{
	if(m_type != 25)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_crls) = c_crls;
	m_isOk=true;
	return true;
}

const mVector< PKI_CRL > & AdminResponseBody::get_crls() const
{
	if((m_type != 25) || !m_crls)
	{
		return mVector< PKI_CRL >::EmptyInstance;
	}
	return (*m_crls);
}

mVector< PKI_CRL > & AdminResponseBody::get_crls()
{
	if((m_type != 25) || !m_crls)
	{
		return mVector< PKI_CRL >::EmptyInstance;
	}
	return (*m_crls);
}

bool AdminResponseBody::set_csr(const PKI_CSR & c_csr)
{
	if(m_type != 24)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_csr) = c_csr;
	m_isOk=true;
	return true;
}

const PKI_CSR & AdminResponseBody::get_csr() const
{
	if((m_type != 24) || !m_csr)
	{
		return PKI_CSR::EmptyInstance;
	}
	return (*m_csr);
}

PKI_CSR & AdminResponseBody::get_csr()
{
	if((m_type != 24) || !m_csr)
	{
		return PKI_CSR::EmptyInstance;
	}
	return (*m_csr);
}

bool AdminResponseBody::set_dn(const X509_NAME * c_dn)
{
	if(m_type != 38 && m_type != 41)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(m_dn)
		ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
	m_dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), (void*)c_dn);
	if(!m_dn)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	m_isOk=true;
	return true;
}

const X509_NAME * AdminResponseBody::get_dn() const
{
	if((m_type != 38 && m_type != 41) || !m_dn)
	{
		return NULL;
	}
	if(!m_dn)
		((AdminResponseBody*)this)->m_dn = (X509_NAME*)ASN1_item_new(ASN1_ITEM_rptr(X509_NAME));
	return m_dn;
}

X509_NAME * AdminResponseBody::get_dn()
{
	if((m_type != 38 && m_type != 41) || !m_dn)
	{
		return NULL;
	}
	if(!m_dn)
		m_dn = (X509_NAME*)ASN1_item_new(ASN1_ITEM_rptr(X509_NAME));
	return m_dn;
}

bool AdminResponseBody::set_eeUsers(const mVector< NewpkiEeUser > & c_eeUsers)
{
	if(m_type != 42)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_eeUsers) = c_eeUsers;
	m_isOk=true;
	return true;
}

const mVector< NewpkiEeUser > & AdminResponseBody::get_eeUsers() const
{
	if((m_type != 42) || !m_eeUsers)
	{
		return mVector< NewpkiEeUser >::EmptyInstance;
	}
	return (*m_eeUsers);
}

mVector< NewpkiEeUser > & AdminResponseBody::get_eeUsers()
{
	if((m_type != 42) || !m_eeUsers)
	{
		return mVector< NewpkiEeUser >::EmptyInstance;
	}
	return (*m_eeUsers);
}

bool AdminResponseBody::set_entities(const mVector< EntityEntryInfo > & c_entities)
{
	if(m_type != 4)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_entities) = c_entities;
	m_isOk=true;
	return true;
}

const mVector< EntityEntryInfo > & AdminResponseBody::get_entities() const
{
	if((m_type != 4) || !m_entities)
	{
		return mVector< EntityEntryInfo >::EmptyInstance;
	}
	return (*m_entities);
}

mVector< EntityEntryInfo > & AdminResponseBody::get_entities()
{
	if((m_type != 4) || !m_entities)
	{
		return mVector< EntityEntryInfo >::EmptyInstance;
	}
	return (*m_entities);
}

bool AdminResponseBody::set_entitiesLinks(const mVector< EntityLinks > & c_entitiesLinks)
{
	if(m_type != 20)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_entitiesLinks) = c_entitiesLinks;
	m_isOk=true;
	return true;
}

const mVector< EntityLinks > & AdminResponseBody::get_entitiesLinks() const
{
	if((m_type != 20) || !m_entitiesLinks)
	{
		return mVector< EntityLinks >::EmptyInstance;
	}
	return (*m_entitiesLinks);
}

mVector< EntityLinks > & AdminResponseBody::get_entitiesLinks()
{
	if((m_type != 20) || !m_entitiesLinks)
	{
		return mVector< EntityLinks >::EmptyInstance;
	}
	return (*m_entitiesLinks);
}

bool AdminResponseBody::set_entityConf(const EntityConfBody & c_entityConf)
{
	if(m_type != 17)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_entityConf) = c_entityConf;
	m_isOk=true;
	return true;
}

const EntityConfBody & AdminResponseBody::get_entityConf() const
{
	if((m_type != 17) || !m_entityConf)
	{
		return EntityConfBody::EmptyInstance;
	}
	return (*m_entityConf);
}

EntityConfBody & AdminResponseBody::get_entityConf()
{
	if((m_type != 17) || !m_entityConf)
	{
		return EntityConfBody::EmptyInstance;
	}
	return (*m_entityConf);
}

bool AdminResponseBody::set_errors(const mVector< ErrorEntry > & c_errors)
{
	if(m_type != 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_errors) = c_errors;
	m_isOk=true;
	return true;
}

const mVector< ErrorEntry > & AdminResponseBody::get_errors() const
{
	if((m_type != 0) || !m_errors)
	{
		return mVector< ErrorEntry >::EmptyInstance;
	}
	return (*m_errors);
}

mVector< ErrorEntry > & AdminResponseBody::get_errors()
{
	if((m_type != 0) || !m_errors)
	{
		return mVector< ErrorEntry >::EmptyInstance;
	}
	return (*m_errors);
}

bool AdminResponseBody::set_groups(const mVector< UsersGroup > & c_groups)
{
	if(m_type != 26)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_groups) = c_groups;
	m_isOk=true;
	return true;
}

const mVector< UsersGroup > & AdminResponseBody::get_groups() const
{
	if((m_type != 26) || !m_groups)
	{
		return mVector< UsersGroup >::EmptyInstance;
	}
	return (*m_groups);
}

mVector< UsersGroup > & AdminResponseBody::get_groups()
{
	if((m_type != 26) || !m_groups)
	{
		return mVector< UsersGroup >::EmptyInstance;
	}
	return (*m_groups);
}

bool AdminResponseBody::set_id(unsigned long c_id)
{
	if(m_type != 27 && m_type != 33 && m_type != 39)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_id) = c_id;
	m_isOk=true;
	return true;
}

unsigned long AdminResponseBody::get_id() const
{
	if((m_type != 27 && m_type != 33 && m_type != 39) || !m_id)
	{
		return 0;
	}
	return (*m_id);
}

bool AdminResponseBody::set_knownObjects(const ObjectsListSynchro & c_knownObjects)
{
	if(m_type != 32)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_knownObjects) = c_knownObjects;
	m_isOk=true;
	return true;
}

const ObjectsListSynchro & AdminResponseBody::get_knownObjects() const
{
	if((m_type != 32) || !m_knownObjects)
	{
		return ObjectsListSynchro::EmptyInstance;
	}
	return (*m_knownObjects);
}

ObjectsListSynchro & AdminResponseBody::get_knownObjects()
{
	if((m_type != 32) || !m_knownObjects)
	{
		return ObjectsListSynchro::EmptyInstance;
	}
	return (*m_knownObjects);
}

bool AdminResponseBody::set_ldapResults(const mVector< LdapResult > & c_ldapResults)
{
	if(m_type != 36)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_ldapResults) = c_ldapResults;
	m_isOk=true;
	return true;
}

const mVector< LdapResult > & AdminResponseBody::get_ldapResults() const
{
	if((m_type != 36) || !m_ldapResults)
	{
		return mVector< LdapResult >::EmptyInstance;
	}
	return (*m_ldapResults);
}

mVector< LdapResult > & AdminResponseBody::get_ldapResults()
{
	if((m_type != 36) || !m_ldapResults)
	{
		return mVector< LdapResult >::EmptyInstance;
	}
	return (*m_ldapResults);
}

bool AdminResponseBody::set_localEntityConf(const EntityConf & c_localEntityConf)
{
	if(m_type != 31)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_localEntityConf) = c_localEntityConf;
	m_isOk=true;
	return true;
}

const EntityConf & AdminResponseBody::get_localEntityConf() const
{
	if((m_type != 31) || !m_localEntityConf)
	{
		return EntityConf::EmptyInstance;
	}
	return (*m_localEntityConf);
}

EntityConf & AdminResponseBody::get_localEntityConf()
{
	if((m_type != 31) || !m_localEntityConf)
	{
		return EntityConf::EmptyInstance;
	}
	return (*m_localEntityConf);
}

bool AdminResponseBody::set_logs(const mVector< LogEntry > & c_logs)
{
	if(m_type != 5)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_logs) = c_logs;
	m_isOk=true;
	return true;
}

const mVector< LogEntry > & AdminResponseBody::get_logs() const
{
	if((m_type != 5) || !m_logs)
	{
		return mVector< LogEntry >::EmptyInstance;
	}
	return (*m_logs);
}

mVector< LogEntry > & AdminResponseBody::get_logs()
{
	if((m_type != 5) || !m_logs)
	{
		return mVector< LogEntry >::EmptyInstance;
	}
	return (*m_logs);
}

bool AdminResponseBody::set_logsType(const mVector< unsigned long > & c_logsType)
{
	if(m_type != 16)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_logsType) = c_logsType;
	m_isOk=true;
	return true;
}

const mVector< unsigned long > & AdminResponseBody::get_logsType() const
{
	if((m_type != 16) || !m_logsType)
	{
		return mVector< unsigned long >::EmptyInstance;
	}
	return (*m_logsType);
}

mVector< unsigned long > & AdminResponseBody::get_logsType()
{
	if((m_type != 16) || !m_logsType)
	{
		return mVector< unsigned long >::EmptyInstance;
	}
	return (*m_logsType);
}

bool AdminResponseBody::set_mailConf(const EmailConf & c_mailConf)
{
	if(m_type != 18)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_mailConf) = c_mailConf;
	m_isOk=true;
	return true;
}

const EmailConf & AdminResponseBody::get_mailConf() const
{
	if((m_type != 18) || !m_mailConf)
	{
		return EmailConf::EmptyInstance;
	}
	return (*m_mailConf);
}

EmailConf & AdminResponseBody::get_mailConf()
{
	if((m_type != 18) || !m_mailConf)
	{
		return EmailConf::EmptyInstance;
	}
	return (*m_mailConf);
}

bool AdminResponseBody::set_myAcl(const ASN1_BIT_STRING * c_myAcl)
{
	if(m_type != 12)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(m_myAcl)
		ASN1_item_free((ASN1_VALUE*)m_myAcl, ASN1_ITEM_rptr(ASN1_BIT_STRING));
	m_myAcl = (ASN1_BIT_STRING*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_BIT_STRING), (void*)c_myAcl);
	if(!m_myAcl)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	m_isOk=true;
	return true;
}

const ASN1_BIT_STRING * AdminResponseBody::get_myAcl() const
{
	if((m_type != 12) || !m_myAcl)
	{
		return NULL;
	}
	if(!m_myAcl)
		((AdminResponseBody*)this)->m_myAcl = (ASN1_BIT_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_BIT_STRING));
	return m_myAcl;
}

ASN1_BIT_STRING * AdminResponseBody::get_myAcl()
{
	if((m_type != 12) || !m_myAcl)
	{
		return NULL;
	}
	if(!m_myAcl)
		m_myAcl = (ASN1_BIT_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_BIT_STRING));
	return m_myAcl;
}

bool AdminResponseBody::set_myConf(const EntityConfCrypted & c_myConf)
{
	if(m_type != 14)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_myConf) = c_myConf;
	m_isOk=true;
	return true;
}

const EntityConfCrypted & AdminResponseBody::get_myConf() const
{
	if((m_type != 14) || !m_myConf)
	{
		return EntityConfCrypted::EmptyInstance;
	}
	return (*m_myConf);
}

EntityConfCrypted & AdminResponseBody::get_myConf()
{
	if((m_type != 14) || !m_myConf)
	{
		return EntityConfCrypted::EmptyInstance;
	}
	return (*m_myConf);
}

bool AdminResponseBody::set_objectReqs(const CryptedNewpkiRequests & c_objectReqs)
{
	if(m_type != 21)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_objectReqs) = c_objectReqs;
	m_isOk=true;
	return true;
}

const CryptedNewpkiRequests & AdminResponseBody::get_objectReqs() const
{
	if((m_type != 21) || !m_objectReqs)
	{
		return CryptedNewpkiRequests::EmptyInstance;
	}
	return (*m_objectReqs);
}

CryptedNewpkiRequests & AdminResponseBody::get_objectReqs()
{
	if((m_type != 21) || !m_objectReqs)
	{
		return CryptedNewpkiRequests::EmptyInstance;
	}
	return (*m_objectReqs);
}

bool AdminResponseBody::set_objectResps(const CryptedNewpkiResponses & c_objectResps)
{
	if(m_type != 29)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_objectResps) = c_objectResps;
	m_isOk=true;
	return true;
}

const CryptedNewpkiResponses & AdminResponseBody::get_objectResps() const
{
	if((m_type != 29) || !m_objectResps)
	{
		return CryptedNewpkiResponses::EmptyInstance;
	}
	return (*m_objectResps);
}

CryptedNewpkiResponses & AdminResponseBody::get_objectResps()
{
	if((m_type != 29) || !m_objectResps)
	{
		return CryptedNewpkiResponses::EmptyInstance;
	}
	return (*m_objectResps);
}

bool AdminResponseBody::set_other(const ASN1_NULL * c_other)
{
	if(m_type != 1)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(m_other)
		ASN1_item_free((ASN1_VALUE*)m_other, ASN1_ITEM_rptr(ASN1_NULL));
	m_other = (ASN1_NULL*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_NULL), (void*)c_other);
	if(!m_other)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	m_isOk=true;
	return true;
}

const ASN1_NULL * AdminResponseBody::get_other() const
{
	if((m_type != 1) || !m_other)
	{
		return NULL;
	}
	if(!m_other)
		((AdminResponseBody*)this)->m_other = (ASN1_NULL*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_NULL));
	return m_other;
}

ASN1_NULL * AdminResponseBody::get_other()
{
	if((m_type != 1) || !m_other)
	{
		return NULL;
	}
	if(!m_other)
		m_other = (ASN1_NULL*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_NULL));
	return m_other;
}

bool AdminResponseBody::set_p7b(const PKI_P7B & c_p7b)
{
	if(m_type != 23)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_p7b) = c_p7b;
	m_isOk=true;
	return true;
}

const PKI_P7B & AdminResponseBody::get_p7b() const
{
	if((m_type != 23) || !m_p7b)
	{
		return PKI_P7B::EmptyInstance;
	}
	return (*m_p7b);
}

PKI_P7B & AdminResponseBody::get_p7b()
{
	if((m_type != 23) || !m_p7b)
	{
		return PKI_P7B::EmptyInstance;
	}
	return (*m_p7b);
}

bool AdminResponseBody::set_profiles(const mVector< NewpkiProfileDatas > & c_profiles)
{
	if(m_type != 30)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_profiles) = c_profiles;
	m_isOk=true;
	return true;
}

const mVector< NewpkiProfileDatas > & AdminResponseBody::get_profiles() const
{
	if((m_type != 30) || !m_profiles)
	{
		return mVector< NewpkiProfileDatas >::EmptyInstance;
	}
	return (*m_profiles);
}

mVector< NewpkiProfileDatas > & AdminResponseBody::get_profiles()
{
	if((m_type != 30) || !m_profiles)
	{
		return mVector< NewpkiProfileDatas >::EmptyInstance;
	}
	return (*m_profiles);
}

bool AdminResponseBody::set_repositories(const mVector< RepEntryInfo > & c_repositories)
{
	if(m_type != 15)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_repositories) = c_repositories;
	m_isOk=true;
	return true;
}

const mVector< RepEntryInfo > & AdminResponseBody::get_repositories() const
{
	if((m_type != 15) || !m_repositories)
	{
		return mVector< RepEntryInfo >::EmptyInstance;
	}
	return (*m_repositories);
}

mVector< RepEntryInfo > & AdminResponseBody::get_repositories()
{
	if((m_type != 15) || !m_repositories)
	{
		return mVector< RepEntryInfo >::EmptyInstance;
	}
	return (*m_repositories);
}

bool AdminResponseBody::set_signEntity(const EntitySignatureResp & c_signEntity)
{
	if(m_type != 7)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_signEntity) = c_signEntity;
	m_isOk=true;
	return true;
}

const EntitySignatureResp & AdminResponseBody::get_signEntity() const
{
	if((m_type != 7) || !m_signEntity)
	{
		return EntitySignatureResp::EmptyInstance;
	}
	return (*m_signEntity);
}

EntitySignatureResp & AdminResponseBody::get_signEntity()
{
	if((m_type != 7) || !m_signEntity)
	{
		return EntitySignatureResp::EmptyInstance;
	}
	return (*m_signEntity);
}

bool AdminResponseBody::set_status(unsigned long c_status)
{
	if(m_type != 10 && m_type != 35 && m_type != 37)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_status) = c_status;
	m_isOk=true;
	return true;
}

unsigned long AdminResponseBody::get_status() const
{
	if((m_type != 10 && m_type != 35 && m_type != 37) || !m_status)
	{
		return 0;
	}
	return (*m_status);
}

bool AdminResponseBody::set_transactionid(const Asn1OctetString & c_transactionid)
{
	if(m_type != 28)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_transactionid) = c_transactionid;
	m_isOk=true;
	return true;
}

const Asn1OctetString & AdminResponseBody::get_transactionid() const
{
	if((m_type != 28) || !m_transactionid)
	{
		return Asn1OctetString::EmptyInstance;
	}
	return (*m_transactionid);
}

Asn1OctetString & AdminResponseBody::get_transactionid()
{
	if((m_type != 28) || !m_transactionid)
	{
		return Asn1OctetString::EmptyInstance;
	}
	return (*m_transactionid);
}

bool AdminResponseBody::set_users(const mVector< UserEntry > & c_users)
{
	if(m_type != 3)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_users) = c_users;
	m_isOk=true;
	return true;
}

const mVector< UserEntry > & AdminResponseBody::get_users() const
{
	if((m_type != 3) || !m_users)
	{
		return mVector< UserEntry >::EmptyInstance;
	}
	return (*m_users);
}

mVector< UserEntry > & AdminResponseBody::get_users()
{
	if((m_type != 3) || !m_users)
	{
		return mVector< UserEntry >::EmptyInstance;
	}
	return (*m_users);
}

bool AdminResponseBody::set_usersCert(const mVector< UserEntryCert > & c_usersCert)
{
	if(m_type != 8)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_usersCert) = c_usersCert;
	m_isOk=true;
	return true;
}

const mVector< UserEntryCert > & AdminResponseBody::get_usersCert() const
{
	if((m_type != 8) || !m_usersCert)
	{
		return mVector< UserEntryCert >::EmptyInstance;
	}
	return (*m_usersCert);
}

mVector< UserEntryCert > & AdminResponseBody::get_usersCert()
{
	if((m_type != 8) || !m_usersCert)
	{
		return mVector< UserEntryCert >::EmptyInstance;
	}
	return (*m_usersCert);
}

bool AdminResponseBody::set_usertype(unsigned long c_usertype)
{
	if(m_type != 2)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_usertype) = c_usertype;
	m_isOk=true;
	return true;
}

unsigned long AdminResponseBody::get_usertype() const
{
	if((m_type != 2) || !m_usertype)
	{
		return 0;
	}
	return (*m_usertype);
}

bool AdminResponseBody::set_webuserCerts(const mVector< NewpkiProfileDatasCert > & c_webuserCerts)
{
	if(m_type != 40)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_webuserCerts) = c_webuserCerts;
	m_isOk=true;
	return true;
}

const mVector< NewpkiProfileDatasCert > & AdminResponseBody::get_webuserCerts() const
{
	if((m_type != 40) || !m_webuserCerts)
	{
		return mVector< NewpkiProfileDatasCert >::EmptyInstance;
	}
	return (*m_webuserCerts);
}

mVector< NewpkiProfileDatasCert > & AdminResponseBody::get_webuserCerts()
{
	if((m_type != 40) || !m_webuserCerts)
	{
		return mVector< NewpkiProfileDatasCert >::EmptyInstance;
	}
	return (*m_webuserCerts);
}

bool AdminResponseBody::malloc_byType (int c_type)
{
	switch(m_type)
	{
		case 11:
			m_aclsConf = new X509Acl();
			if(!m_aclsConf)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 19:
			m_audits = new mVector< EntityAuditEntry >;
			if(!m_audits)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 34:
			m_cert = new PKI_CERT();
			if(!m_cert)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 22:
			m_certs = new mVector< InternalCaCert >;
			if(!m_certs)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 13:
			m_conf = new ExportedPkiConf();
			if(!m_conf)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 6:
			m_creEntity = new EntityCreationResp();
			if(!m_creEntity)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 9:
			m_createPkiUser = new CreatePkiUserResponse();
			if(!m_createPkiUser)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 25:
			m_crls = new mVector< PKI_CRL >;
			if(!m_crls)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 24:
			m_csr = new PKI_CSR();
			if(!m_csr)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 38:
		case 41:
			m_dn = (X509_NAME*)ASN1_item_new(ASN1_ITEM_rptr(X509_NAME));
			if(!m_dn)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 42:
			m_eeUsers = new mVector< NewpkiEeUser >;
			if(!m_eeUsers)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 4:
			m_entities = new mVector< EntityEntryInfo >;
			if(!m_entities)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 20:
			m_entitiesLinks = new mVector< EntityLinks >;
			if(!m_entitiesLinks)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 17:
			m_entityConf = new EntityConfBody();
			if(!m_entityConf)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 0:
			m_errors = new mVector< ErrorEntry >;
			if(!m_errors)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 26:
			m_groups = new mVector< UsersGroup >;
			if(!m_groups)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 27:
		case 33:
		case 39:
			m_id = (unsigned long*)malloc(sizeof(unsigned long));
			if(!m_id)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 32:
			m_knownObjects = new ObjectsListSynchro();
			if(!m_knownObjects)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 36:
			m_ldapResults = new mVector< LdapResult >;
			if(!m_ldapResults)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 31:
			m_localEntityConf = new EntityConf();
			if(!m_localEntityConf)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 5:
			m_logs = new mVector< LogEntry >;
			if(!m_logs)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 16:
			m_logsType = new mVector< unsigned long >;
			if(!m_logsType)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 18:
			m_mailConf = new EmailConf();
			if(!m_mailConf)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 12:
			m_myAcl = (ASN1_BIT_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_BIT_STRING));
			if(!m_myAcl)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 14:
			m_myConf = new EntityConfCrypted();
			if(!m_myConf)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 21:
			m_objectReqs = new CryptedNewpkiRequests();
			if(!m_objectReqs)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 29:
			m_objectResps = new CryptedNewpkiResponses();
			if(!m_objectResps)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 1:
			m_other = (ASN1_NULL*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_NULL));
			if(!m_other)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 23:
			m_p7b = new PKI_P7B();
			if(!m_p7b)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 30:
			m_profiles = new mVector< NewpkiProfileDatas >;
			if(!m_profiles)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 15:
			m_repositories = new mVector< RepEntryInfo >;
			if(!m_repositories)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 7:
			m_signEntity = new EntitySignatureResp();
			if(!m_signEntity)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 10:
		case 35:
		case 37:
			m_status = (unsigned long*)malloc(sizeof(unsigned long));
			if(!m_status)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 28:
			m_transactionid = new Asn1OctetString();
			if(!m_transactionid)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 3:
			m_users = new mVector< UserEntry >;
			if(!m_users)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 8:
			m_usersCert = new mVector< UserEntryCert >;
			if(!m_usersCert)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 2:
			m_usertype = (unsigned long*)malloc(sizeof(unsigned long));
			if(!m_usertype)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 40:
			m_webuserCerts = new mVector< NewpkiProfileDatasCert >;
			if(!m_webuserCerts)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
	}

	return true;
}

AdminResponseBody::AdminResponseBody():NewPKIObject()
{
	resetAll();
}

AdminResponseBody::AdminResponseBody(const AdminResponseBody & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

AdminResponseBody::~AdminResponseBody()
{
	Clear();
}

void AdminResponseBody::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void AdminResponseBody::freeAll()
{
	if(m_aclsConf)
	{
		delete m_aclsConf;
	}
	if(m_audits)
	{
		delete m_audits;
	}
	if(m_cert)
	{
		delete m_cert;
	}
	if(m_certs)
	{
		delete m_certs;
	}
	if(m_conf)
	{
		delete m_conf;
	}
	if(m_creEntity)
	{
		delete m_creEntity;
	}
	if(m_createPkiUser)
	{
		delete m_createPkiUser;
	}
	if(m_crls)
	{
		delete m_crls;
	}
	if(m_csr)
	{
		delete m_csr;
	}
	if(m_dn)
	{
		ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
	}
	if(m_eeUsers)
	{
		delete m_eeUsers;
	}
	if(m_entities)
	{
		delete m_entities;
	}
	if(m_entitiesLinks)
	{
		delete m_entitiesLinks;
	}
	if(m_entityConf)
	{
		delete m_entityConf;
	}
	if(m_errors)
	{
		delete m_errors;
	}
	if(m_groups)
	{
		delete m_groups;
	}
	if(m_id)
	{
		free(m_id);
	}
	if(m_knownObjects)
	{
		delete m_knownObjects;
	}
	if(m_ldapResults)
	{
		delete m_ldapResults;
	}
	if(m_localEntityConf)
	{
		delete m_localEntityConf;
	}
	if(m_logs)
	{
		delete m_logs;
	}
	if(m_logsType)
	{
		delete m_logsType;
	}
	if(m_mailConf)
	{
		delete m_mailConf;
	}
	if(m_myAcl)
	{
		ASN1_item_free((ASN1_VALUE*)m_myAcl, ASN1_ITEM_rptr(ASN1_BIT_STRING));
	}
	if(m_myConf)
	{
		delete m_myConf;
	}
	if(m_objectReqs)
	{
		delete m_objectReqs;
	}
	if(m_objectResps)
	{
		delete m_objectResps;
	}
	if(m_other)
	{
		ASN1_item_free((ASN1_VALUE*)m_other, ASN1_ITEM_rptr(ASN1_NULL));
	}
	if(m_p7b)
	{
		delete m_p7b;
	}
	if(m_profiles)
	{
		delete m_profiles;
	}
	if(m_repositories)
	{
		delete m_repositories;
	}
	if(m_signEntity)
	{
		delete m_signEntity;
	}
	if(m_status)
	{
		free(m_status);
	}
	if(m_transactionid)
	{
		delete m_transactionid;
	}
	if(m_users)
	{
		delete m_users;
	}
	if(m_usersCert)
	{
		delete m_usersCert;
	}
	if(m_usertype)
	{
		free(m_usertype);
	}
	if(m_webuserCerts)
	{
		delete m_webuserCerts;
	}
}

void AdminResponseBody::resetAll()
{
	m_type = -1;
	m_aclsConf = NULL;
	m_audits = NULL;
	m_cert = NULL;
	m_certs = NULL;
	m_conf = NULL;
	m_creEntity = NULL;
	m_createPkiUser = NULL;
	m_crls = NULL;
	m_csr = NULL;
	m_dn = NULL;
	m_eeUsers = NULL;
	m_entities = NULL;
	m_entitiesLinks = NULL;
	m_entityConf = NULL;
	m_errors = NULL;
	m_groups = NULL;
	m_id = NULL;
	m_knownObjects = NULL;
	m_ldapResults = NULL;
	m_localEntityConf = NULL;
	m_logs = NULL;
	m_logsType = NULL;
	m_mailConf = NULL;
	m_myAcl = NULL;
	m_myConf = NULL;
	m_objectReqs = NULL;
	m_objectResps = NULL;
	m_other = NULL;
	m_p7b = NULL;
	m_profiles = NULL;
	m_repositories = NULL;
	m_signEntity = NULL;
	m_status = NULL;
	m_transactionid = NULL;
	m_users = NULL;
	m_usersCert = NULL;
	m_usertype = NULL;
	m_webuserCerts = NULL;
}

bool AdminResponseBody::load_Datas(const ADMIN_RESPONSE_BODY * Datas)
{
	Clear();
	NEWPKI_PROFILE_DATAS_CERT * currwebuserCerts;
	USER_ENTRY_CERT * currusersCert;
	USER_ENTRY * currusers;
	REP_ENTRY_INFO * currrepositories;
	NEWPKI_PROFILE_DATAS * currprofiles;
	ASN1_INTEGER * currlogsType;
	LOG_ENTRY * currlogs;
	LDAP_RESULT * currldapResults;
	USERS_GROUP * currgroups;
	ERROR_ENTRY * currerrors;
	ENTITY_LINKS * currentitiesLinks;
	ENTITY_ENTRY_INFO * currentities;
	NEWPKI_EE_USER * curreeUsers;
	X509_CRL * currcrls;
	INTERNAL_CA_CERT * currcerts;
	int i;
	ENTITY_AUDIT_ENTRY * curraudits;
	if(!set_type(Datas->type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(Datas->type)
	{
		case 11:
			if(Datas->d.acls_conf)
			{
				if(!(*m_aclsConf).load_Datas(Datas->d.acls_conf))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 19:
			if(Datas->d.audits)
			{
				for(i=0; i<SKM_sk_num(ENTITY_AUDIT_ENTRY, Datas->d.audits); i++)
				{
					curraudits = SKM_sk_value(ENTITY_AUDIT_ENTRY, Datas->d.audits, i);
					if(!curraudits)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_audits).insert((*m_audits).begin() + i);
					if(!(*m_audits)[i].load_Datas(curraudits))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
		case 34:
			if(Datas->d.cert)
			{
				if(!(*m_cert).load_Datas(Datas->d.cert))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 22:
			if(Datas->d.certs)
			{
				for(i=0; i<SKM_sk_num(INTERNAL_CA_CERT, Datas->d.certs); i++)
				{
					currcerts = SKM_sk_value(INTERNAL_CA_CERT, Datas->d.certs, i);
					if(!currcerts)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_certs).insert((*m_certs).begin() + i);
					if(!(*m_certs)[i].load_Datas(currcerts))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
		case 13:
			if(Datas->d.conf)
			{
				if(!(*m_conf).load_Datas(Datas->d.conf))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 6:
			if(Datas->d.cre_entity)
			{
				if(!(*m_creEntity).load_Datas(Datas->d.cre_entity))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 9:
			if(Datas->d.create_pki_user)
			{
				if(!(*m_createPkiUser).load_Datas(Datas->d.create_pki_user))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 25:
			if(Datas->d.crls)
			{
				for(i=0; i<SKM_sk_num(X509_CRL, Datas->d.crls); i++)
				{
					currcrls = SKM_sk_value(X509_CRL, Datas->d.crls, i);
					if(!currcrls)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_crls).insert((*m_crls).begin() + i);
					if(!(*m_crls)[i].load_Datas(currcrls))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
		case 24:
			if(Datas->d.csr)
			{
				if(!(*m_csr).load_Datas(Datas->d.csr))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 38:
		case 41:
			if(Datas->d.dn)
			{
				if(m_dn)
					ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
				m_dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), Datas->d.dn);
				if(!m_dn)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			break;
		case 42:
			if(Datas->d.ee_users)
			{
				for(i=0; i<SKM_sk_num(NEWPKI_EE_USER, Datas->d.ee_users); i++)
				{
					curreeUsers = SKM_sk_value(NEWPKI_EE_USER, Datas->d.ee_users, i);
					if(!curreeUsers)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_eeUsers).insert((*m_eeUsers).begin() + i);
					if(!(*m_eeUsers)[i].load_Datas(curreeUsers))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
		case 4:
			if(Datas->d.entities)
			{
				for(i=0; i<SKM_sk_num(ENTITY_ENTRY_INFO, Datas->d.entities); i++)
				{
					currentities = SKM_sk_value(ENTITY_ENTRY_INFO, Datas->d.entities, i);
					if(!currentities)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_entities).insert((*m_entities).begin() + i);
					if(!(*m_entities)[i].load_Datas(currentities))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
		case 20:
			if(Datas->d.entities_links)
			{
				for(i=0; i<SKM_sk_num(ENTITY_LINKS, Datas->d.entities_links); i++)
				{
					currentitiesLinks = SKM_sk_value(ENTITY_LINKS, Datas->d.entities_links, i);
					if(!currentitiesLinks)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_entitiesLinks).insert((*m_entitiesLinks).begin() + i);
					if(!(*m_entitiesLinks)[i].load_Datas(currentitiesLinks))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
		case 17:
			if(Datas->d.entity_conf)
			{
				if(!(*m_entityConf).load_Datas(Datas->d.entity_conf))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 0:
			if(Datas->d.errors)
			{
				for(i=0; i<SKM_sk_num(ERROR_ENTRY, Datas->d.errors); i++)
				{
					currerrors = SKM_sk_value(ERROR_ENTRY, Datas->d.errors, i);
					if(!currerrors)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_errors).insert((*m_errors).begin() + i);
					if(!(*m_errors)[i].load_Datas(currerrors))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
		case 26:
			if(Datas->d.groups)
			{
				for(i=0; i<SKM_sk_num(USERS_GROUP, Datas->d.groups); i++)
				{
					currgroups = SKM_sk_value(USERS_GROUP, Datas->d.groups, i);
					if(!currgroups)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_groups).insert((*m_groups).begin() + i);
					if(!(*m_groups)[i].load_Datas(currgroups))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
		case 27:
		case 33:
		case 39:
			if(Datas->d.id)
			{
				(*m_id) = ASN1_INTEGER_GET(Datas->d.id);
			}
			break;
		case 32:
			if(Datas->d.known_objects)
			{
				if(!(*m_knownObjects).load_Datas(Datas->d.known_objects))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 36:
			if(Datas->d.ldap_results)
			{
				for(i=0; i<SKM_sk_num(LDAP_RESULT, Datas->d.ldap_results); i++)
				{
					currldapResults = SKM_sk_value(LDAP_RESULT, Datas->d.ldap_results, i);
					if(!currldapResults)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_ldapResults).insert((*m_ldapResults).begin() + i);
					if(!(*m_ldapResults)[i].load_Datas(currldapResults))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
		case 31:
			if(Datas->d.local_entity_conf)
			{
				if(!(*m_localEntityConf).load_Datas(Datas->d.local_entity_conf))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 5:
			if(Datas->d.logs)
			{
				for(i=0; i<SKM_sk_num(LOG_ENTRY, Datas->d.logs); i++)
				{
					currlogs = SKM_sk_value(LOG_ENTRY, Datas->d.logs, i);
					if(!currlogs)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_logs).insert((*m_logs).begin() + i);
					if(!(*m_logs)[i].load_Datas(currlogs))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
		case 16:
			if(Datas->d.logs_type)
			{
				for(i=0; i<SKM_sk_num(ASN1_INTEGER, Datas->d.logs_type); i++)
				{
					currlogsType = SKM_sk_value(ASN1_INTEGER, Datas->d.logs_type, i);
					if(!currlogsType)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_logsType).insert((*m_logsType).begin() + i);
					(*m_logsType)[i] = ASN1_INTEGER_GET(currlogsType);
				}
			}
			break;
		case 18:
			if(Datas->d.mail_conf)
			{
				if(!(*m_mailConf).load_Datas(Datas->d.mail_conf))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 12:
			if(Datas->d.my_acl)
			{
				if(m_myAcl)
					ASN1_item_free((ASN1_VALUE*)m_myAcl, ASN1_ITEM_rptr(ASN1_BIT_STRING));
				m_myAcl = (ASN1_BIT_STRING*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_BIT_STRING), Datas->d.my_acl);
				if(!m_myAcl)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			break;
		case 14:
			if(Datas->d.my_conf)
			{
				if(!(*m_myConf).load_Datas(Datas->d.my_conf))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 21:
			if(Datas->d.object_reqs)
			{
				if(!(*m_objectReqs).load_Datas(Datas->d.object_reqs))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 29:
			if(Datas->d.object_resps)
			{
				if(!(*m_objectResps).load_Datas(Datas->d.object_resps))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 1:
			if(Datas->d.other)
			{
				if(m_other)
					ASN1_item_free((ASN1_VALUE*)m_other, ASN1_ITEM_rptr(ASN1_NULL));
				m_other = (ASN1_NULL*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_NULL), Datas->d.other);
				if(!m_other)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			break;
		case 23:
			if(Datas->d.p7b)
			{
				if(!(*m_p7b).load_Datas(Datas->d.p7b))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 30:
			if(Datas->d.profiles)
			{
				for(i=0; i<SKM_sk_num(NEWPKI_PROFILE_DATAS, Datas->d.profiles); i++)
				{
					currprofiles = SKM_sk_value(NEWPKI_PROFILE_DATAS, Datas->d.profiles, i);
					if(!currprofiles)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_profiles).insert((*m_profiles).begin() + i);
					if(!(*m_profiles)[i].load_Datas(currprofiles))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
		case 15:
			if(Datas->d.repositories)
			{
				for(i=0; i<SKM_sk_num(REP_ENTRY_INFO, Datas->d.repositories); i++)
				{
					currrepositories = SKM_sk_value(REP_ENTRY_INFO, Datas->d.repositories, i);
					if(!currrepositories)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_repositories).insert((*m_repositories).begin() + i);
					if(!(*m_repositories)[i].load_Datas(currrepositories))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
		case 7:
			if(Datas->d.sign_entity)
			{
				if(!(*m_signEntity).load_Datas(Datas->d.sign_entity))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 10:
		case 35:
		case 37:
			if(Datas->d.status)
			{
				(*m_status) = ASN1_INTEGER_GET(Datas->d.status);
			}
			break;
		case 28:
			if(Datas->d.transactionID)
			{
				if(!(*m_transactionid).load_Datas(Datas->d.transactionID))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 3:
			if(Datas->d.users)
			{
				for(i=0; i<SKM_sk_num(USER_ENTRY, Datas->d.users); i++)
				{
					currusers = SKM_sk_value(USER_ENTRY, Datas->d.users, i);
					if(!currusers)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_users).insert((*m_users).begin() + i);
					if(!(*m_users)[i].load_Datas(currusers))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
		case 8:
			if(Datas->d.users_cert)
			{
				for(i=0; i<SKM_sk_num(USER_ENTRY_CERT, Datas->d.users_cert); i++)
				{
					currusersCert = SKM_sk_value(USER_ENTRY_CERT, Datas->d.users_cert, i);
					if(!currusersCert)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_usersCert).insert((*m_usersCert).begin() + i);
					if(!(*m_usersCert)[i].load_Datas(currusersCert))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
		case 2:
			if(Datas->d.usertype)
			{
				(*m_usertype) = ASN1_INTEGER_GET(Datas->d.usertype);
			}
			break;
		case 40:
			if(Datas->d.webuser_certs)
			{
				for(i=0; i<SKM_sk_num(NEWPKI_PROFILE_DATAS_CERT, Datas->d.webuser_certs); i++)
				{
					currwebuserCerts = SKM_sk_value(NEWPKI_PROFILE_DATAS_CERT, Datas->d.webuser_certs, i);
					if(!currwebuserCerts)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					(*m_webuserCerts).insert((*m_webuserCerts).begin() + i);
					if(!(*m_webuserCerts)[i].load_Datas(currwebuserCerts))
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
						return false;
					}
				}
			}
			break;
	}
	m_isOk=true;
	return true;
}

bool AdminResponseBody::give_Datas(ADMIN_RESPONSE_BODY ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ADMIN_RESPONSE_BODY*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
			NEWPKI_PROFILE_DATAS_CERT * currwebuserCerts;
			USER_ENTRY_CERT * currusersCert;
			USER_ENTRY * currusers;
			REP_ENTRY_INFO * currrepositories;
			NEWPKI_PROFILE_DATAS * currprofiles;
			ASN1_INTEGER * currlogsType;
			LOG_ENTRY * currlogs;
			LDAP_RESULT * currldapResults;
			USERS_GROUP * currgroups;
			ERROR_ENTRY * currerrors;
			ENTITY_LINKS * currentitiesLinks;
			ENTITY_ENTRY_INFO * currentities;
			NEWPKI_EE_USER * curreeUsers;
			X509_CRL * currcrls;
			INTERNAL_CA_CERT * currcerts;
			ENTITY_AUDIT_ENTRY * curraudits;
	size_t i;
	(*Datas)->type = m_type;
	switch(m_type)
	{
		case 11:
			if(!((*Datas)->d.acls_conf = (X509_ACL*)ASN1_item_new(ASN1_ITEM_rptr(X509_ACL))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_aclsConf).give_Datas(&(*Datas)->d.acls_conf))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.acls_conf, ASN1_ITEM_rptr(X509_ACL));
				(*Datas)->d.acls_conf = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 19:
			if(!((*Datas)->d.audits = SKM_sk_new_null(ENTITY_AUDIT_ENTRY)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_audits).size() ; i++)
			{
				curraudits = NULL;
				if(!(*m_audits)[i].give_Datas(&curraudits))
				{
					ASN1_item_free((ASN1_VALUE*)curraudits, ASN1_ITEM_rptr(ENTITY_AUDIT_ENTRY));
					curraudits = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(ENTITY_AUDIT_ENTRY, (*Datas)->d.audits, curraudits) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)curraudits, ASN1_ITEM_rptr(ENTITY_AUDIT_ENTRY));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 34:
			if(!((*Datas)->d.cert = (X509*)ASN1_item_new(ASN1_ITEM_rptr(X509))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_cert).give_Datas(&(*Datas)->d.cert))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.cert, ASN1_ITEM_rptr(X509));
				(*Datas)->d.cert = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 22:
			if(!((*Datas)->d.certs = SKM_sk_new_null(INTERNAL_CA_CERT)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_certs).size() ; i++)
			{
				currcerts = NULL;
				if(!(*m_certs)[i].give_Datas(&currcerts))
				{
					ASN1_item_free((ASN1_VALUE*)currcerts, ASN1_ITEM_rptr(INTERNAL_CA_CERT));
					currcerts = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(INTERNAL_CA_CERT, (*Datas)->d.certs, currcerts) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)currcerts, ASN1_ITEM_rptr(INTERNAL_CA_CERT));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 13:
			if(!((*Datas)->d.conf = (EXPORTED_PKI_CONF*)ASN1_item_new(ASN1_ITEM_rptr(EXPORTED_PKI_CONF))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_conf).give_Datas(&(*Datas)->d.conf))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.conf, ASN1_ITEM_rptr(EXPORTED_PKI_CONF));
				(*Datas)->d.conf = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 6:
			if(!((*Datas)->d.cre_entity = (ENTITY_CREATION_RESP*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_CREATION_RESP))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_creEntity).give_Datas(&(*Datas)->d.cre_entity))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.cre_entity, ASN1_ITEM_rptr(ENTITY_CREATION_RESP));
				(*Datas)->d.cre_entity = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 9:
			if(!((*Datas)->d.create_pki_user = (CREATE_PKI_USER_RESPONSE*)ASN1_item_new(ASN1_ITEM_rptr(CREATE_PKI_USER_RESPONSE))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_createPkiUser).give_Datas(&(*Datas)->d.create_pki_user))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.create_pki_user, ASN1_ITEM_rptr(CREATE_PKI_USER_RESPONSE));
				(*Datas)->d.create_pki_user = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 25:
			if(!((*Datas)->d.crls = SKM_sk_new_null(X509_CRL)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_crls).size() ; i++)
			{
				currcrls = NULL;
				if(!(*m_crls)[i].give_Datas(&currcrls))
				{
					ASN1_item_free((ASN1_VALUE*)currcrls, ASN1_ITEM_rptr(X509_CRL));
					currcrls = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(X509_CRL, (*Datas)->d.crls, currcrls) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)currcrls, ASN1_ITEM_rptr(X509_CRL));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 24:
			if(!((*Datas)->d.csr = (X509_REQ*)ASN1_item_new(ASN1_ITEM_rptr(X509_REQ))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_csr).give_Datas(&(*Datas)->d.csr))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.csr, ASN1_ITEM_rptr(X509_REQ));
				(*Datas)->d.csr = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 38:
		case 41:
			if(m_dn)
			{
				if((*Datas)->d.dn)
					ASN1_item_free((ASN1_VALUE*)(*Datas)->d.dn, ASN1_ITEM_rptr(X509_NAME));
				if(!((*Datas)->d.dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), (ASN1_VALUE*)m_dn)))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			else
			{
				if(!(*Datas)->d.dn)
				{
					(*Datas)->d.dn = (X509_NAME*)ASN1_item_new(ASN1_ITEM_rptr(X509_NAME));
					if(!(*Datas)->d.dn)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
						return false;
					}
				}
			}
			break;
		case 42:
			if(!((*Datas)->d.ee_users = SKM_sk_new_null(NEWPKI_EE_USER)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_eeUsers).size() ; i++)
			{
				curreeUsers = NULL;
				if(!(*m_eeUsers)[i].give_Datas(&curreeUsers))
				{
					ASN1_item_free((ASN1_VALUE*)curreeUsers, ASN1_ITEM_rptr(NEWPKI_EE_USER));
					curreeUsers = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(NEWPKI_EE_USER, (*Datas)->d.ee_users, curreeUsers) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)curreeUsers, ASN1_ITEM_rptr(NEWPKI_EE_USER));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 4:
			if(!((*Datas)->d.entities = SKM_sk_new_null(ENTITY_ENTRY_INFO)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_entities).size() ; i++)
			{
				currentities = NULL;
				if(!(*m_entities)[i].give_Datas(&currentities))
				{
					ASN1_item_free((ASN1_VALUE*)currentities, ASN1_ITEM_rptr(ENTITY_ENTRY_INFO));
					currentities = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(ENTITY_ENTRY_INFO, (*Datas)->d.entities, currentities) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)currentities, ASN1_ITEM_rptr(ENTITY_ENTRY_INFO));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 20:
			if(!((*Datas)->d.entities_links = SKM_sk_new_null(ENTITY_LINKS)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_entitiesLinks).size() ; i++)
			{
				currentitiesLinks = NULL;
				if(!(*m_entitiesLinks)[i].give_Datas(&currentitiesLinks))
				{
					ASN1_item_free((ASN1_VALUE*)currentitiesLinks, ASN1_ITEM_rptr(ENTITY_LINKS));
					currentitiesLinks = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(ENTITY_LINKS, (*Datas)->d.entities_links, currentitiesLinks) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)currentitiesLinks, ASN1_ITEM_rptr(ENTITY_LINKS));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 17:
			if(!((*Datas)->d.entity_conf = (ENTITY_CONF_BODY*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_CONF_BODY))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_entityConf).give_Datas(&(*Datas)->d.entity_conf))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.entity_conf, ASN1_ITEM_rptr(ENTITY_CONF_BODY));
				(*Datas)->d.entity_conf = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 0:
			if(!((*Datas)->d.errors = SKM_sk_new_null(ERROR_ENTRY)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_errors).size() ; i++)
			{
				currerrors = NULL;
				if(!(*m_errors)[i].give_Datas(&currerrors))
				{
					ASN1_item_free((ASN1_VALUE*)currerrors, ASN1_ITEM_rptr(ERROR_ENTRY));
					currerrors = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(ERROR_ENTRY, (*Datas)->d.errors, currerrors) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)currerrors, ASN1_ITEM_rptr(ERROR_ENTRY));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 26:
			if(!((*Datas)->d.groups = SKM_sk_new_null(USERS_GROUP)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_groups).size() ; i++)
			{
				currgroups = NULL;
				if(!(*m_groups)[i].give_Datas(&currgroups))
				{
					ASN1_item_free((ASN1_VALUE*)currgroups, ASN1_ITEM_rptr(USERS_GROUP));
					currgroups = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(USERS_GROUP, (*Datas)->d.groups, currgroups) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)currgroups, ASN1_ITEM_rptr(USERS_GROUP));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 27:
		case 33:
		case 39:
			if(!((*Datas)->d.id = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(ASN1_INTEGER_set((*Datas)->d.id, (*m_id)) <= 0)
			{
				ASN1_INTEGER_free((*Datas)->d.id);
				(*Datas)->d.id = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
				return false;
			}
			break;
		case 32:
			if(!((*Datas)->d.known_objects = (OBJECTS_LIST_SYNCHRO*)ASN1_item_new(ASN1_ITEM_rptr(OBJECTS_LIST_SYNCHRO))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_knownObjects).give_Datas(&(*Datas)->d.known_objects))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.known_objects, ASN1_ITEM_rptr(OBJECTS_LIST_SYNCHRO));
				(*Datas)->d.known_objects = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 36:
			if(!((*Datas)->d.ldap_results = SKM_sk_new_null(LDAP_RESULT)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_ldapResults).size() ; i++)
			{
				currldapResults = NULL;
				if(!(*m_ldapResults)[i].give_Datas(&currldapResults))
				{
					ASN1_item_free((ASN1_VALUE*)currldapResults, ASN1_ITEM_rptr(LDAP_RESULT));
					currldapResults = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(LDAP_RESULT, (*Datas)->d.ldap_results, currldapResults) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)currldapResults, ASN1_ITEM_rptr(LDAP_RESULT));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 31:
			if(!((*Datas)->d.local_entity_conf = (ENTITY_CONF*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_CONF))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_localEntityConf).give_Datas(&(*Datas)->d.local_entity_conf))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.local_entity_conf, ASN1_ITEM_rptr(ENTITY_CONF));
				(*Datas)->d.local_entity_conf = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 5:
			if(!((*Datas)->d.logs = SKM_sk_new_null(LOG_ENTRY)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_logs).size() ; i++)
			{
				currlogs = NULL;
				if(!(*m_logs)[i].give_Datas(&currlogs))
				{
					ASN1_item_free((ASN1_VALUE*)currlogs, ASN1_ITEM_rptr(LOG_ENTRY));
					currlogs = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(LOG_ENTRY, (*Datas)->d.logs, currlogs) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)currlogs, ASN1_ITEM_rptr(LOG_ENTRY));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 16:
			if(!((*Datas)->d.logs_type = SKM_sk_new_null(ASN1_INTEGER)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_logsType).size() ; i++)
			{
				currlogsType = ASN1_INTEGER_new();
				if(!currlogsType)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
				if(ASN1_INTEGER_set(currlogsType, (*m_logsType)[i]) <= 0)
				{
					ASN1_INTEGER_free(currlogsType);
					currlogsType = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
				if(SKM_sk_push(ASN1_INTEGER, (*Datas)->d.logs_type, currlogsType) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)currlogsType, ASN1_ITEM_rptr(ASN1_INTEGER));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 18:
			if(!((*Datas)->d.mail_conf = (EMAIL_CONF*)ASN1_item_new(ASN1_ITEM_rptr(EMAIL_CONF))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_mailConf).give_Datas(&(*Datas)->d.mail_conf))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.mail_conf, ASN1_ITEM_rptr(EMAIL_CONF));
				(*Datas)->d.mail_conf = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 12:
			if(m_myAcl)
			{
				if((*Datas)->d.my_acl)
					ASN1_item_free((ASN1_VALUE*)(*Datas)->d.my_acl, ASN1_ITEM_rptr(ASN1_BIT_STRING));
				if(!((*Datas)->d.my_acl = (ASN1_BIT_STRING*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_BIT_STRING), (ASN1_VALUE*)m_myAcl)))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			else
			{
				if(!(*Datas)->d.my_acl)
				{
					(*Datas)->d.my_acl = (ASN1_BIT_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_BIT_STRING));
					if(!(*Datas)->d.my_acl)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
						return false;
					}
				}
			}
			break;
		case 14:
			if(!((*Datas)->d.my_conf = (ENTITY_CONF_CRYPTED*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_CONF_CRYPTED))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_myConf).give_Datas(&(*Datas)->d.my_conf))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.my_conf, ASN1_ITEM_rptr(ENTITY_CONF_CRYPTED));
				(*Datas)->d.my_conf = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 21:
			if(!((*Datas)->d.object_reqs = (CRYPTED_NEWPKI_REQUESTS*)ASN1_item_new(ASN1_ITEM_rptr(CRYPTED_NEWPKI_REQUESTS))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_objectReqs).give_Datas(&(*Datas)->d.object_reqs))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.object_reqs, ASN1_ITEM_rptr(CRYPTED_NEWPKI_REQUESTS));
				(*Datas)->d.object_reqs = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 29:
			if(!((*Datas)->d.object_resps = (CRYPTED_NEWPKI_RESPONSES*)ASN1_item_new(ASN1_ITEM_rptr(CRYPTED_NEWPKI_RESPONSES))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_objectResps).give_Datas(&(*Datas)->d.object_resps))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.object_resps, ASN1_ITEM_rptr(CRYPTED_NEWPKI_RESPONSES));
				(*Datas)->d.object_resps = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 1:
			if(m_other)
			{
				if((*Datas)->d.other)
					ASN1_item_free((ASN1_VALUE*)(*Datas)->d.other, ASN1_ITEM_rptr(ASN1_NULL));
				if(!((*Datas)->d.other = (ASN1_NULL*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_NULL), (ASN1_VALUE*)m_other)))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			else
			{
				if(!(*Datas)->d.other)
				{
					(*Datas)->d.other = (ASN1_NULL*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_NULL));
					if(!(*Datas)->d.other)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
						return false;
					}
				}
			}
			break;
		case 23:
			if(!((*Datas)->d.p7b = (PKCS7*)ASN1_item_new(ASN1_ITEM_rptr(PKCS7))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_p7b).give_Datas(&(*Datas)->d.p7b))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.p7b, ASN1_ITEM_rptr(PKCS7));
				(*Datas)->d.p7b = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 30:
			if(!((*Datas)->d.profiles = SKM_sk_new_null(NEWPKI_PROFILE_DATAS)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_profiles).size() ; i++)
			{
				currprofiles = NULL;
				if(!(*m_profiles)[i].give_Datas(&currprofiles))
				{
					ASN1_item_free((ASN1_VALUE*)currprofiles, ASN1_ITEM_rptr(NEWPKI_PROFILE_DATAS));
					currprofiles = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(NEWPKI_PROFILE_DATAS, (*Datas)->d.profiles, currprofiles) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)currprofiles, ASN1_ITEM_rptr(NEWPKI_PROFILE_DATAS));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 15:
			if(!((*Datas)->d.repositories = SKM_sk_new_null(REP_ENTRY_INFO)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_repositories).size() ; i++)
			{
				currrepositories = NULL;
				if(!(*m_repositories)[i].give_Datas(&currrepositories))
				{
					ASN1_item_free((ASN1_VALUE*)currrepositories, ASN1_ITEM_rptr(REP_ENTRY_INFO));
					currrepositories = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(REP_ENTRY_INFO, (*Datas)->d.repositories, currrepositories) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)currrepositories, ASN1_ITEM_rptr(REP_ENTRY_INFO));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 7:
			if(!((*Datas)->d.sign_entity = (ENTITY_SIGNATURE_RESP*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_SIGNATURE_RESP))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_signEntity).give_Datas(&(*Datas)->d.sign_entity))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.sign_entity, ASN1_ITEM_rptr(ENTITY_SIGNATURE_RESP));
				(*Datas)->d.sign_entity = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 10:
		case 35:
		case 37:
			if(!((*Datas)->d.status = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(ASN1_INTEGER_set((*Datas)->d.status, (*m_status)) <= 0)
			{
				ASN1_INTEGER_free((*Datas)->d.status);
				(*Datas)->d.status = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
				return false;
			}
			break;
		case 28:
			if(!((*Datas)->d.transactionID = (ASN1_OCTET_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_OCTET_STRING))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_transactionid).give_Datas(&(*Datas)->d.transactionID))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.transactionID, ASN1_ITEM_rptr(ASN1_OCTET_STRING));
				(*Datas)->d.transactionID = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 3:
			if(!((*Datas)->d.users = SKM_sk_new_null(USER_ENTRY)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_users).size() ; i++)
			{
				currusers = NULL;
				if(!(*m_users)[i].give_Datas(&currusers))
				{
					ASN1_item_free((ASN1_VALUE*)currusers, ASN1_ITEM_rptr(USER_ENTRY));
					currusers = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(USER_ENTRY, (*Datas)->d.users, currusers) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)currusers, ASN1_ITEM_rptr(USER_ENTRY));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 8:
			if(!((*Datas)->d.users_cert = SKM_sk_new_null(USER_ENTRY_CERT)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_usersCert).size() ; i++)
			{
				currusersCert = NULL;
				if(!(*m_usersCert)[i].give_Datas(&currusersCert))
				{
					ASN1_item_free((ASN1_VALUE*)currusersCert, ASN1_ITEM_rptr(USER_ENTRY_CERT));
					currusersCert = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(USER_ENTRY_CERT, (*Datas)->d.users_cert, currusersCert) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)currusersCert, ASN1_ITEM_rptr(USER_ENTRY_CERT));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
		case 2:
			if(!((*Datas)->d.usertype = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(ASN1_INTEGER_set((*Datas)->d.usertype, (*m_usertype)) <= 0)
			{
				ASN1_INTEGER_free((*Datas)->d.usertype);
				(*Datas)->d.usertype = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
				return false;
			}
			break;
		case 40:
			if(!((*Datas)->d.webuser_certs = SKM_sk_new_null(NEWPKI_PROFILE_DATAS_CERT)))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			for(i = 0 ; i < (*m_webuserCerts).size() ; i++)
			{
				currwebuserCerts = NULL;
				if(!(*m_webuserCerts)[i].give_Datas(&currwebuserCerts))
				{
					ASN1_item_free((ASN1_VALUE*)currwebuserCerts, ASN1_ITEM_rptr(NEWPKI_PROFILE_DATAS_CERT));
					currwebuserCerts = NULL;
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(SKM_sk_push(NEWPKI_PROFILE_DATAS_CERT, (*Datas)->d.webuser_certs, currwebuserCerts) < 0)
				{
					ASN1_item_free((ASN1_VALUE*)currwebuserCerts, ASN1_ITEM_rptr(NEWPKI_PROFILE_DATAS_CERT));
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
					return false;
				}
			}
			break;
	}
	return true;
}

bool AdminResponseBody::operator=(const AdminResponseBody & other)
{
	Clear();
	if(!set_type(other.m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(other.m_type)
	{
		case 11:
			if(!other.m_aclsConf)
				break;
			(*m_aclsConf) = *(other.m_aclsConf);
			break;
		case 19:
			if(!other.m_audits)
				break;
			(*m_audits) = *(other.m_audits);
			break;
		case 34:
			if(!other.m_cert)
				break;
			(*m_cert) = *(other.m_cert);
			break;
		case 22:
			if(!other.m_certs)
				break;
			(*m_certs) = *(other.m_certs);
			break;
		case 13:
			if(!other.m_conf)
				break;
			(*m_conf) = *(other.m_conf);
			break;
		case 6:
			if(!other.m_creEntity)
				break;
			(*m_creEntity) = *(other.m_creEntity);
			break;
		case 9:
			if(!other.m_createPkiUser)
				break;
			(*m_createPkiUser) = *(other.m_createPkiUser);
			break;
		case 25:
			if(!other.m_crls)
				break;
			(*m_crls) = *(other.m_crls);
			break;
		case 24:
			if(!other.m_csr)
				break;
			(*m_csr) = *(other.m_csr);
			break;
		case 38:
		case 41:
			if(other.m_dn)
			{
				if(m_dn)
					ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
				m_dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), (void*)other.m_dn);
				if(!m_dn)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 42:
			if(!other.m_eeUsers)
				break;
			(*m_eeUsers) = *(other.m_eeUsers);
			break;
		case 4:
			if(!other.m_entities)
				break;
			(*m_entities) = *(other.m_entities);
			break;
		case 20:
			if(!other.m_entitiesLinks)
				break;
			(*m_entitiesLinks) = *(other.m_entitiesLinks);
			break;
		case 17:
			if(!other.m_entityConf)
				break;
			(*m_entityConf) = *(other.m_entityConf);
			break;
		case 0:
			if(!other.m_errors)
				break;
			(*m_errors) = *(other.m_errors);
			break;
		case 26:
			if(!other.m_groups)
				break;
			(*m_groups) = *(other.m_groups);
			break;
		case 27:
		case 33:
		case 39:
			if(!other.m_id)
				break;
			(*m_id) = *(other.m_id);
			break;
		case 32:
			if(!other.m_knownObjects)
				break;
			(*m_knownObjects) = *(other.m_knownObjects);
			break;
		case 36:
			if(!other.m_ldapResults)
				break;
			(*m_ldapResults) = *(other.m_ldapResults);
			break;
		case 31:
			if(!other.m_localEntityConf)
				break;
			(*m_localEntityConf) = *(other.m_localEntityConf);
			break;
		case 5:
			if(!other.m_logs)
				break;
			(*m_logs) = *(other.m_logs);
			break;
		case 16:
			if(!other.m_logsType)
				break;
			(*m_logsType) = *(other.m_logsType);
			break;
		case 18:
			if(!other.m_mailConf)
				break;
			(*m_mailConf) = *(other.m_mailConf);
			break;
		case 12:
			if(other.m_myAcl)
			{
				if(m_myAcl)
					ASN1_item_free((ASN1_VALUE*)m_myAcl, ASN1_ITEM_rptr(ASN1_BIT_STRING));
				m_myAcl = (ASN1_BIT_STRING*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_BIT_STRING), (void*)other.m_myAcl);
				if(!m_myAcl)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 14:
			if(!other.m_myConf)
				break;
			(*m_myConf) = *(other.m_myConf);
			break;
		case 21:
			if(!other.m_objectReqs)
				break;
			(*m_objectReqs) = *(other.m_objectReqs);
			break;
		case 29:
			if(!other.m_objectResps)
				break;
			(*m_objectResps) = *(other.m_objectResps);
			break;
		case 1:
			if(other.m_other)
			{
				if(m_other)
					ASN1_item_free((ASN1_VALUE*)m_other, ASN1_ITEM_rptr(ASN1_NULL));
				m_other = (ASN1_NULL*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_NULL), (void*)other.m_other);
				if(!m_other)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 23:
			if(!other.m_p7b)
				break;
			(*m_p7b) = *(other.m_p7b);
			break;
		case 30:
			if(!other.m_profiles)
				break;
			(*m_profiles) = *(other.m_profiles);
			break;
		case 15:
			if(!other.m_repositories)
				break;
			(*m_repositories) = *(other.m_repositories);
			break;
		case 7:
			if(!other.m_signEntity)
				break;
			(*m_signEntity) = *(other.m_signEntity);
			break;
		case 10:
		case 35:
		case 37:
			if(!other.m_status)
				break;
			(*m_status) = *(other.m_status);
			break;
		case 28:
			if(!other.m_transactionid)
				break;
			(*m_transactionid) = *(other.m_transactionid);
			break;
		case 3:
			if(!other.m_users)
				break;
			(*m_users) = *(other.m_users);
			break;
		case 8:
			if(!other.m_usersCert)
				break;
			(*m_usersCert) = *(other.m_usersCert);
			break;
		case 2:
			if(!other.m_usertype)
				break;
			(*m_usertype) = *(other.m_usertype);
			break;
		case 40:
			if(!other.m_webuserCerts)
				break;
			(*m_webuserCerts) = *(other.m_webuserCerts);
			break;
	}
	m_isOk=true;
	return true;
}



const ASN1_ITEM * AdminResponseBody::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ADMIN_RESPONSE_BODY);
}
AdminResponse AdminResponse::EmptyInstance;
bool AdminResponse::set_body(const AdminResponseBody & c_body)
{
	m_body = c_body;
	return true;
}

const AdminResponseBody & AdminResponse::get_body() const
{
	return m_body;
}

AdminResponseBody & AdminResponse::get_body()
{
	return m_body;
}

bool AdminResponse::set_sigAlgo(const X509_ALGOR * c_sigAlgo)
{
	if(m_sigAlgo)
		ASN1_item_free((ASN1_VALUE*)m_sigAlgo, ASN1_ITEM_rptr(X509_ALGOR));
	m_sigAlgo = (X509_ALGOR*)ASN1_item_dup(ASN1_ITEM_rptr(X509_ALGOR), (void*)c_sigAlgo);
	if(!m_sigAlgo)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const X509_ALGOR * AdminResponse::get_sigAlgo() const
{
	if(!m_sigAlgo)
		((AdminResponse*)this)->m_sigAlgo = (X509_ALGOR*)ASN1_item_new(ASN1_ITEM_rptr(X509_ALGOR));
	return m_sigAlgo;
}

X509_ALGOR * AdminResponse::get_sigAlgo()
{
	if(!m_sigAlgo)
		m_sigAlgo = (X509_ALGOR*)ASN1_item_new(ASN1_ITEM_rptr(X509_ALGOR));
	return m_sigAlgo;
}

bool AdminResponse::set_signature(const ASN1_BIT_STRING * c_signature)
{
	if(m_signature)
		ASN1_item_free((ASN1_VALUE*)m_signature, ASN1_ITEM_rptr(ASN1_BIT_STRING));
	m_signature = (ASN1_BIT_STRING*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_BIT_STRING), (void*)c_signature);
	if(!m_signature)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const ASN1_BIT_STRING * AdminResponse::get_signature() const
{
	if(!m_signature)
		((AdminResponse*)this)->m_signature = (ASN1_BIT_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_BIT_STRING));
	return m_signature;
}

ASN1_BIT_STRING * AdminResponse::get_signature()
{
	if(!m_signature)
		m_signature = (ASN1_BIT_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_BIT_STRING));
	return m_signature;
}

bool AdminResponse::set_signingCert(const PKI_CERT & c_signingCert)
{
	m_signingCert = c_signingCert;
	return true;
}

const PKI_CERT & AdminResponse::get_signingCert() const
{
	return m_signingCert;
}

PKI_CERT & AdminResponse::get_signingCert()
{
	return m_signingCert;
}

AdminResponse::AdminResponse():NewPKIObject()
{
	resetAll();
}

AdminResponse::AdminResponse(const AdminResponse & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

AdminResponse::~AdminResponse()
{
	Clear();
}

void AdminResponse::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void AdminResponse::freeAll()
{
	if(m_sigAlgo)
	{
		ASN1_item_free((ASN1_VALUE*)m_sigAlgo, ASN1_ITEM_rptr(X509_ALGOR));
	}
	if(m_signature)
	{
		ASN1_item_free((ASN1_VALUE*)m_signature, ASN1_ITEM_rptr(ASN1_BIT_STRING));
	}
}

void AdminResponse::resetAll()
{
	m_body.Clear();
	m_sigAlgo = NULL;
	m_signature = NULL;
	m_signingCert.Clear();
}

bool AdminResponse::load_Datas(const ADMIN_RESPONSE * Datas)
{
	Clear();
	if(Datas->body)
	{
		if(!m_body.load_Datas(Datas->body))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->sig_algo)
	{
		if(m_sigAlgo)
			ASN1_item_free((ASN1_VALUE*)m_sigAlgo, ASN1_ITEM_rptr(X509_ALGOR));
		m_sigAlgo = (X509_ALGOR*)ASN1_item_dup(ASN1_ITEM_rptr(X509_ALGOR), Datas->sig_algo);
		if(!m_sigAlgo)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	if(Datas->signature)
	{
		if(m_signature)
			ASN1_item_free((ASN1_VALUE*)m_signature, ASN1_ITEM_rptr(ASN1_BIT_STRING));
		m_signature = (ASN1_BIT_STRING*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_BIT_STRING), Datas->signature);
		if(!m_signature)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	if(Datas->signing_cert)
	{
		if(!m_signingCert.load_Datas(Datas->signing_cert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool AdminResponse::give_Datas(ADMIN_RESPONSE ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ADMIN_RESPONSE*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->body && !((*Datas)->body = (ADMIN_RESPONSE_BODY*)ASN1_item_new(ASN1_ITEM_rptr(ADMIN_RESPONSE_BODY))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_body.give_Datas(&(*Datas)->body))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->body, ASN1_ITEM_rptr(ADMIN_RESPONSE_BODY));
		(*Datas)->body = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(m_sigAlgo)
	{
		if((*Datas)->sig_algo)
			ASN1_item_free((ASN1_VALUE*)(*Datas)->sig_algo, ASN1_ITEM_rptr(X509_ALGOR));
		if(!((*Datas)->sig_algo = (X509_ALGOR*)ASN1_item_dup(ASN1_ITEM_rptr(X509_ALGOR), (ASN1_VALUE*)m_sigAlgo)))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	else
	{
		if(!(*Datas)->sig_algo)
		{
			(*Datas)->sig_algo = (X509_ALGOR*)ASN1_item_new(ASN1_ITEM_rptr(X509_ALGOR));
			if(!(*Datas)->sig_algo)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
		}
	}
	if(m_signature)
	{
		if((*Datas)->signature)
			ASN1_item_free((ASN1_VALUE*)(*Datas)->signature, ASN1_ITEM_rptr(ASN1_BIT_STRING));
		if(!((*Datas)->signature = (ASN1_BIT_STRING*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_BIT_STRING), (ASN1_VALUE*)m_signature)))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	else
	{
		if(!(*Datas)->signature)
		{
			(*Datas)->signature = (ASN1_BIT_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_BIT_STRING));
			if(!(*Datas)->signature)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
		}
	}
	if(!(*Datas)->signing_cert && !((*Datas)->signing_cert = (X509*)ASN1_item_new(ASN1_ITEM_rptr(X509))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_signingCert.give_Datas(&(*Datas)->signing_cert))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->signing_cert, ASN1_ITEM_rptr(X509));
		(*Datas)->signing_cert = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool AdminResponse::operator=(const AdminResponse & other)
{
	Clear();
	m_body = other.m_body;
	if(other.m_sigAlgo)
	{
		if(m_sigAlgo)
			ASN1_item_free((ASN1_VALUE*)m_sigAlgo, ASN1_ITEM_rptr(X509_ALGOR));
		m_sigAlgo = (X509_ALGOR*)ASN1_item_dup(ASN1_ITEM_rptr(X509_ALGOR), (void*)other.m_sigAlgo);
		if(!m_sigAlgo)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(other.m_signature)
	{
		if(m_signature)
			ASN1_item_free((ASN1_VALUE*)m_signature, ASN1_ITEM_rptr(ASN1_BIT_STRING));
		m_signature = (ASN1_BIT_STRING*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_BIT_STRING), (void*)other.m_signature);
		if(!m_signature)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_signingCert = other.m_signingCert;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * AdminResponse::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ADMIN_RESPONSE);
}
