/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// NewPKIObject.cpp: implementation of the NewPKIObject class.
//
//////////////////////////////////////////////////////////////////////

#include "NewPKIObject.h"
#include <PKI_ERR.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

NewPKIObject::NewPKIObject()
{
	m_isOk = false;
}

NewPKIObject::~NewPKIObject()
{

}

bool NewPKIObject::STACK_cpy(const ASN1_ITEM *item, const STACK *in, STACK *out)
{
	void * currValue;
	int i;

	STACK_empty(item, out);

	for(i=0; i<sk_num(in); i++)
	{
		currValue = sk_value(in, i);
		if(!currValue)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			STACK_empty(item, out);
			return false;
		}

		currValue = ASN1_item_dup(item, currValue);
		if(!currValue)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			STACK_empty(item, out);
			return false;
		}
		if(sk_push(out, (char*)currValue) < 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ASN1_item_free((ASN1_VALUE*)currValue, item);
			return false;
		}
	}
	return true;
}

void NewPKIObject::STACK_empty(const ASN1_ITEM *item, STACK *in)
{
	void * currValue;
	int i;
	for(i=0; i<sk_num(in); i++)
	{
		currValue = sk_value(in, i);
		if(currValue)
			ASN1_item_free((ASN1_VALUE*)currValue, item);
	}
}

void NewPKIObject::STACK_free(const ASN1_ITEM *item, STACK *in)
{
	STACK_empty(item, in);
	sk_free(in);
}

STACK * NewPKIObject::STACK_dup(const ASN1_ITEM *item, const STACK *in)
{
	STACK * out;
	out = sk_new_null();
	if(!out)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
		return NULL;
	}
	if(!STACK_cpy(item, in, out))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		STACK_empty(item, out);
		sk_free(out);
		return NULL;
	}

	return out;
}

NewPKIObject::operator int() const
{
	return (isOK()?1:0);
}



bool NewPKIObject::isOK() const
{
	return m_isOk;
}

void NewPKIObject::set_isOK()
{
	m_isOk = true;
}


bool NewPKIObject::Private_toPEM(const mString & name, const ASN1_ITEM *it, ASN1_VALUE * datas, mString &pem_datas) const
{
	unsigned char * buffer;
	unsigned char * pem;
	unsigned char * p;
	int bufferlen;
	int outl, outdataslen;
	EVP_ENCODE_CTX ctx;

	if((bufferlen = ASN1_item_i2d(datas, NULL, it)) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(buffer = (unsigned char*)malloc(bufferlen)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	p = buffer;
	if((bufferlen = ASN1_item_i2d(datas, &p, it)) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		free(buffer);
		return false;
	}

	if(!(pem = (unsigned char*)malloc(bufferlen*2)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		free(buffer);
		return false;
	}

	EVP_EncodeInit(&ctx);
	EVP_EncodeUpdate(&ctx, (unsigned char*)pem, &outdataslen, buffer, bufferlen);
	EVP_EncodeFinal(&ctx, (unsigned char*)&pem[outdataslen], &outl);
	free(buffer);

	pem_datas = "-----BEGIN ";
	pem_datas += name;
	pem_datas += "-----\n";
	pem_datas += pem;
	pem_datas += "-----END ";
	pem_datas += name;
	pem_datas += "-----\n";
	free(pem);
	return true;
}


bool NewPKIObject::Private_fromPEM(const mString & name, const ASN1_ITEM *it, ASN1_VALUE ** datas, const mString &pem_datas)
{
	unsigned char * buffer;
	unsigned char * p;
	int outl, i;
	EVP_ENCODE_CTX ctx;
	mString tmpStr;
	int from;
	int to;

	tmpStr = "-----BEGIN ";
	tmpStr += name;
	tmpStr += "-----\n";
	from = tmpStr.size();
	if(!(pem_datas.Left(from) == tmpStr))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	tmpStr = "-----END ";
	tmpStr += name;
	tmpStr += "-----\n";
	to = tmpStr.size();
	if(!(pem_datas.Right(to) == tmpStr))
	{
		tmpStr = pem_datas.Right(to);
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	tmpStr = pem_datas.Mid(from, pem_datas.size() - from - to);

	buffer=(unsigned char *)malloc(tmpStr.size());
	if(!buffer)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	
	EVP_DecodeInit(&ctx);
	if(EVP_DecodeUpdate(&ctx, buffer, &outl,(unsigned char *)tmpStr.c_str(), tmpStr.size()) == -1)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		free(buffer);
		return false;
	}

	if(EVP_DecodeFinal(&ctx, &(buffer[outl]), &i) == -1)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		free(buffer);
		return false;
	}
	outl += i;


	p = buffer;
	if(! (*datas = ASN1_item_d2i(NULL, &p, outl, it)) )
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		free(buffer);
		return false;
	}
	free(buffer);
	
	return true;
}
