/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// CriticalSection.cpp: implementation of the CriticalSection class.
//
//////////////////////////////////////////////////////////////////////

#include "CriticalSection.h"
#include <assert.h>
#include <string.h>

#ifndef _WIN32
	#include "../config.h"
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CriticalSection::CriticalSection()
{
	#ifdef WIN32
		hMutex = CreateMutex(NULL, FALSE, NULL);
		assert(hMutex != NULL);
	#else
		pthread_mutexattr_init(&recmutex);
		pthread_mutexattr_settype(&recmutex, PT_MUTEX_RECURSIVE);
		pthread_mutex_init(&lpCriticalSection, &recmutex);
	#endif
	#ifdef _DEBUG
		m_stIndex = 0;
		memset(&m_stLockers, 0, sizeof(m_stLockers));
	#endif
}

CriticalSection::~CriticalSection()
{
	#ifdef WIN32
		if(hMutex)
			CloseHandle(hMutex);
	#else
		pthread_mutex_destroy(&lpCriticalSection);
		pthread_mutexattr_destroy(&recmutex);
	#endif
}

#ifdef _DEBUG
	void CriticalSection::DoEnterCS(const char * file, int line) const
#else
	void CriticalSection::EnterCS() const
#endif
{
	#ifdef WIN32
		assert(hMutex != NULL);
		if(hMutex)
			WaitForSingleObject(hMutex, INFINITE);

	#else
		pthread_mutex_lock((pthread_mutex_t*)&lpCriticalSection);
	#endif
	#ifdef _DEBUG
		((CriticalSection*)this)->m_stLockers[m_stIndex].file = file;
		((CriticalSection*)this)->m_stLockers[m_stIndex].line = line;
		((CriticalSection*)this)->m_stIndex++;
	#endif
}

void CriticalSection::LeaveCS() const
{
	#ifdef _DEBUG
		assert(m_stIndex > 0);
		((CriticalSection*)this)->m_stIndex--;
//		((CriticalSection*)this)->m_stLockers[m_stIndex].file = NULL;
//		((CriticalSection*)this)->m_stLockers[m_stIndex].line = 0;
	#endif
	#ifdef WIN32
		assert(hMutex != NULL);
		if(hMutex)
			ReleaseMutex(hMutex);
	#else
		pthread_mutex_unlock((pthread_mutex_t*)&lpCriticalSection);
	#endif
}
