/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// HashCorrelation.cpp: implementation of the HashCorrelation class.
//
//////////////////////////////////////////////////////////////////////
#include <NewPKI.h>
#include "HashCorrelation.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

HashCorrelation::HashCorrelation()
{
}

HashCorrelation::~HashCorrelation()
{
}

void HashCorrelation::SetCert(const PKI_CERT & cert)
{
	m_cert = cert;
}

bool HashCorrelation::SetCurrentHash(const mString & hash)
{
	Asn1EncryptSign crypted_hash;
	if(!hash.size())
		return true;
	if(!m_cert)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	
	if(!crypted_hash.from_PEM(hash))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	//We now have the hash, let's decrypt it
	const PKI_RSA & rootKey = m_cert.GetPrivateKey();
	if(!rootKey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}

	//Decrypt the body and verify signature
	if(!ActualHash.from_SignEncrypt(crypted_hash, rootKey.GetRsaKey(), rootKey.GetRsaKey()))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool HashCorrelation::AddEntry(const char * id, int id_len, const char * datas, int dataslen) const
{
	char * entry_hash;
	char * full_hash;

	// We first compute the id and the datas together
	entry_hash = HashDatas(id, id_len, datas, dataslen, id_len);
	if(!entry_hash)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!ActualHash)
	{
		((HashCorrelation*)this)->ActualHash.get_hash().Assign((unsigned char*)entry_hash, id_len);
	}
	else
	{
		full_hash = HashDatas(entry_hash, id_len, (char*)ActualHash.get_hash().get_Buffer(), ActualHash.get_hash().get_BufferLen(), id_len);
		if(!full_hash)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			free(entry_hash);
			return false;
		}
		((HashCorrelation*)this)->ActualHash.get_hash().Assign((unsigned char*)full_hash, id_len);
		free(entry_hash);
	}
	return true;
}

bool HashCorrelation::AddEntry(unsigned long id, const char * datas, int dataslen) const
{
	int str_id_len;
	char str_id[STR_LONG_LEN];

	str_id_len = sprintf(str_id, "%ld", id);
	return AddEntry(str_id, str_id_len, datas, dataslen);
}

char * HashCorrelation::HashDatas(const char *datas1, int len1, const char *datas2, int len2, int & resultlen) const
{
	int i;
	const char * dst_datas;
	const char * src_datas;
	char * str_hash;
	int min;

	if(len1 > len2)
	{
		dst_datas = datas1;
		src_datas = datas2;
		resultlen = len1;
		min = len2;
	}
	else
	{
		dst_datas = datas2;
		src_datas = datas1;
		resultlen = len2;
		min = len1;
	}

	str_hash = (char*)malloc(resultlen);
	if(!str_hash)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return NULL;
	}
	memcpy(str_hash, dst_datas, resultlen);

	for(i=0; i<min; i++)
	{
		str_hash[i] ^= src_datas[i];
	}

	return str_hash;
}

bool HashCorrelation::operator==(const HashCorrelation &other)
{
	if(!ActualHash.get_hash().get_BufferLen() || !other.ActualHash.get_hash().get_BufferLen())
		return false;

	if(ActualHash.get_hash().get_BufferLen() != other.ActualHash.get_hash().get_BufferLen())
		return false;

	return (memcmp(ActualHash.get_hash().get_Buffer(), other.ActualHash.get_hash().get_Buffer(), other.ActualHash.get_hash().get_BufferLen())==0)?true:false;
}

HashCorrelation::operator int()
{
	if(!m_cert)
		return 0;
	else
		return 1;
}


bool HashCorrelation::GetHash(mString & pem_hash) const
{
	Asn1EncryptSign crypted_hash;

	if(!m_cert)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	
	pem_hash = "";
	if(!ActualHash)
		return true;


	const PKI_RSA & rootKey = m_cert.GetPrivateKey();
	if(!rootKey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}

	//Ok, now we're all ready to sign and crypt for ourself the whole think
	//Encrypt the body and generate signature
	if(!ActualHash.to_SignEncrypt(crypted_hash, rootKey.GetRsaKey(), rootKey.GetRsaKey(), EVP_sha1(), EVP_des_ede3_cbc()) ) 
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//We have everything we can now convert it to PEM
	if(!crypted_hash.to_PEM(pem_hash))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

void HashCorrelation::Clear()
{
	ActualHash.Clear();
}
