/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// HashTable_Dn.cpp: implementation of the HashTable_Dn class.
//
//////////////////////////////////////////////////////////////////////

#include "HashTable_Dn.h"
#include <PKI_ERR.h>


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


HashTable_Dn::HashTable_Dn()
{
	AllowDuplicateNames();
}

HashTable_Dn::~HashTable_Dn()
{
	Clear();
}

const char * HashTable_Dn::Get(long Pos) const
{
	return (const char *)m_GetPTR(Pos);
}

bool HashTable_Dn::Add(const char *Name, const char *Value)
{
	Name = FormatObject(Name);
	return m_Add(Name, Value, strlen(Value)+1);
}

bool HashTable_Dn::Modify(long Pos, const char *Value)
{
	return m_Modify(Pos, Value, strlen(Value)+1);
}

bool HashTable_Dn::operator=(const HashTable_Dn &other)
{
	return PKI_HashTable::operator=(other);
}


bool HashTable_Dn::From_X509_NAME(const X509_NAME * name)
{
	int i;
	ASN1_OBJECT *obj;
	X509_NAME_ENTRY *ne;
	ASN1_STRING *str;
	char cf_name[80];
	char * cf_value;
	Clear();

	for (i=0; i<X509_NAME_entry_count((X509_NAME*)name); i++)
	{
		ne=(X509_NAME_ENTRY *)X509_NAME_get_entry((X509_NAME*)name,i);
		obj=X509_NAME_ENTRY_get_object(ne);

		if(OBJ_obj2txt(cf_name, sizeof(cf_name), obj, 0) <= 0)
		{
			i2t_ASN1_OBJECT(cf_name, sizeof(cf_name),obj);	
		}
		str=X509_NAME_ENTRY_get_data(ne);

		cf_value=(char *)malloc(str->length+1);
		if(!cf_value)
		{
			return false;
		}
		memcpy(cf_value,str->data,str->length);
		cf_value[str->length]=0;
		if(!*cf_value && str->length)
		{
			free(cf_value);
			return false;
		}

		if(!Add(cf_name, cf_value))
		{
			free(cf_value);
			return false;
		}
		free(cf_value);
	}
	
	return true;
}

bool HashTable_Dn::To_X509_NAME(X509_NAME *name) const
{
	int nid;
	int NumFields;
	const char * DN_field;
	const char * DN_value;
	int i;
	const char * p;
	X509_NAME_ENTRY * entry;

	while( (entry = X509_NAME_delete_entry(name, 0)) )
	{
		X509_NAME_ENTRY_free(entry);
	}

	NumFields = EntriesCount();
	for(i=0; i < NumFields; i++)
	{
		DN_field=GetName(i);
		//Si le champ du DN est inconnu on ne l'inclus pas
		if (!IsValidObject(DN_field)) continue;

		p = FormatObject(DN_field);
		if ((nid=OBJ_txt2nid(p)) == NID_undef) continue;


		DN_value=Get(i);
		if(!DN_value) continue;

		if(!X509_NAME_add_entry_by_NID(name, nid, MBSTRING_ASC,(unsigned char *)DN_value, -1,-1,0))
		{
			return false;
		}
	}

	if (X509_NAME_entry_count(name) == 0)
	{
		return false;
	}
	return true;
}


bool HashTable_Dn::ValidateAgainstPolicy(const HashTable_String &Policies)
{
	const char * pol_field_name;
	const char * pol_field_value;
	const char * real_field_name;
	int i;
	const char * dn_field_value;
	char * MatchEntry;
	bool WasFound;
	bool IsPresent;
	DN_TYPE TypeDn;
	long LastPos;
	long LastIndex;
	HashTable_Dn RequestDN;

	if(!EntriesCount() || !Policies.EntriesCount())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	RequestDN = *this;
	if(RequestDN.EntriesCount() != EntriesCount())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}

	for(i=0; i < Policies.EntriesCount(); i++)
	{
		real_field_name=Policies.GetName(i);
		if(!real_field_name)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_POLICY_FORMAT);
			return false;
		}
		pol_field_value=Policies.Get(i);
		if(!pol_field_value)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_POLICY_FORMAT);
			return false;
		}

		//Let's get the type of policy
		TypeDn = HashTable_String::IsValidPolicyField(real_field_name, pol_field_value);
		if(TypeDn == DN_TYPE_UNKNOWN)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		pol_field_name = FormatObject(real_field_name);

		LastPos = HASHTABLE_NOT_FOUND;
		IsPresent = false;

		LastIndex = 0;
		while( (LastPos = RequestDN.SeekEntryName(pol_field_name, LastPos)) != HASHTABLE_NOT_FOUND)
		{
			LastIndex = LastPos;
			// We seek the DN value
			dn_field_value=RequestDN.Get(LastPos);

			// Is the field set ?
			IsPresent = (dn_field_value && strlen(dn_field_value));
		
			// If he must be supplied but is not, it's an error
			if(!IsPresent && (TypeDn == DN_TYPE_MATCH || TypeDn == DN_TYPE_SUPPLIED))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_DN_FIELD_UNPRESENT);
				ERR_set_error_data(strdup(real_field_name), ERR_TXT_MALLOCED | ERR_TXT_STRING);
				return false;
			}
			else
			{
				if(!IsPresent && TypeDn == DN_TYPE_OPTIONAL)
				{
					// C'est un champ optionel
					continue;
				}
			}

			if(TypeDn == DN_TYPE_MATCH)
			{
				WasFound=false;
				// On verifie que la valeur du dn corresponde a un champs possible
				MatchEntry=strtok((char*)&(pol_field_value[strlen(STR_MATCH)]), "," );
				while( MatchEntry != NULL )
				{
					if(strcmp(MatchEntry,dn_field_value)==0)
					{
						WasFound=true;
						break;
					}
					MatchEntry = strtok( NULL, "," );
				}
				if(!WasFound)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_DN_FIELD_BAD_MATCH);
					ERR_set_error_data(strdup((char*)&(pol_field_value[strlen(STR_MATCH)])), ERR_TXT_MALLOCED | ERR_TXT_STRING);
					return false;
				}
			}
		}

		// S'il est obligatoire et qu'il n'est pas present
		// C'est une erreur		
		if(!IsPresent && (TypeDn == DN_TYPE_MATCH || TypeDn == DN_TYPE_SUPPLIED))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_DN_FIELD_UNPRESENT);
			ERR_set_error_data(strdup(real_field_name), ERR_TXT_MALLOCED | ERR_TXT_STRING);
			return false;
		}
		if(LastIndex)
			RequestDN.Delete(LastIndex);
	}

	return true;
}
