/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PKI_EXT.cpp: implementation of the PKI_EXT class.
//
//////////////////////////////////////////////////////////////////////

#include "PKI_EXT.h"
#include <ctype.h>
#include <mString.h>
#ifndef __STDC__
	#define __STDC__ 1
	#define UNDEF__STDC__
#endif
extern "C" {
#include <regex.h>
}
#ifdef UNDEF__STDC__
	#undef __STDC__
#endif
#include "lbintl.h"
#include <PKI_ERR.h>
#include <ASN1/Asn1Helper.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


PKI_EXT::PKI_EXT()
{
}

PKI_EXT::~PKI_EXT()
{
}


char * PKI_EXT::newpki_strip_spaces(char *name)
{
	char *p, *q;
	/* Skip over leading spaces */
	p = name;
	while(*p && isspace((unsigned char)*p)) p++;
	if(!*p) return NULL;
	q = p + strlen(p) - 1;
	while((q != p) && isspace((unsigned char)*q)) q--;
	if(p != q) q[1] = 0;
	if(!*p) return NULL;
	return p;
}


STACK_OF(CONF_VALUE) * PKI_EXT::newpki_X509V3_parse_list(const char *line)
{
	char *p, *q, *ptr, c;
	char *ntmp, *vtmp;
	STACK_OF(CONF_VALUE) *values = NULL;
	char *linebuf;
	int state;
	/* We are going to modify the line so copy it first */
	linebuf = BUF_strdup(line);
	state = 1;
	ntmp = NULL;
	/* Go through all characters */
	for(p = linebuf, q = linebuf; (c = *p) && (c!='\r') && (c!='\n'); p++) {

		switch(state) {
			case 1:
			if(c == ':') {
				state = 2;
				*p = 0;
				ntmp = newpki_strip_spaces(q);
				if(!ntmp) {
					X509V3err(X509V3_F_X509V3_PARSE_LIST, X509V3_R_INVALID_NULL_NAME);
					goto err;
				}
				q = p + 1;
			} else if(c == ',') {
				*p = 0;
				ntmp = newpki_strip_spaces(q);
				q = p + 1;
				if(!ntmp) {
					X509V3err(X509V3_F_X509V3_PARSE_LIST, X509V3_R_INVALID_NULL_NAME);
					goto err;
				}
				X509V3_add_value(ntmp, NULL, &values);
			}
			break ;

			case 2:
			if(c == ',')
			{
				if(*(p+1) != ',')
				{
					state = 1;
					*p = 0;
					vtmp = newpki_strip_spaces(q);

					if(!vtmp) {
						X509V3err(X509V3_F_X509V3_PARSE_LIST, X509V3_R_INVALID_NULL_VALUE);
						goto err;
					}
					X509V3_add_value(ntmp, vtmp, &values);
					ntmp = NULL;
					q = p + 1;
				}
				else
				{
					for(ptr = p+1; *ptr; ptr++)
					{
						*ptr = *(ptr+1);
					}
				}
			}
		}
	}

	if(state == 2) {
		vtmp = newpki_strip_spaces(q);
		if(!vtmp) {
			X509V3err(X509V3_F_X509V3_PARSE_LIST, X509V3_R_INVALID_NULL_VALUE);
			goto err;
		}
		X509V3_add_value(ntmp, vtmp, &values);
	} else {
		ntmp = newpki_strip_spaces(q);
		if(!ntmp) {
			X509V3err(X509V3_F_X509V3_PARSE_LIST, X509V3_R_INVALID_NULL_NAME);
			goto err;
		}
		X509V3_add_value(ntmp, NULL, &values);
	}
OPENSSL_free(linebuf);
return values;

err:
OPENSSL_free(linebuf);
sk_CONF_VALUE_pop_free(values, X509V3_conf_free);
return NULL;

}


X509_EXTENSION * PKI_EXT::newpki_do_ext_i2d(X509V3_EXT_METHOD *method, int ext_nid,  int crit, void *ext_struc)
{
	unsigned char *ext_der;
	int ext_len;
	ASN1_OCTET_STRING *ext_oct;
	X509_EXTENSION *ext;
	/* Convert internal representation to DER */
	if (method->it)
	{
		ext_der = NULL;
		ext_len = ASN1_item_i2d((ASN1_VALUE *)ext_struc, &ext_der, ASN1_ITEM_ptr(method->it));
		if (ext_len < 0) goto merr;
	}
	 else
	{
		unsigned char *p;
		ext_len = method->i2d(ext_struc, NULL);
		if(!(ext_der = (unsigned char*)OPENSSL_malloc(ext_len)))
			goto merr;
		p = ext_der;
		method->i2d(ext_struc, &p);
	}
	if (!(ext_oct = M_ASN1_OCTET_STRING_new()))
		goto merr;
	ext_oct->data = ext_der;
	ext_oct->length = ext_len;

	ext = X509_EXTENSION_create_by_NID(NULL, ext_nid, crit, ext_oct);
	if (!ext) goto merr;
	M_ASN1_OCTET_STRING_free(ext_oct);

	return ext;

	merr:
	X509V3err(X509V3_F_DO_EXT_I2D,ERR_R_MALLOC_FAILURE);
	return NULL;
}



X509_EXTENSION * PKI_EXT::newpki_do_ext(X509V3_CTX *ctx, int ext_nid, int crit, char *value)
{
	X509V3_EXT_METHOD *method;
	X509_EXTENSION *ext;
	STACK_OF(CONF_VALUE) *nval;
	void *ext_struc;
	if (ext_nid == NID_undef)
	{
		X509V3err(X509V3_F_DO_EXT_CONF,X509V3_R_UNKNOWN_EXTENSION_NAME);
		return NULL;
	}
	if (!(method = X509V3_EXT_get_nid(ext_nid)))
	{
		X509V3err(X509V3_F_DO_EXT_CONF,X509V3_R_UNKNOWN_EXTENSION);
		return NULL;
	}
	/* Now get internal extension representation based on type */
	if (method->v2i)
	{
		nval = newpki_X509V3_parse_list(value);
		if(!nval)
		{
			X509V3err(X509V3_F_X509V3_EXT_CONF,X509V3_R_INVALID_EXTENSION_STRING);
			ERR_add_error_data(4, "name=", OBJ_nid2sn(ext_nid), ",section=", value);
			return NULL;
		}
		ext_struc = method->v2i(method, ctx, nval);
		sk_CONF_VALUE_pop_free(nval, X509V3_conf_free);
		if(!ext_struc)
			return NULL;
	}
	else if(method->s2i)
	{
		if(!(ext_struc = method->s2i(method, ctx, value)))
			return NULL;
	}
	else if(method->r2i)
	{
		if(!ctx->db)
		{
			X509V3err(X509V3_F_X509V3_EXT_CONF,X509V3_R_NO_CONFIG_DATABASE);
			return NULL;
		}
		if(!(ext_struc = method->r2i(method, ctx, value)))
			return NULL;
	}
	else
	{
		X509V3err(X509V3_F_X509V3_EXT_CONF,X509V3_R_EXTENSION_SETTING_NOT_SUPPORTED);
		ERR_add_error_data(2, "name=", OBJ_nid2sn(ext_nid));
		return NULL;
	}

	ext = newpki_do_ext_i2d(method, ext_nid, crit, ext_struc);
	if(method->it)
		ASN1_item_free((ASN1_VALUE *)ext_struc, ASN1_ITEM_ptr(method->it));
	else
		method->ext_free(ext_struc);
	return ext;
}


int PKI_EXT::newpki_v3_check_critical(char **value)
{
	char *p = *value;
	if ((strlen(p) < 9) || strncmp(p, "critical,", 9)) return 0;
	p+=9;
	while(isspace((unsigned char)*p)) p++;
	*value = p;
	return 1;
}

int PKI_EXT::newpki_v3_check_generic(char **value)
{
	char *p = *value;
	if ((strlen(p) < 4) || strncmp(p, "DER:,", 4)) return 0;
	p+=4;
	while (isspace((unsigned char)*p)) p++;
	*value = p;
	return 1;
}

X509_EXTENSION * PKI_EXT::newpki_v3_generic_extension(const char *ext, char *value, int crit, int type)
{
	unsigned char *ext_der=NULL;
	long ext_len;
	ASN1_OBJECT *obj=NULL;
	ASN1_OCTET_STRING *oct=NULL;
	X509_EXTENSION *extension=NULL;
	if (!(obj = OBJ_txt2obj(ext, 0)))
		{
		goto err;
		}

	if (!(ext_der = string_to_hex(value, &ext_len)))
	{
		goto err;
		}

	if (!(oct = M_ASN1_OCTET_STRING_new()))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		goto err;
	}

	oct->data = ext_der;
	oct->length = ext_len;
	ext_der = NULL;

	extension = X509_EXTENSION_create_by_OBJ(NULL, obj, crit, oct);

	err:
	ASN1_OBJECT_free(obj);
	M_ASN1_OCTET_STRING_free(oct);
	if(ext_der) OPENSSL_free(ext_der);
	return extension;
}


X509_EXTENSION * PKI_EXT::GetExtensionValue(X509V3_CTX *ctx, const char * Name, const char * Value)
{
	int crit;
	int ext_type;
	X509_EXTENSION *ret;
	crit = newpki_v3_check_critical((char**)&Value);

	if ( (ext_type = newpki_v3_check_generic((char**)&Value)) ) 
	{
		return newpki_v3_generic_extension((char*)Name, (char*)Value, crit, ext_type);
	}

	ret = newpki_do_ext(ctx, OBJ_sn2nid((char*)Name), crit, (char*)Value);
	if (!ret)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return NULL;
	}
	return ret;
}


bool copy_dn_field(X509V3_CTX *ctx, const mString & Name, mString & Value, int move_p)
{
	X509_NAME *nm;
	ASN1_IA5STRING * tmpValue = NULL;
	X509_NAME_ENTRY *ne;
	int i;
	int nid;

	if(ctx->flags == CTX_TEST) return 1;
	if(!ctx || (!ctx->subject_cert && !ctx->subject_req))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}


	nid = OBJ_txt2nid(Name.c_str());
	if(nid == NID_undef)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		ERR_add_error_data(1, Name.c_str());
		return false;
	}

	/* Find the subject name */
	if(ctx->subject_cert)
		nm = X509_get_subject_name(ctx->subject_cert);
	else
		nm = X509_REQ_get_subject_name(ctx->subject_req);

	if(!nm)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	/* Now add any email address(es) to STACK */
	i = X509_NAME_get_index_by_NID(nm, nid, -1);
	if(i < 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		ERR_add_error_data(1, Name.c_str());
		return false;
	}
	
	ne = X509_NAME_get_entry(nm, i);
	if(!ne)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}


	tmpValue = X509_NAME_ENTRY_get_data(ne);
	if(!tmpValue)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}

	Value = ASN1_STRING_GET(tmpValue);
	if(!Value.size())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		ERR_add_error_data(1, Name.c_str());
		return false;
	}

	if(move_p)
	{
		X509_NAME_delete_entry(nm, i);
	}

	return true;
}

const char * PKI_EXT::GetExtensionName(ASN1_OBJECT *obj)
{
	int nid;
	nid = OBJ_obj2nid(obj);
	static char buffer[512];

	if(nid == NID_undef)
	{
		// This is an OID
		if(OBJ_obj2txt(buffer, sizeof(buffer), obj, 1) <= 0)
			return NULL;
		return buffer;
	}
	else
	{
		return OBJ_nid2sn(nid);
	}
}

X509_EXTENSION * PKI_EXT::GetExtension(const STACK_OF(X509_EXTENSION) * Exts, const char * name)
{
	int i;
	ASN1_OBJECT *obj;
	X509_EXTENSION * ex;
	const char * extname;
	
	for (i=0; i<sk_X509_EXTENSION_num(Exts); i++)
	{
		ex=sk_X509_EXTENSION_value(Exts, i);

		obj=X509_EXTENSION_get_object(ex);
		if(!obj)
			continue;
		if(! (extname = GetExtensionName(obj)) )
			continue;
		if(strcmp(extname, name) == 0)
			return ex;

	}
	return NULL;
}

bool PKI_EXT::Add_CertExtensions(const HashTable_String * Ext, X509V3_CTX * ctx, bool keep_csr_exts, bool csr_exts_overwrite, X509 * cert)
{
	X509_EXTENSION *ext;
	long i,j;
	const char * name;
	const char * name2;
	const char * value;
	regex_t dn_regexp;
	regmatch_t pm[4];
	int res;
	mString dnValue;
	mString eValue;
	mString dnName;
	mString dnAction;
	int move;
	bool ret = true;
	bool freeExt;
	STACK_OF(X509_EXTENSION) * CsrExts;
	ASN1_OBJECT *obj;

	// Clear up any previous extensions
	if (cert->cert_info->extensions != NULL)
	{
		sk_X509_EXTENSION_pop_free(cert->cert_info->extensions,X509_EXTENSION_free);
		cert->cert_info->extensions = NULL;
	}

	// We directly copy the CSR's exts
	if(!Ext && keep_csr_exts)
	{
		if(ctx->subject_req)
			cert->cert_info->extensions = X509_REQ_get_extensions(ctx->subject_req);
		return true;
	}
	
	// Let's compile the regular expression.
	if( (res = regcomp(&dn_regexp, DN_EXT_RE, REG_EXTENDED)) != 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		ERR_add_error_data(1, _lb("Failed to compile regexp"));
		return false;
	}

	// Get the CSR's extensions
	if(keep_csr_exts && ctx->subject_req)
	{
		CsrExts = X509_REQ_get_extensions(ctx->subject_req);
	}
	else
	{
		CsrExts = NULL;
	}

	for(i=0;i<Ext->EntriesCount();i++)
	{
		name=Ext->GetName(i);
		value=Ext->Get(i);
		if(!name || !value)
			continue;
		name = FormatObject(name);

		// Are we supposed to handle the CSR's extensions ?
		if(CsrExts && !csr_exts_overwrite)
		{
			// Get extension value from CSR
			ext = GetExtension(CsrExts, name);
			freeExt = false;
		}
		else
		{
			// We expand the extension
			ext = NULL;
		}

		if(!ext)
		{
			dnValue = value;
			// Is it a special value ?
			if( (res = regexec(&dn_regexp, dnValue.c_str(), 4, pm, 0)) == 0)
			{
				dnName = FormatObject(dnValue.Mid(pm[2].rm_so, pm[2].rm_eo - pm[2].rm_so).c_str());
				dnAction = dnValue.Mid(pm[3].rm_so, pm[3].rm_eo - pm[3].rm_so);
				dnValue = dnValue.Mid(pm[1].rm_so, pm[1].rm_eo - pm[1].rm_so);
				if(dnAction == "copy")
					move = 0;
				else if(dnAction == "move")
					move = 1;
				else
					move = 0;

				if(!copy_dn_field(ctx, dnName, eValue, move))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					ret = false;
					break;
				}
				dnValue+=eValue;
			}
			// Verify the extension
			if(!VerifyExtension(name, dnValue.c_str()))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_EXT_SYNTAX);
				ERR_add_error_data(1, name);
				ret = false;
				break;
			}
			// Expand the extension
			if(!(ext = GetExtensionValue(ctx, name, dnValue.c_str())))
			{
				ret = false;
				break;
			}
			freeExt = true;
		}
		
		// Add expanded extension to certificate
		if(cert && !X509_add_ext(cert, ext, -1))
		{
			if(freeExt)
				X509_EXTENSION_free(ext);
			ret = false;
			break;
		}
		if(freeExt)
			X509_EXTENSION_free(ext);
	}
	// Now add the CSR extensions
	if(CsrExts)
	{
		// Now add the CSR's extensions that haven't yet been
		// added
		for (i=0; i<sk_X509_EXTENSION_num(CsrExts); i++)
		{
			ext=sk_X509_EXTENSION_value(CsrExts, i);

			obj=X509_EXTENSION_get_object(ext);
			if(!obj)
				continue;

			if(! (name = GetExtensionName(obj)) )
				continue;

			// Search if extension already present
			for(j=0;j<Ext->EntriesCount();j++)
			{
				name2=Ext->GetName(j);
				if(!name2)
					continue;
				name2 = FormatObject(name2);
				if(strcmp(name, name2) == 0)
					break;
			}
			// Extension already present
			if(j != Ext->EntriesCount())
				continue;


			// Add expanded extension to certificate
			if(cert && !X509_add_ext(cert, ext, -1))
			{
				ret = false;
				break;
			}
		}
		sk_X509_EXTENSION_pop_free(CsrExts,X509_EXTENSION_free);
	}
	regfree(&dn_regexp);
	return ret;
}

bool PKI_EXT::Add_CrlExtensions(const HashTable_String * Ext, X509V3_CTX *ctx, X509_CRL *crl)
{
	X509_EXTENSION *ext;
	long i;
	const char * name;
	const char * value;

	for(i=0;i<Ext->EntriesCount();i++)
	{
		name=Ext->GetName(i);
		value=Ext->Get(i);
		if(!name || !value) continue;
		name = FormatObject(name);

		if(!VerifyExtension(name, value))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_EXT_SYNTAX);
			ERR_add_error_data(1, name);
			return false;
		}


		if(!(ext = GetExtensionValue(ctx, name, value)))
		{
			return false;
		}

		if(crl)
		{
			if(!X509_CRL_add_ext(crl, ext, -1))
			{
				X509_EXTENSION_free(ext);
				return false;
			}
		}
		X509_EXTENSION_free(ext);
	}
	
	return true;
}

bool PKI_EXT::VerifyExtensionValue(const char *FieldName, const char *FieldValue)
{
	X509V3_CTX ctx;
	X509V3_set_ctx_test(&ctx);
	X509_EXTENSION *ext;
	regex_t dn_regexp;
	regmatch_t pm[4];
	int res;
	ERR_STATE * es;
	int top;

	es = ERR_get_state();
	top = es->top;
	if(!IsValidObject(FieldName))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		ERR_add_error_data(2, _lb("Invalid object"), FieldName);
		return false;
	}
	FieldName = FormatObject(FieldName);

	
	// Let's compile the regular expression.
	if( (res = regcomp(&dn_regexp, DN_EXT_RE, REG_EXTENDED)) != 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		ERR_add_error_data(1, _lb("Failed to compile regexp"));
		return false;
	}

	// We validate the value only if it's not special
	if( (res = regexec(&dn_regexp, FieldValue, 4, pm, 0)) != 0)
	{
		if (!(ext = GetExtensionValue(&ctx, FieldName, FieldValue)))
		{
			regfree(&dn_regexp);
			ERR_clear_error();
			return false;
		}
		X509_EXTENSION_free(ext);
	}
	regfree(&dn_regexp);
	es->top = top;
	return true;
}


bool PKI_EXT::VerifyExtension(const char *FieldName, const char *FieldValue)
{
	X509V3_CTX ctx;
	X509V3_set_ctx_test(&ctx);
	X509_EXTENSION *ext;
	ERR_STATE * es;
	int top;

	es = ERR_get_state();
	top = es->top;

	if(!IsValidObject(FieldName))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		ERR_add_error_data(2, _lb("Invalid object"), FieldName);
		return false;
	}
	FieldName = FormatObject(FieldName);

	if (!(ext = GetExtensionValue(&ctx, FieldName, FieldValue)))
	{
		es->top = top;
		return false;
	}
	X509_EXTENSION_free(ext);

	es->top = top;
	return true;
}


bool PKI_EXT::Load(const STACK_OF(X509_EXTENSION) * Exts)
{
	int i;
	ASN1_OBJECT *obj;
	X509_EXTENSION * ex;
	char * extvalue;
	const char * extname;
	BIO * membio;


	m_Exts.Clear();
	m_Exts.AllowDuplicateNames();
	
	for (i=0; i<sk_X509_EXTENSION_num(Exts); i++)
	{
		ex=sk_X509_EXTENSION_value(Exts, i);

		obj=X509_EXTENSION_get_object(ex);
		if(!obj) continue;

		if(! (extname = GetExtensionName(obj)) )
			continue;

		membio = BIO_new(BIO_s_mem());
		if(!membio) continue;

		if(X509_EXTENSION_get_critical(ex))
		{
			BIO_printf(membio, "%s", "critical, ");
		}
					
		if(!X509V3_EXT_print(membio, ex, X509V3_EXT_MULTILINE, 0))
		{
			M_ASN1_OCTET_STRING_print(membio,ex->value);
		}

		extvalue = (char *)malloc(BIO_number_written(membio) + 1);
		if(!extvalue)
		{
			BIO_free_all(membio);
			continue;
		}

		BIO_read(membio, extvalue, BIO_number_written(membio));
		extvalue[BIO_number_written(membio)]=0;
					
		BIO_free_all(membio);

		m_Exts.Add(extname, extvalue);
		free(extvalue);
	}

	return true;
}

const HashTable_String & PKI_EXT::GetExts() const
{
	return m_Exts;
}

void PKI_EXT::Clear()
{
	m_Exts.Clear();
}

bool PKI_EXT::operator=( const PKI_EXT &other )
{
	m_Exts = other.m_Exts;

	return true;
}
