/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/

// PKI_PLUG_CA.h: interface for AC plug
//
//////////////////////////////////////////////////////////////////////

#ifndef PKI_PLUG_CA_H
#define PKI_PLUG_CA_H

#include <PKI_CERT.h>
#include <PKI_CSR.h>
#include <PKI_CRL.h>
#include <HashTable/HashTable_String.h>
#include "SmtpClient.h"

/*!
  This class allows to access a CA Plug-in
*/
class PKI_PLUG_CA
{
public:
	/*! \brief This is the destructor.
	 */
	virtual ~PKI_PLUG_CA()
	{
	}
	
	/*! \brief This function is called to validate the plugin options.
	 *  \param Options [IN] The options.
	 *  \return true on success, false on failure.
	 */
	virtual bool ValidateOptions(HashTable_String & Options)=0;

	/*! \brief This function is called when a new certification request arrives to the CA.
	 *  \param Options [IN] The options.
	 *  \param CaCert [IN] The CA cert.
	 *  \param Mailer [IN] The SMTP client.
	 *  \param Csr [IN] The CSR.
	 *  \return true on success, false on failure.
	 */
	virtual bool OnNewRequest(HashTable_String & Options, const PKI_CERT & CaCert, const SmtpClient & Mailer, const PKI_CSR & Csr)=0;

	/*! \brief This function is called when a new certificate has been generated.
	 *  \param Options [IN] The options.
	 *  \param CaCert [IN] The CA cert.
	 *  \param Mailer [IN] The SMTP client.
	 *  \param Cert [IN] The certificate.
	 *  \return true on success, false on failure.
	 */
	virtual bool OnNewCertificate(HashTable_String & Options, const PKI_CERT & CaCert, const SmtpClient & Mailer, const PKI_CERT & Cert)=0;

	/*! \brief This function is called when a new revocation request arrives to the CA.
	 *  \param Options [IN] The options.
	 *  \param CaCert [IN] The CA cert.
	 *  \param Mailer [IN] The SMTP client.
	 *  \param Serial [IN] The serial to revoke.
	 *  \return true on success, false on failure.
	 */
	virtual bool OnNewRevocation(HashTable_String & Options, const PKI_CERT & CaCert, const SmtpClient & Mailer, unsigned long Serial)=0;

	/*! \brief This function is called when a new CRL has been generated.
	 *  \param Options [IN] The options.
	 *  \param CaCert [IN] The CA cert.
	 *  \param Mailer [IN] The SMTP client.
	 *  \param Crl [IN] The CRL.
	 *  \return true on success, false on failure.
	 */
	virtual bool OnNewCRL(HashTable_String & Options, const PKI_CERT & CaCert, const SmtpClient & Mailer, const PKI_CRL & Crl)=0;
};

typedef PKI_PLUG_CA * (*GET_PKI_PLUG_CA)(); //!< Function is called to request the creation of a CA plugin
typedef void (*FREE_PKI_PLUG_CA)(PKI_PLUG_CA * plug); //!< Function is called to request the destruction of a CA plugin

#endif
