/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PkiClient.cpp: implementation of the PkiClient class.
//
//////////////////////////////////////////////////////////////////////
#ifdef _WIN32
#pragma warning(disable:4786)
#endif

#include "PkiClient.h"
#include <openssl/md5.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

vector<SOCKET> PkiClient::ListAllConnections;
CriticalSection PkiClient::ListAllConnectionsLock;
CriticalSection PkiClient::DatasCounterLock;
unsigned long PkiClient::m_RcvDatas=0;
unsigned long PkiClient::m_SndDatas=0;
SessionsCache PkiClient::m_clientSessions = SessionsCache();


PkiClient::PkiClient(PROC_WAIT_CALLBACK * WaitFunction)
{
	m_connection = NULL;
	m_ConnParam.m_MsAuthCert = NULL;
	m_WaitFunction = WaitFunction;
	m_ReadTimeOut = 0;

}

PkiClient::~PkiClient()
{
	CloseConnection();
#if defined(_WIN32) && !defined(NO_BIO_MS_SSL)
	if(m_ConnParam.m_MsAuthCert)
		CertFreeCertificateContext(m_ConnParam.m_MsAuthCert);;
#endif
	ClearErrors();
}

void PkiClient::SetReadTimeOut(int ReadTimeOut)
{
	m_ReadTimeOut = ReadTimeOut;
}

bool PkiClient::Connect(const mString & Server, unsigned int Port, const PKI_CERT & AuthCert, PCCERT_CONTEXT MsAuthCert)
{
	ERR_clear_error();

	m_ConnParam.Server = Server;
	m_ConnParam.Port = Port;
#if defined(_WIN32) && !defined(NO_BIO_MS_SSL)
	m_ConnParam.m_MsAuthCert = CertDuplicateCertificateContext(MsAuthCert);;
#else
	m_ConnParam.m_MsAuthCert = NULL;
#endif
	if(AuthCert)
		m_ConnParam.m_AuthCert = AuthCert;
	
	return Reconnect();
}


const char * PkiClient::GetError()
{
	size_t i;
	
	ERR_clear_error();

	for(i=0; i<errors.size(); i++)
	{
		ERR_put_error(errors[i].get_lib(), errors[i].get_function(), errors[i].get_code(), errors[i].get_file().c_str(), errors[i].get_line());
		if(errors[i].get_data().size())
		{
			ERR_add_error_data(1, errors[i].get_data().c_str());
		}
	}

	ERR_to_mstring(str_Error);

	return str_Error.c_str();
}


void PkiClient::ClearErrors()
{
	str_Error = "";
	errors.clear();
	ERR_clear_error();
}

void PkiClient::PackThreadErrors()
{
	ErrorEntry newError;
	const char *file,*data=NULL;
	unsigned long lastError;
	int flags,m_line;

	errors.clear();
	while( (lastError = ERR_get_error_line_data(&file,&m_line,&data,&flags)) )
	{
		newError.set_lib(ERR_GET_LIB(lastError));
		newError.set_code(ERR_GET_REASON(lastError));
		newError.set_function(ERR_GET_FUNC(lastError));
		if(data && (flags & ERR_TXT_STRING) ) newError.set_data(data);
		if(file) newError.set_file(file);
		newError.set_line(m_line);
		errors.push_back(newError);
	}
	ERR_clear_error();
}


void PkiClient::GetPeerCertificate(PKI_CERT & PeerCert)
{
	PeerCert = m_PeerCert;
}

void PkiClient::GetEntityCert(PKI_CERT & entity_cert)
{
	if(!m_connection) return;
	entity_cert = m_connection->get_EntityCert();
}

const mString & PkiClient::GetRemoteServer()
{
	return m_ConnParam.Server;
}

unsigned int PkiClient::GetRemotePort()
{
	return m_ConnParam.Port;
}

bool PkiClient::Reconnect()
{
	return DoNetworkConnection(&m_ConnParam);
}

const AdminReqLogin & PkiClient::GetLoginInfo()
{
	return m_loginInfo;
}

bool PkiClient::IsConnected()
{
	return (m_connection != NULL);
}

bool PkiClient::DoNetworkExchange(RequestDatas * ClientParam)
{
	ClearErrors();

	if(m_WaitFunction)
	{
		NewpkiThread ThreadId;
		ThreadId.Create(ThreadProcSocket, ClientParam);
		ClientParam->m_Finished = false;
		ClientParam->me_this = this;


		if(!ThreadId.Start())
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			PackThreadErrors();
			return false;
		}
		
		while(!ClientParam->m_Finished)
		{
			m_WaitFunction();
			NewpkiThread::Sleep(10);
		}
		ThreadId.Stop();
		return ClientParam->m_Success;
	}
	else
	{
		return DoBufferSend(ClientParam);
	}
}


bool PkiClient::DoNetworkConnection(ConnectionParam * ConnParam)
{
	ClearErrors();

	if(m_WaitFunction)
	{
		NewpkiThread ThreadId;
		ThreadId.Create(ThreadProcConnection, ConnParam);

		ConnParam->me_this = this;
		ConnParam->Finished = false;

		if(!ThreadId.Start())
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			PackThreadErrors();
			return false;
		}
		while(!ConnParam->Finished)
		{
			m_WaitFunction();
			NewpkiThread::Sleep(10);
		}
		ThreadId.Stop();
		return ConnParam->Success;
	}
	else
	{
		return DoConnection(ConnParam);
	}
}


void PkiClient::ThreadProcSocket(const NewpkiThread * Thread, void *param)
{
	RequestDatas * ClientParam = (RequestDatas *)param;
	ClientParam->m_Success = ClientParam->me_this->DoBufferSend(ClientParam);
	ClientParam->m_Finished = true;
}

void PkiClient::ThreadProcConnection(const NewpkiThread * Thread, void *param)
{
	ConnectionParam * ConnParam = (ConnectionParam *)param;
	ConnParam->Success = ConnParam->me_this->DoConnection(ConnParam);
	ConnParam->Finished = true;
}


bool PkiClient::DoBufferSend(RequestDatas * ClientParam)
{
	ClearErrors();


	if(!m_connection->SendRequest(ClientParam->m_request, ClientParam->m_response, m_ReadTimeOut))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(ClientParam->m_response.get_body().get_type() == ADMIN_RESP_TYPE_ERRORS)
	{
		errors = ClientParam->m_response.get_body().get_errors();
		ClientParam->m_response.Clear();
		return false;
	}
	return true;
}

bool PkiClient::DoConnection(ConnectionParam * ConnParam)
{
	ClearErrors();
	CloseConnection();


#if defined(_WIN32) && !defined(NO_BIO_MS_SSL)
	SspiConnection * sspi;
	//We use MS SSL API
	if(ConnParam->m_MsAuthCert)
	{
		if( !(sspi = new SspiConnection()) )
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			PackThreadErrors();
			CloseConnection();
			return false;
		}

		sspi->set_host(ConnParam->Server);
		sspi->set_port(ConnParam->Port);

		if(!sspi->set_certificate(ConnParam->m_MsAuthCert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			PackThreadErrors();
			CloseConnection();
			delete sspi;
			return false;
		}
		if(!sspi->connect())
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			PackThreadErrors();
			CloseConnection();
			delete sspi;
			return false;
		}
		if(!sspi->GetPeerCertificate(m_PeerCert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			PackThreadErrors();
			CloseConnection();
			delete sspi;
			return false;
		}
		m_connection = sspi;
	}
	else
#endif
	{
		SslConnection * ssl;
		try
		{
			if( !(ssl = new SslConnection(m_clientSessions)) )
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				PackThreadErrors();
				CloseConnection();
				return false;
			}
		}
		catch(ExceptionNewPKI e)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			PackThreadErrors();
			CloseConnection();
			return false;
		}

		ssl->enable_Cache(true);
		ssl->set_host(ConnParam->Server);
		ssl->set_port(ConnParam->Port);
		
		if(ConnParam->m_AuthCert && !ssl->set_certificate(ConnParam->m_AuthCert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			PackThreadErrors();
			CloseConnection();
			delete ssl;
			return false;
		}
		if(!ssl->connect())
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			PackThreadErrors();
			CloseConnection();
			delete ssl;
			return false;
		}
		if(!ssl->GetPeerCertificate(m_PeerCert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			PackThreadErrors();
			CloseConnection();
			delete ssl;
			return false;
		}
		m_connection = ssl;
	}

	ListAllConnectionsLock.EnterCS();
	ListAllConnections.push_back(m_connection->get_socket());
	ListAllConnectionsLock.LeaveCS();

	return true;
}


bool PkiClient::UserLogin(const AdminReqLogin & Login, int & UserType)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_LOGIN);

	m_loginInfo = Login;

	if(!Request.get_body().set_login(Login))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_USER_TYPE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	UserType = NET_PARAM_RESPONSE.get_body().get_usertype();
	return true;
}

bool PkiClient::EnumEntities(mVector<EntityEntryInfo> & entities)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_ENUM_ENTITIES);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_ENUM_ENTITIES)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	entities = NET_PARAM_RESPONSE.get_body().get_entities();
	return true;
}

bool PkiClient::CreateEntity(const EntityCreationReq & cre_entity, EntityCreationResp & creation_resp)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_CREATE_ENTITY);

	if(!Request.get_body().set_creEntity(cre_entity))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_CREATE_ENTITY)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	creation_resp = NET_PARAM_RESPONSE.get_body().get_creEntity();
	return true;
}

bool PkiClient::DeleteEntity(const mString & Name)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_DELETE_ENTITY);

	if(!Request.get_body().set_entityName(Name))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::LoadEntity(const mString & Name)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_LOAD_ENTITY);

	if(!Request.get_body().set_entityName(Name))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::UnloadEntity(const mString & Name)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_UNLOAD_ENTITY);

	if(!Request.get_body().set_entityName(Name))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::EnumLOG(const AdminReqEnumLogs & Filters, mVector<LogEntry> & Logs)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_ENUM_LOGS);

	if(!Request.get_body().set_enumLogs(Filters))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_ENUM_LOGS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	Logs = NET_PARAM_RESPONSE.get_body().get_logs();
	return true;
}

bool PkiClient::GetLogsCount(const AdminReqEnumLogs & Filters, unsigned long & LogsCount)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_LOGS_COUNT);

	if(!Request.get_body().set_enumLogs(Filters))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_LOGS_COUNT)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	LogsCount = NET_PARAM_RESPONSE.get_body().get_status();
	return true;
}

bool PkiClient::EnumUsers(mVector<UserEntry> & users)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_ENUM_USERS);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_ENUM_USERS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	users = NET_PARAM_RESPONSE.get_body().get_users();
	return true;
}

bool PkiClient::CreateUser(const UserInfo & user)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_CREATE_USER);

	if(!Request.get_body().set_creUser(user))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::UpdateUser(const UserEntry & user)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_UPDATE_USER);

	if(!Request.get_body().set_updUser(user))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::ChangeUserPassword(const ChangeUserPasswd & Password)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_CHG_USER_PASSWD);

	if(!Request.get_body().set_chgUserPwd(Password))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::ChangePassword(const ChangePasswd & Password)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_CHG_PASSWD);

	if(!Request.get_body().set_chgPwd(Password))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::SignEntity(const EntitySignatureReq & sign_entity, EntitySignatureResp & resp)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_SIGN_ENTITY);

	if(!Request.get_body().set_signEntity(sign_entity))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_SIGN_ENTITY)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	resp = NET_PARAM_RESPONSE.get_body().get_signEntity();
	return true;
}

bool PkiClient::EnumUsersCert(mVector<UserEntryCert> & users_cert)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_ENUM_USERS);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_ENUM_USERS_CERT)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	users_cert = NET_PARAM_RESPONSE.get_body().get_usersCert();
	return true;
}

bool PkiClient::CreatePkiUser(const CreatePkiUserRequest & req, CreatePkiUserResponse & resp)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_CREATE_PKI_USER);

	if(!Request.get_body().set_createPkiUser(req))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_CREATE_PKI_USER)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	resp = NET_PARAM_RESPONSE.get_body().get_createPkiUser();
	return true;
}

bool PkiClient::RevokeUserCert(unsigned long serial)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_REVOKE_USER);

	if(!Request.get_body().set_serial(serial))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::RevokeEntityCert(unsigned long serial)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_REVOKE_ENTITY);

	if(!Request.get_body().set_serial(serial))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::PushConfiguration(const ExportedPkiConf & conf)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_PUSH_CONFIG);

	if(!Request.get_body().set_conf(conf))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::SetOfflineState(bool offline)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_SET_OFFLINE);

	if(!Request.get_body().set_offline(offline?1:0))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::GetOfflineState(bool & offline)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_OFFLINE);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_STATUS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	offline = NET_PARAM_RESPONSE.get_body().get_status()?true:false;
	return true;
}

bool PkiClient::GetMyACL(ASN1_BIT_STRING **acl)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_MY_ACL);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_MY_ACL)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	*acl = (ASN1_BIT_STRING *)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_BIT_STRING), (void*)NET_PARAM_RESPONSE.get_body().get_myAcl());
	return true;
}

bool PkiClient::GetConfiguration(ExportedPkiConf & conf)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_CONFIG);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_CONF)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	conf = NET_PARAM_RESPONSE.get_body().get_conf();
	return true;
}

bool PkiClient::InitEntity(const EntityInitReq & init_entity)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_INIT_ENTITY);

	if(!Request.get_body().set_initEntity(init_entity))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::GetRepositoriesList(mVector<RepEntryInfo> & repositories)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_REPOSITORIES);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_GET_REPOSITORIES)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	repositories = NET_PARAM_RESPONSE.get_body().get_repositories();
	return true;
}

bool PkiClient::SetRepositoriesList(const mVector<RepEntryInfo> & repositories)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_SET_REPOSITORIES);

	if(!Request.get_body().set_repositories(repositories))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}


bool PkiClient::GetLogsType(mVector<unsigned long> & LogsType)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_LOGS_TYPE);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_LOGS_TYPE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	LogsType = NET_PARAM_RESPONSE.get_body().get_logsType();
	return true;
}


bool PkiClient::GetEntityConf(const PKI_CERT & entity_cert, EntityConfBody & conf)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_ENTITY_CONF);

	if(!Request.get_body().set_entityCert(entity_cert))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_ENTITY_CONF)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	conf = NET_PARAM_RESPONSE.get_body().get_entityConf();
	return true;
}

bool PkiClient::ResourceLock(const mString & ResourceName)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_LOCK_RESOURCE);

	if(!Request.get_body().set_resourceName(ResourceName))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::ResourceUnlock(const mString & ResourceName)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_UNLOCK_RESOURCE);

	if(!Request.get_body().set_resourceName(ResourceName))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::GetEntityAcl(const PKI_CERT & entity_cert, X509Acl & acl)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_ENTITY_ACL);

	if(!Request.get_body().set_entityCert(entity_cert))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_ACLS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	acl = NET_PARAM_RESPONSE.get_body().get_aclsConf();
	return true;
}

bool PkiClient::GetEntityMailConf(const PKI_CERT & entity_cert, EmailConf & mail_conf)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_ENTITY_MAIL_CONF);

	if(!Request.get_body().set_entityCert(entity_cert))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_MAIL_CONF)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	mail_conf = NET_PARAM_RESPONSE.get_body().get_mailConf();
	return true;
}

bool PkiClient::SetEntityMailConf(const PKI_CERT & entity_cert, const EmailConf & mail_conf)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_SET_ENTITY_MAIL_CONF);

	if(!Request.get_body().get_entityMailConf().set_entityCert(entity_cert))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}
	if(!Request.get_body().get_entityMailConf().set_mailConf(mail_conf))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::SetEntityAcl(const PKI_CERT & entity_cert, const mVector<AclEntry> & acl_entries)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_SET_ENTITY_ACL);

	if(!Request.get_body().get_entityAcl().set_entityCert(entity_cert))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}
	if(!Request.get_body().get_entityAcl().set_aclEntries(acl_entries))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::SetEntityConf(const PKI_CERT & entity_cert, const EntityConfBody & conf)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_SET_ENTITY_CONF);

	if(!Request.get_body().get_entityConf().set_entityCert(entity_cert))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}
	if(!Request.get_body().get_entityConf().set_conf(conf))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::AdminSendMail(const MailDatas & mail)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_SEND_ADMIN_MAIL);

	if(!Request.get_body().set_adminMail(mail))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}


bool PkiClient::GetEntityAudits(const PKI_CERT & entity_cert, mVector<EntityAuditEntry> & audits)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_ENTITY_AUDITS);

	if(!Request.get_body().set_entityCert(entity_cert))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_AUDITS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	audits = NET_PARAM_RESPONSE.get_body().get_audits();
	return true;
}

bool PkiClient::SetEntityAudits(const PKI_CERT & entity_cert, const mVector<EntityAuditEntry> & audits)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_SET_ENTITY_AUDITS);

	if(!Request.get_body().get_entityAudits().set_entityCert(entity_cert))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}
	if(!Request.get_body().get_entityAudits().set_audits(audits))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::GetEntityLogsType(int EntityType, mVector<unsigned long> & LogsType)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_ENTITY_LOGS_TYPE);

	if(!Request.get_body().set_entityType(EntityType))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_LOGS_TYPE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	LogsType = NET_PARAM_RESPONSE.get_body().get_logsType();
	return true;
}

bool PkiClient::SetEntitiesLinks(const mVector<EntityLinks> & links)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_SET_ENTITIES_LINKS);

	if(!Request.get_body().set_entitiesLinks(links))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::GetEntitiesLinks(mVector<EntityLinks> &links)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_ENTITIES_LINKS);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_ENTITIES_LINKS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	links = NET_PARAM_RESPONSE.get_body().get_entitiesLinks();
	return true;
}

bool PkiClient::GetMyConf(EntityConfCrypted & myConf)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_MY_CONFIG);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_MY_CONF)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	myConf = NET_PARAM_RESPONSE.get_body().get_myConf();
	return true;
}

bool PkiClient::PushWaitingObject(const WaitingNewpkiObject & obj, int & Status)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_PUSH_NEWPKI_OBJECTS);

	if(!Request.get_body().set_waitingObj(obj))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_OBJECT_STATUS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	Status = NET_PARAM_RESPONSE.get_body().get_status();
	return true;
}

bool PkiClient::GetCaStatus(unsigned long & status)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_CA_STATUS);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_STATUS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	status = NET_PARAM_RESPONSE.get_body().get_status();
	return true;
}

bool PkiClient::CreateCaRoot(const ReqCreateRootCa & ca_info)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_CREATE_ROOT_CA);

	if(!Request.get_body().set_createRootCa(ca_info))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::GetMyRequests(const TransactionIds & transactions_ids, CryptedNewpkiRequests & objects)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_MY_REQS);

	if(!Request.get_body().set_transactionsIds(transactions_ids))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_REQS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	objects = NET_PARAM_RESPONSE.get_body().get_objectReqs();
	return true;
}

bool PkiClient::GetMyResponses(const TransactionIds & transactions_ids, CryptedNewpkiResponses & objects)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_MY_RESPS);

	if(!Request.get_body().set_transactionsIds(transactions_ids))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_RESPS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	objects = NET_PARAM_RESPONSE.get_body().get_objectResps();
	return true;
}

bool PkiClient::SendRequest(const CryptedNewpkiRequest & req)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_PUSH_REQ);

	if(!Request.get_body().set_newReq(req))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}


bool PkiClient::SendResponse(const CryptedNewpkiResponse & resp)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_PUSH_RESP);

	if(!Request.get_body().set_newResp(resp))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::DeleteResponse(const Asn1OctetString & transactionID)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_DELETE_RESPONSE);

	if(!Request.get_body().set_transactionId(transactionID))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

void PkiClient::CancelAllConnections()
{
	ListAllConnectionsLock.EnterCS();
	SOCKET hSocket;
	unsigned int i;	
	for(i=0; i<ListAllConnections.size(); i++)
	{
		hSocket = ListAllConnections[i];
		if(!hSocket) continue;
		closesocket(hSocket);
	}
	ListAllConnections.clear();
	ListAllConnectionsLock.LeaveCS();
	m_clientSessions.clean();
}

void PkiClient::CloseConnection()
{
	unsigned int i;
	if(m_connection)
	{
		m_connection->close();
		ListAllConnectionsLock.EnterCS();
		for(i=0; i<ListAllConnections.size(); i++)
		{
			if(ListAllConnections[i] == m_connection->get_socket())
			{
				ListAllConnections.erase(ListAllConnections.begin()+i);
				break;
			}
		}
		ListAllConnectionsLock.LeaveCS();
		delete m_connection;
		m_connection = NULL;
	}
}

bool PkiClient::EnumCERT(long index, long num, CERT_STATE state, mVector<InternalCaCert> & certs)
{
	ObjectsEnum enumobjs;
	INIT_NET_PARAM(ADMIN_REQ_TYPE_ENUM_CERTS);

	enumobjs.set_index(index);
	enumobjs.set_state(state);
	enumobjs.set_num(num);

	if(!Request.get_body().set_enumObjects(enumobjs))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_CERTS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	certs = NET_PARAM_RESPONSE.get_body().get_certs();
	return true;
}

bool PkiClient::EnumCrls(long index, long num, mVector<PKI_CRL> & crls)
{
	ObjectsEnum enumobjs;
	INIT_NET_PARAM(ADMIN_REQ_TYPE_ENUM_CRLS);

	enumobjs.set_state(0);
	enumobjs.set_index(index);
	enumobjs.set_num(num);

	if(!Request.get_body().set_enumObjects(enumobjs))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_CRLS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	crls = NET_PARAM_RESPONSE.get_body().get_crls();
	return true;
}


bool PkiClient::SignCSR(const PKI_CSR & req, int Days, const mString & uid, PKI_P7B & p7b)
{
	SignCsr signreq;
	INIT_NET_PARAM(ADMIN_REQ_TYPE_SIGN_CSR);

	signreq.set_days(Days);
	signreq.set_uid(uid);
	if(!signreq.set_request(req))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}
	if(!Request.get_body().set_signCsr(signreq))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_P7B)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	p7b = NET_PARAM_RESPONSE.get_body().get_p7b();
	return true;
}

bool PkiClient::CreateCaChild(const ReqCreateChildCa & ca_info, PKI_CSR & csr)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_CREATE_CHILD_CA);

	if(!Request.get_body().set_createChildCa(ca_info))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_CSR)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	csr = NET_PARAM_RESPONSE.get_body().get_csr();
	return true;
}


bool PkiClient::ImportChildCaCert(const PKI_P7B & certs)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_IMPORT_CHILD_CA_CERT);

	if(!Request.get_body().set_p7b(certs))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::ImportCaP12(const PKI_PKCS12 & p12, const mString & password)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_IMPORT_CA_P12);

	if(!Request.get_body().get_p12Import().set_p12(p12))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}
	if(!Request.get_body().get_p12Import().set_password(password))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::RevokeCert(unsigned long serial)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_REVOKE_CERT);

	if(!Request.get_body().set_serial(serial))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::SuspendCert(unsigned long serial)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_SUSPEND_CERT);

	if(!Request.get_body().set_serial(serial))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::UnsuspendCert(unsigned long serial)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_UNSUSPEND_CERT);

	if(!Request.get_body().set_serial(serial))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::GetGroups(mVector<UsersGroup> & Groups)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_GROUPS);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_GROUPS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	Groups = NET_PARAM_RESPONSE.get_body().get_groups();
	return true;
}

bool PkiClient::SetGroups(const mVector<UsersGroup> & Groups)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_SET_GROUPS);

	if(!Request.get_body().set_groups(Groups))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::AddGroup(const mString & Name)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_ADD_GROUP);

	if(!Request.get_body().set_groupName(Name))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::DelGroup(unsigned long id)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_DEL_GROUP);

	if(!Request.get_body().set_groupId(id))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::RenameGroup(unsigned long id, const mString & Name)
{
	SRenameGroup groupname;
	INIT_NET_PARAM(ADMIN_REQ_TYPE_REN_GROUP);

	groupname.set_id(id);
	groupname.set_name(Name);

	if(!Request.get_body().set_renameGroup(groupname))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}


void PkiClient::AddSentBytes(unsigned long bytes)
{
	DatasCounterLock.EnterCS();
	m_SndDatas+=bytes;
	DatasCounterLock.LeaveCS();
}

void PkiClient::AddRecvBytes(unsigned long bytes)
{
	DatasCounterLock.EnterCS();
	m_RcvDatas+=bytes;
	DatasCounterLock.LeaveCS();
}

unsigned long PkiClient::GetRecvBytes()
{
	return m_RcvDatas;
}

unsigned long PkiClient::GetSentBytes()
{
	return m_SndDatas;
}

bool PkiClient::CheckLogsIntegrity()
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_CHECK_LOGS);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::ImportProfile(const NewpkiProfile & Profile, unsigned long & ProfileId)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_IMPORT_PROFILE);

	if(!Request.get_body().set_profile(Profile))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_PROFILE_ID)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	ProfileId = NET_PARAM_RESPONSE.get_body().get_id();
	return true;
}

bool PkiClient::GetLocalConf(EntityConf & conf)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_LOCAL_CONF);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_LOCAL_ENTITY_CONF)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	conf = NET_PARAM_RESPONSE.get_body().get_localEntityConf();
	return true;
}

bool PkiClient::EnumProfiles(long index, long num, 
							 int state, const mString & filter, 
							 mVector<NewpkiProfileDatas> & Profiles)
{
	ObjectsEnum enumobjs;
	INIT_NET_PARAM(ADMIN_REQ_TYPE_ENUM_PROFILES);

	enumobjs.set_index(index);
	enumobjs.set_state(state);
	enumobjs.set_num(num);
	enumobjs.set_filter(filter);

	if(!Request.get_body().set_enumObjects(enumobjs))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_PROFILES)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	Profiles = NET_PARAM_RESPONSE.get_body().get_profiles();
	return true;
}


bool PkiClient::EnumEeUsers(long index, long num, 
							 int state, const mString & filter, 
							 mVector<NewpkiEeUser> & Users)
{
	ObjectsEnum enumobjs;
	INIT_NET_PARAM(ADMIN_REQ_TYPE_ENUM_EE_USERS);

	enumobjs.set_index(index);
	enumobjs.set_state(state);
	enumobjs.set_num(num);
	enumobjs.set_filter(filter);

	if(!Request.get_body().set_enumObjects(enumobjs))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_EE_USERS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	Users = NET_PARAM_RESPONSE.get_body().get_eeUsers();
	return true;
}


bool PkiClient::RequestCertificate(const NewpkiRequestCert & cert_request, unsigned long & ReqId)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_REQUEST_CERT);

	if(!Request.get_body().set_certRequest(cert_request))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_REQ_ID)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	ReqId = NET_PARAM_RESPONSE.get_body().get_id();
	return true;
}

bool PkiClient::GenerateCRL()
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GENERATE_CRL);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::SynchObjects(const ObjectsListSynchro & local, ObjectsListSynchro & remote)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_SYNCH_OBJECTS);

	if(!Request.get_body().set_knownObjects(local))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_KNOWN_OBJECTS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}
	remote = NET_PARAM_RESPONSE.get_body().get_knownObjects();
	return true;
}

bool PkiClient::GetCaP7b(PKI_P7B & p7b)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_CA_P7B);

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_P7B)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}
	p7b = NET_PARAM_RESPONSE.get_body().get_p7b();
	return true;
}


bool PkiClient::DeletePkcs12(unsigned long CertId)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_DELETE_PKCS12);

	if(!Request.get_body().set_serial(CertId))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}
	return true;
}

bool PkiClient::SearchLdap(const mString & Filters, mVector<LdapResult> & Results)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_SEARCH_LDAP);

	if(!Request.get_body().set_ldapSearch(Filters))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_LDAP_RESULTS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}
	Results = NET_PARAM_RESPONSE.get_body().get_ldapResults();
	return true;
}

bool PkiClient::ChangeProfileUID(unsigned long ProfileId, const mString &uid)
{
	ProfileChangeUid changeuid;
	INIT_NET_PARAM(ADMIN_REQ_TYPE_CHANGE_PROFILE_UID);

	changeuid.set_ldapUid(uid);
	changeuid.set_profileId(ProfileId);

	if(!Request.get_body().set_profileUid(changeuid))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::ChangeProfileOwner(unsigned long ProfileId, unsigned long group_id)
{
	ProfileChangeOwner changeowner;
	INIT_NET_PARAM(ADMIN_REQ_TYPE_CHANGE_PROFILE_OWNER);

	changeowner.set_profileId(ProfileId);
	changeowner.set_ownerGroupSerial(group_id);

	if(!Request.get_body().set_profileOwner(changeowner))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::DeleteProfile(unsigned long ProfileId)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_DELETE_PROFILE);

	if(!Request.get_body().set_profileId(ProfileId))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}


bool PkiClient::ChangeProfileDN(unsigned long ProfileId, const X509_NAME * dn)
{
	ProfileChangeDn changedn;
	INIT_NET_PARAM(ADMIN_REQ_TYPE_CHANGE_PROFILE_DN);

	changedn.set_profileId(ProfileId);
	if(!changedn.set_dn(dn))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}
	if(!Request.get_body().set_profileDn(changedn))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::DeleteRequest(unsigned long ReqId)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_DELETE_REQUEST);

	if(!Request.get_body().set_requestId(ReqId))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

X509_NAME * PkiClient::GetDnValidation(const mString & Id)
{
	X509_NAME * dn;
	INIT_NET_PARAM(ADMIN_REQ_TYPE_GET_DN_VALIDATION);

	if(!Request.get_body().set_dnId(Id))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return NULL;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return NULL;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_DN_VALIDATION)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return NULL;
	}

	dn = NET_PARAM_RESPONSE.get_body().get_dn();
	if(!dn)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return NULL;
	}

	dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), (void*)dn);
	if(!dn)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return NULL;
	}
	return dn;
}

bool PkiClient::SetDnValidation(const mString & Id, const X509_NAME * dn)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_DN_VALIDATION);

	if(!Request.get_body().get_dnVal().set_id(Id))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}
	if(!Request.get_body().get_dnVal().set_dn(dn))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}
	return true;
}

bool PkiClient::WebUserLogin(const mString & email, const mString & password, unsigned long & UserId)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_WEBUSER_LOGIN);

	if(!Request.get_body().get_weblogin().set_email(email))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}
	if(!Request.get_body().get_weblogin().set_password(password))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_WEBUSER_ID)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	UserId = NET_PARAM_RESPONSE.get_body().get_id();
	return true;
}

bool PkiClient::WebUserCreate(const mString & email, const mString & password, const X509_NAME * dn)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_WEBUSER_CREATE);

	if(!Request.get_body().get_webcreate().set_email(email))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}
	if(!Request.get_body().get_webcreate().set_password(password))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}
	if(!Request.get_body().get_webcreate().set_dn(dn))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}
	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}
	return true;
}

bool PkiClient::WebUserActivate(const mString & otp)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_WEBUSER_ACTIVATE);

	if(!Request.get_body().set_webactivate(otp))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}
	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}
	return true;
}

bool PkiClient::WebUserRevokeCertificate(const WebuserRevokeCert & rev_request)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_EE_REVOKE_CERT);

	if(!Request.get_body().set_eeCertRevoke(rev_request))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::WebUserRequestCertificate(const NewpkiEeRequestCert & cert_request)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_EE_REQUEST_CERT);

	if(!Request.get_body().set_eeCertRequest(cert_request))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	return true;
}

bool PkiClient::AcceptProfile(unsigned long profile_id)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_ACCEPT_PROFILE);

	if(!Request.get_body().set_profileId(profile_id))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}
	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}
	return true;
}

bool PkiClient::RejectProfile(unsigned long profile_id, const mString & reason)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_REJECT_PROFILE);

	if(!Request.get_body().get_rejectProfile().set_profileId(profile_id))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}
	if(!Request.get_body().get_rejectProfile().set_reason(reason))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}
	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}
	return true;
}

bool PkiClient::WebUserEnumCerts(unsigned long user_id, mVector<NewpkiProfileDatasCert> & certs)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_WEBUSER_ENUM_CERTS);

	if(!Request.get_body().set_profileId(user_id))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_WEBUSER_CERTS)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}	
	certs = NET_PARAM_RESPONSE.get_body().get_webuserCerts();
	return true;
}

bool PkiClient::WebUserGetDn(unsigned long user_id, HashTable_Dn & dn)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_WEBUSER_GET_DN);

	if(!Request.get_body().set_profileId(user_id))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_DN)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}
	if(!dn.From_X509_NAME(NET_PARAM_RESPONSE.get_body().get_dn()))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}
	return true;
}

bool PkiClient::WebUserChangePassword(unsigned long user_id, const mString & oldPassword, const mString & Password)
{
	INIT_NET_PARAM(ADMIN_REQ_TYPE_WEBUSER_CHANGE_PASSWD);

	Request.get_body().get_webChgPasswd().set_userId(user_id);
	if(!Request.get_body().get_webChgPasswd().set_oldPassword(oldPassword))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}
	if(!Request.get_body().get_webChgPasswd().set_password(Password))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		PackThreadErrors();
		return false;
	}

	if(!DoNetworkExchange(&NET_PARAM))
	{
		return false;
	}

	if(NET_PARAM_RESPONSE.get_body().get_type() != ADMIN_RESP_TYPE_NONE)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		PackThreadErrors();
		return false;
	}
	return true;
}
