/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


/* ssl/bio_ms_ssl.c */
#if defined(_WIN32) && !defined(NO_BIO_MS_SSL)

#include "bio_ms_ssl.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <wintrust.h>
#include <schannel.h>

#define SECURITY_WIN32
#include <security.h>
#include <sspi.h>

#define IO_BUFFER_SIZE  0x10000

#define DLL_NAME TEXT("Secur32.dll")
#define NT4_DLL_NAME TEXT("Security.dll")


#include <openssl/bio.h>
#include <openssl/x509.h>
#include <openssl/err.h>
#include "PKI_ERR.h"


#define READ_BUFFER_SIZE 8192

typedef struct bio_ms_ssl_st
{
	HMODULE g_hSecurity ;
	PSecurityFunctionTable g_pSSPI;
	CredHandle phCreds;
	int phCredsInitialized;
	CtxtHandle phContext;
	int phContextInitialized;
	int handshaked;
	BIO * rwbio;
	X509 * serverCert;
    SecPkgContext_StreamSizes Sizes;
    PBYTE pbIoBuffer;
	DWORD cbIoBufferLength;
    PBYTE pbReadBuffer;
	DWORD cbReadBuffer;
    PBYTE pbCurrReadBuffer;
	DWORD cbCurrReadBuffer;
}BIO_MS_SSL;


static void reset_ssl(BIO_MS_SSL * ms_ssl);
static int ms_ssl_write(BIO *h, const char *buf, int num);
static int ms_ssl_read(BIO *h, char *buf, int size);
static int ms_ssl_puts(BIO *h, const char *str);
static long ms_ssl_ctrl(BIO *h, int cmd, long arg1, void *arg2);
static int ms_ssl_new(BIO *h);
static int ms_ssl_free(BIO *data);
static long ms_ssl_callback_ctrl(BIO *h, int cmd, bio_info_cb *fp);

static int LoadSecurityLibrary(BIO_MS_SSL * ms_ssl);
static int CreateCredentials(BIO_MS_SSL * ms_ssl, PCERT_CONTEXT pCertContext);
static int ClientHandshakeLoop(BIO * b, BIO_MS_SSL * ms_ssl, BOOL fDoInitialRead, SecBuffer * pExtraData);

static int PerformClientHandshake(
	BIO			*	b,
    BIO_MS_SSL *	ms_ssl,
    LPSTR           pszServerName,  // in
    SecBuffer *     pExtraData);    // out
static int DoHandShake(BIO * a, BIO_MS_SSL * ms_ssl);

static BIO_METHOD methods_ms_sslp=
{
	BIO_TYPE_SSL,"ms_ssl",
	ms_ssl_write,
	ms_ssl_read,
	ms_ssl_puts,
	NULL, /* ms_ssl_gets, */
	ms_ssl_ctrl,
	ms_ssl_new,
	ms_ssl_free,
	ms_ssl_callback_ctrl,
};

BIO_METHOD *BIO_f_ms_ssl(void)
{
	return(&methods_ms_sslp);
}

static int ms_ssl_new(BIO *bi)
{
	BIO_MS_SSL *ms_ssl;

	ms_ssl=(BIO_MS_SSL *)malloc(sizeof(BIO_MS_SSL));
	if (ms_ssl == NULL)
	{
		BIOerr(BIO_F_SSL_NEW,ERR_R_MALLOC_FAILURE);
		return(0);
	}
	bi->ptr=(char *)ms_ssl;

	memset(ms_ssl,0,sizeof(BIO_MS_SSL));

	ms_ssl->pbReadBuffer = (PBYTE)malloc(READ_BUFFER_SIZE+1);
	ms_ssl->pbCurrReadBuffer = ms_ssl->pbReadBuffer;
	ms_ssl->cbCurrReadBuffer = 0;
	ms_ssl->cbReadBuffer = READ_BUFFER_SIZE;

	if(!LoadSecurityLibrary(ms_ssl))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return(0);
	}

	bi->init=1;
	bi->flags=0;
	return(1);
}

static int ms_ssl_free(BIO *a)
{
	BIO_MS_SSL * ms_ssl;
	if (a == NULL)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return(0);
	}
	ms_ssl=(BIO_MS_SSL *)a->ptr;
	if(!ms_ssl)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return(0);
	}

	reset_ssl(ms_ssl);
	if(ms_ssl->g_hSecurity) FreeLibrary(ms_ssl->g_hSecurity);
	if(ms_ssl->pbIoBuffer) free(ms_ssl->pbIoBuffer);
	if(ms_ssl->rwbio)
	{
		BIO_free_all(ms_ssl->rwbio);
		ms_ssl->rwbio = NULL;
	}

	free(ms_ssl);

	a->ptr=NULL;

	return(1);
}
	
static int ms_ssl_read(BIO *b, char *out, int outl)
{
	BIO_MS_SSL * ms_ssl;
    SECURITY_STATUS scRet;
	INT cbData;
    DWORD cbIoBuffer;
    SecBuffer       Buffers[4];
    SecBuffer *     pExtraBuffer;
    SecBuffer       ExtraBuffer;
    SecBufferDesc   Message;
    INT   i;
	DWORD readSize;
	LPVOID lpMsgBuf;



	if (b == NULL)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return(0);
	}
	ms_ssl=(BIO_MS_SSL *)b->ptr;
	if(!ms_ssl)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return(0);
	}

	
	if(!ms_ssl->handshaked)
	{
		if(!DoHandShake(b, ms_ssl))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
			return 0;
		}
		ms_ssl->handshaked = 1;
	}


	if(outl <= (long)ms_ssl->cbCurrReadBuffer)
	{
		memcpy(out, ms_ssl->pbCurrReadBuffer, outl);
		ms_ssl->cbCurrReadBuffer -= outl;
		if(!ms_ssl->cbCurrReadBuffer)
		{
			ms_ssl->pbCurrReadBuffer = ms_ssl->pbReadBuffer;
		}
		else
		{
			ms_ssl->pbCurrReadBuffer += outl;
		}
		return outl;
	}


	
	cbIoBuffer = 0;
	scRet = SEC_E_INCOMPLETE_MESSAGE;
    while(scRet == SEC_E_INCOMPLETE_MESSAGE)
    {	
		// Read some data.
		if(cbIoBuffer == 0 || scRet == SEC_E_INCOMPLETE_MESSAGE)
		{
			cbData = BIO_read(ms_ssl->rwbio,
						  ms_ssl->pbIoBuffer + cbIoBuffer, 
						  ms_ssl->cbIoBufferLength - cbIoBuffer);
			if(cbData <= 0)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
				return(0);
			}
			cbIoBuffer += cbData;
		}


		// Attempt to decrypt the received data.
		Buffers[0].pvBuffer     = ms_ssl->pbIoBuffer;
		Buffers[0].cbBuffer     = cbIoBuffer;
		Buffers[0].BufferType   = SECBUFFER_DATA;

		Buffers[1].BufferType   = SECBUFFER_EMPTY;
		Buffers[2].BufferType   = SECBUFFER_EMPTY;
		Buffers[3].BufferType   = SECBUFFER_EMPTY;

		Message.ulVersion       = SECBUFFER_VERSION;
		Message.cBuffers        = 4;
		Message.pBuffers        = Buffers;

		scRet = ms_ssl->g_pSSPI->DecryptMessage(&ms_ssl->phContext, &Message, 0, NULL);


		
		//Need to read more datas before decryption
		if(scRet == SEC_E_INCOMPLETE_MESSAGE)
		{
			continue;
		}
		else if( scRet != SEC_E_OK && scRet != SEC_I_RENEGOTIATE)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			FormatMessage(	FORMAT_MESSAGE_FROM_SYSTEM | 
							FORMAT_MESSAGE_IGNORE_INSERTS |
							FORMAT_MESSAGE_ALLOCATE_BUFFER, NULL, scRet, 
							MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
							(LPTSTR) &lpMsgBuf, 0, NULL);
			ERR_add_error_data(1, (LPTSTR)&lpMsgBuf);
			LocalFree( lpMsgBuf );
			return(0);
		}


		// Locate data and (optional) extra buffers.
		pExtraBuffer = NULL;
		for(i = 1; i < 4; i++)
		{

			if(Buffers[i].BufferType == SECBUFFER_DATA)
			{
				readSize = ms_ssl->pbCurrReadBuffer - ms_ssl->pbReadBuffer;

				//Do we need to allocate more memmory for read bufffer ?
				if( (readSize + Buffers[i].cbBuffer) >= ms_ssl->cbReadBuffer)
				{
					ms_ssl->cbReadBuffer += READ_BUFFER_SIZE;
					ms_ssl->pbReadBuffer = (PBYTE)realloc(ms_ssl->pbReadBuffer, ms_ssl->cbReadBuffer);
					if(!ms_ssl->pbReadBuffer)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
						return(0);
					}
					ms_ssl->pbCurrReadBuffer = ms_ssl->pbReadBuffer + readSize;
				}

				//Copy datas to the read buffer
				memcpy(ms_ssl->pbCurrReadBuffer + ms_ssl->cbCurrReadBuffer, Buffers[i].pvBuffer, Buffers[i].cbBuffer);
				//There are more datas to read from read buffer
				ms_ssl->cbCurrReadBuffer += Buffers[i].cbBuffer;
			}

			if(pExtraBuffer == NULL && Buffers[i].BufferType == SECBUFFER_EXTRA)
			{
				pExtraBuffer = &Buffers[i];
			}
		}
		// Move any "extra" data to the input buffer.
		if(pExtraBuffer)
		{
			memcpy(ms_ssl->pbIoBuffer, pExtraBuffer->pvBuffer, pExtraBuffer->cbBuffer);
			cbIoBuffer = pExtraBuffer->cbBuffer;
		}
		else
		{
			cbIoBuffer = 0;
		}

		if(scRet == SEC_I_RENEGOTIATE)
		{
			// The server wants to perform another handshake
			// sequence.
			scRet = ClientHandshakeLoop(b, 
										ms_ssl, 
										FALSE, 
										&ExtraBuffer);
			if(scRet != SEC_E_OK)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				FormatMessage(	FORMAT_MESSAGE_FROM_SYSTEM | 
								FORMAT_MESSAGE_IGNORE_INSERTS |
								FORMAT_MESSAGE_ALLOCATE_BUFFER, NULL, scRet, 
								MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
								(LPTSTR) &lpMsgBuf, 0, NULL);
				ERR_add_error_data(1, (LPTSTR)&lpMsgBuf);
				LocalFree( lpMsgBuf );
				return(0);
			}
			// Move any "extra" data to the input buffer.
			if(ExtraBuffer.pvBuffer)
			{
				memcpy(ms_ssl->pbIoBuffer, ExtraBuffer.pvBuffer, ExtraBuffer.cbBuffer);
				cbIoBuffer = ExtraBuffer.cbBuffer;
			}
		}
	}

	memcpy(out, ms_ssl->pbCurrReadBuffer, outl);
	ms_ssl->cbCurrReadBuffer -= outl;
	if(!ms_ssl->cbCurrReadBuffer)
	{
		ms_ssl->pbCurrReadBuffer = ms_ssl->pbReadBuffer;
	}
	else
	{
		ms_ssl->pbCurrReadBuffer += outl;
	}
	return(outl);
}

static int ms_ssl_write(BIO *b, const char *out, int outl)
{
	BIO_MS_SSL * ms_ssl;
    SECURITY_STATUS scRet;
    SecBufferDesc   Message;
    SecBuffer       Buffers[4];
    PBYTE pbMessage;
	LPVOID lpMsgBuf;
    INT i;


	if (b == NULL)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return(0);
	}
	ms_ssl=(BIO_MS_SSL *)b->ptr;
	if(!ms_ssl)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return(0);
	}

	BIO_clear_retry_flags(b);


	if(!ms_ssl->handshaked)
	{
		if(!DoHandShake(b, ms_ssl))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return 0;
		}
		ms_ssl->handshaked = 1;
	}


	ms_ssl->pbCurrReadBuffer = ms_ssl->pbReadBuffer;
	ms_ssl->cbCurrReadBuffer = 0;




    // This enables Schannel to perform the encryption in place,
    // which is a significant performance win.
    pbMessage = ms_ssl->pbIoBuffer + ms_ssl->Sizes.cbHeader;

	memcpy(pbMessage, out, outl);

    Buffers[0].pvBuffer     = ms_ssl->pbIoBuffer;
    Buffers[0].cbBuffer     = ms_ssl->Sizes.cbHeader;
    Buffers[0].BufferType   = SECBUFFER_STREAM_HEADER;

    Buffers[1].pvBuffer     = pbMessage;
    Buffers[1].cbBuffer     = outl;
    Buffers[1].BufferType   = SECBUFFER_DATA;

    Buffers[2].pvBuffer     = pbMessage + outl;
    Buffers[2].cbBuffer     = ms_ssl->Sizes.cbTrailer;
    Buffers[2].BufferType   = SECBUFFER_STREAM_TRAILER;

    Buffers[3].BufferType   = SECBUFFER_EMPTY;

    Message.ulVersion       = SECBUFFER_VERSION;
    Message.cBuffers        = 4;
    Message.pBuffers        = Buffers;

    scRet = ms_ssl->g_pSSPI->EncryptMessage(&ms_ssl->phContext, 0, &Message, 0);

    if(FAILED(scRet))
    {
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		FormatMessage(	FORMAT_MESSAGE_FROM_SYSTEM | 
						FORMAT_MESSAGE_IGNORE_INSERTS |
						FORMAT_MESSAGE_ALLOCATE_BUFFER, NULL, scRet, 
						MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
						(LPTSTR) &lpMsgBuf, 0, NULL);
		ERR_add_error_data(1, (LPTSTR)&lpMsgBuf);
		LocalFree( lpMsgBuf );
		return(0);
    }


    // 
    // Send the encrypted data to the server.
    //
	i = BIO_write(ms_ssl->rwbio, 
				ms_ssl->pbIoBuffer,
				Buffers[0].cbBuffer + Buffers[1].cbBuffer + Buffers[2].cbBuffer);

	if(i <= 0)
	{
		return(0);
	}

	return(outl);
}

static long ms_ssl_ctrl(BIO *b, int cmd, long num, void *ptr)
{
	BIO_MS_SSL *ms_ssl;
	long ret=1;

	ms_ssl=(BIO_MS_SSL *)b->ptr;

	switch (cmd)
	{
		case BIO_CTRL_RESET:
			break;
		case BIO_CTRL_INFO:
			ret=0;
			break;
		case BIO_CTRL_GET_CLOSE:
			ret=b->shutdown;
			break;
		case BIO_CTRL_SET_CLOSE:
			b->shutdown=(int)num;
			break;
		case BIO_C_SET_BIO:
			if(ms_ssl->rwbio) BIO_free_all(ms_ssl->rwbio);
			reset_ssl(ms_ssl);
			ms_ssl->rwbio = (BIO*)ptr;
			return 1;
			break;
		case BIO_C_SET_CERT:
			reset_ssl(ms_ssl);
			ret = CreateCredentials(ms_ssl, (PCERT_CONTEXT)ptr);
			if(!ret)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			}
			break;
		case BIO_C_GET_BIO:
			return (long)ms_ssl->rwbio;
			break;
		case BIO_C_GET_PEER:
			if(ms_ssl->serverCert)
			{
				CRYPTO_add(&ms_ssl->serverCert->references, 1, CRYPTO_LOCK_X509);
				return (long)ms_ssl->serverCert;
			}
			else
			{
				return (long)NULL;
			}
			break;
		case BIO_C_DO_CONNECT:
			if (b == NULL) return(0);
			ret = DoHandShake(b, ms_ssl);
			if(ret)
				ms_ssl->handshaked = 1;
			else
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			break;
		default:
			if(ms_ssl->rwbio) return BIO_ctrl(ms_ssl->rwbio, cmd, num, ptr);
			break;
	}

	return(ret);
}

static void reset_ssl(BIO_MS_SSL * ms_ssl)
{
	ms_ssl->handshaked = 0;
	ms_ssl->pbCurrReadBuffer = ms_ssl->pbReadBuffer;
	ms_ssl->cbCurrReadBuffer = 0;

	if(ms_ssl->pbIoBuffer)
	{
		free(ms_ssl->pbIoBuffer);
		ms_ssl->pbIoBuffer = NULL;
	}
	if(ms_ssl->serverCert)
	{
		X509_free(ms_ssl->serverCert);
		ms_ssl->serverCert = NULL;
	}
    if(ms_ssl->phContextInitialized)
	{
		ms_ssl->g_pSSPI->DeleteSecurityContext(&ms_ssl->phContext);
		ms_ssl->phContextInitialized = 0;
	}
	if(ms_ssl->phCredsInitialized)
	{
		ms_ssl->g_pSSPI->FreeCredentialsHandle(&ms_ssl->phCreds);
		ms_ssl->phCredsInitialized = 0;
	}
}


static long ms_ssl_callback_ctrl(BIO *b, int cmd, bio_info_cb *fp)
{
	long ret=1;
	return(ret);
}

static int ms_ssl_puts(BIO *bp, const char *str)
{
	int n,ret;

	n=strlen(str);
	ret=BIO_write(bp,str,n);
	return(ret);
}

BIO *BIO_new_ms_ssl(BIO * rwbio, PCERT_CONTEXT pCert)
{
	BIO *ret;
	BIO_MS_SSL *ms_ssl;


	if ((ret=BIO_new(BIO_f_ms_ssl())) == NULL)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return(NULL);
	}

	ms_ssl=(BIO_MS_SSL *)ret->ptr;
	if(!CreateCredentials(ms_ssl, pCert))
	{
		BIO_free_all(ret);
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return(0);
	}
	ms_ssl->rwbio = rwbio;

	return(ret);
}



static int LoadSecurityLibrary(BIO_MS_SSL * ms_ssl)
{
    INIT_SECURITY_INTERFACE         pInitSecurityInterface;
    OSVERSIONINFO VerInfo;
    UCHAR lpszDLL[MAX_PATH];
	LPVOID lpMsgBuf;

    //
    //  Find out which security DLL to use, depending on
    //  whether we are on Win2K, NT or Win9x
    //
    VerInfo.dwOSVersionInfoSize = sizeof (OSVERSIONINFO);
    if (!GetVersionEx (&VerInfo))   
    {
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
        return(0);
    }

    if (VerInfo.dwPlatformId == VER_PLATFORM_WIN32_NT && VerInfo.dwMajorVersion == 4)
    {
        strcpy ((char*)lpszDLL, NT4_DLL_NAME );
    }
    else if (VerInfo.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS || VerInfo.dwPlatformId == VER_PLATFORM_WIN32_NT )
    {
        strcpy ((char*)lpszDLL, DLL_NAME );
    }
    else
    {
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ERR_add_error_data(1, "Bad Windows Version");
        return(0);
    }

    //
    //  Load Security DLL
    //

    ms_ssl->g_hSecurity = LoadLibrary((char*)lpszDLL);
    if(!ms_ssl->g_hSecurity)
    {
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		FormatMessage(	FORMAT_MESSAGE_FROM_SYSTEM | 
						FORMAT_MESSAGE_IGNORE_INSERTS |
						FORMAT_MESSAGE_ALLOCATE_BUFFER, NULL, GetLastError(), 
						MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
						(LPTSTR) &lpMsgBuf, 0, NULL);
		ERR_add_error_data(1, (LPTSTR)&lpMsgBuf);
		LocalFree( lpMsgBuf );
        return(0);
    }

    pInitSecurityInterface = (INIT_SECURITY_INTERFACE)GetProcAddress(ms_ssl->g_hSecurity, "InitSecurityInterfaceA");
	if(!pInitSecurityInterface)
    {
		FreeLibrary(ms_ssl->g_hSecurity);
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		FormatMessage(	FORMAT_MESSAGE_FROM_SYSTEM | 
						FORMAT_MESSAGE_IGNORE_INSERTS |
						FORMAT_MESSAGE_ALLOCATE_BUFFER, NULL, GetLastError(), 
						MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
						(LPTSTR) &lpMsgBuf, 0, NULL);
		ERR_add_error_data(1, (LPTSTR)&lpMsgBuf);
		LocalFree( lpMsgBuf );
        return(0);
    }

    ms_ssl->g_pSSPI = pInitSecurityInterface();
    if(!ms_ssl->g_pSSPI)
    {
		FreeLibrary(ms_ssl->g_hSecurity);
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		FormatMessage(	FORMAT_MESSAGE_FROM_SYSTEM | 
						FORMAT_MESSAGE_IGNORE_INSERTS |
						FORMAT_MESSAGE_ALLOCATE_BUFFER, NULL, GetLastError(), 
						MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
						(LPTSTR) &lpMsgBuf, 0, NULL);
		ERR_add_error_data(1, (LPTSTR)&lpMsgBuf);
		LocalFree( lpMsgBuf );
        return(0);
    }

    return 1;
}

static int CreateCredentials(BIO_MS_SSL * ms_ssl, PCERT_CONTEXT pCertContext)
{
    TimeStamp       tsExpiry;
    SECURITY_STATUS Status;
	SCHANNEL_CRED   SchannelCred;
	LPVOID lpMsgBuf;

    //
    // Build Schannel credential structure. Currently, this sample only
    // specifies the protocol to be used (and optionally the certificate, 
    // of course). Real applications may wish to specify other parameters 
    // as well.
    //

    ZeroMemory(&SchannelCred, sizeof(SchannelCred));

    SchannelCred.dwVersion  = SCHANNEL_CRED_VERSION;
    SchannelCred.cCreds     = 1;
    SchannelCred.paCred     = (PCCERT_CONTEXT*)&pCertContext;
    SchannelCred.grbitEnabledProtocols = SP_PROT_SSL3_CLIENT;
    SchannelCred.dwFlags |= SCH_CRED_NO_DEFAULT_CREDS;

    // The SCH_CRED_MANUAL_CRED_VALIDATION flag is specified because
    // this sample verifies the server certificate manually. 
    // Applications that expect to run on WinNT, Win9x, or WinME 
    // should specify this flag and also manually verify the server
    // certificate. Applications running on newer versions of Windows can
    // leave off this flag, in which case the InitializeSecurityContext
    // function will validate the server certificate automatically.
    SchannelCred.dwFlags |= SCH_CRED_MANUAL_CRED_VALIDATION;

    //
    // Create an SSPI credential.
    //

    Status = ms_ssl->g_pSSPI->AcquireCredentialsHandleA(
                        NULL,                   // Name of principal    
                        UNISP_NAME_A,           // Name of package
                        SECPKG_CRED_OUTBOUND,   // Flags indicating use
                        NULL,                   // Pointer to logon ID
                        &SchannelCred,  // Package specific data
                        NULL,                   // Pointer to GetKey() func
                        NULL,                   // Value to pass to GetKey()
                        &ms_ssl->phCreds,       // (out) Cred Handle
                        &tsExpiry);             // (out) Lifetime (optional)

	if(Status==SEC_E_OK)
	{
		ms_ssl->phCredsInitialized = 1;
		return(1);
	}

	NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
	FormatMessage(	FORMAT_MESSAGE_FROM_SYSTEM | 
					FORMAT_MESSAGE_IGNORE_INSERTS |
					FORMAT_MESSAGE_ALLOCATE_BUFFER, NULL, Status, 
					MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
					(LPTSTR) &lpMsgBuf, 0, NULL);
	ERR_add_error_data(1, (LPTSTR)&lpMsgBuf);
	LocalFree( lpMsgBuf );
	
    return (0);
}

static int DoHandShake(BIO * a, BIO_MS_SSL * ms_ssl)
{
	SecBuffer pExtraData;
	PCERT_CONTEXT pRemoteCertContext;
	unsigned char * p;
	LPVOID lpMsgBuf;
	SECURITY_STATUS scRet;

	char uid[100];
	sprintf(uid, "%ld", ms_ssl);

	if(!PerformClientHandshake(a, ms_ssl, uid, &pExtraData))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return(0);
	}
    if(ms_ssl->g_pSSPI->QueryContextAttributes(&ms_ssl->phContext, SECPKG_ATTR_REMOTE_CERT_CONTEXT, (PVOID)&pRemoteCertContext) == SEC_E_OK)
    {
		p = pRemoteCertContext->pbCertEncoded;
		ms_ssl->serverCert = d2i_X509(NULL, &p, pRemoteCertContext->cbCertEncoded);
		CertFreeCertificateContext(pRemoteCertContext);
    }

	
    //
    // Read stream encryption properties.
    //
    if( (scRet = ms_ssl->g_pSSPI->QueryContextAttributes(&ms_ssl->phContext,
                                   SECPKG_ATTR_STREAM_SIZES,
                                   &ms_ssl->Sizes)) != SEC_E_OK)
    {
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		FormatMessage(	FORMAT_MESSAGE_FROM_SYSTEM | 
						FORMAT_MESSAGE_IGNORE_INSERTS |
						FORMAT_MESSAGE_ALLOCATE_BUFFER, NULL, scRet, 
						MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
						(LPTSTR) &lpMsgBuf, 0, NULL);
		ERR_add_error_data(1, (LPTSTR)&lpMsgBuf);
		LocalFree( lpMsgBuf );
        return(0);
    }
    //
    // Allocate a working buffer. The plaintext sent to EncryptMessage
    // should never be more than 'Sizes.cbMaximumMessage', so a buffer 
    // size of this plus the header and trailer sizes should be safe enough.
    // 
    ms_ssl->cbIoBufferLength = ms_ssl->Sizes.cbHeader + 
                       ms_ssl->Sizes.cbMaximumMessage +
                       ms_ssl->Sizes.cbTrailer;

    ms_ssl->pbIoBuffer = (PBYTE)malloc(ms_ssl->cbIoBufferLength+1);
    if(ms_ssl->pbIoBuffer == NULL)
    {
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
        return(0);
    }
	return(1);
}

static int PerformClientHandshake(
	BIO			*	b,
    BIO_MS_SSL *	ms_ssl,
    LPSTR           pszServerName,  // in
    SecBuffer *     pExtraData)     // out
{
    SecBufferDesc   OutBuffer;
    SecBuffer       OutBuffers[1];
    DWORD           dwSSPIFlags;
    DWORD           dwSSPIOutFlags;
    TimeStamp       tsExpiry;
    SECURITY_STATUS scRet;
    DWORD           cbData;
	LPVOID lpMsgBuf;

    dwSSPIFlags = ISC_REQ_SEQUENCE_DETECT   |
                  ISC_REQ_REPLAY_DETECT     |
                  ISC_REQ_CONFIDENTIALITY   |
                  ISC_RET_EXTENDED_ERROR    |
                  ISC_REQ_ALLOCATE_MEMORY   |
                  ISC_REQ_STREAM;

    //
    //  Initiate a ClientHello message and generate a token.
    //

    OutBuffers[0].pvBuffer   = NULL;
    OutBuffers[0].BufferType = SECBUFFER_TOKEN;
    OutBuffers[0].cbBuffer   = 0;

    OutBuffer.cBuffers = 1;
    OutBuffer.pBuffers = OutBuffers;
    OutBuffer.ulVersion = SECBUFFER_VERSION;

    scRet = ms_ssl->g_pSSPI->InitializeSecurityContextA(
                    &ms_ssl->phCreds,
                    NULL,
                    pszServerName,
                    dwSSPIFlags,
                    0,
                    SECURITY_NATIVE_DREP,
                    NULL,
                    0,
                    &ms_ssl->phContext,
                    &OutBuffer,
                    &dwSSPIOutFlags,
                    &tsExpiry);

    if(scRet != SEC_I_CONTINUE_NEEDED)
    {
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		FormatMessage(	FORMAT_MESSAGE_FROM_SYSTEM | 
						FORMAT_MESSAGE_IGNORE_INSERTS |
						FORMAT_MESSAGE_ALLOCATE_BUFFER, NULL, scRet, 
						MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
						(LPTSTR) &lpMsgBuf, 0, NULL);
		ERR_add_error_data(1, (LPTSTR)&lpMsgBuf);
		LocalFree( lpMsgBuf );
        return (0);
    }

    // Send response to server if there is one.
    if(OutBuffers[0].cbBuffer != 0 && OutBuffers[0].pvBuffer != NULL)
    {
        cbData = BIO_write(ms_ssl->rwbio,
                      OutBuffers[0].pvBuffer,
                      OutBuffers[0].cbBuffer);
        if(cbData <= 0)
        {
            ms_ssl->g_pSSPI->FreeContextBuffer(OutBuffers[0].pvBuffer);
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
            return(0);
        }
        // Free output buffer.
        ms_ssl->g_pSSPI->FreeContextBuffer(OutBuffers[0].pvBuffer);
        OutBuffers[0].pvBuffer = NULL;
    }


    return ClientHandshakeLoop(b, ms_ssl, TRUE, pExtraData);
}

/*****************************************************************************/
static
int
ClientHandshakeLoop(
	BIO			*	b,
    BIO_MS_SSL *	ms_ssl,
    BOOL            fDoInitialRead, // in
    SecBuffer *     pExtraData)     // out
{
    SecBufferDesc   InBuffer;
    SecBuffer       InBuffers[2];
    SecBufferDesc   OutBuffer;
    SecBuffer       OutBuffers[1];
    DWORD           dwSSPIFlags;
    DWORD           dwSSPIOutFlags;
    TimeStamp       tsExpiry;
    SECURITY_STATUS scRet;
    INT				cbData;

    PUCHAR          IoBuffer;
    DWORD           cbIoBuffer;
    BOOL            fDoRead;
	LPVOID			lpMsgBuf;


    dwSSPIFlags = ISC_REQ_SEQUENCE_DETECT   |
                  ISC_REQ_REPLAY_DETECT     |
                  ISC_REQ_CONFIDENTIALITY   |
                  ISC_RET_EXTENDED_ERROR    |
                  ISC_REQ_ALLOCATE_MEMORY   |
                  ISC_REQ_STREAM;

    //
    // Allocate data buffer.
    //

    IoBuffer = (PUCHAR)malloc(IO_BUFFER_SIZE+1);
    if(IoBuffer == NULL)
    {
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
        return(0);
    }
    cbIoBuffer = 0;

    fDoRead = fDoInitialRead;


    // 
    // Loop until the handshake is finished or an error occurs.
    //

    scRet = SEC_I_CONTINUE_NEEDED;

    while(scRet == SEC_I_CONTINUE_NEEDED        ||
          scRet == SEC_E_INCOMPLETE_MESSAGE     ||
          scRet == SEC_I_INCOMPLETE_CREDENTIALS) 
   {

        //
        // Read data from server.
        //

        if(0 == cbIoBuffer || scRet == SEC_E_INCOMPLETE_MESSAGE)
        {
            if(fDoRead)
            {
				cbData = BIO_read(ms_ssl->rwbio, IoBuffer + cbIoBuffer, IO_BUFFER_SIZE - cbIoBuffer);
                if(cbData <= 0)
                {
                    break;
                }
                cbIoBuffer += cbData;
            }
            else
            {
                fDoRead = TRUE;
            }
        }


        //
        // Set up the input buffers. Buffer 0 is used to pass in data
        // received from the server. Schannel will consume some or all
        // of this. Leftover data (if any) will be placed in buffer 1 and
        // given a buffer type of SECBUFFER_EXTRA.
        //

        InBuffers[0].pvBuffer   = IoBuffer;
        InBuffers[0].cbBuffer   = cbIoBuffer;
        InBuffers[0].BufferType = SECBUFFER_TOKEN;

        InBuffers[1].pvBuffer   = NULL;
        InBuffers[1].cbBuffer   = 0;
        InBuffers[1].BufferType = SECBUFFER_EMPTY;

        InBuffer.cBuffers       = 2;
        InBuffer.pBuffers       = InBuffers;
        InBuffer.ulVersion      = SECBUFFER_VERSION;

        //
        // Set up the output buffers. These are initialized to NULL
        // so as to make it less likely we'll attempt to free random
        // garbage later.
        //

        OutBuffers[0].pvBuffer  = NULL;
        OutBuffers[0].BufferType= SECBUFFER_TOKEN;
        OutBuffers[0].cbBuffer  = 0;

        OutBuffer.cBuffers      = 1;
        OutBuffer.pBuffers      = OutBuffers;
        OutBuffer.ulVersion     = SECBUFFER_VERSION;

        //
        // Call InitializeSecurityContext.
        //

        scRet = ms_ssl->g_pSSPI->InitializeSecurityContextA(&ms_ssl->phCreds,
                                          &ms_ssl->phContext,
                                          NULL,
                                          dwSSPIFlags,
                                          0,
                                          SECURITY_NATIVE_DREP,
                                          &InBuffer,
                                          0,
                                          NULL,
                                          &OutBuffer,
                                          &dwSSPIOutFlags,
                                          &tsExpiry);

        //
        // If InitializeSecurityContext was successful (or if the error was 
        // one of the special extended ones), send the contends of the output
        // buffer to the server.
        //

        if(scRet == SEC_E_OK                ||
           scRet == SEC_I_CONTINUE_NEEDED   ||
           FAILED(scRet) && (dwSSPIOutFlags & ISC_RET_EXTENDED_ERROR))
        {
            if(OutBuffers[0].cbBuffer != 0 && OutBuffers[0].pvBuffer != NULL)
            {
                cbData = BIO_write(ms_ssl->rwbio,
                              OutBuffers[0].pvBuffer,
                              OutBuffers[0].cbBuffer);
                if(cbData <= 0)
                {
                    ms_ssl->g_pSSPI->FreeContextBuffer(OutBuffers[0].pvBuffer);
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
                    return(0);
                }

                // Free output buffer.
                ms_ssl->g_pSSPI->FreeContextBuffer(OutBuffers[0].pvBuffer);
                OutBuffers[0].pvBuffer = NULL;
            }
        }


        //
        // If InitializeSecurityContext returned SEC_E_INCOMPLETE_MESSAGE,
        // then we need to read more data from the server and try again.
        //

        if(scRet == SEC_E_INCOMPLETE_MESSAGE)
        {
            continue;
        }


        //
        // If InitializeSecurityContext returned SEC_E_OK, then the 
        // handshake completed successfully.
        //

        if(scRet == SEC_E_OK)
        {
            //
            // If the "extra" buffer contains data, this is encrypted application
            // protocol layer stuff. It needs to be saved. The application layer
            // will later decrypt it with DecryptMessage.
            //

            if(InBuffers[1].BufferType == SECBUFFER_EXTRA)
            {
                pExtraData->pvBuffer = malloc(InBuffers[1].cbBuffer+1);
                if(pExtraData->pvBuffer == NULL)
                {
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
                    return(0);
                }

                MoveMemory(pExtraData->pvBuffer,
                           IoBuffer + (cbIoBuffer - InBuffers[1].cbBuffer),
                           InBuffers[1].cbBuffer);

                pExtraData->cbBuffer   = InBuffers[1].cbBuffer;
                pExtraData->BufferType = SECBUFFER_TOKEN;
            }
            else
            {
                pExtraData->pvBuffer   = NULL;
                pExtraData->cbBuffer   = 0;
                pExtraData->BufferType = SECBUFFER_EMPTY;
            }

            //
            // Bail out to quit
            //

            break;
        }


        //
        // Check for fatal error.
        //
        if(FAILED(scRet))
        {
            break;
        }
        //
        // If InitializeSecurityContext returned SEC_I_INCOMPLETE_CREDENTIALS,
        // then the server just requested client authentication. 
        //
        if(scRet == SEC_I_INCOMPLETE_CREDENTIALS)
        {
            break;
        }


        //
        // Copy any leftover data from the "extra" buffer, and go around
        // again.
        //

        if ( InBuffers[1].BufferType == SECBUFFER_EXTRA )
        {
            MoveMemory(IoBuffer,
                       IoBuffer + (cbIoBuffer - InBuffers[1].cbBuffer),
                       InBuffers[1].cbBuffer);

            cbIoBuffer = InBuffers[1].cbBuffer;
        }
        else
        {
            cbIoBuffer = 0;
        }
    }

    // Delete the security context in the case of a fatal error.
    if(FAILED(scRet))
    {
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		FormatMessage(	FORMAT_MESSAGE_FROM_SYSTEM | 
						FORMAT_MESSAGE_IGNORE_INSERTS |
						FORMAT_MESSAGE_ALLOCATE_BUFFER, NULL, scRet, 
						MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
						(LPTSTR) &lpMsgBuf, 0, NULL);
		ERR_add_error_data(1, (LPTSTR)&lpMsgBuf);
		LocalFree( lpMsgBuf );
		free(IoBuffer);
		return(0);
    }

    free(IoBuffer);

	ms_ssl->phContextInitialized = 1;
    return(1);
}

#endif //_WIN32 NO_BIO_MS_SSL
