/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef CONF_H
#define CONF_H

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <NewPKI.h>

#ifdef _WIN32
	#include <direct.h>
	#include <windows.h>
#else
	#include <syslog.h>
	#define stricmp strcasecmp
	#include <ctype.h>
#endif

#include <FileLog.h>
#include <fstream>


/*!
  This class represents the configuration loaded from config.conf
*/
class Config
{
public:
	/*! \brief This is the constructor.
	 */
	Config();

	/*! \brief This is the constructor.
	 *  \param other [IN] Load datas from another instance.
	 */
	Config(const Config &other);

	/*! \brief This is the destructor.
	 */
	~Config();

	/*! \brief This function reads of configuration from a file.
	 *  \param ConfFile [IN] The name of the file  to read the config from.
	 */
	bool LoadConf(const mString & ConfFile);

	/*! \brief This function clears the conf.
	 */
	void Clear();

	/*! \brief This function sets the IP for the NewPKI server to bind on.
	 *  \param BindAddress [IN] The IP for the NewPKI server to bind on.
	 */
	void set_BindAddress(const mString & BindAddress);

	/*! \brief This function returns the IP for the NewPKI server to bind on.
	 *  \\return The IP for the NewPKI server to bind on.
	 */
	const mString & get_BindAddress() const;

	/*! \brief This function sets the port for the NewPKI server to listen on.
	 *  \param LocalPort [IN] The port for the NewPKI server to listen on.
	 */
	void set_LocalPort(unsigned int LocalPort);

	/*! \brief This function returns the port for the NewPKI server to listen on.
	 *  \return The port for the NewPKI server to listen on.
	 */
	unsigned int get_LocalPort() const;

	/*! \brief This function sets the ip or name where to find the sql server.
	 *  \param SqlServer [IN] The ip or name where to find the sql server.
	 */
	void set_SqlServer(const mString & SqlServer);

	/*! \brief This function returns the ip or name where to find the sql server.
	 *  \return The ip or name where to find the sql server.
	 */
	const mString & get_SqlServer() const;

	/*! \brief This function sets the port for the sql server.
	 *  \param SqlPort [IN] The port for the sql server.
	 */
	void set_SqlPort(unsigned int SqlPort);

	/*! \brief This function returns the port for the sql server.
	 *  \return The port for the sql server.
	 */
	unsigned int get_SqlPort() const;

	/*! \brief This function sets the username used by NewPKI to connect to the sql server.
	 *  \param SqlUsername [IN] The username used by NewPKI to connect to the sql server.
	 */
	void set_SqlUsername(const mString & SqlUsername);

	/*! \brief This function returns the username used by NewPKI to connect to the sql server.
	 *  \return The username used by NewPKI to connect to the sql server.
	 */
	const mString & get_SqlUsername() const;

	/*! \brief This function sets the password for the sql account.
	 *  \param SqlPassword [IN] The password for the sql account.
	 */
	void set_SqlPassword(const mString & SqlPassword);

	/*! \brief This function returns the password for the sql account.
	 *  \return The password for the sql account.
	 */
	const mString & get_SqlPassword() const;

	/*! \brief This function sets the log file path.
	 *  \param LogFile [IN] The log file path.
	 */
	void set_LogFile(const mString & LogFile);

	/*! \brief This function returns the log file path.
	 *  \return The log file path.
	 */
	const mString & get_LogFile() const;

	/*! \brief This function sets the name of the ENGINE to use (if a CA private key is "ENGINE:keyid" a value must be set).
	 *  \param Engine [IN] The name of the ENGINE to use.
	 */
	void set_Engine(const mString & Engine);

	/*! \brief This function returns the name of the ENGINE to use.
	 *  \return The name of the ENGINE to use.
	 */
	const mString & get_Engine() const;

	/*! \brief This function sets the ENGINE specific commands to execute after the ENGINE load. The format of the command is COMAND1:VALUE, COMMAND2, COMMAND3:VALUE (eg: SO_PATH:etpkcs11.dll,PIN:1234567890).
	 *  \param EngineCmd [IN] The ENGINE specific commands.
	 */
	void set_EngineCmd(const mString & EngineCmd);

	/*! \brief This function returns the ENGINE specific commands.
	 *  \return The ENGINE specific commands.
	 */
	const mString & get_EngineCmd() const;

	/*! \brief This function sets the key ID in the engine for the SSL certificate to generate/use.
	 *  \param SSL_KeyId [IN] The key ID.
	 */
	void set_SSL_KeyId(const mString & SSL_KeyId);

	/*! \brief This function returns the key ID in the engine for the SSL certificate.
	 *  \return The key ID.
	 */
	const mString & get_SSL_KeyId() const;

	/*! \brief This function sets the path to the authentification lib.
	 *  \param AuthLib [IN] The path to the authentification lib.
	 */
	void set_AuthLib(const mString & AuthLib);

	/*! \brief This function returns the path to the authentification lib.
	 *  \return The path to the authentification lib.
	 */
	const mString & get_AuthLib() const;

	/*! \brief This function sets the datas for the authentification lib.
	 *  \param AuthCmd [IN] The datas for the authentification lib.
	 */
	void set_AuthCmd(const mString & AuthCmd);

	/*! \brief This function returns the datas for the authentification lib.
	 *  \return The datas for the authentification lib.
	 */
	const mString & get_AuthCmd() const;

	/*! \brief This function returns the debug level.
	 *  \param DebugLevel [IN] The debug level.
	 */
	void set_DebugLevel(LOG_LEVEL DebugLevel);

	/*! \brief This function sets the debug level.
	 *  \return The debug level.
	 */
	LOG_LEVEL get_DebugLevel() const;

	/*! \brief This operator copies one Config into another.
	 *  \param other [IN] The other Config to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const Config &other);
private:
	#define ENV_DB_SERVER		"server"
	#define ENV_DB_PORT			"port"
	#define ENV_DB_USERNAME		"username"
	#define ENV_DB_PASSWORD		"password"
	#define ENV_LOCALPORT		"localport"
	#define ENV_BIND_ADDRESS	"bindaddress"
	#define ENV_ENGINE_CMD		"enginecmd"
	#define ENV_LOGFILE			"logfile"
	#define ENV_ENGINE			"engine"
	#define ENV_ENGINE_CMD		"enginecmd"
	#define ENV_SSL_KEY_ID		"ssl_keyid"
	#define ENV_AUTH_LIB		"authlib"
	#define ENV_AUTH_CMD		"authcmd"
	#define ENV_DBG_LEVEL		"debuglevel"

	void LogError(char * strError, ...);
	void ReadLine(fstream & infile, mString & Line);

	mString m_BindAddress;
	unsigned int m_LocalPort;
	mString m_SqlServer;
	unsigned int m_SqlPort;
	mString m_SqlUsername;
	mString m_SqlPassword;
	mString m_LogFile;
	mString m_Engine;
	mString m_EngineCmd;
	mString m_SSL_KeyId;
	mString m_AuthLib;
	mString m_AuthCmd;
	LOG_LEVEL m_DebugLevel;
};

#endif //CONF_H

