/*  VER 071  TAB P   $Id: exec.c,v 1.12.2.1 2002/10/01 08:54:59 egil Exp $
 *
 *  run a script
 *
 *  copyright 1996, 1997 Egil Kvaleberg, egil@kvaleberg.no
 *  the GNU General Public License applies
 *
 *  $Log: exec.c,v $
 *  Revision 1.12.2.1  2002/10/01 08:54:59  egil
 *  Changed sprintf to snprintf
 *
 *  Revision 1.12  1999/04/07 21:07:59  src
 *  Debugged config support
 *
 *  Revision 1.11  1999/04/07 20:09:16  src
 *  Implemented configuration reading, and eval mechanism
 *
 *  Revision 1.10  1999/03/31 03:38:28  src
 *  Moved errno.h to common.h
 *
 *  Revision 1.9  1999/03/07 14:58:18  src
 *  Read newsconfig supported. Storage API supported.
 *
 *  Revision 1.8  1998/11/21 19:14:23  src
 *  Added --filter option
 *
 *  Revision 1.7  1998/09/09 07:32:11  src
 *  Version 1.1
 *
 *  Revision 1.6  1998/09/03 02:49:29  src
 *  Fixed stuff detected by -Wall
 *
 *  Revision 1.5  1998/07/12 09:39:27  src
 *  newsx version 1.0
 */

#include "common.h"
#include "proto.h"
#include "options.h"
#include "news.h"
#include "newsconfig.h"

#include <fcntl.h>
#if HAVE_SYS_WAIT_H
#  include <sys/wait.h>
#endif

#ifndef STDIN
#define STDIN  0
#define STDOUT 1
#define STDERR 2
#endif

/*
 *  perform a fork()
 */
static int 
do_fork(void)
{
    int pid;

    if ((pid = fork()) < 0) {
	log_msg(L_ERRno,"could not fork");
	exit_cleanup(1);
    }
    return pid;
}

/*
 *  wait for pid to finish
 *  return status 
 */
int        
wait_pid(int pid)
{
    int sts;

#if 0 /* HAVE_WAITPID */
    while (waitpid(pid, &sts, 0) < 0) {
	if (errno == EINTR)
	    continue;
	log_msg(L_ERRno,"error waiting for connect process");
	exit_cleanup(1);
    }
#else
    int id;

    while ((id=wait(&sts)) != pid) {
	if (id < 0) {
	    if (errno == EINTR)
		continue;
	    log_msg(L_ERRno,"error waiting for connect process");
	    exit_cleanup(1);
	}
    }
#endif

    return sts;
}

/*
 *  run ctlinnd with two arguments
 *  return status 
 *  BUG: what about stderr...
 */
int 
ctlinnd(char *arg1, char *arg2)
{
    char ctlpath[PATH_MAX];
    char t_opt[30];
    int pid;
	
    /*
     * options:
     *  -s silent if everything OK
     *  -t<n> timeout
     */
    snprintf(t_opt,sizeof(t_opt),"-t%ld",timeout ? timeout : LOCKTIME);
	
    build_alt_filename(ctlpath,cfg_newsbin,INN_CTL,ctlpgm);

    if ((pid=do_fork()) == 0) {
	/* slave */
	setuid(getuid());
	setgid(getgid());
	execl(ctlpath, ctlpath, "-s", t_opt, arg1, arg2, (char *)0);
	log_msg(L_ERRno,"could not exec %s",ctlpath);
	exit(5);
    }

    return wait_pid(pid);
}

/*
 *  run a program with files in and out plumbed as stdin and stdout
 *  return program status
 */
int 
script(char *program, int in, int out)
{
    int err;
    int pid;

    if ((pid=do_fork()) == 0) {
	dup2(in, STDIN);
	dup2(out, STDOUT);
	/* BUG: do something better... */
	err = open("/tmp/connect.errors", O_WRONLY|O_APPEND|O_CREAT, 0644);
	if (err >= 0) dup2(err, STDERR);
	setuid(getuid());
	setgid(getgid());
	/* BUG: configure this... */
	execl("/bin/sh", "sh", "-c", program, (char *)0);
	log_msg(L_ERRno,"could not exec /bin/sh");
	exit_cleanup(1);
    }

    return wait_pid(pid);
}

/*
 *  start a program with in and out plumbed to FILE pair
 */
void 
program_open(char *program, SOCKET_D *sock)
{
    int err;
    int pid;
    int p_in[2];
    int p_out[2];

    if (pipe(p_in) < 0
     || pipe(p_out) < 0) {
	log_msg(L_ERRno,"could not pipe");
	exit_cleanup(1);
    }

    if ((pid=do_fork()) == 0) {
	/* child */
	close(p_in[1]); /* only read on this */
	close(p_out[0]); /* only write on this */
	dup2(p_in[0], STDIN);
	dup2(p_out[1], STDOUT);

	/* BUG: do something better... */
	err = open("/tmp/connect.errors", O_WRONLY|O_APPEND|O_CREAT, 0644);
	if (err >= 0) dup2(err, STDERR);
	setuid(getuid());
	setgid(getgid());
	/* BUG: config /bin/sh */
	execl("/bin/sh", "sh", "-c", program, (char *)0);
	log_msg(L_ERRno,"could not exec /bin/sh");
	exit_cleanup(1);
    } else {
	/* parent */
	close(p_in[0]); /* only write on this */
	close(p_out[1]); /* only read on this */
	sock->r_fd = p_out[0]; 
	sock->w_fd = p_in[1];  
	if (!(sock->r_str = fdopen(p_out[0],"r"))) {
	    log_msg(L_ERRno,"could not open pipe for read");
	}
	if (!(sock->w_str = fdopen(p_in[1],"w"))) {
	    log_msg(L_ERRno,"could not open pipe for write");
	}
    }

    /* do not wait for child */
}

/*
 *  fork a process
 *  open up a stream to write to it
 */
FILE *
fork_write(char *path,int *pidp,char *arg1,char *arg2)
{ 
    FILE *f;
    pid_t pid;
    int pipe_fd[2];

    /* normal mode */
    if (pipe(pipe_fd) == -1) {
	log_msg(L_ERRno,"can't pipe: %s",path);
	return 0;
    }
	
    if (!(pid = fork())) {
	/* child */
	close(STDIN);       /* disconnect stdin */ 
	dup2(pipe_fd[0],STDIN); /* connect stdin to the pipe */
	close(pipe_fd[1]);
	
	log_msg(L_DEBUG,"exec: %s args: %s %s",path,arg1?arg1:"",
						  arg2?arg2:"");
	execl(path,path,arg1,arg2,NULL);

	/* here on exec failure: reject with "system down" status */
	log_msg(L_ERRno,"can't exec: %s",path);
	return 0;
    }

    /* parent process */
    close(pipe_fd[0]);
    if (!(f = fdopen(pipe_fd[1],"w"))) {
	log_msg(L_ERRno,"can't connect write pipe: %s",path);
	return 0;
    }
    *pidp = pid;
    return f;
}

/*
 *  fork a process
 *  open up a stream to read from it
 */
FILE *
fork_read(char *path,int *pidp,char *arg)
{ 
    FILE *f;
    pid_t pid;
    int pipe_fd[2];

    /* normal mode */
    if (pipe(pipe_fd) == -1) {
	log_msg(L_ERRno,"can't pipe: %s",path);
	return 0;
    }
	
    if (!(pid = fork())) {
	/* child */
	close(STDOUT);      /* disconnect stdout */
	dup2(pipe_fd[1],STDOUT); /* connect stdout to the pipe */
	close(pipe_fd[0]);
	
	log_msg(L_DEBUG,"exec: %s arg: %s",path,arg?arg:"");
	execl(path,path,arg,NULL,NULL);

	/* here on exec failure: reject with "system down" status */
	log_msg(L_ERRno,"can't exec: %s",path);
	return 0;
    }

    /* parent process */
    close(pipe_fd[1]);
    if (!(f = fdopen(pipe_fd[0],"r"))) {
	log_msg(L_ERRno,"can't connect read pipe: %s",path);
	return 0;
    }
    *pidp = pid;
    return f;
}

/*
 *  open a file or pipe to a process
 *  if the first character is a '|', pipe to process is assumed
 *  returns stream to write to 
 */
FILE *
open_any(char *any,int *pidp)
{ 
    FILE *f;

    *pidp = -1;
    if (*any == '|') {
	return fork_write("/bin/sh",pidp,"-c",any+1);
    } else if (*any == '-' && !*(any+1)) {
	return stdout;
    } else {
	if (!(f = fopen(any,"w"))) {
	    log_msg(L_ERRno,"can't write to: %s",any);
	}
	return f;
    }
}

/*
 *  open a file or pipe to a process
 *  if the first character is a '|', pipe to process is assumed
 *  returns stream to write to 
 */
int     
close_any(FILE *f,int pid)
{ 
    int ok = 0;

    if (f) {
	fflush(f);
	if (f != stdout) {
	    if ((ok=fclose(f)) < 0) {
		log_msg(L_ERRno,"can't close");
	    }
	}
    }
    if (pid == -1) {
	return ok;
    } else {
	/* await process to finish */
	return wait_pid(pid);
    }
}

