/* ntfs-conf - tool to enable/disable write support for NTFS.
 *
 * Copyright (C) 2007 Mertens Florent <flomertens@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <glade/glade.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <string.h>
#include <stdio.h>
#include <mntent.h>
#include <unistd.h>

#include "main.h"

#define MOUNT               "/bin/mount"
#define UMOUNT              "/bin/umount"

typedef struct
{
    GtkWidget           *dialog;
    GtkWidget           *value;
} Prog;

typedef struct
{
    GtkWidget           *dialog;
    GtkWidget           *text;
} Retry;


static int
ask_retry (gchar *device, gchar *serr)
{
    Retry           *retry;
    GtkTextBuffer   *buffer;
    gchar           *text;
    int             res;
    GladeXML        *xml;

    xml = glade_xml_new (GLADE_FILE, NULL, NULL);

    if ( !xml )
    {
        g_warning ("Could not open " GLADE_FILE);
        return FALSE;
    }

    retry = g_new0 (Retry, 1);
    retry->dialog = glade_xml_get_widget (xml, "dialog_retry");
    retry->text = glade_xml_get_widget (xml, "textview");

    gtk_window_set_title (GTK_WINDOW (retry->dialog), "ntfs-config");

    buffer = gtk_text_buffer_new(NULL);
    text = g_strdup_printf (_("%s\n\nError Message :\n%s"), device, serr);
    gtk_text_buffer_set_text(buffer, text, -1);
    gtk_text_view_set_buffer( GTK_TEXT_VIEW (retry->text), buffer);

    res = gtk_dialog_run (GTK_DIALOG (retry->dialog));

    gtk_widget_destroy (retry->dialog);
    
    g_free(text);
    g_free(retry);

    return res;
}

static int
check_device_mounted (gchar *device)
{
    FILE*       f;
    struct      mntent *entry;
 
    g_debug("Searching %s in mtab...", device);
    if( !( f = fopen ( MTAB, "r") ) ) 
    {
        g_warning(_("Error : Can't open /etc/mtab in read mode.\n"));
        return 2;
    }

    while( ( entry = getmntent( f ) ) != NULL )
    {
        if( !strcmp(entry->mnt_dir, device) )
        {
            fclose(f);
			g_debug("%s is mounted.", entry->mnt_dir);
            return 0;
        }
    }
	
    fclose(f);
	
    return 1;
}

void
remount_all (GPtrArray *devices)
{
    GError      *err = NULL;
	char        *args[3]= {NULL, NULL, NULL};
    int         i = 0;
    char        *sout = NULL;
    char        *serr = NULL;
    int         exit_status = 0;
    char        **envp = {NULL};
    char        *device;
    float       per;
	Prog		*progress;
    GladeXML	*xml;

    xml = glade_xml_new (GLADE_FILE, NULL, NULL);

    if ( !xml )
    {
        g_warning ("Could not open " GLADE_FILE);
        return;
    }

    progress = g_new0 (Prog, 1);
	progress->dialog = glade_xml_get_widget (xml, "dialog_progress");
    progress->value = glade_xml_get_widget (xml, "progressbar");
	
	for ( i = 0; i < devices->len; i++ )
    {
        gtk_window_set_title (GTK_WINDOW (progress->dialog), "ntfs-config");
		
        gtk_widget_show_now (progress->dialog);
	
        while ( gtk_events_pending () )
            gtk_main_iteration ();
	
        gtk_progress_set_percentage( GTK_PROGRESS(progress->value), 0);
		
        device = g_ptr_array_index (devices, i);
	
        if( !check_device_mounted(device) )
        {
            g_debug("Unmounting %s...", device);
            args[0] = UMOUNT;
            args[1] = device;
            g_spawn_sync ("/", 
                    args, 
                    envp, 
                    0, 
                    NULL,
                    NULL, 
                    &sout, 
                    &serr, 
                    &exit_status, 
                    &err);
            if ( exit_status )
            {
                if ( ask_retry(device, serr) == -5 )
                {
                    i = i-1;
                    continue;
                }
                continue;
            }
            sleep(1);
        }

        per = ((float)(i+1.))/ ((float)devices->len);
        gtk_progress_set_percentage( GTK_PROGRESS(progress->value), per);
        if (per >= 1)
            gtk_progress_set_percentage( GTK_PROGRESS(progress->value), 1);

        while ( gtk_events_pending () )
            gtk_main_iteration ();

        g_debug("Mounting %s...", device);
        args[0] = MOUNT;
        args[1] = device;
        g_spawn_sync ("/", 
                args, 
                envp, 
                0, 
                NULL,
                NULL, 
                &sout, 
                &serr, 
                &exit_status, 
                &err);
        if ( exit_status )
        {
            gchar* message;
            message = g_strdup_printf (_("Mounting %s failed.\n\n%s"), device , serr);
            err = g_error_new(1, -1, message);
            show_error(err);
            continue;
        }
    }

    gtk_widget_destroy(progress->dialog);
    
    g_free(progress);
    
    return;
}
