/* ntfs-conf - tool to enable/disable write support for NTFS.
 *
 * Copyright (C) 2007 Mertens Florent <flomertens@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <glade/glade.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <glib/gstdio.h>
#include <string.h>
#include <stdio.h>
#include <mntent.h>
#include <libhal-storage.h>

#include "main.h"
#include "prober.h"
#include "mounter.h"

typedef struct
{
    GtkWidget           *dialog;
    GtkWidget           *treeview;
    GtkButton           *apply;
    GtkCellRenderer     *renderer1;
    GtkCellRenderer     *renderer2;
    GtkCellRenderer     *renderer3;
} New_devices;

enum
{
    CHECKED_COLUMN,
    CHECKABLE,
    DEVICE_NAME,
    EDITABLE1,
    MOUNT_POINT,
    EDITABLE2,
    N_COLUMNS
};

const gchar         *start = "<Click here to set a mount point>";
static New_devices  *new_devices;

static gboolean
add_in_fstab (gchar* device, gchar* mount_point)
{
    FILE            *f;
    struct mntent   *entry;
    gchar           options[32];
    
    g_debug ("Creating %s", mount_point);
    
    if ( g_mkdir (mount_point, 0744) != 0 )
    {
        g_warning ("Failed to create %s.\n", mount_point);
        return FALSE;
    }
    
    g_snprintf (options, 32, "defaults,locale=%s", setlocale( LC_ALL, "" ));
    
    entry = g_new0 (struct mntent, 1);

    entry->mnt_fsname = device;
    entry->mnt_dir = mount_point;
    entry->mnt_type = "ntfs-3g";
    entry->mnt_opts = options;
    entry->mnt_freq = 0;
    entry->mnt_passno = 0;
    
    if( !( f = fopen ( FSTAB, "a") ) ) 
    {
        g_warning ("Error : Can't open /etc/fstab in write mode.\n");
        g_free(entry);
        fclose(f);
        return FALSE;
    }
    
    g_debug ("Adding : %s %s %s %s %i %i in /etc/fstab.", 
                entry->mnt_fsname,
                entry->mnt_dir,
                entry->mnt_type,
                entry->mnt_opts,
                entry->mnt_freq,
                entry->mnt_passno);

    addmntent (f, entry);
    
    g_free (entry);
    
    fclose (f);
    
    return TRUE;
}

static gboolean
check_mount_point (gchar* mount_point)
{
    FILE            *f1, *f2;
    struct mntent   *entry;
    
    /* Check in /etc/fstab */
    if( !( f1 = fopen ( FSTAB, "r") ) ) 
    {
        g_warning ("Error : Can't open /etc/fstab in read mode.\n");
        return 2;
    }

    g_debug ("Searching %s in %s...", mount_point, FSTAB);
    
    while( ( entry = getmntent ( f1 ) ) != NULL )
    {
        if( !strcmp(entry->mnt_dir, mount_point) )
        {
            fclose(f1);
            return 1;
        }
    }
    fclose(f1);
    
    /* Check in /etc/mtab */
    if( !( f2 = fopen ( MTAB, "r") ) ) 
    {
        g_warning ("Error : Can't open /etc/mtab in read mode.\n");
        return 2;
    }

    g_debug("Searching %s in %s...", mount_point, MTAB);
    
    while( ( entry = getmntent( f2 ) ) != NULL )
    {
        if( !strcmp (entry->mnt_dir, mount_point) )
        {
            fclose(f2);
            return 1;
        }
    }
    fclose(f2);
    
    /* Check in /media. */
    if ( g_file_test (mount_point, G_FILE_TEST_EXISTS) )
        return 1;
    
    return 0;
}   

static void
set_apply_button (GtkTreeModel *model, GtkTreeIter iter)
{
    gboolean    state;
    gchar       *mount_point;
    gboolean    sensitivity = TRUE;
    
    g_debug("Setting sensitivity...");
    
    gtk_tree_model_get_iter_first (model, &iter);
    gtk_tree_model_get (model, &iter, 
                CHECKED_COLUMN, &state,
                MOUNT_POINT, &mount_point, -1);
    do
    {
        gtk_tree_model_get (model, &iter, 
                CHECKED_COLUMN, &state,
                MOUNT_POINT, &mount_point, -1);
        if( state != 0 )
            if ( !( strcmp (mount_point,start) ) )
            {
                sensitivity = FALSE;
                g_debug("Set apply button insensitive.");
                break;
            } 
    }
    while ( ( gtk_tree_model_iter_next (model, &iter) ) );
    gtk_widget_set_sensitive (GTK_WIDGET (new_devices->apply), sensitivity);
}

static void
on_toggled (GtkCellRendererToggle *cell_render,
        gchar *path, GtkTreeModel *model)
{
    GtkTreeIter iter;
    gboolean    state;
    
    gtk_tree_model_get_iter_from_string (model, &iter, path);
    gtk_tree_model_get (model, &iter, CHECKED_COLUMN, &state, -1);
    
    if ( state)
        gtk_list_store_set (GTK_LIST_STORE(model), &iter, CHECKED_COLUMN, FALSE, -1);
    else
        gtk_list_store_set (GTK_LIST_STORE(model), &iter, CHECKED_COLUMN, TRUE, -1);
    
    set_apply_button (model, iter);
}

static void
on_edited (GtkCellRendererText *cell_render,
    gchar *path, gchar *new_text, GtkTreeModel *model)
{
    GtkTreeIter     iter;
    gchar           mount_point[64];
    GError          *err;
    
    /* Check that the name don't contain "/" */
    if ( g_strrstr ( new_text, "/") != NULL)
    {
        err = g_error_new (1, -1,_("%s contains an invalid caracter.\
\nyou must choose a name, not a directory."), new_text);
        show_error (err);
        return;
    }
    
    if ( !( strcmp (new_text,start) ) )
        return;
    
    g_snprintf (mount_point, 64, "/media/%s", new_text);
    
    if ( check_mount_point (mount_point) )
    { 
        err = g_error_new (1, -1, _("/media/%s is already in use.\
\nPlease choose another one."), new_text);
        show_error (err);
        return;
    }
    
    gtk_tree_model_get_iter_from_string (model, &iter, path);
    
    gtk_list_store_set (GTK_LIST_STORE(model), &iter, MOUNT_POINT, mount_point, -1);
    
    set_apply_button (model, iter);
}

static void
ask_new_devices (GPtrArray* devices, GPtrArray* devices_name)
{
    GError              *err;
    GtkListStore        *store;
    GtkTreeIter         iter;
    GtkTreeViewColumn   *column;
    GladeXML            *xml;
    int                 i, res;
    gboolean            wrong_name = FALSE;

    new_devices = g_new0 (New_devices, 1);
    
    xml = glade_xml_new (GLADE_FILE, NULL, NULL);

    if ( !xml )
    {
        g_warning ("Could not open %s.", GLADE_FILE);
        return;
    }
    
    new_devices->dialog = glade_xml_get_widget (xml, "dialog_new");
    new_devices->treeview = glade_xml_get_widget (xml, "treeview");
    new_devices->apply = GTK_BUTTON (glade_xml_get_widget (xml, "buttonapply1"));
    
    gtk_window_set_title (GTK_WINDOW (new_devices->dialog), "ntfs-config");
    gtk_window_set_icon_name (GTK_WINDOW (new_devices->dialog), 
                                "gnome-dev-harddisk");
    
    store = gtk_list_store_new (N_COLUMNS, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN,
                G_TYPE_STRING, G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_BOOLEAN);
    
    for ( i = 0; i < devices->len; i++ )
    {
        gtk_list_store_append (store, &iter);
        gtk_list_store_set (store, &iter,
                    CHECKED_COLUMN, FALSE,
                    CHECKABLE, TRUE,
                    DEVICE_NAME, g_ptr_array_index (devices, i),
                    EDITABLE2, FALSE,
                    MOUNT_POINT, g_ptr_array_index (devices_name, i),
                    EDITABLE2, TRUE, -1);
    }
    
    gtk_tree_view_set_model(GTK_TREE_VIEW (new_devices->treeview), 
                        GTK_TREE_MODEL (store));
    
    g_object_unref (G_OBJECT (store));
    
    new_devices->renderer1 = gtk_cell_renderer_toggle_new ();
    column = gtk_tree_view_column_new_with_attributes (
                        "Add", new_devices->renderer1,
                        "active", CHECKED_COLUMN,
                        "activatable", CHECKABLE,
                        NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (new_devices->treeview), column);
    
    new_devices->renderer2 = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (
                        "Device", new_devices->renderer2,
                        "text", DEVICE_NAME,
                        "editable" , EDITABLE1,
                        NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (new_devices->treeview), column);
    
    new_devices->renderer3 = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (
                        "Mount point", new_devices->renderer3,
                        "text", MOUNT_POINT,
                        "editable" , EDITABLE2,
                        NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (new_devices->treeview), column);
    
    g_signal_connect (new_devices->renderer1, "toggled",
                G_CALLBACK (on_toggled), GTK_TREE_MODEL(store));
    g_signal_connect (new_devices->renderer3, "edited",
                G_CALLBACK (on_edited), GTK_TREE_MODEL(store));

    set_apply_button (GTK_TREE_MODEL(store), iter);
    
    res = gtk_dialog_run (GTK_DIALOG (new_devices->dialog));
        
    if ( res == 0 )
    {
        gtk_widget_destroy (new_devices->dialog);
        return;
    }
    else
    {
        gboolean    state;
        gchar       *mount_point;
        GPtrArray   *to_remount;
        
        to_remount = g_ptr_array_new ();
        
        gtk_tree_model_get_iter_first (GTK_TREE_MODEL (store), &iter);
        for ( i = 0; i < devices->len; i++ )
        { 
            gtk_tree_model_get (GTK_TREE_MODEL (store), &iter, 
                        CHECKED_COLUMN, &state,
                        MOUNT_POINT, &mount_point, -1);
            
            if ( state != 0 )
            {
                if ( !( strcmp (mount_point,start) ) )
                    wrong_name = TRUE;
                else
                {
                    if ( !( add_in_fstab (g_ptr_array_index (devices, i), 
                                        mount_point) ) )
                    {
                        err = g_error_new (1, -1, _("Error : An error occured \
when trying to configure\n %s, please retry. Thanks."), mount_point);
                        show_error (err);
                    }
                    else
                        g_ptr_array_add (to_remount, mount_point);
                }
            }
                    
            gtk_tree_model_iter_next (GTK_TREE_MODEL (store), &iter);          
        }
        
        remount_all (to_remount);
        
        g_ptr_array_free (to_remount, TRUE);
    }
    
    gtk_widget_destroy (new_devices->dialog);
    
    g_free (new_devices);
    
    if ( wrong_name )
    {
        err = g_error_new (1, -1, "Error : It seems that one of your \
mount\npoint name is wrong. Please retry.\n Thanks.");
        show_error (err);
        search_new_devices ();
    }
    
    return;
}

static int
check_in_file (const gchar *device, gchar *file)
{
    FILE*           f;
    struct mntent   *entry;
 
    if( !( f = fopen ( file, "r") ) ) 
    {
        g_warning (_("Error : Can't open %s in read mode.\n"), file);
        return 2;
    }

    g_debug("Searching %s in %s...", device, file);
    
    while( ( entry = getmntent ( f ) ) != NULL )
    {
        if( !strcmp (entry->mnt_fsname, device) )
        {
            fclose (f);
			g_debug ("%s found in %s.", entry->mnt_fsname, file);
            return 0;
        }
    }
	
    fclose (f);
    
    return 1;
}

gboolean
search_new_devices ()
{
    LibHalContext*  hal_ctx;
    LibHalVolume*   volume;
    LibHalDrive*    drive;
    DBusError       error;
    DBusConnection* dbus_conn;
    GPtrArray*      devices;
    GPtrArray*      devices_name;
    gchar**         all_devices;
    gchar*          uuid;
    int             nbs, i, j;

    devices = g_ptr_array_new ();
    devices_name = g_ptr_array_new ();
    
    g_debug("Probing new NTFS devices...");

    /* Initialize hal connection */
    dbus_error_init ( &error );
    dbus_conn = dbus_bus_get ( DBUS_BUS_SYSTEM, &error );
    if( dbus_conn == NULL ) {
        g_warning ("Error: could not connect to dbus: %s: %s\n",
                    error.name, error.message );
        return FALSE;
    }

    hal_ctx = libhal_ctx_new();
    if( !hal_ctx ) {
        g_warning ("Error: libhal_ctx_new\n" );
        return FALSE;
    }

    if( !libhal_ctx_set_dbus_connection ( hal_ctx, dbus_conn ) ) {
        g_warning ("Error: libhal_ctx_set_dbus_connection: %s: %s\n",
                    error.name, error.message );
        return FALSE;
    }
    if( !libhal_ctx_init ( hal_ctx, &error ) ) {
        g_warning ("Error: libhal_ctx_init: %s: %s\n", error.name,
                    error.message );
        return FALSE;
    }
    
    /* Search for NTFS volume not already configure, and not removable */
    all_devices = libhal_manager_find_device_string_match (hal_ctx,
                "volume.fstype", "ntfs", &nbs, &error);

    for ( j=0; j <=1; j++)
    {
        for( i=0; i < nbs; i++)
        {
            const gchar*    device_name;
            const gchar*    device_label;
            gchar*          device;
            gchar*          name;
        
            g_debug ("New partitions detected : %s", all_devices[i]);
            volume = libhal_volume_from_udi (hal_ctx, all_devices[i]);
            drive = libhal_drive_from_udi (hal_ctx, 
                    libhal_volume_get_storage_device_udi (volume));
        
            if( libhal_drive_is_hotpluggable (drive) )
            {
                g_debug ("%s is removable.", libhal_drive_get_device_file (drive));
                continue;
            }
        
            uuid = g_strdup_printf ("UUID=%s", libhal_volume_get_uuid (volume));
            device_name = libhal_volume_get_device_file (volume);
            if( !( check_in_file (device_name, FSTAB) && check_in_file (uuid, FSTAB) 
                            && check_in_file (device_name, MTAB) ) )
            {
                g_free(uuid);
                continue;
            }   
            g_free(uuid);
        
            device = g_strdup (device_name);
            g_debug ("Add in 'to configure' : %s", device);
            g_ptr_array_add (devices, device);
            
            /* Set the name to it's label if available */
            device_label = libhal_volume_get_label (volume);
            if ( device_label )
            {
                gchar  tmp[64];
                g_snprintf (tmp, 64, "/media/%s", device_label);
                name = g_strdup (tmp);
            }
            else
                name = g_strdup (start);
            g_debug("Set label to  : %s", name);
            g_ptr_array_add (devices_name, name);
            
        }
        /* Search also ntfs-3g device */
        all_devices = libhal_manager_find_device_string_match (hal_ctx,
                "volume.fstype", "ntfs-3g", &nbs, &error);
    }
          
    
    /* Shutdown hal connection */
    libhal_ctx_shutdown ( hal_ctx, &error );
    libhal_ctx_free ( hal_ctx );
    
    if ( devices->len != 0)
        ask_new_devices (devices, devices_name);
    
    g_ptr_array_free (devices, TRUE);
    g_ptr_array_free (devices_name, TRUE);
    
    g_debug("Probing new NTFS device exit with succes !");
    
    return TRUE;
}
