/* NVClock 0.8 - Linux overclocker for NVIDIA cards
 *
 * site: http://nvclock.sourceforge.net
 *
 * Copyright(C) 2001-2005 Roderick Colenbrander
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * LM99 hardware monitoring
 */
#include <string.h>
#include "i2c.h"
#include "nvclock.h"


/* various defines for register offsets and such are needed */

#define LM99_REG_LOCAL_TEMP 0x0
#define LM99_REG_REMOTE_TEMP 0x1
#define LM99_REG_MAN_ID 0xfe
	#define NATSEM_MAN_ID 0x1
	#define MAXIM_MAN_ID 0x4d
#define LM99_REG_CHIP_ID 0xff

/* This function should return the chip type .. */
int lm99_detect(I2CDevPtr dev)
{
	I2CByte man_id, chip_id;

	xf86I2CReadByte  (dev, LM99_REG_MAN_ID, &man_id); 
	xf86I2CReadByte  (dev, LM99_REG_CHIP_ID, &chip_id); 
  
	switch(man_id)
	{
		/* National Semiconductor LM99; needs offset? */
		case NATSEM_MAN_ID:
			dev->chip_id = LM99;
			dev->chip_name = (char*)strdup("National Semiconductor LM99");
			break;
		/* Unknown vendor; this chip was used in a FX5700Go laptop and looks similar to the MAx6659 */
		case 0x47:
		/* Maxim; likely a 655x model */
		case MAXIM_MAN_ID:
			dev->chip_id = MAX6559;
			dev->chip_name = (char*)strdup("Maxim MAX6659");
			break;
		default:
			return 0;
	}
  
	return 1;
}

int lm99_get_board_temp(I2CDevPtr dev)
{
	I2CByte temp;
	xf86I2CReadByte(dev, LM99_REG_LOCAL_TEMP, &temp);
	return temp;
}

int lm99_get_gpu_temp(I2CDevPtr dev)
{
	I2CByte temp;
	
	xf86I2CReadByte(dev, LM99_REG_REMOTE_TEMP, &temp);

	/* Cards with lm99 chips need an offset of 16C according to the datasheets.
	/  Further an extra offset of 10C seems to be needed on Geforce6800 cards to match nvidia-settings.
	/  Last but not least Geforce6600GT boards containing an LM99 sensor seem to need a total offset of 21C.
	*/
	if(dev->chip_id == LM99)
	{
		if(dev->arch == NV43)
			temp += 21;
		else if(dev->arch & NV4X)
			temp += 26;
		else
			temp += 16;
	}
	/* Geforce6 boards need an extra offset of +10C on both LM99 and MAX6659 chipsets.
	/  The code below is only executed for MAX6659 as we already handle the extra 10C above.
	*/	
	else if(dev->arch & NV4X)
		temp += 10;
	
	return temp;
}
