#!/usr/bin/perl
#
# $Id: IPStatefulPH.pm,v 1.1 2001/10/06 22:19:14 levine Exp $
#
# Copyright (C) 2001  James D. Levine (jdl@vinecorp.com)
#
#
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your option) any later version.
# 
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
# 
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
#   02111-1307, USA.
#
####################################################################


#
# A stateful PacketHandler for IP traffic.  Creates a separate
# NWatch::state_machine instance for each source/dest/proto combination.
#
# 
#
#
#
#

use NWatch::StateMachine;
use NWatch::IPPacketHandler;
use PortScan::ScannedHost;
use PortScan::ScanSet;
use NWatch::TCPPacket;
use NWatch::UDPPacket;

use strict;

package NWatch::IPStatefulPH;

@NWatch::IPStatefulPH::ISA = qw( NWatch::IPPacketHandler );

sub new
{
    my( $type, $template_scanset, $accept_any_host ) = @_;
    my $self = NWatch::IPPacketHandler::new( $type, $template_scanset, $accept_any_host );

    $self->{machines} = {};	# hash of state machine instances
    
    bless $self, $type;

    $self->compile_tcp;
    $self->compile_udp;
    $self;
}

sub machines { $_[0]->{machines}; }
sub tcp_compiled_def { $_[0]->{tcp_compiled_def}; }
sub udp_compiled_def { $_[0]->{udp_compiled_def}; }


sub handle_packet
{
    my( $self, $packet, $scanset ) = @_;

    # return if an uninteresting protocol
    return if( ! $packet->proto_isa( "ethernet:ipv4" ) );
    (print "IPStatefulPH: igorning packet\n", return)
	if( ( ! $packet->proto_isa( "ethernet:ipv4:tcp" ) ) &&
	    ( ! $packet->proto_isa( "ethernet:ipv4:udp" ) ) &&
	    ( ! $packet->proto_isa( "ethernet:ipv4:icmp4" ) )
	    );


    # return if an uninteresting source/destination
    # for now, assume all source & destinations are interesting


    my $dest = $packet->field_path( "ipv4:destination_address_text" );
    my $source = $packet->field_path( "ipv4:source_address_text" );
    my $protocol = $packet->field_path( "ipv4:protocol" );


    my $machine_id;		# id for specific state machine instance
    my $machine_id_reverse;	# reverse the source/dest so lookup in either direction finds 
                                # the same state machine for the conversation

    if( $packet->proto_isa( "ethernet:ipv4:tcp" ) )
    {
	( $machine_id, $machine_id_reverse ) =
	    $self->gen_machine_names
		( "ipv4:tcp", $source, 
		  $packet->field_path( "ipv4:tcp:source_port" ),
		  $dest, 
		  $packet->field_path( "ipv4:tcp:destination_port" )
		  );
    }
    elsif( $packet->proto_isa( "ethernet:ipv4:udp" ) )
    {
#	print" is udp\n";

	( $machine_id, $machine_id_reverse ) =
	    $self->gen_machine_names
		( "ipv4:udp", $source, 
		  $packet->field_path( "ipv4:udp:source_port" ),
		  $dest, 
		  $packet->field_path( "ipv4:udp:destination_port" )
		  );
    }
    elsif( $packet->proto_isa( "ethernet:ipv4:icmp4" ) )
    {
#	print "IPStatefulPH: it's an icmp4 packet\n";

	if(
	   $packet->proto_isa( "ethernet:ipv4:icmp4:destination_unreachable" )  ||
	   $packet->proto_isa( "ethernet:ipv4:icmp4:time_exceeded" )  ||
	   $packet->proto_isa( "ethernet:ipv4:icmp4:parameter_problem" )  ||
	   $packet->proto_isa( "ethernet:ipv4:icmp4:source_quench" )  ||
	   $packet->proto_isa( "ethernet:ipv4:icmp4:redirect" ) 
	   )
	{
	    # need top_get
	    # my $p = $packet->field_path( "ipv4:icmp4:destination_unreachable:packet" );

	    my $p = $packet->top_field( 'packet' );

	    if( $p->proto_isa( "ipv4:tcp" ) )
	    {
		( $machine_id, $machine_id_reverse ) =
		    $self->gen_machine_names
			( "ipv4:tcp",
			  $p->field_path( "source_address_text" ),
			  $p->field_path( "tcp:source_port" ),
			  $p->field_path( "destination_address_text" ),
			  $p->field_path( "tcp:destination_port" )
			  );
	    }
	    elsif( $p->proto_isa( "ipv4:udp" ) )
	    {
		( $machine_id, $machine_id_reverse ) =
		    $self->gen_machine_names
			( "ipv4:udp",
			  $p->field_path( "source_address_text" ),
			  $p->field_path( "udp:source_port" ),
			  $p->field_path( "destination_address_text" ),
			  $p->field_path( "udp:destination_port" )
			  );
	    }
	    else
	    {
		return;
	    }
	}

#	print "icmp4: $machine_id, \n$machine_id_reverse\n";
     }
    

    my $machine = $self->machines->{ $machine_id };

    if( ( ! ( defined $machine ) ) &&
	( ! $packet->proto_isa( "ethernet:ipv4:icmp4" ) )
	  )
    {
	    $machine = new NWatch::state_machine( $self->tcp_compiled_def, $self )
		if( $packet->proto_isa( "ethernet:ipv4:tcp" ) );
	    $machine = new NWatch::state_machine( $self->udp_compiled_def, $self )
		if( $packet->proto_isa( "ethernet:ipv4:udp" ) );

	    $self->machines->{ $machine_id } = $machine;
	    $self->machines->{ $machine_id_reverse } = $machine;
#	    print "IPStatefulPH: $machine_id was undefined\n"; 
    }
    else
    {
#	print "IPStatefulPH: $machine_id was defined\n"; 
    }

    return if !defined $machine; # catch-all

    # apply the present packet to the state machine 
    $machine->input( $packet );
}

sub gen_machine_names
{
    my( $self, $proto, $source, $source_port, $destination, $destination_port ) = @_;

    ( "ipv4:$proto:$source-$source_port-$destination-$destination_port",
      "ipv4:$proto:$destination-$destination_port-$source-$source_port"
      );
}


sub set_host
{
      my( $self, $proto, $addr, $port, $state ) = @_;

#      print "set_host:  $self, $proto, $addr, $port, $state\n";

      my $host = $self->get_or_create_host( $addr, $self->accept_any_host );
#      print "set_host: host is $host . \n";
      return if !(defined $host);

      my $s = $host->get_state( $proto, $port );

#      print "set_host: existing state is $s \n";

      if( $s eq 'open' )
      {
  	# leave as is - we care most to know if the port was ever open
      }
      elsif( $s eq 'closed' )
      {
  	# only open overrides closed
  	$host->set_port( $port, $state, $proto, "", "", "", "" ) if $state eq 'open';
      }
      elsif( $s eq 'filtered' )	#  unknown
      {
	  # open or closed override filtered
	  $host->set_port( $port, $state, $proto, "", "", "", "" ) if $state =~ /open|closed/;
      }
      elsif( $s eq 'unknown' )	# accept any state if was previously unknown
      {
  	$host->set_port( $port, $state, $proto, "", "", "", "" );
      }
      else
      {
  	$host->set_port( $port, $state, $proto, "", "", "", "" );
      }
}


sub get_or_create_host
{
    my( $self, $address, $accept_any_host ) = @_;

    my $scanset = $self->output_scanset;

    my $host =  $scanset->get_host( $address );

#    print "goch: template " . $self->template_scanset() . "\n";
#    print "goch: host " . $host . "\n";


    if( ! defined( $host ) )
    {
	my $template_host =  $self->template_scanset()->get_host( $address );
	
	if( $accept_any_host || (defined $template_host) ) 
	{
	    my $new_host = new PortScan::ScannedHost( $address );
	    $new_host->addr( $address );
	    $new_host->default_state( "unknown" );

	    my $date = ` date "+%D %H:%M:%S" `;
	    chomp $date;
	    print "$date adding new host $address\n";
	    $scanset->add_host( $new_host );

	    return $new_host;
	}
	else
	{
#	    print "goch: skipping host $address \n";
	    return undef;
	}
    }


#    print "goch: $address - $host \n";

    $host;
}




sub compile_tcp
{
    my $self = shift;

    $self->{tcp_compiled_def} =

  NWatch::state_machine::compile
      (
       [

	[ 'start', '#print "tcp1 " . %p:ipv4:source_address_text% . " -> " . %p:ipv4:destination_address_text% . "\n";', 
	  [
	   [ '%p:ipv4:tcp:syn% && (! %p:ipv4:tcp:ack%)', 'syn-sent' ]
	   ]
	  ],

	# syn-sent: someone tries to open a connection
	[ 'syn-sent', '%s:destination_addr% = %p:ipv4:destination_address_text%; %s:destination_port% = %p:ipv4:tcp:destination_port%; #print "syn-sent " . %p:ipv4:source_address_text% . " -> " . %p:ipv4:destination_address_text% . "\n";', 
	  [
	   [ '%p:ipv4:tcp:ack%  &&  %p:ipv4:tcp:syn%', 'syn-ack-sent' ],
	   [ '%p:ipv4:tcp:rst%',                       'syn-rst' ],
	   [ '%p:eot% && ( $self->age > 10 )',         'port-filtered' ],

	   ]
	  ],

	# syn-ack-sent: we can deduce the port is open
	[ 'syn-ack-sent', '$self->handler->set_host( "tcp", %s:destination_addr%, %s:destination_port%, "open" ); # print "syn-ack-sent " . %s:destination_addr% . " " . %s:destination_port% . " is open\n";  ', 
	  [
	   [ '1', 'nothing' ],
	   ]
	  ],
	
	# syn-rst: we can deduce the port is closed
	[ 'syn-rst', '$self->handler->set_host( "tcp", %s:destination_addr%, %s:destination_port%, "closed" ); # print "syn-ack-sent " . %s:destination_addr% . " " . %s:destination_port% . " is closed\n"; ', 
	  [
	   [ '1', 'nothing' ],
	   ]
	  ],

	# port-filtered: we can deduce the port is filtered
	[ 'port-filtered', '$self->handler->set_host( "tcp", %s:destination_addr%, %s:destination_port%, "filtered" );  # print "syn-ack-sent " . %s:destination_addr% . " " . %s:destination_port% . " is filtered\n"; ', 
	  [
	   [ '1', 'nothing' ],
	   ]
	  ],

	# do nothing
	[ 'nothing', '',
	  [
	   [ '1', 'nothing' ],
	   ],
	  ]

	]
       );
}



sub compile_udp
{
    my $self = shift;

    $self->{udp_compiled_def} =

  NWatch::state_machine::compile
      (
       [

	[ 'start', '#print "udp " . %p:ipv4:source_address_text% . " -> " . %p:ipv4:destination_address_text% . "\n";  ',
	  [
	   ['1', 'udp-sent']
	   ]
	  ],

	[ 'udp-sent', '%s:destination_addr% = %p:ipv4:destination_address_text%; %s:destination_port% = %p:ipv4:udp:destination_port%; ', # name, entry_expr
	  [
	   [ '$p->proto_isa( "ethernet:ipv4:icmp4:destination_unreachable" )', 'port-closed' ],
	   [ '%p:eot% && ( $self->age > 10 )',         'port-open' ],
	   [ '1', 'udp-sent' ],	
	   ]		
	  ],		


	[ 'port-closed', '$self->handler->set_host( "udp", %s:destination_addr%, %s:destination_port%, "closed" ); #print "udp " . %s:destination_addr% . " " . %s:destination_port% . " is closed\n";', # name, entry_expr
	  [		
	   [ '1', 'nothing' ]
	   ]		
	  ],		


	[ 'port-open', '$self->handler->set_host( "udp", %s:destination_addr%, %s:destination_port%, "open" ); #print "udp " . %s:destination_addr% . " " . %s:destination_port% . " is open\n";', # name, entry_expr
	  [		
	   [ '1', 'nothing' ]
	   ]		
	  ],		

	
	# do nothing
	[ 'nothing', '',
	  [
	   [ '1', 'nothing' ],
	   ],
	  ]


	]
       );
}



sub finish_interval
{
    my $self = shift;
#    print "IPStatefulPH: finish_interval entry \n";

    # there are two references for each state machine in the machines
    # hash so track which state machines have been notified with the
    # eot_packet, and do not repeat the notification.
    my $notified = {};

    my $eot = new NWatch::EOT_packet;

    # iterate over all state machines, sending an eot_packet to each
    foreach my $m ( values %{$self->machines} )
    {
	if( $notified->{ $m } != 1 )
	{
	    $m->input( $eot );
	    $notified->{ $m } = 1;
	}

    }

}


1;




















