# --------------------------------------------------------
# OpenDb New Installation script for MySQL
# --------------------------------------------------------

#
# System Item Type Group
#
# system_ind - indicates if a group should be used by the system
# to group s_item_type's or if the grouping is only for use in
# listings functionality.
#
DROP TABLE IF EXISTS s_item_type_group;
CREATE TABLE s_item_type_group (
	s_item_type_group	VARCHAR(10) NOT NULL,
	description		VARCHAR(60) NOT NULL,
	system_ind		VARCHAR(1) NOT NULL DEFAULT 'N',
	PRIMARY KEY ( s_item_type_group )
) TYPE=MyISAM COMMENT='System Item Type Group';

#
# System Item Type Group Relationship
#
DROP TABLE IF EXISTS s_item_type_group_rltshp;
CREATE TABLE s_item_type_group_rltshp (
	s_item_type_group	VARCHAR(10) NOT NULL,
	s_item_type		VARCHAR(10) NOT NULL,
	PRIMARY KEY ( s_item_type_group, s_item_type )
) TYPE=MyISAM COMMENT='System Item Type Group Relationship';

#
# System Item Type table
#
DROP TABLE IF EXISTS s_item_type;
CREATE TABLE s_item_type (
  s_item_type		varchar(10) NOT NULL,
  description		varchar(60) NOT NULL,
  image				varchar(255),
  order_no			tinyint(2),
  PRIMARY KEY ( s_item_type )
) TYPE=MyISAM COMMENT='System Item Type table';

#
# System Attribute Type table
#
DROP TABLE IF EXISTS s_attribute_type;
CREATE TABLE s_attribute_type (
  s_attribute_type	varchar(10) NOT NULL,
  description		varchar(60) NOT NULL,
  prompt			varchar(30) default NULL,
  input_type		varchar(255),
  display_type		varchar(255),
  s_field_type		varchar(10),
  site_type			varchar(10),
  PRIMARY KEY ( s_attribute_type )
) TYPE=MyISAM COMMENT='System Attribute table';

#
# System Item Attribute Type relationship table
#
DROP TABLE IF EXISTS s_item_attribute_type;
CREATE TABLE s_item_attribute_type (
  s_item_type		varchar(10) NOT NULL,
  s_attribute_type	varchar(10) NOT NULL,
  order_no			tinyint(3) unsigned NOT NULL,
  prompt			varchar(30),
  compulsory_ind	varchar(1),
  PRIMARY KEY ( s_item_type, s_attribute_type, order_no )
) TYPE=MyISAM COMMENT='System Item Attribute table';

#
# System Attribute Type Lookup table
#
DROP TABLE IF EXISTS s_attribute_type_lookup;
CREATE TABLE s_attribute_type_lookup (
  s_attribute_type		varchar(10) NOT NULL,
  order_no				tinyint(3) unsigned,
  value					varchar(50) NOT NULL,
  display				varchar(255),
  img					varchar(255),
  checked_ind			varchar(1),
  PRIMARY KEY ( s_attribute_type, value )
) TYPE=MyISAM COMMENT='System Attribute Type Lookup table';

#
# System Item Status table
#
DROP TABLE IF EXISTS s_status_type;
CREATE table s_status_type (
  s_status_type				varchar(1) NOT NULL default 'Y',
  description				varchar(30) NOT NULL,
  img						varchar(255),
  insert_ind				varchar(1) NOT NULL default 'Y',
  update_ind				varchar(1) NOT NULL default 'Y',
  delete_ind				varchar(1) NOT NULL default 'Y',
  change_owner_ind			varchar(1) NOT NULL default 'N',
  min_display_user_type		varchar(1),
  min_create_user_type		varchar(1),
  new_owner_instance_ind	varchar(1) NOT NULL default 'Y',
  new_not_owner_instance_ind	varchar(1) NOT NULL default 'Y',
  borrow_ind				varchar(1) NOT NULL default 'Y',
  status_comment_ind		varchar(1) NOT NULL default 'N',
  default_ind				varchar(1),
  closed_ind				varchar(1) NOT NULL default 'N',
  PRIMARY KEY ( s_status_type )
) TYPE=MyISAM COMMENT='System Item Status table';

#
# System Address type table
#
DROP TABLE IF EXISTS s_address_type;
CREATE TABLE s_address_type (
  s_address_type			varchar(10) NOT NULL,
  description				varchar(30) NOT NULL,
  display_order				tinyint(2),
  min_create_user_type		varchar(1) NOT NULL default 'B', # borrower
  min_display_user_type		varchar(1) NOT NULL default 'N', # normal
  compulsory_for_user_type	varchar(1) NOT NULL default 'B', # normal
  closed_ind				varchar(1) NOT NULL default 'N',
  PRIMARY KEY ( s_address_type )
) TYPE=MyISAM COMMENT='System address type';
#
# System Address Type Attribute relationship table
#
DROP TABLE IF EXISTS s_addr_attribute_type_rltshp;
CREATE TABLE s_addr_attribute_type_rltshp (
  s_address_type		varchar(10) NOT NULL,
  s_attribute_type		varchar(10) NOT NULL,
  order_no				tinyint(3) unsigned NOT NULL,
  # override for s_attribute_type prompt field
  prompt				varchar(30),
  min_create_user_type		varchar(1),
  min_display_user_type		varchar(1),
  compulsory_for_user_type	varchar(1),
  closed_ind			varchar(1) NOT NULL default 'N',
  PRIMARY KEY ( s_address_type, s_attribute_type, order_no )
) TYPE=MyISAM COMMENT='System address attribute type relationship';

#
# User table
#
DROP TABLE IF EXISTS user;
CREATE TABLE user (
  user_id		varchar(20) NOT NULL,
  fullname		varchar(100) NOT NULL,
  pwd			varchar(40) NOT NULL,
  type			varchar(1) NOT NULL default 'N',
  language		varchar(20),
  theme			varchar(20),
  lastvisit		datetime NOT NULL default '0000-00-00 00:00:00',
  active_ind	varchar(1) NOT NULL default 'Y',
  PRIMARY KEY ( user_id )
) TYPE=MyISAM COMMENT='User table';

#
# User address
#
DROP TABLE IF EXISTS user_address;
CREATE TABLE user_address (
  sequence_number	integer(10) unsigned NOT NULL auto_increment,
  user_id			varchar(20) NOT NULL,
  s_address_type	varchar(10) NOT NULL,
  start_dt			date NOT NULL default '0000-00-00',
  end_dt			date,
  update_on			timestamp(14) NOT NULL,
  PRIMARY KEY ( sequence_number ),
  KEY user_address_idx ( user_id, s_address_type, start_dt )
) TYPE=MyISAM COMMENT='User address';

#
# User address attribute
#
DROP TABLE IF EXISTS user_address_attribute;
CREATE TABLE user_address_attribute (
  ua_sequence_number	integer(10) unsigned NOT NULL,
  s_attribute_type		varchar(10) NOT NULL,
  order_no				tinyint(3) unsigned NOT NULL,
  lookup_attribute_val 	varchar(50) NOT NULL,
  attribute_val			text,
  update_on				timestamp(14) NOT NULL,
  PRIMARY KEY ( ua_sequence_number, s_attribute_type, order_no, lookup_attribute_val )
) TYPE=MyISAM COMMENT='User address attribute';

#
# Table structure for table 'item'
#
DROP TABLE IF EXISTS item;
CREATE TABLE item (
  id			integer(10) unsigned NOT NULL auto_increment,
  parent_id		integer(10) unsigned,
  title			varchar(255) NOT NULL,
  category		varchar(100),
  s_item_type	varchar(10) NOT NULL,
  PRIMARY KEY ( id ),
  KEY title_idx ( title ),
  KEY s_item_type_idx ( s_item_type )
) TYPE=MyISAM COMMENT='Item table';

#
# Item instance table.
#
DROP TABLE IF EXISTS item_instance;
CREATE table item_instance (
  item_id			bigint(10) unsigned NOT NULL,
  instance_no		smallint(5) unsigned NOT NULL,
  owner_id			varchar(20) NOT NULL,
  borrow_duration	smallint(3) unsigned,
  s_status_type		varchar(1) NOT NULL default 'Y',
  status_comment 	varchar(255),
  update_on			timestamp(14) NOT NULL,
  PRIMARY KEY ( item_id, instance_no ),
  KEY owner_id_idx ( owner_id ),
  KEY s_status_type_idx ( s_status_type )
) TYPE=MyISAM COMMENT='Item Instance table';

#
# Item Attribute table
#
DROP TABLE IF EXISTS item_attribute;
CREATE TABLE item_attribute (
  item_id			integer(10) unsigned NOT NULL,
  s_attribute_type	varchar(10) NOT NULL,
  order_no			tinyint(3) unsigned NOT NULL,
  lookup_attribute_val varchar(50) NOT NULL,
  attribute_val		text,
  update_on			timestamp(14) NOT NULL,
  PRIMARY KEY ( item_id, s_attribute_type, order_no, lookup_attribute_val )
) TYPE=MyISAM COMMENT='Item Attribute table';

#
# Borrowed Item table
#
DROP TABLE IF EXISTS borrowed_item;
CREATE TABLE borrowed_item (
  sequence_number	integer(10) unsigned NOT NULL auto_increment,
  item_id			integer(10) unsigned NOT NULL,
  instance_no		smallint(5) unsigned NOT NULL,
  borrower_id		varchar(20) NOT NULL,
  borrow_duration	smallint(3) unsigned,
  total_duration	smallint(3) unsigned,
  status			varchar(1) NOT NULL,
  update_on			timestamp(14) NOT NULL,
  PRIMARY KEY ( sequence_number ),
  KEY borrower_idx ( borrower_id ),
  KEY item_instance_idx ( item_id, instance_no )
) TYPE=MyISAM COMMENT='Borrowed Item table';

#
# Table structure for table 'review'
#
DROP TABLE IF EXISTS review;
CREATE TABLE review (
  sequence_number	integer(10) unsigned NOT NULL auto_increment,
  author_id			varchar(20) NOT NULL,
  item_id			integer(10) unsigned NOT NULL,
  comment			text,
  rating			varchar(1) NOT NULL,
  update_on			timestamp(14) NOT NULL,
  PRIMARY KEY ( sequence_number ),
  KEY author_idx ( author_id ),
  KEY item_idx ( item_id )
) TYPE=MyISAM COMMENT='Item Review table';

#
# Site Plugins (v2 architecture) infrastructure tables
#

#
# Site Plugin Configuration
#
DROP TABLE IF EXISTS s_site_plugin;
CREATE TABLE s_site_plugin (
	site_type	VARCHAR(10) NOT NULL,
	classname	VARCHAR(50) NOT NULL,
	order_no	TINYINT(2) UNSIGNED NOT NULL,
	title		VARCHAR(50) NOT NULL,
	image		VARCHAR(255) NOT NULL,
	description	VARCHAR(255) NOT NULL,
	external_url	VARCHAR(255) NOT NULL,
	items_per_page	TINYINT(3) UNSIGNED NOT NULL,
	more_info_url	VARCHAR(255),
	PRIMARY KEY ( site_type )
) TYPE=MyISAM COMMENT='Site Plugin Configuration';

#
# This table provides any site plugin specific variable configuration,
# and a plugin should provide defaults for all such conf variables
# when installed, so that the user can correctly configure them
# if required based on the description field.
#
DROP TABLE IF EXISTS s_site_plugin_conf;
CREATE TABLE s_site_plugin_conf (
	site_type	VARCHAR(10) NOT NULL,
	name		VARCHAR(50) NOT NULL,
	keyid		VARCHAR(50) NOT NULL DEFAULT '0',
	description	VARCHAR(255),
	value		VARCHAR(255),
	PRIMARY KEY ( site_type, name, keyid )
) TYPE=MyISAM COMMENT='Site Plugin Configuration';

#
# Site Plugin Input Field
#
# This table will define the input fields generated for
# the plugin in the item_add (or item_input if we
# merge this into one location) script.
#
DROP TABLE IF EXISTS s_site_plugin_input_field;
CREATE TABLE s_site_plugin_input_field (
	site_type	VARCHAR(10) NOT NULL,
	field		VARCHAR(20) NOT NULL,
	order_no	TINYINT(2) UNSIGNED NOT NULL,
	description	VARCHAR(255) NOT NULL,
	prompt		VARCHAR(30) NOT NULL,
	field_type	VARCHAR(10) NOT NULL DEFAULT 'text',
	default_value		VARCHAR(50),
	refresh_mask	VARCHAR(255),
	PRIMARY KEY ( site_type, field )
) TYPE=MyISAM COMMENT='Site Plugin Input Field';

#
# Site Plugin Attribute Type Map
#
DROP TABLE IF EXISTS s_site_plugin_s_attribute_type_map;
CREATE TABLE s_site_plugin_s_attribute_type_map (
	sequence_number		INTEGER(10) UNSIGNED NOT NULL AUTO_INCREMENT,
	site_type		VARCHAR(10) NOT NULL,
	s_item_type_group	VARCHAR(10) NOT NULL DEFAULT '*',
	s_item_type		VARCHAR(10) NOT NULL DEFAULT '*',
	variable		VARCHAR(20) NOT NULL,
	s_attribute_type	VARCHAR(10) NOT NULL,
	PRIMARY KEY ( sequence_number ),
	UNIQUE KEY ( site_type, s_item_type_group, s_item_type, variable, s_attribute_type )
) TYPE=MyISAM COMMENT='Site Plugin Attribute Type Map';

#
# Site Plugin Attribute Type Lookup Map
#
DROP TABLE IF EXISTS s_site_plugin_s_attribute_type_lookup_map;
CREATE TABLE s_site_plugin_s_attribute_type_lookup_map (
	sequence_number		INTEGER(10) UNSIGNED NOT NULL AUTO_INCREMENT,
	site_type		VARCHAR(10) NOT NULL,
	s_attribute_type	VARCHAR(10) NOT NULL,
	value			VARCHAR(255) NOT NULL,
	lookup_attribute_val	VARCHAR(50) NOT NULL,
	PRIMARY KEY ( sequence_number ),
	UNIQUE KEY ( site_type, s_attribute_type, value, lookup_attribute_val )
) TYPE=MyISAM COMMENT='Site Plugin Attribute Type Lookup Map';

#
# Site Plugin Link
#
DROP TABLE IF EXISTS s_site_plugin_link;
CREATE TABLE s_site_plugin_link (
	sequence_number		INTEGER(10) UNSIGNED NOT NULL AUTO_INCREMENT,
	site_type		VARCHAR(10) NOT NULL,
	s_item_type_group	VARCHAR(10) NOT NULL DEFAULT '*',
	s_item_type		VARCHAR(10) NOT NULL DEFAULT '*',
	order_no		TINYINT(2) UNSIGNED NOT NULL,
	description		VARCHAR(50),
	url			VARCHAR(255),
	title_url		VARCHAR(255),
	PRIMARY KEY ( sequence_number ),
	UNIQUE KEY ( site_type, s_item_type_group, s_item_type, order_no )
) TYPE=MyISAM COMMENT='Site Plugin Link';

#
# Create Import Cache table
#
CREATE TABLE import_cache (
  sequence_number	INTEGER(10) unsigned NOT NULL auto_increment,
  user_id			VARCHAR(20) NOT NULL,
  plugin_name		VARCHAR(100),
  content_length	INTEGER(10) unsigned NOT NULL,
  PRIMARY KEY ( sequence_number )
) TYPE=MyISAM COMMENT='Import Cache table';

#
# Create File Cache table
#
CREATE TABLE file_cache (
  sequence_number	INTEGER(10) unsigned NOT NULL auto_increment,
  cache_type		VARCHAR(10) NOT NULL DEFAULT 'HTTP',
  cache_date		DATETIME NOT NULL,
  expire_date		DATETIME NOT NULL,
  url				MEDIUMTEXT NOT NULL,
  # if url was redirected, this stores the redirected URL
  location			MEDIUMTEXT,
  content_type		VARCHAR(100) NOT NULL,
  # we do not want to calculate the length at 
  # runtime of the content column
  content_length	INTEGER(10) unsigned NOT NULL,
  # if GZIP compression used, this will record what level.  A
  # zero (0) level means no compression.  This allows the
  # config variable to be changed without effecting existing
  # cache entries.
  gzcompress_level	tinyint(1) NOT NULL DEFAULT 0,
  content			BLOB,
  PRIMARY KEY ( sequence_number )
) TYPE=MyISAM COMMENT='File Cache table';

# -----------------------------------------------------------------------------------------------
# Default Installation System Data
# -----------------------------------------------------------------------------------------------
INSERT INTO s_status_type ( s_status_type, description, img, insert_ind, update_ind, delete_ind, change_owner_ind, min_display_user_type, min_create_user_type, new_owner_instance_ind, new_not_owner_instance_ind, borrow_ind, status_comment_ind, default_ind, closed_ind )
VALUES ('A', 'Available', 'avail.gif', 'Y', 'Y', 'Y', 'N', 'G', 'N', 'Y', 'Y', 'Y', 'N', 'Y', 'N');

INSERT INTO s_status_type ( s_status_type, description, img, insert_ind, update_ind, delete_ind, change_owner_ind, min_display_user_type, min_create_user_type, new_owner_instance_ind, new_not_owner_instance_ind, borrow_ind, status_comment_ind, default_ind, closed_ind )
VALUES ('H', 'Hidden', 'hidden.gif', 'Y', 'Y', 'Y', 'N', 'A', 'N', 'Y', 'Y', 'Y',  'N', '', 'N');

INSERT INTO s_status_type ( s_status_type, description, img, insert_ind, update_ind, delete_ind, change_owner_ind, min_display_user_type, min_create_user_type, new_owner_instance_ind, new_not_owner_instance_ind, borrow_ind, status_comment_ind, default_ind, closed_ind )
VALUES ('N', 'Inactive', 'inactive.gif', 'Y', 'Y', 'Y', 'N', 'G', 'N', 'Y', 'Y', 'N',  'Y', '', 'N');

INSERT INTO s_status_type ( s_status_type, description, img, insert_ind, update_ind, delete_ind, change_owner_ind, min_display_user_type, min_create_user_type, new_owner_instance_ind, new_not_owner_instance_ind, borrow_ind, status_comment_ind, default_ind, closed_ind )
VALUES ('O', 'Ordered', 'ordered.gif', 'Y', 'Y', 'Y', 'N', 'G', 'N', 'N', 'Y', 'X',  'Y', '', 'N');

INSERT INTO s_status_type ( s_status_type, description, img, insert_ind, update_ind, delete_ind, change_owner_ind, min_display_user_type, min_create_user_type, new_owner_instance_ind, new_not_owner_instance_ind, borrow_ind, status_comment_ind, default_ind, closed_ind )
VALUES ('W', 'Wishlist', 'wishlist.gif', 'Y', 'Y', 'Y', 'N', 'G', 'N', 'N', 'Y', 'X',  'Y', '', 'N');

INSERT INTO s_status_type ( s_status_type, description, img, insert_ind, update_ind, delete_ind, change_owner_ind, min_display_user_type, min_create_user_type, new_owner_instance_ind, new_not_owner_instance_ind, borrow_ind, status_comment_ind, default_ind, closed_ind )
VALUES ('X', 'External', 'external.gif', 'N', 'Y', 'N', 'N', 'G', 'N', 'Y', 'Y', 'B',  'H', '', 'N');

#
# System Item Type
#
INSERT INTO s_item_type (s_item_type, description, image, order_no) VALUES ( 'DVD', 'Digital Versatile Disc', 'dvd.gif', '0');
INSERT INTO s_item_type (s_item_type, description, image, order_no) VALUES ( 'VHS', 'VHS Video', 'vhs.gif', '1');
INSERT INTO s_item_type (s_item_type, description, image, order_no) VALUES ( 'CD', 'Compact Disc', 'cd.gif', '2');
INSERT INTO s_item_type (s_item_type, description, image, order_no) VALUES ( 'DIVX', 'DivX Video', 'divX.gif', '3');
INSERT INTO s_item_type (s_item_type, description, image, order_no) VALUES ( 'VCD', 'Video Disc', 'vcd.gif', '4');
INSERT INTO s_item_type (s_item_type, description, image, order_no) VALUES ( 'LD', 'Laser Disc', 'ld.gif', '5');

#
# System Item Type Group
#
INSERT INTO s_item_type_group(s_item_type_group, description, system_ind) VALUES('AUDIO', 'Audio Item Types', 'Y');
INSERT INTO s_item_type_group(s_item_type_group, description, system_ind) VALUES('VIDEO', 'Video Item Types', 'Y');
INSERT INTO s_item_type_group(s_item_type_group, description, system_ind) VALUES('OTHER', 'Miscellaneous Item Types', 'N');

#
# System Item Type Group Relationship
#
INSERT INTO s_item_type_group_rltshp(s_item_type_group, s_item_type) VALUES('AUDIO', 'CD');
INSERT INTO s_item_type_group_rltshp(s_item_type_group, s_item_type) VALUES('AUDIO', 'MP3');
INSERT INTO s_item_type_group_rltshp(s_item_type_group, s_item_type) VALUES('VIDEO', 'DVD');
INSERT INTO s_item_type_group_rltshp(s_item_type_group, s_item_type) VALUES('VIDEO', 'VHS');
INSERT INTO s_item_type_group_rltshp(s_item_type_group, s_item_type) VALUES('VIDEO', 'VCD');
INSERT INTO s_item_type_group_rltshp(s_item_type_group, s_item_type) VALUES('VIDEO', 'LD');
INSERT INTO s_item_type_group_rltshp(s_item_type_group, s_item_type) VALUES('VIDEO', 'DIVX');
INSERT INTO s_item_type_group_rltshp(s_item_type_group, s_item_type) VALUES('OTHER', 'BOOK');
INSERT INTO s_item_type_group_rltshp(s_item_type_group, s_item_type) VALUES('OTHER', 'GAME');

#
# System Attribute Types
#

# System attributes
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'S_TITLE', 'Item Title', 'Title', 'text(50,255)', 'hidden', 'TITLE', NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'S_STATUS', 'System Status Type', 'Status Type', NULL, 'hidden', 'STATUSTYPE', NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'S_STATCMNT', 'System Status Comment', 'Status Comment', 'textarea(50,5)', 'hidden', 'STATUSCMNT', NULL);

# DVD,DIVX,VCD,VHS,LD attributes
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'ALT_TITLE', 'Alternate Title', 'Alternate Title', 'text(50)', 'display(%value%)', '', NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'ACTORS', 'List of Actors in a movie', 'Actors', 'text(50)', 'list(plain, ",", list-link)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'DIRECTOR', 'Director of a Movie', 'Director', 'text(50)', 'list(plain, ",", list-link)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'RUN_TIME', 'Running time','Length', 'number(4, %field% <i>minutes</i>)', 'format_mins(%h %H %m %M)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'AUDIO_LANG', 'Audio Languages', 'Language', 'checkbox_grid(%img% %display%, 3)', 'display(%img% %display%, list-link)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'SUBTITLES', 'Subtitle languages', 'Subtitles', 'checkbox_grid(%img% %display%, 4)', 'display(%img% %display%, list-link)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'AGE_RATING', 'Age Recommendation', 'Age', 'radio_grid(%img% %display%, 1)', 'display(%img%, list-link)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'MOVIEGENRE', 'Movie Genre', 'Genre', 'checkbox_grid(%display%, 4)', 'category(%display%, list-link)', 'CATEGORY', NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'IMAGEURL', 'Item Image URL', 'Image', 'upload_or_saveurl(50,*,"gif,jpg,jpeg,png")', 'hidden', 'IMAGE', NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'DVD_EXTRAS', 'DVD Extra Features Details', 'Extras', 'textarea(50,5)', 'list(ticks)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'DVD_REGION','DVD Region', 'Region', 'checkbox_grid(%value% - %display%, 1)', 'list(plain, " ", list-link)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'RATIO', 'Aspect Ratio of Movie', 'Aspect Ratio', 'checkbox_grid(%value%, *)', 'list(plain, " ", list-link)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'VID_FORMAT', 'Encoding Format of Video','Video Format', 'single_select(%value%)', 'display(%value%, list-link)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'VHS_TYPE', 'Video Type', 'VHS Format', 'single_select(%display%)', 'display(%display%, list-link)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'VIDQUALITY', 'Quality of Video', 'Quality', 'radio_grid(%display%, *)', 'display(%display%, list-link)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'COMMENTS', 'Extra Comments', 'Comments', 'textarea(50,5)', 'display(%value%)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'MOVIE_PLOT', 'Plot of a Movie', 'Plot', 'textarea(50,5)', 'display(%value%)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'YEAR', 'Year of Release', 'Year', 'number(4)', 'display(%value%, list-link)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'ANAMORPHIC','Anamorphic indicator', 'Anamorphic', 'checkbox(Y,,)', '%value%', NULL, NULL);

# CD Attributes
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'ARTIST', 'Music Artist', 'Artist', 'text(50)', 'list(plain, ",", list-link)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'CDTRACK', 'CD Track Title', 'Track', 'text(30)', 'display(%value%)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'CDTIME', 'CD Play Length', 'Time', 'text(6)', 'display(%value%)', NULL, NULL);
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'MUSICGENRE', 'Music Style', 'Style', 'single_select(%display%)', 'category(%display%, list-link)', 'CATEGORY', NULL);

# DivX Attributes
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'DIVXVIDVER', 'Video Codec Version', 'Video Codec', 'single_select(%display%)', 'display(%display%, list-link)', NULL, NULL); 
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'DIVXAUDVER', 'Audio Codec Version', 'Audio Codec', 'single_select(%display%)', 'display(%display%, list-link)', NULL, NULL); 
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'DIVXQUALIT', 'DIVX Quality', 'Quality', 'single_select(%display%)', 'display(%display%, list-link)', NULL, NULL); 

#
# This attribute is reserved for use in item_review.  Please do not use it for your own s_item_attribute_type structures.
# Note: display_type will be ignored when review actually displayed!!!
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ('S_RATING', 'Item Rating', 'Rating', 'review_options(%display%, VERTICAL)', NULL, 'RATING', NULL);

#
# This attribute is reserved for use in Borrow Duration functionality.  Please do not use it for your own s_item_attribute_type structures.
# If you want you could run this update to change the input type for S_DURATION a numeric input field instead:
# UPDATE s_attribute_type SET input_type = 'number(3, %field% <i>days</i>)' WHERE s_attribute_type = 'S_DURATION'
#
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ('S_DURATION', 'Borrow Duration', 'Borrow Duration', 'single_select(%display%)', 'display(%display%)', 'DURATION', NULL);

#
# Display item.id
#
INSERT INTO s_attribute_type (s_attribute_type, description, prompt, input_type, display_type, s_field_type, site_type) VALUES ( 'S_ITEM_ID', 'OpenDb Item ID', 'ID#', 'hidden', 'hidden', 'ITEM_ID', NULL);

#
# Address type attributes
#
INSERT INTO s_attribute_type ( s_attribute_type, description, prompt, input_type, display_type, s_field_type )
VALUES ( 'ADDR_LINE', 'Address Line', 'Address', 'text(50,255)', 'display(%value%)', 'ADDRESS' );

INSERT INTO s_attribute_type ( s_attribute_type, description, prompt, input_type, display_type, s_field_type )
VALUES ( 'CITY', 'City', 'City', 'text(50,100)', 'display(%value%)', 'ADDRESS' );

INSERT INTO s_attribute_type ( s_attribute_type, description, prompt, input_type, display_type, s_field_type )
VALUES ( 'STATE', 'State', 'State', 'text(20,100)', 'display(%display%)', 'ADDRESS' );

INSERT INTO s_attribute_type ( s_attribute_type, description, prompt, input_type, display_type, s_field_type )
VALUES ( 'POSTCODE', 'Post code', 'Postcode', 'number(10)', 'display(%value%)', 'ADDRESS' );

INSERT INTO s_attribute_type ( s_attribute_type, description, prompt, input_type, display_type, s_field_type )
VALUES ( 'COUNTRY', 'Country', 'Country', 'single_select(%display%)', 'display(%value% - %display%)', 'ADDRESS' );

INSERT INTO s_attribute_type ( s_attribute_type, description, prompt, input_type, display_type, s_field_type )
VALUES ( 'PHONE_NO', 'Phone Number', 'Phone', 'filtered(20,50,0-9 \\\\-+)', 'display(%value%)', 'ADDRESS' );

INSERT INTO s_attribute_type ( s_attribute_type, description, prompt, input_type, display_type, s_field_type )
VALUES ( 'EMAIL_ADDR', 'Email address', 'Email', 'email(30,50)', 'display(%value%)', 'ADDRESS' );

#
# System Address Type
# 
INSERT INTO s_address_type ( s_address_type, description, display_order, min_create_user_type, min_display_user_type, compulsory_for_user_type )
VALUES ( 'EMAIL', 'Email Address', '1', 'B', 'N', 'B' );

INSERT INTO s_address_type ( s_address_type, description, display_order, min_create_user_type, min_display_user_type, compulsory_for_user_type )
VALUES ( 'SNAIL', 'Postal Address', '2', 'B', 'B', 'B' );

#
# System Address Type relationship
# 
INSERT INTO s_addr_attribute_type_rltshp (s_address_type, s_attribute_type, order_no, prompt, min_create_user_type, min_display_user_type, compulsory_for_user_type, closed_ind)
VALUES ( 'EMAIL', 'EMAIL_ADDR', '1', NULL, NULL, NULL, NULL, 'N' );
INSERT INTO s_addr_attribute_type_rltshp (s_address_type, s_attribute_type, order_no, prompt, min_create_user_type, min_display_user_type, compulsory_for_user_type, closed_ind)
VALUES ( 'SNAIL', 'ADDR_LINE', '1', 'Address Line 1', NULL, NULL, NULL, 'N' );
INSERT INTO s_addr_attribute_type_rltshp (s_address_type, s_attribute_type, order_no, prompt, min_create_user_type, min_display_user_type, compulsory_for_user_type, closed_ind)
VALUES ( 'SNAIL', 'ADDR_LINE', '2', 'Address Line 2', NULL, NULL, '*', 'N' );
INSERT INTO s_addr_attribute_type_rltshp (s_address_type, s_attribute_type, order_no, prompt, min_create_user_type, min_display_user_type, compulsory_for_user_type, closed_ind)
VALUES ( 'SNAIL', 'CITY', '3', NULL, NULL, NULL, NULL, 'N' );
INSERT INTO s_addr_attribute_type_rltshp (s_address_type, s_attribute_type, order_no, prompt, min_create_user_type, min_display_user_type, compulsory_for_user_type, closed_ind)
VALUES ( 'SNAIL', 'STATE', '4', NULL, NULL, NULL, NULL, 'N' );
INSERT INTO s_addr_attribute_type_rltshp (s_address_type, s_attribute_type, order_no, prompt, min_create_user_type, min_display_user_type, compulsory_for_user_type, closed_ind)
VALUES ( 'SNAIL', 'POSTCODE', '5', NULL, NULL, NULL, NULL, 'N' );
INSERT INTO s_addr_attribute_type_rltshp (s_address_type, s_attribute_type, order_no, prompt, min_create_user_type, min_display_user_type, compulsory_for_user_type, closed_ind)
VALUES ( 'SNAIL', 'COUNTRY', '6', NULL, NULL, NULL, NULL, 'N' );
INSERT INTO s_addr_attribute_type_rltshp (s_address_type, s_attribute_type, order_no, prompt, min_create_user_type, min_display_user_type, compulsory_for_user_type, closed_ind)
VALUES ( 'SNAIL', 'PHONE_NO', '10', 'Home Phone', NULL, NULL, NULL, 'N' );
INSERT INTO s_addr_attribute_type_rltshp (s_address_type, s_attribute_type, order_no, prompt, min_create_user_type, min_display_user_type, compulsory_for_user_type, closed_ind)
VALUES ( 'SNAIL', 'PHONE_NO', '11', 'Work Phone', NULL, NULL, '*', 'N' );
INSERT INTO s_addr_attribute_type_rltshp (s_address_type, s_attribute_type, order_no, prompt, min_create_user_type, min_display_user_type, compulsory_for_user_type, closed_ind)
VALUES ( 'SNAIL', 'PHONE_NO', '12', 'Mobile Phone', NULL, NULL, '*', 'N' );

#
# System Item Attribute type relationship
#
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'S_ITEM_ID', '0', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'S_TITLE', '1', NULL, 'Y');
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'IMAGEURL', '2', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'MOVIE_PLOT', '10', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'DIRECTOR', '20', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'ACTORS', '30', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'MOVIEGENRE', '40', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'YEAR', '50', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'RUN_TIME', '60', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'AUDIO_LANG', '70', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'SUBTITLES', '80', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'AGE_RATING', '90', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'DVD_REGION', '100', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'RATIO', '110', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'ANAMORPHIC', '115', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'VID_FORMAT', '120', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'DVD_EXTRAS', '130', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'S_DURATION', '200', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'S_STATUS', '254', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DVD', 'S_STATCMNT', '255', NULL, NULL);

INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'S_ITEM_ID', '0', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'S_TITLE', '1', NULL, 'Y');
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'IMAGEURL', '2', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'MOVIE_PLOT', '10', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'DIRECTOR', '20', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'ACTORS', '30', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'MOVIEGENRE', '40', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'YEAR', '50', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'RUN_TIME', '60', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'AUDIO_LANG', '70', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'SUBTITLES', '80', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'AGE_RATING', '90', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'RATIO', '100', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'VID_FORMAT', '110', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'COMMENTS', '120', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'S_DURATION', '200', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'S_STATUS', '254', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'LD', 'S_STATCMNT', '255', NULL, NULL);

INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'S_ITEM_ID', '0', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'S_TITLE', '1', NULL, 'Y');
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'IMAGEURL', '2', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'MOVIE_PLOT', '10', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'DIRECTOR', '20', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'ACTORS', '30', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'MOVIEGENRE', '40', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'YEAR', '50', NULL,NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'RUN_TIME', '60', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'SUBTITLES', '70', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'AGE_RATING', '80', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'VID_FORMAT', '90', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'VIDQUALITY', '100', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'COMMENTS', '110', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'S_DURATION', '200', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'S_STATUS', '254', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VCD', 'S_STATCMNT', '255', NULL, NULL);

INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'S_ITEM_ID', '0', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'S_TITLE', '1', NULL, 'Y');
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'IMAGEURL', '2', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'MOVIE_PLOT', '10', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'DIRECTOR', '20', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'ACTORS', '30', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'MOVIEGENRE', '40', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'YEAR', '50', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'RUN_TIME', '60', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'SUBTITLES', '70', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'AGE_RATING', '80', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'VID_FORMAT', '90', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'VHS_TYPE', '100', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'VIDQUALITY', '110', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'COMMENTS', '120', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'S_DURATION', '200', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'S_STATUS', '254', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'VHS', 'S_STATCMNT', '255', NULL, NULL);

INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'S_ITEM_ID', '0', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'S_TITLE', '1', 'CD Title', 'Y');
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'IMAGEURL', '2', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'ARTIST', '10', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'MUSICGENRE', '20', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTIME', '30', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'YEAR', '40', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '50', 'Track 1', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '51', 'Track 2', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '52', 'Track 3', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '53', 'Track 4', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '54', 'Track 5', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '55', 'Track 6', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '56', 'Track 7', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '57', 'Track 8', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '58', 'Track 9', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '59', 'Track 10', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '60', 'Track 11', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '61', 'Track 12', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '62', 'Track 13', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '63', 'Track 14', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '64', 'Track 15', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '65', 'Track 16', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '66', 'Track 17', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '67', 'Track 18', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '68', 'Track 19', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'CDTRACK', '69', 'Track 20', NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'S_DURATION', '200', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'S_STATUS', '254', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'CD', 'S_STATCMNT', '255', NULL, NULL);

INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'S_ITEM_ID', '0', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'S_TITLE', '1', NULL, 'Y');
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'IMAGEURL', '2', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'MOVIE_PLOT', '20', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'DIRECTOR', '30', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'ACTORS', '40', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'MOVIEGENRE', '50', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'YEAR', '60', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'RUN_TIME', '70', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'AUDIO_LANG', '80', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'SUBTITLES', '90', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'AGE_RATING', '100', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'RATIO', '110', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'DIVXVIDVER', '120', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'DIVXAUDVER', '130', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'DIVXQUALIT', '140', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'DVD_FORMAT', '150', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'DVD_EXTRAS', '160', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'S_DURATION', '200', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'S_STATUS', '254', NULL, NULL);
INSERT INTO s_item_attribute_type (s_item_type, s_attribute_type, order_no, prompt, compulsory_ind) VALUES ( 'DIVX', 'S_STATCMNT', '255', NULL, NULL);

#
# System Attribute Type Lookup
#
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DVD_REGION', NULL, '1', 'United States & Canada', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DVD_REGION', NULL, '2', 'Europe, Near East, South Africa & Japan', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DVD_REGION', NULL, '3', 'South East Asia', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DVD_REGION', NULL, '4', 'Australasia, Middle & South America', NULL, 'Y');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DVD_REGION', NULL, '5', 'Africa, Asia, Eastern Europe', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DVD_REGION', NULL, '6', 'People\'s Republic of China', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DVD_REGION', NULL, '0', 'No Region Encoding', NULL, NULL);

INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'RATIO', NULL, '1.33', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'RATIO', NULL, '1.78', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'RATIO', NULL, '1.85', NULL, NULL, 'Y');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'RATIO', NULL, '2.35', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'RATIO', NULL, '2.78', NULL, NULL, NULL);

INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'VID_FORMAT', NULL, 'NTSC', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'VID_FORMAT', NULL, 'PAL', NULL, NULL, 'Y');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'VID_FORMAT', NULL, 'SECAM', NULL, NULL, NULL);

INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'VHS_TYPE', NULL, 'PIRATED', 'Pirated/Copied', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'VHS_TYPE', NULL, 'TV', 'TV Recording', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'VHS_TYPE', NULL, 'PRERECORD', 'Pre-Recorded', NULL, 'Y');

INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'VIDQUALITY', NULL, '1', 'Excellent', NULL, 'Y');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'VIDQUALITY', NULL, '2', 'Good', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'VIDQUALITY', NULL, '3', 'Fair', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'VIDQUALITY', NULL, '4', 'Poor', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'VIDQUALITY', NULL, '5', 'Awful', NULL, NULL);

INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AGE_RATING', '0', 'G', 'General Exhibition', 'G.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AGE_RATING', '1', 'PG', 'Parental Guidance Recommended', 'PG.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AGE_RATING', '2', 'M', 'Mature Audience Recommended', 'M.gif', 'Y');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AGE_RATING', '3', 'MA', 'Mature Audience', 'MA.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AGE_RATING', '4', 'R', 'Restricted 18+', 'R.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AGE_RATING', '5', 'X', 'Adult Videos', 'X.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AGE_RATING', '6', 'NR', 'Not Rated Material', 'NR.gif', NULL);

INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Action', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Adventure', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Adult', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Animation', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Biblical', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Comedy', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Documentary', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Drama', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Horror', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Fantasy', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Music', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Musical', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Mystery', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Romance', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'ScienceFiction', 'Science Fiction', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Suspense', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Thriller', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'War', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Western', NULL, NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MOVIEGENRE', NULL, 'Other', NULL, NULL, NULL);

INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'ENGLISH_6.1_DTS_ES', 'English (6.1 DTS ES)', 'dts.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'ENGLISH_6.1_EX', 'English(6.1 EX)', 'dolby.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'ENGLISH_6.1', 'English(6.1)', 'dolby.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'ENGLISH_5.1', 'English(5.1)', 'dolby.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'ENGLISH_DTS', 'English(DTS)', 'dts.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'ENGLISH_THX', 'English(THX)', 'thx.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'ENGLISH_5.0', 'English(5.0)', 'dolby.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'ENGLISH_SR', 'English(Surround)', 'english.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'ENGLISH_2.0', 'English(2.0)', 'english.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'ENGLISH', 'English', 'english.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'GERMAN', 'German', 'german.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'FRENCH', 'French', 'french.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'SPANISH', 'Spanish', 'spanish.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'ITALIAN', 'Italian', 'italian.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'AUDIO_LANG', NULL, 'DIR_COMMENT', 'Director\'s Commentary', 'director.gif', NULL);

INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'SUBTITLES', NULL, 'ENGLISH', 'English', 'english.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'SUBTITLES', NULL, 'FRENCH', 'French', 'french.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'SUBTITLES', NULL, 'GERMAN', 'German', 'german.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'SUBTITLES', NULL, 'SPANISH', 'Spanish', 'spanish.gif', NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'SUBTITLES', NULL, 'ITALIAN', 'Italian', 'italian.gif', NULL);

# These actually match with the CDDB defined genres.
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MUSICGENRE', NULL, 'blues', 'Blues', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MUSICGENRE', NULL, 'classical', 'Classical', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MUSICGENRE', NULL, 'country', 'Country', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MUSICGENRE', NULL, 'folk', 'Folk', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MUSICGENRE', NULL, 'jazz', 'Jazz', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MUSICGENRE', NULL, 'newage', 'New Age', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MUSICGENRE', NULL, 'reggae', 'Reggae', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MUSICGENRE', NULL, 'rock', 'Rock', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MUSICGENRE', NULL, 'pop', 'Popular', NULL, 'Y');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MUSICGENRE', NULL, 'soundtrack', 'Soundtrack', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'MUSICGENRE', NULL, 'misc', 'Miscellaneous', NULL, NULL);

# DIVX Audio codec
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DIVXAUDVER', NULL, 'mp3', 'Mp3', NULL, 'Y');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DIVXAUDVER', NULL, 'mp2', 'Mp2', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DIVXAUDVER', NULL, 'ogg', 'Ogg', NULL, NULL);

# DIVX Video codec
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DIVXVIDVER', NULL, 'divx3', 'Divx3 3.11 Alpha', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DIVXVIDVER', NULL, 'divx4', 'Divx4', NULL, 'Y');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DIVXVIDVER', NULL, 'divx5', 'Divx5', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DIVXVIDVER', NULL, 'xvid', 'Xvid', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DIVXVIDVER', NULL, 'angelpotion', 'Angel Potion Codec', NULL, NULL);

# DIVX Quality
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DIVXQUALIT', NULL, 'poor', 'Poor', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DIVXQUALIT', NULL, 'medium', 'Medium', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'DIVXQUALIT', NULL, 'Good', 'Good', NULL, NULL);

#
# This attribute is reserved for use in item_review.  Please do not use it for your own s_item_attribute_type structures.
#
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'S_RATING', NULL, '1', 'Disgraceful!', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'S_RATING', NULL, '2', 'Terrible!', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'S_RATING', NULL, '3', 'Decent!', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'S_RATING', NULL, '4', 'Great!', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'S_RATING', NULL, '5', 'Sensational!', NULL, NULL);

#
# Duration support
#
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ('S_DURATION', '0', '', 'Undefined', NULL, 'Y');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ('S_DURATION', '1', '1', 'One Day', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ('S_DURATION', '2', '3', 'Three Days', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ('S_DURATION', '3', '7', 'One Week', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ('S_DURATION', '4', '14', 'Two Weeks', NULL, NULL);
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ('S_DURATION', '5', '28', 'One Month', NULL, NULL);

#
# Country lookups
#
# cleanup
DELETE FROM s_attribute_type_lookup WHERE s_attribute_type IN('COUNTRY');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AF', 'AFGHANISTAN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AL', 'ALBANIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'DZ', 'ALGERIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AS', 'AMERICAN SAMOA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AD', 'ANDORRA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AO', 'ANGOLA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AI', 'ANGUILLA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AQ', 'ANTARCTICA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AG', 'ANTIGUA AND BARBUDA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AZ', 'AZERBAIJAN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AR', 'ARGENTINA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AM', 'ARMENIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AW', 'ARUBA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AU', 'AUSTRALIA', '', 'Y');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AT', 'AUSTRIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BS', 'BAHAMAS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BH', 'BAHRAIN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BD', 'BANGLADESH', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BB', 'BARBADOS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BY', 'BELARUS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BE', 'BELGIUM', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BZ', 'BELIZE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BJ', 'BENIN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BM', 'BERMUDA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BT', 'BHUTAN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BO', 'BOLIVIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BA', 'BOSNIA AND HERZEGOWINA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BW', 'BOTSWANA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BV', 'BOUVET ISLAND', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BR', 'BRAZIL', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'IO', 'BRITISH INDIAN OCEAN TERRITORY', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BN', 'BRUNEI DARUSSALAM', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BG', 'BULGARIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BF', 'BURKINA FASO', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'BI', 'BURUNDI', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CA', 'CANADA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'KH', 'CAMBODIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CM', 'CAMEROON', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CV', 'CAPE VERDE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CF', 'CENTRAL AFRICAN REPUBLIC', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TD', 'CHAD', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CL', 'CHILE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CN', 'CHINA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CX', 'CHRISTMAS ISLAND', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CC', 'COCOS (KEELING) ISLANDS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CO', 'COLOMBIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'KM', 'COMOROS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CG', 'CONGO', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CD', 'CONGO, THE DEMOCRATIC REPUBLIC OF THE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CK', 'COOK ISLANDS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CR', 'COSTA RICA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CI', 'COTE D\'IVOIRE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'HR', 'CROATIA (localname:Hrvatska)', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CU', 'CUBA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CY', 'CYPRUS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CZ', 'CZECH REPUBLIC', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'DE', 'GERMANY', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'DK', 'DENMARK', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'DJ', 'DJIBOUTI', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'DM', 'DOMINICA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'DO', 'DOMINICAN REPUBLIC', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TP', 'EAST TIMOR', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'EC', 'ECUADOR', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'EG', 'EGYPT', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SV', 'ELSALVADOR', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GQ', 'EQUATORIAL GUINEA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'ER', 'ERITREA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'EE', 'ESTONIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'ET', 'ETHIOPIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'FK', 'FALKLAND ISLANDS (MALVINAS)', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'FO', 'FAROE ISLANDS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'FJ', 'FIJI', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'FI', 'FINLAND', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'FR', 'FRANCE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'FX', 'FRANCE, METROPOLITAN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GF', 'FRENCH GUIANA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'PF', 'FRENCH POLYNESIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TF', 'FRENCH SOUTHERN TERRITORIES', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GM', 'GAMBIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GA', 'GABON', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GE', 'GEORGIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GH', 'GHANA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GI', 'GIBRALTAR', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GR', 'GREECE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GL', 'GREENLAND', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GD', 'GRENADA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GP', 'GUADELOUPE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GU', 'GUAM', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GT', 'GUATEMALA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GN', 'GUINEA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GW', 'GUINEA-BISSAU', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GY', 'GUYANA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'HT', 'HAITI', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'HM', 'HEARD AND MCDONALD ISLANDS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'HN', 'HONDURAS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'HK', 'HONGKONG', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'HU', 'HUNGARY', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'IS', 'ICELAND', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'ID', 'INDONESIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'IL', 'ISRAEL', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'IN', 'INDIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'IR', 'IRAN (ISLAMIC REPUBLIC OF)', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'IQ', 'IRAQ', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'IE', 'IRELAND', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'IT', 'ITALY', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'JM', 'JAMAICA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'JP', 'JAPAN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'JO', 'JORDAN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'KZ', 'KAZAKHSTAN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'KY', 'CAYMAN ISLANDS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'KE', 'KENYA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'KI', 'KIRIBATI', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'KP', 'KOREA, DEMOCRATIC PEOPLE\'S REPUBLIC OF', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'KR', 'KOREA, REPUBLIC OF', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'KW', 'KUWAIT', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'KG', 'KYRGYZSTAN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'LV', 'LATVIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'LB', 'LEBANON', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'LS', 'LESOTHO', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'LR', 'LIBERIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'LY', 'LIBYAN ARAB JAMAHIRIYA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'LI', 'LIECHTENSTEIN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'LT', 'LITHUANIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'LA', 'LAO PEOPLE\'S DEMOCRATIC REPUBLIC', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'LU', 'LUXEMBOURG', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MK', 'MACEDONIA, THE FORMER YUGOSLAV REPUBLIC OF', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MG', 'MADAGASCAR', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MW', 'MALAWI', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MY', 'MALAYSIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MV', 'MALDIVES', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'ML', 'MALI', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MH', 'MARSHALL ISLANDS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MQ', 'MARTINIQUE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MD', 'MOLDOVA, REPUBLIC OF', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MP', 'NORTHERN MARIANA ISLANDS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MA', 'MOROCCO', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MR', 'MAURITANIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MU', 'MAURITIUS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'YT', 'MAYOTTE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MX', 'MEXICO', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'FM', 'MICRONESIA, FEDERATED STATES OF', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MN', 'MONGOLIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MO', 'MACAU', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MS', 'MONTSERRAT', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MT', 'MALTA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MC', 'MONACO', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MZ', 'MOZAMBIQUE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'MM', 'MYANMAR', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'NR', 'NAURU', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'NE', 'NIGER', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'NP', 'NEPAL', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'NL', 'NETHERLANDS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AN', 'NETHERLANDS ANTILLES', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'NZ', 'NEW ZEALAND', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'NF', 'NORFOLK ISLAND', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'NI', 'NICARAGUA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'NG', 'NIGERIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'NU', 'NIUE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'NC', 'NEW CALEDONIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'NO', 'NORWAY', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'NA', 'NAMIBIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'OM', 'OMAN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'PK', 'PAKISTAN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'PW', 'PALAU', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'PS', 'PALESTINIAN TERRITORY, OCCUPIED', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'PG', 'PAPUA NEW GUINEA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'PY', 'PARAGUAY', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'PA', 'PANAMA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'PH', 'PHILIPPINES', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'PN', 'PITCAIRN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'PL', 'POLAND', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'PT', 'PORTUGAL', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'PE', 'PERU', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'PR', 'PUERTO RICO', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'QA', 'QATAR', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'RE', 'REUNION', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'RO', 'ROMANIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'RU', 'RUSSIAN FEDERATION', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'RW', 'RWANDA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'KN', 'SAINT KITTS AND NEVIS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'LC', 'SAINT LUCIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'VC', 'SAINT VINCENT AND THE GRENADINES', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'WS', 'SAMOA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SM', 'SANMARINO', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'ST', 'SAO TOME AND PRINCIPE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SK', 'SLOVAKIA (Slovak Republic)', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SA', 'SAUDI ARABIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SN', 'SENEGAL', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SL', 'SIERRA LEONE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SG', 'SINGAPORE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SI', 'SLOVENIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SB', 'SOLOMON ISLANDS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SO', 'SOMALIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SC', 'SEYCHELLES', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'ZA', 'SOUTH AFRICA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GS', 'SOUTH GEORGIA AND THE SOUTHS AND WICH ISLANDS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SD', 'SUDAN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'ES', 'SPAIN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'LK', 'SRI LANKA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SH', 'ST.HELENA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'PM', 'ST.PIERRE AND MIQUELON', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SR', 'SURINAME', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SJ', 'SVALBARD AND JANMAYEN ISLANDS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SZ', 'SWAZILAND', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SE', 'SWEDEN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'CH', 'SWITZERLAND', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'SY', 'SYRIAN ARAB REPUBLIC', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TW', 'TAIWAN, PROVINCE OF CHINA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TJ', 'TAJIKISTAN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TZ', 'TANZANIA, UNITED REPUBLIC OF', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TN', 'TUNISIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TH', 'THAILAND', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TG', 'TOGO', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TK', 'TOKELAU', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TO', 'TONGA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TT', 'TRINIDAD AND TOBAGO', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TR', 'TURKEY', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TM', 'TURKMENISTAN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TC', 'TURKS AND CAICOS ISLANDS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'TV', 'TUVALU', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'UG', 'UGANDA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'UA', 'UKRAINE', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'AE', 'UNITED ARAB EMIRATES', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'GB', 'UNITED KINGDOM', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'US', 'UNITED STATES', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'UM', 'UNITED STATES MINOR OUTLYING ISLANDS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'UY', 'URUGUAY', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'UZ', 'UZBEKISTAN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'VU', 'VANUATU', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'VE', 'VENEZUELA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'VN', 'VIETNAM', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'VA', 'HOLY SEE (VATICAN CITY STATE)', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'VG', 'VIRGIN ISLANDS (BRITISH)', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'VI', 'VIRGIN ISLANDS (U.S.)', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'WF', 'WALLIS AND FUTUNA ISLANDS', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'EH', 'WESTERN SAHARA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'YE', 'YEMEN', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'YU', 'YUGOSLAVIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'ZM', 'ZAMBIA', '', '');
INSERT INTO s_attribute_type_lookup (s_attribute_type, order_no, value, display, img, checked_ind) VALUES ( 'COUNTRY', NULL, 'ZW', 'ZIMBABWE', '', '');


# --------------------------------------------------------
# Test user installation
# --------------------------------------------------------
INSERT INTO user (user_id, fullname, pwd, type) VALUES ( 'test', 'Test User', '098f6bcd4621d373cade4e832627b4f6', 'A');
#
# Insert User Email address
# 
INSERT INTO user_address (sequence_number, user_id, s_address_type, start_dt, end_dt)
VALUES (1, 'test', 'EMAIL', now(), NULL);

INSERT INTO user_address_attribute (ua_sequence_number, s_attribute_type, order_no, attribute_val, lookup_attribute_val)
VALUES (1, 'EMAIL_ADDR', 1, 'jasonpell@hotmail.com', '');
