'encoding UTF-8  Do not remove or change this line!
'*************************************************************************
'*
'*  OpenOffice.org - a multi-platform office productivity suite
'*
'*  $RCSfile: treelist_tools.inc,v $
'*
'*  $Revision: 1.11 $
'*
'*  last change: $Author: jsk $ $Date: 2006/01/05 13:07:33 $
'*
'*  The Contents of this file are made available subject to
'*  the terms of GNU Lesser General Public License Version 2.1.
'*
'*
'*    GNU Lesser General Public License Version 2.1
'*    =============================================
'*    Copyright 2005 by Sun Microsystems, Inc.
'*    901 San Antonio Road, Palo Alto, CA 94303, USA
'*
'*    This library is free software; you can redistribute it and/or
'*    modify it under the terms of the GNU Lesser General Public
'*    License version 2.1, as published by the Free Software Foundation.
'*
'*    This library is distributed in the hope that it will be useful,
'*    but WITHOUT ANY WARRANTY; without even the implied warranty of
'*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
'*    Lesser General Public License for more details.
'*
'*    You should have received a copy of the GNU Lesser General Public
'*    License along with this library; if not, write to the Free Software
'*    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
'*    MA  02111-1307  USA
'*
'/************************************************************************
'*
'* owner : joerg.skottke@sun.com
'*
'* short description :
'*
'*******************************************************************************
' **
' #1 hGetNodeCount        ' returns the number of visible nodes in a treelist
' #1 hSelectTopNode       ' jumps to the upper root-node in a treelist
' #1 hSelectNextNode      ' just move selection down one item
' #1 hGetNodeName         ' gets the name of a specified node
' #1 hExpandNode          ' expands a specified node
' #1 hExpandAllNodes      ' expands all nodes in a treelist
' #1 hGetVisibleNodeNames ' feeds the names of all nodes into an array
' #1 hSelectNode          ' selects a node by index
' #1 hSelectNodeByName    ' selects a node by a given name
' #1 hSelectTheLastNode   ' selects the absolute last node in a treelist
' #1 hVerifyNodeName      ' compare nodename against a string
' #1 hWaitForTreeList     ' Wait for the treelist to get populated
' #1 hGetListItems        ' Store all items from a list into an array
' **
'\******************************************************************************

function hGetNodeCount( oControl as object ) as integer

   '///<h3>Retrieve the number of visible (open) nodes from a treelist</h3>

   hGetNodeCount() = oControl.getItemCount()
   
end function

'*******************************************************************************

function hSelectTopNode( oControl as object ) as boolean

    '///<h3>Move selection to the first node in a treelist</h3>
    '///<ul>
    '///+<li>Type the &quot;Home&quot;-key in a treelist, to select index 1</li>
    '///</ul>

   oControl.typeKeys( "<HOME>" )
   
   if ( oControl.getSelIndex() = 1 ) then
   
      hSelectTopNode() = true
      
   else 
   
      hSelectTopNode() = false
      
   endif
   
end function

'*******************************************************************************

function hSelectNextNode( oControl as object ) as integer

    '///<h3>Move one node down in a treelist</h3>
    '///<ul>
    '///+<li>Returns new position or 0 on error</li>
    '///</ul>
    
    dim iPosBefore as integer
    dim iPosAfter as integer
    
    iPosBefore = oControl.getSelIndex()
    
    oControl.typeKeys( "<DOWN>" )
    
    iPosAfter = oControl.getSelIndex()
    
    if ( iPosAfter = iPosBefore ) then
        hSelectNextNode() = 0
    endif
    
    if ( iPosAfter = ( iPosBefore + 1 ) ) then
        hSelectnextNode() = iPosAfter
    endif
    
end function

'*******************************************************************************

function hGetNodeName( oControl as object , iNode as integer ) as string

    '///<h3>Retrieve the name of a node in a treelist specified by index</h3>
    '///<ul>
    '///+<li>Returns the name of the node or empty string on error</li>
    '///</ul>
    const CFN = "hGetNodeName::"
    dim iItemCount as integer
    
    iItemCount = hGetNodeCount( oControl )
    if ( iNode > iItemCount ) then
        warnlog( CFN & "Selected node out of range, aborting" )
        hGetNodeName() = ""
    else 
        oControl.select( iNode )
        hGetNodeName() = oControl.getSelText()
    endif
   
end function

'*******************************************************************************

function hExpandNode( oControl as object, iNode as integer ) as integer

    '///<h3>Expand a node in a treelist specified by index</h3>
    '///<ul>
    '///+<li>Returns new nodecount or 0 on error</li>
    '///</ul>
   
    const CFN = "hExpandNode::"
   
    dim iOldNodeCount as integer
    dim iNewNodeCOunt as integer
   
    iOldNodeCount = hGetNodeCount( oControl )
    if ( iNode > iOldNodeCount ) then
        warnlog( CFN & "Selected node out of bounds, aborting" )
        hExpandNode() = 0
        exit function
    endif

    ' 0 = currently selected node, else open node 'node' (absolute)
    if ( iNode > 0 ) then
        hSelectNode( oControl , iNode )
    endif

    ' DEBUG: printlog( CFN & "Expanding: " & oControl.getSelText() )
    oControl.typekeys( "<RIGHT>" )
    iNewNodeCount = hGetNodeCount( oControl )
   
    if ( iNewNodeCount > iOldNodeCount ) then
        hExpandNode() = iNewNodeCount 
    else
        hExpandNode() = 0
    endif
   
end function

'*******************************************************************************

function hExpandAllNodes( oControl as object ) as integer

    '///<h3>Expand all nodes of treelist</h3>
    '///<ul>
    '///+<li>Run through all items in the treelist, expand every node</li>
    '///+<li>Returns the number of all nodes in the treelist</li>
    '///</ul>

   dim iNode as integer
   dim iNodeCount as integer
   
   iNode = 0
   iNodeCount = 1000

   hSelectTopNode( oControl )
   
   do while ( iNode < iNodeCount )
   
      iNode = iNode + 1
      hExpandNode( oControl , iNode )
      iNodeCount = hGetNodeCount( oControl )
      
   loop
       
   hExpandAllNodes() =  iNodeCount
      
end function

'*******************************************************************************

function hGetVisibleNodeNames( oControl as object , _
                               lsList() as string ) as integer
                               
    '///<h3>Retrieve the names of all nodes in a treelist</h3>
    '///<ul>
    '///+<li>Expand all nodes of a treelist</li>
    '///+<li>Store all node names into an array</li>
    '///+<li>Return the number of nodes read (listcount), 0 on error</li>
    '///</ul>
            
   ' Get the list of all visible (expanded) nodes and store it
   ' in lsList(). if _id > ubound lsList() an error is given and lsList remains
   ' empty, thus returning "0"
                               
   dim iNodeCount as integer
   dim iThisNode as integer

   iNodeCount = hGetNodeCount( oControl )
   
    if ( iNodeCount > ubound( lsList() ) ) then
        warnlog( "Array to small to hold: " & iNodeCount & " items" )
    else
        hSelectTopNode( oControl )
   
        for iThisNode = 1 to iNodeCount

            listappend( lsList() , hGetNodeName( oControl , iThisNode )

        next iThisNode
        
    endif
   
    hGetVisibleNodeNames() = listcount( lsList() )

end function  

'*******************************************************************************

function hSelectNode( oControl as object , _id as integer ) as string

    '///<h3>Select a node in a treelist by index</h3>
    '///<ul>
    '///+<li>Return the name of the selected node</li>
    '///</ul>

    if ( _id <= hGetNodeCount( oControl ) ) then
        oControl.select( _id )
        hSelectNode() = hGetNodeName( oControl , _id )
    else
        warnlog( "Selected ID of node is too large" )
        hSelectNode() = ""
    endif

end function

'*******************************************************************************

function hSelectNodeByName( oControl as object , _name as string ) as integer
                            
    '///<h3>Select a node by name in treelist (first occurrence)</h3>
    '///<ul>
    '///+<li>Try to find a node by name - directly</li>
    '///+<li>If the node is not visible, expand all and search the tree</li>
    '///+<li>Returns index of requested node or 0 on failure</li>
    '///</ul>
    
    const CFN = "hSelectNodeByName::"

    dim iThisNode as integer
    dim iCurrentNode as integer
    dim iNodeCount as integer    

    dim cNodeName as string

    iThisNode = 0
    
    ' First we try to jump to the node directly, if this fails we use the 
    ' slower but safer method (expand all nodes and step through)
    try
    
        oControl.select( _name )
        iThisNode = oControl.getSelIndex()
        hSelectNodeByName() = iThisNode
        
    catch
    
        printlog( CFN & "Node not visible: Using search method" )
        iNodeCount = hExpandAllNodes( oControl )
    
        for iCurrentNode = 1 to iNodeCount
        
            oControl.select( iCurrentNode )
            cNodeName = oControl.getSelText()
            
            if ( cNodeName = _name ) then
                iThisNode = iCurrentNode
                
                exit for
            endif
    
        next iCurrentNode
    
    endcatch
    
    if ( iThisNode = 0 ) then
        printlog( CFN & "Node not found." )
    else
        printlog( CFN & "Node selected at pos: " & iThisNode )
    endif
    
    hSelectNodeByName() = iThisNode
   
end function

'*******************************************************************************

function hSelectTheLastNode( oControl as object ) as integer

    '///<h3>Select the (absolute) last node in a treelist</h3>
    '///<ul>
    '///+<li>Expand all nodes</li>
    '///+<li>Go to the last node, select it</li>
    '///+<li>Return the number of the last node in the treelist</li>
    '///</ul>
    
    const CFN = "hSelectTheLastNode::"
    dim iPos as integer
    dim bNodeExpanded as boolean
        bNodeExpanded = true
    dim cNodeName as string
    
    while( bNodeExpanded )
                         
        iPos = hGetNodeCount( oControl )
        cNodeName = hSelectNode( oControl , iPos )
        printlog( CFN & cNodeName )
        bNodeExpanded = hExpandNode( oControl , 0 )
    
    wend 

    hSelectTheLastNode() = iPos
    
end function

'*******************************************************************************

function hVerifyNodeName( oControl as object , cName as string ) as boolean

    '///<h3>Compare the name of the current node from a treelist to a reference string</h3>
    '///<ul>
    '///+<li>Returns true on success, false on failure</li>
    '///</ul>

    if ( oControl.getSelText() = cName ) then
        hVerifyNodeName() = true
    else
        hVerifyNodeName() = false
    endif
    
end function

'*******************************************************************************

function hWaitForTreelist( oTreeList as object, _
                           cContext as string, _
                           iItemCount as integer ) as boolean
                           
    '///<h3>Wait for a treelist to get populated (java related delay)</h3>
    '///<b>IMPORTANT: Do not use unless absolutely necessary</b>
    '///+<p>Retrieve the number of items from the treelist until a specified
    '///+ number has been reached.</p>

    const CFN = "hWaitForTreelist::"
    dim iTry as integer
    dim iObjects as integer
    dim brc as boolean
    
    brc = false
    
    kontext cContext
    for iTry = 1 to 10
    
        iObjects = hGetNodeCount( oTreeList )
        
        if ( iObjects = iItemCount ) then
            printlog( CFN & "Treelist has been populated: " & iObjects )
            brc = true
            exit for
        else
            oTreeList.select( 1 )
            wait( 500 )
        endif
        
    next iTry
    
    if ( not brc ) then
        warnlog( CFN & "It takes too long to populate the treelist" )
    endif
    
    hWaitForTreelist() = brc
        
end function

'******************************************************************************

function hGetListItems( oControl as object, aList() as string ) as integer

    '///<h3>Retrieve the names of all nodes from a plain (linear) list</h3>
    '///<ul>
    '///+<li>Cycle through a list, append all entries to an array</li>
    '///+<li>Returns number of items or 0 on error</li>
    '///</ul>
    
    const CFN = "hGetListItems::"
    
    dim iItemCount as integer
    dim iCurrentItem as integer
    dim cCurrentItem as string    
    
    iItemCount = oControl.getItemCount()
    if ( iItemCount > ubound( aList() ) ) then
        printlog( CFN & "Array too small, needed: " & iItemCount )
        hGetListItems() = 0
        exit function
    endif
    
    for iCurrentItem = 1 to iItemCount
    
        oControl.select( iCurrentItem )
        cCurrentItem = oControl.getSelText()        
        hListAppend( cCurrentItem, aList() )
        
    next iCurrentItem
    
    hGetListItems() = iItemCount
    
end function
