/* Sinusoidal.java
 * Copyright (C) 1996 by William Giel
 *
 * E-mail: rvdi@usa.nai.net
 * WWW: http://www.nai.net/~rvdi/home.htm
 *
 ***************************************************************************
 * Abstract
 * --------
 * Sinusoidal projection methods (forward and inverse), extends MapProjection
 * class
 *
 * Formulas and code for general sinusoidal projection on ellipsoid adapted 
 * from GN_SINU.C and MLFN.C contained in source for PROJ, a general
 * cartographic projection program obtained from United States Geological
 * Survey (USGS, ftp://kai.er.usgs.gov/pub/PROJ.4/)
 
 ***************************************************************************
 * Permission to use, copy, modify, and distribute this software and its
 * documentation without fee for NON-COMMERCIAL purposes is hereby granted.
 * 
 * THE AUTHOR MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY
 * OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. THE AUTHOR SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 ***************************************************************************/


import java.awt.*;
import MapProjection;

public class Sinusoidal extends MapProjection
{
    static final double C00     =   1.0;
    static final double C02     =   .25;
    static final double C04     =   .046875;
    static final double C06     =   .01953125;
    static final double C08     =   .01068115234375;
    static final double C22     =   .75;
    static final double C44     =   .46875;
    static final double C46     =   .01302083333333333333;
    static final double C48     =   .00712076822916666666;
    static final double C66     =   .36458333333333333333;
    static final double C68     =   .00569661458333333333;
    static final double C88     =   .3076171875;
    static final double EPS     =   1.0e-11;
    static final double EPS10   =   1.0e-10;
    static final int MAX_ITER   =   10;
    static final double HALFPI  =   Math.PI/2;
    static final int EN_SIZE    =   5;

    double en[]=new double[EN_SIZE];
    double es;


    
    Sinusoidal(mappingData data)
    {
        
        super(data);
        
        double t;
        es=data.eccentricity*data.eccentricity;
		en[0] = C00 - es * (C02 + es * (C04 + es * (C06 + es * C08)));
		en[1] = es * (C22 - es * (C04 + es * (C06 + es * C08)));
		en[2] = (t = es * es) * (C44 - es * (C46 + es * C48));
		en[3] = (t *= es) * (C66 - es * C68);
		en[4] = t * es * C88;        
               
    }

	double pj_mlfn(double phi, double sphi, double cphi)
	{
	    cphi *= sphi;
	    sphi *= sphi;
	    return(en[0] * phi - cphi * (en[1] + sphi*(en[2]
		    + sphi*(en[3] + sphi*en[4]))));
    }

	double pj_inv_mlfn(double arg)
	{
	    int i;
	    double  s,
	            t,
	            phi,
	            k;
	            
	    k = 1.0/(1.0-es);

	    phi = arg;
	    for (i = MAX_ITER; i>0 ; --i){
	    	s = Math.sin(phi);
		    t = 1.0 - es * s * s;
		    t = (pj_mlfn(phi, s, Math.cos(phi)) - arg) * (t * Math.sqrt(t)) * k;
		    phi -= t;
		    if (Math.abs(t) < EPS)
			    return phi;
	    }
	    return phi;
    }

    public Point forward(geoPos latlong)
    {
        double x,y;
		double latitude = latlong.latitude * Math.PI/180.0;
		double longitude = (latlong.longitude-PRIME_MERIDIAN) * Math.PI/180.0;
    	double s = Math.sin(latitude);
    	double c = Math.cos(latitude);

	    y = FALSE_NORTHING - (a*pj_mlfn(latitude, s, c ))/MAP_SCALE;
	    x = FALSE_EASTING -(a*(longitude * c / Math.sqrt(1. - es * s * s)))/MAP_SCALE;

	    return new Point((int)x,(int)y);
    }

    public geoPos inverse(Point xy)
    {
 
    	double lam;

    	xy.x = (int)FALSE_EASTING - xy.x;
    	xy.y = (int)FALSE_NORTHING -xy.y;

    	double X=(double)xy.x * MAP_SCALE/a;
    	double Y=(double)xy.y * MAP_SCALE/a;

    	  	
    	double phi = pj_inv_mlfn(Y);
    	double s = Math.abs(phi);

	    if (s < HALFPI) {
		    s = Math.sin(phi);
		    lam = X * Math.sqrt(1. - es * s * s) / Math.cos(phi);
	    }
	    else lam = 0;

        return new geoPos(phi*180.0/Math.PI,lam * 180.0/Math.PI + PRIME_MERIDIAN);
    }


}

